# Struts2 Security Testing - Windows Setup Script
# This script automatically sets up your Windows environment for security testing
# Prerequisites: OWASP ZAP must be installed manually first

param(
    [switch]$SkipJava,
    [switch]$SkipWSL
)

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Struts2 Security Testing - Windows Setup" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Check if running as Administrator
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)

if (-not $isAdmin) {
    Write-Host "[!] This script requires Administrator privileges for WSL installation" -ForegroundColor Red
    Write-Host "[*] Please run PowerShell as Administrator" -ForegroundColor Yellow
    exit 1
}

Write-Host "[*] Administrator privileges confirmed" -ForegroundColor Green
Write-Host ""

# Step 1: Check/Install WSL
if (-not $SkipWSL) {
    Write-Host "========================================" -ForegroundColor Cyan
    Write-Host "Step 1: Windows Subsystem for Linux (WSL)" -ForegroundColor Cyan
    Write-Host "========================================" -ForegroundColor Cyan
    
    $wslInstalled = $false
    try {
        $wslCheck = wsl --status 2>&1
        $wslInstalled = $true
        Write-Host "[+] WSL feature is already enabled" -ForegroundColor Green
    } catch {
        Write-Host "[*] WSL not found, installing..." -ForegroundColor Yellow
    }
    
    if (-not $wslInstalled) {
        Write-Host "[*] Installing WSL... (this will require a restart)" -ForegroundColor Yellow
        wsl --install
        
        Write-Host ""
        Write-Host "[!] IMPORTANT: Computer restart required!" -ForegroundColor Red
        Write-Host "[*] After restart, run this script again" -ForegroundColor Yellow
        Write-Host "[*] Press any key to restart now, or Ctrl+C to restart later..."
        $null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
        Restart-Computer
        exit 0
    }
    
    # Check if Ubuntu distribution is installed
    Write-Host "[*] Checking for Ubuntu distribution..." -ForegroundColor Yellow
    $distroInstalled = $false
    try {
        $distros = wsl --list --quiet 2>&1
        if ($distros -match "Ubuntu") {
            $distroInstalled = $true
            Write-Host "[+] Ubuntu distribution is already installed" -ForegroundColor Green
        }
    } catch {
        # No distros found
    }
    
    if (-not $distroInstalled) {
        Write-Host "[*] Installing Ubuntu distribution..." -ForegroundColor Yellow
        Write-Host "[*] This may take a few minutes depending on your internet connection" -ForegroundColor Cyan
        wsl --install -d Ubuntu
        
        Write-Host "[+] Ubuntu installation started" -ForegroundColor Green
        Write-Host "[!] You may need to create a username and password for Ubuntu" -ForegroundColor Yellow
        Write-Host "[*] After Ubuntu setup completes, run this script again" -ForegroundColor Yellow
        Write-Host ""
        Write-Host "Press any key to continue..."
        $null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
    }
    
    Write-Host ""
}

# Step 2: Install jq in WSL
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Step 2: Install jq (JSON parser) in WSL" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan

Write-Host "[*] Checking jq installation in WSL..." -ForegroundColor Yellow
$jqInstalled = wsl bash -c "command -v jq" 2>$null

if ($jqInstalled) {
    Write-Host "[+] jq is already installed in WSL" -ForegroundColor Green
} else {
    Write-Host "[*] Installing jq in WSL..." -ForegroundColor Yellow
    wsl bash -c "sudo apt-get update && sudo apt-get install -y jq"
    Write-Host "[+] jq installed successfully" -ForegroundColor Green
}
Write-Host ""

# Step 3: Check/Install Java 17
if (-not $SkipJava) {
    Write-Host "========================================" -ForegroundColor Cyan
    Write-Host "Step 3: Java 17 (Required by OWASP ZAP)" -ForegroundColor Cyan
    Write-Host "========================================" -ForegroundColor Cyan
    
    $javaInstalled = $false
    try {
        $javaVersion = & "C:\Program Files\Eclipse Adoptium\jdk-17.0.16.8-hotspot\bin\java.exe" -version 2>&1
        if ($javaVersion -match "17\.0") {
            $javaInstalled = $true
            Write-Host "[+] Java 17 is already installed" -ForegroundColor Green
        }
    } catch {
        # Java not found
    }
    
    if (-not $javaInstalled) {
        Write-Host "[*] Installing Java 17..." -ForegroundColor Yellow
        winget install --id EclipseAdoptium.Temurin.17.JDK -e --accept-source-agreements --accept-package-agreements
        Write-Host "[+] Java 17 installed successfully" -ForegroundColor Green
    }
    Write-Host ""
}

# Step 4: Verify ZAP Installation
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Step 4: Verify OWASP ZAP Installation" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan

$zapPath = "C:\Program Files\ZAP\Zed Attack Proxy\zap.bat"
if (Test-Path $zapPath) {
    Write-Host "[+] OWASP ZAP is installed" -ForegroundColor Green
} else {
    Write-Host "[!] OWASP ZAP not found!" -ForegroundColor Red
    Write-Host "[*] Please install ZAP manually:" -ForegroundColor Yellow
    Write-Host "    1. Download from: https://www.zaproxy.org/download/" -ForegroundColor Gray
    Write-Host "    2. Install to default location" -ForegroundColor Gray
    Write-Host "    3. Run this script again" -ForegroundColor Gray
    exit 1
}
Write-Host ""

# Step 5: Create ZAP startup script
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Step 5: Create ZAP Startup Helper" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan

$startZapScript = @"
@echo off
SET JAVA_HOME=C:\Program Files\Eclipse Adoptium\jdk-17.0.16.8-hotspot
SET PATH=%JAVA_HOME%\bin;%PATH%
cd /d "C:\Program Files\ZAP\Zed Attack Proxy"
echo Starting OWASP ZAP daemon on port 8090...
echo Please wait 30-60 seconds for ZAP to fully start...
echo.
zap.bat -daemon -host 0.0.0.0 -port 8090 -config api.key=changeme123456 -config api.addrs.addr.name=.* -config api.addrs.addr.regex=true
"@

$scriptPath = Join-Path $PSScriptRoot "start-zap.bat"
$startZapScript | Out-File -FilePath $scriptPath -Encoding ASCII -Force

Write-Host "[+] Created start-zap.bat" -ForegroundColor Green
Write-Host ""

# Step 6: Get Windows Host IP
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Step 6: Network Configuration" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan

$hostIP = (Get-NetIPAddress -AddressFamily IPv4 | Where-Object { 
    $_.InterfaceAlias -like "*Ethernet*" -or $_.InterfaceAlias -like "*Wi-Fi*" 
} | Select-Object -First 1).IPAddress

if (-not $hostIP) {
    $hostIP = (Get-NetIPConfiguration | Where-Object { $_.IPv4DefaultGateway -ne $null }).IPv4Address.IPAddress
}

Write-Host "[*] Your Windows Host IP: $hostIP" -ForegroundColor Cyan
Write-Host "[*] WSL/Docker will use this IP to connect to ZAP" -ForegroundColor Cyan
Write-Host ""

# Step 7: Create helper scripts
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Step 7: Create Test Runner Scripts" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan

# Quick test script
$quickTestScript = @"
@echo off
REM Quick security test (no ZAP, ~5 minutes)
wsl bash -c "cd `$(wslpath '%cd%') && bash struts2-comprehensive-security-test.sh %1 quick"
pause
"@

$quickTestScript | Out-File -FilePath (Join-Path $PSScriptRoot "struts2vulnerability-quick-test.bat") -Encoding ASCII -Force
Write-Host "[+] Updated struts2vulnerability-quick-test.bat" -ForegroundColor Green

# Full test script
$fullTestScript = @"
@echo off
REM Full security test with ZAP (~20-60 minutes)
REM Make sure ZAP is running first: start-zap.bat

wsl bash -c "cd `$(wslpath '%cd%') && export ZAP_HOST=$hostIP && export ZAP_PORT=8090 && bash struts2-comprehensive-security-test.sh %1 full"
pause
"@

$fullTestScript | Out-File -FilePath (Join-Path $PSScriptRoot "struts2vulnerability-full-test.bat") -Encoding ASCII -Force
Write-Host "[+] Updated struts2vulnerability-full-test.bat" -ForegroundColor Green
Write-Host ""

# Summary
Write-Host "========================================" -ForegroundColor Green
Write-Host "Setup Complete!" -ForegroundColor Green
Write-Host "========================================" -ForegroundColor Green
Write-Host ""
Write-Host "Created/Updated files in security-scripts\struts2WithAuth directory:" -ForegroundColor Yellow
Write-Host "  - start-zap.bat                        (Start ZAP daemon)" -ForegroundColor Gray
Write-Host "  - struts2vulnerability-quick-test.bat  (Quick test, no ZAP)" -ForegroundColor Gray
Write-Host "  - struts2vulnerability-full-test.bat   (Full test with ZAP)" -ForegroundColor Gray
Write-Host ""
Write-Host "Usage (run from security-scripts\struts2WithAuth directory):" -ForegroundColor Yellow
Write-Host ""
Write-Host "1. Quick test (localhost app):" -ForegroundColor Cyan
Write-Host "   struts2vulnerability-quick-test.bat http://localhost:8080/myapp" -ForegroundColor White
Write-Host ""
Write-Host "2. Full test with ZAP (localhost app):" -ForegroundColor Cyan
Write-Host "   a) Double-click: start-zap.bat" -ForegroundColor White
Write-Host "   b) Wait 30-60 seconds for ZAP to start" -ForegroundColor White
Write-Host "   c) struts2vulnerability-full-test.bat http://localhost:8080/myapp" -ForegroundColor White
Write-Host ""
Write-Host "3. Docker app:" -ForegroundColor Cyan
Write-Host "   struts2vulnerability-quick-test.bat http://host.docker.internal:7070/myapp" -ForegroundColor White
Write-Host ""
Write-Host "Notes:" -ForegroundColor Yellow
Write-Host "  - Works on ANY Windows computer after this setup" -ForegroundColor Gray
Write-Host "  - Works with localhost apps or Docker containers" -ForegroundColor Gray
Write-Host "  - ZAP only needed for full mode" -ForegroundColor Gray
Write-Host "  - Your Windows Host IP: $hostIP" -ForegroundColor Gray
Write-Host ""
