#!/bin/bash

# Web Malware Detection Script
# Scans websites for malicious code, backdoors, cryptominers, and signs of compromise
# Part of Security Scanner Desktop - Reconnaissance Features
# Usage: ./web-malware-detection-test.sh [TARGET_URL] [MODE] [CLEAN_OUTPUT]

# Note: Do NOT use 'set -e' as grep returns exit code 1 when no match is found,
# which would terminate the script prematurely during normal scanning

if [[ "$1" == "--help" || "$1" == "-h" ]]; then
    echo "WEB MALWARE DETECTION SCANNER"
    echo "======================================================"
    echo "Scans websites for malicious code and signs of compromise"
    echo ""
    echo "USAGE: $0 <TARGET_URL> [MODE] [CLEAN_OUTPUT]"
    echo ""
    echo "DETECTION CATEGORIES:"
    echo "  1. Malicious JavaScript (obfuscation, cryptominers, keyloggers)"
    echo "  2. PHP Backdoors / Web Shells"
    echo "  3. SEO Spam Injection"
    echo "  4. Defacement Detection"
    echo "  5. Suspicious External Resources"
    echo "  6. Blacklisted Patterns"
    echo ""
    echo "THREAT LEVELS:"
    echo "  CLEAN (0-2 points): No significant threats"
    echo "  LOW (3-5 points): Minor concerns"
    echo "  MEDIUM (6-9 points): Suspicious patterns found"
    echo "  HIGH (10+ points): Multiple malware indicators"
    exit 0
fi

TARGET_URL="${1:-}"
MODE="${2:-full}"
CLEAN_OUTPUT="${3:-false}"

if [ "$CLEAN_OUTPUT" = "true" ]; then
    RED='' GREEN='' YELLOW='' BLUE='' PURPLE='' CYAN='' GRAY='' NC=''
else
    RED='\033[0;31m' GREEN='\033[0;32m' YELLOW='\033[1;33m' BLUE='\033[0;34m'
    PURPLE='\033[0;35m' CYAN='\033[0;36m' GRAY='\033[0;90m' NC='\033[0m'
fi

# Detection results
THREAT_SCORE=0
MAX_SCORE=20
THREAT_LEVEL="CLEAN"
FINDINGS=()

# Detection details
OBFUSCATION_PATTERNS=0
CRYPTOMINER_DETECTED=false
BACKDOORS_FOUND=false
SPAM_KEYWORDS=0
HIDDEN_LINKS=0
DEFACEMENT_DETECTED=false
SUSPICIOUS_TLDS=0
BLACKLISTED_PATTERNS=0

if [ -z "$TARGET_URL" ]; then
    echo -e "${RED}Error: Target URL is required${NC}"
    echo "Usage: $0 <TARGET_URL> [MODE] [CLEAN_OUTPUT]"
    exit 1
fi

[[ ! "$TARGET_URL" =~ ^https?:// ]] && TARGET_URL="https://$TARGET_URL"
TARGET_URL="${TARGET_URL%/}"

# Authentication support
AUTH_TOKEN="${AUTH_TOKEN:-}"
AUTH_COOKIE="${AUTH_COOKIE:-}"

auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

echo -e "${PURPLE}WEB MALWARE DETECTION SCANNER${NC}"
echo -e "${PURPLE}======================================================${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${NC}"
echo -e "${BLUE}Mode: ${MODE}${NC}"
echo -e "${BLUE}Timestamp: $(date)${NC}"
[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication${NC}"

echo ""

# Function to add finding and update score
add_finding() {
    local finding="$1"
    local points="$2"

    FINDINGS+=("$finding")
    THREAT_SCORE=$((THREAT_SCORE + points))
    echo -e "${YELLOW}  FOUND: ${finding} (+${points} points)${NC}"
}

# Fetch page content
fetch_content() {
    echo -e "\n${CYAN}Fetching target content...${NC}"

    PAGE_CONTENT=$(auth_curl -sL "$TARGET_URL" --max-time 30 -A "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36" 2>/dev/null || echo "")

    if [ -z "$PAGE_CONTENT" ]; then
        echo -e "${RED}  Failed to fetch page content${NC}"
        return 1
    fi

    local content_length=${#PAGE_CONTENT}
    echo -e "${GRAY}  Fetched ${content_length} bytes${NC}"
    return 0
}

# Category 1: Malicious JavaScript Detection
detect_malicious_javascript() {
    echo -e "\n${YELLOW}CATEGORY 1: MALICIOUS JAVASCRIPT DETECTION${NC}"

    if [ -z "$PAGE_CONTENT" ]; then
        echo -e "${GRAY}  Skipped - no content available${NC}"
        return
    fi

    # Check for obfuscation patterns
    local obfuscation_patterns=(
        "eval\("
        "unescape\("
        "atob\("
        "String\.fromCharCode"
        "document\.write\(unescape"
        "window\.atob"
        "decodeURIComponent"
        "\\\\x[0-9a-fA-F]{2}"
        "\\\\u[0-9a-fA-F]{4}"
    )

    local obf_count=0
    for pattern in "${obfuscation_patterns[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qiE "$pattern"; then
            obf_count=$((obf_count + 1))
        fi
    done

    OBFUSCATION_PATTERNS=$obf_count

    if [ $obf_count -ge 5 ]; then
        add_finding "High JavaScript obfuscation detected (${obf_count} patterns)" 3
    elif [ $obf_count -ge 3 ]; then
        add_finding "Moderate JavaScript obfuscation (${obf_count} patterns)" 2
        echo -e "${GRAY}    Note: Some obfuscation may be legitimate minification${NC}"
    elif [ $obf_count -ge 1 ]; then
        echo -e "${GRAY}  Low obfuscation level (${obf_count} patterns) - likely normal${NC}"
    fi

    # Check for cryptominer signatures
    local cryptominer_patterns=(
        "CoinHive"
        "coinhive.min.js"
        "coin-hive.com"
        "JSEcoin"
        "Client.Anonymous"
        "CryptoNoter"
        "CryptoLoot"
        "cryptoloot.pro"
        "deepMiner"
        "webminerpool"
        "minero.cc"
        "authedmine"
        "crypto-loot.com"
        "miner.start"
        "WebMR"
        "webmr.js"
    )

    for pattern in "${cryptominer_patterns[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qi "$pattern"; then
            CRYPTOMINER_DETECTED=true
            add_finding "Cryptominer detected: ${pattern}" 5
            break
        fi
    done

    # Check for keylogger patterns
    local keylogger_patterns=(
        "onkeypress.*POST"
        "onkeyup.*send"
        "onkeydown.*ajax"
        "addEventListener.*keypress.*fetch"
        "document.onkeypress"
        "keyCode.*XMLHttpRequest"
    )

    for pattern in "${keylogger_patterns[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qiE "$pattern"; then
            add_finding "Potential keylogger pattern detected" 4
            break
        fi
    done

    # Check for malicious redirects
    if echo "$PAGE_CONTENT" | grep -qiE "location\.(href|replace).*\.(tk|ml|ga|cf|gq)"; then
        add_finding "Suspicious redirect to malicious TLD" 3
    fi

    [ "$CRYPTOMINER_DETECTED" = false ] && [ $obf_count -lt 3 ] && echo -e "${GREEN}  No malicious JavaScript patterns detected${NC}"
}

# Category 2: PHP Backdoor / Web Shell Detection
detect_backdoors() {
    echo -e "\n${YELLOW}CATEGORY 2: PHP BACKDOOR / WEB SHELL DETECTION${NC}"

    # Common web shell paths to check
    local shell_paths=(
        "shell.php"
        "c99.php"
        "r57.php"
        "wso.php"
        "b374k.php"
        "cmd.php"
        "backdoor.php"
        "upload.php"
        "filemanager.php"
        "phpinfo.php"
        ".well-known/shell.php"
        "wp-content/uploads/shell.php"
        "images/shell.php"
    )

    local found_shells=0

    for path in "${shell_paths[@]}"; do
        local status=$(auth_curl -so /dev/null -w "%{http_code}" "${TARGET_URL}/${path}" --max-time 10 2>/dev/null || echo "000")

        if [ "$status" = "200" ]; then
            BACKDOORS_FOUND=true
            found_shells=$((found_shells + 1))
            echo -e "${RED}  Potential web shell found: ${TARGET_URL}/${path} (HTTP ${status})${NC}"
        fi
    done

    if [ $found_shells -gt 0 ]; then
        add_finding "Potential backdoor/shell files accessible (${found_shells} found)" 5
    fi

    # Check for backdoor patterns in page content
    if [ -n "$PAGE_CONTENT" ]; then
        local backdoor_patterns=(
            'eval\s*\(\s*\$_(POST|GET|REQUEST)'
            'system\s*\(\s*\$_(POST|GET|REQUEST)'
            'exec\s*\(\s*\$_(POST|GET|REQUEST)'
            'passthru\s*\(\s*\$_(POST|GET|REQUEST)'
            'shell_exec\s*\(\s*\$_(POST|GET|REQUEST)'
            'assert\s*\(\s*\$_(POST|GET|REQUEST)'
            'preg_replace.*\/e'
            'base64_decode\s*\(\s*\$_(POST|GET|REQUEST)'
        )

        for pattern in "${backdoor_patterns[@]}"; do
            if echo "$PAGE_CONTENT" | grep -qiE "$pattern"; then
                BACKDOORS_FOUND=true
                add_finding "PHP backdoor code pattern detected" 5
                break
            fi
        done
    fi

    [ "$BACKDOORS_FOUND" = false ] && echo -e "${GREEN}  No backdoor/shell patterns detected${NC}"
}

# Category 3: SEO Spam Detection
detect_seo_spam() {
    echo -e "\n${YELLOW}CATEGORY 3: SEO SPAM DETECTION${NC}"

    if [ -z "$PAGE_CONTENT" ]; then
        echo -e "${GRAY}  Skipped - no content available${NC}"
        return
    fi

    # Check for hidden links
    local hidden_count=0
    if echo "$PAGE_CONTENT" | grep -qiE 'display:\s*none.*<a\s|visibility:\s*hidden.*<a\s|font-size:\s*0.*<a\s'; then
        hidden_count=$((hidden_count + 1))
    fi

    if echo "$PAGE_CONTENT" | grep -qiE '<div[^>]*style="[^"]*display:\s*none[^"]*"[^>]*>.*<a\s'; then
        hidden_count=$((hidden_count + 1))
    fi

    HIDDEN_LINKS=$hidden_count

    if [ $hidden_count -gt 0 ]; then
        add_finding "Hidden links detected (${hidden_count} instances)" 2
    fi

    # Check for spam keywords
    local spam_keywords=(
        "buy viagra"
        "cheap cialis"
        "pharmacy online"
        "canadian pharmacy"
        "payday loan"
        "online casino"
        "free casino"
        "poker online"
        "replica watch"
        "louis vuitton"
        "cheap oakley"
        "jordan shoes"
        "ugg boots"
    )

    local spam_count=0
    for keyword in "${spam_keywords[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qi "$keyword"; then
            spam_count=$((spam_count + 1))
        fi
    done

    SPAM_KEYWORDS=$spam_count

    if [ $spam_count -ge 3 ]; then
        add_finding "SEO spam keywords detected (${spam_count} found)" 3
    elif [ $spam_count -ge 1 ]; then
        add_finding "Possible spam keywords (${spam_count} found)" 1
    fi

    # Check for hidden iframes
    if echo "$PAGE_CONTENT" | grep -qiE '<iframe[^>]*style="[^"]*width:\s*0|height:\s*0|display:\s*none'; then
        add_finding "Hidden iframe detected" 3
    fi

    [ $spam_count -eq 0 ] && [ $hidden_count -eq 0 ] && echo -e "${GREEN}  No SEO spam detected${NC}"
}

# Category 4: Defacement Detection
detect_defacement() {
    echo -e "\n${YELLOW}CATEGORY 4: DEFACEMENT DETECTION${NC}"

    if [ -z "$PAGE_CONTENT" ]; then
        echo -e "${GRAY}  Skipped - no content available${NC}"
        return
    fi

    # Defacement signatures
    local defacement_patterns=(
        "hacked by"
        "owned by"
        "pwned by"
        "defaced by"
        "greetz to"
        "h4ck3d"
        "0wn3d"
        "d3f4c3d"
        "Team hackers"
        "cyber army"
        "hacking team"
        "security breach"
        "anonymous collective"
    )

    for pattern in "${defacement_patterns[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qi "$pattern"; then
            DEFACEMENT_DETECTED=true
            add_finding "Defacement signature detected: '${pattern}'" 5
            break
        fi
    done

    # Check for hacktivist group markers
    local hacktivist_groups=(
        "Anonymous"
        "LulzSec"
        "Lizard Squad"
        "Syrian Electronic Army"
        "Ghost Squad"
        "AnonGhost"
    )

    for group in "${hacktivist_groups[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qi "hacked.*$group\|$group.*hack"; then
            DEFACEMENT_DETECTED=true
            add_finding "Hacktivist group marker: ${group}" 4
            break
        fi
    done

    [ "$DEFACEMENT_DETECTED" = false ] && echo -e "${GREEN}  No defacement signatures detected${NC}"
}

# Category 5: Suspicious External Resources
detect_suspicious_resources() {
    echo -e "\n${YELLOW}CATEGORY 5: SUSPICIOUS EXTERNAL RESOURCES${NC}"

    if [ -z "$PAGE_CONTENT" ]; then
        echo -e "${GRAY}  Skipped - no content available${NC}"
        return
    fi

    # Suspicious TLDs commonly used for malware
    local suspicious_tlds=(
        "\\.tk[\"'/\\s>]"
        "\\.ml[\"'/\\s>]"
        "\\.ga[\"'/\\s>]"
        "\\.cf[\"'/\\s>]"
        "\\.gq[\"'/\\s>]"
        "\\.xyz[\"'/\\s>]"
        "\\.top[\"'/\\s>]"
        "\\.work[\"'/\\s>]"
        "\\.click[\"'/\\s>]"
        "\\.download[\"'/\\s>]"
    )

    local tld_count=0
    for tld in "${suspicious_tlds[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qiE "(src|href)=[\"'][^\"']*${tld}"; then
            tld_count=$((tld_count + 1))
        fi
    done

    SUSPICIOUS_TLDS=$tld_count

    if [ $tld_count -ge 3 ]; then
        add_finding "Multiple suspicious TLDs detected (${tld_count} found)" 3
    elif [ $tld_count -ge 1 ]; then
        add_finding "Suspicious TLD detected (${tld_count} found)" 1
    fi

    # Check for mixed content (HTTP resources on HTTPS page)
    if [[ "$TARGET_URL" == https://* ]]; then
        if echo "$PAGE_CONTENT" | grep -qiE '(src|href)="http://[^"]+\.(js|css)"'; then
            add_finding "Mixed content detected (HTTP on HTTPS)" 1
        fi
    fi

    # Check for data URIs (often used for malicious payloads)
    if echo "$PAGE_CONTENT" | grep -qiE 'src="data:text/javascript'; then
        add_finding "Suspicious data URI JavaScript detected" 2
    fi

    # Check for external scripts from unusual sources
    if echo "$PAGE_CONTENT" | grep -qiE 'src="[^"]*pastebin\.com|src="[^"]*ghostbin\.'; then
        add_finding "Script loaded from paste site" 4
    fi

    [ $tld_count -eq 0 ] && echo -e "${GREEN}  No suspicious external resources detected${NC}"
}

# Category 6: Blacklisted Patterns (replaces VirusTotal)
detect_blacklisted_patterns() {
    echo -e "\n${YELLOW}CATEGORY 6: BLACKLISTED PATTERNS${NC}"

    if [ -z "$PAGE_CONTENT" ]; then
        echo -e "${GRAY}  Skipped - no content available${NC}"
        return
    fi

    # Known malicious domains
    local malicious_domains=(
        "malware-traffic-analysis"
        "coinhive.com"
        "coin-hive.com"
        "authedmine.com"
        "crypto-loot.com"
        "cryptoloot.pro"
        "webmine.pro"
        "minero.cc"
        "ppoi.org"
        "kisshentai.net/Content"
        "greenflashplayer"
    )

    local domain_count=0
    for domain in "${malicious_domains[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qi "$domain"; then
            domain_count=$((domain_count + 1))
            echo -e "${RED}  Blacklisted domain found: ${domain}${NC}"
        fi
    done

    BLACKLISTED_PATTERNS=$domain_count

    if [ $domain_count -gt 0 ]; then
        add_finding "Blacklisted domains detected (${domain_count} found)" 5
    fi

    # Check for exploit kit patterns
    local exploit_patterns=(
        "NSIS_Inetc"
        "PowerLoader"
        "Angler"
        "Magnitude"
        "RIG exploit"
        "Neutrino"
        "Nuclear"
        "Sweet Orange"
    )

    for pattern in "${exploit_patterns[@]}"; do
        if echo "$PAGE_CONTENT" | grep -qi "$pattern"; then
            add_finding "Exploit kit signature: ${pattern}" 5
            break
        fi
    done

    # Check for suspicious base64 encoded strings (potential encoded payloads)
    local base64_count=$(echo "$PAGE_CONTENT" | grep -oE '[A-Za-z0-9+/]{100,}={0,2}' | wc -l | tr -d ' ')

    if [ "$base64_count" -gt 5 ]; then
        add_finding "Multiple large Base64 strings detected (${base64_count}) - potential encoded payloads" 2
    fi

    [ $domain_count -eq 0 ] && echo -e "${GREEN}  No blacklisted patterns detected${NC}"
}

# Calculate threat level
calculate_threat_level() {
    if [ $THREAT_SCORE -ge 10 ]; then
        THREAT_LEVEL="HIGH"
    elif [ $THREAT_SCORE -ge 6 ]; then
        THREAT_LEVEL="MEDIUM"
    elif [ $THREAT_SCORE -ge 3 ]; then
        THREAT_LEVEL="LOW"
    else
        THREAT_LEVEL="CLEAN"
    fi
}

# Generate summary
generate_summary() {
    calculate_threat_level

    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}WEB MALWARE DETECTION SUMMARY${NC}"
    echo -e "${PURPLE}======================================================${NC}"

    echo -e "\n${BLUE}Target: ${TARGET_URL}${NC}"
    echo -e "${BLUE}Scan completed: $(date)${NC}\n"

    # Display threat level with color
    case "$THREAT_LEVEL" in
        "HIGH")
            echo -e "${RED}THREAT LEVEL: HIGH${NC}"
            echo -e "${RED}Score: ${THREAT_SCORE}/${MAX_SCORE}${NC}"
            ;;
        "MEDIUM")
            echo -e "${YELLOW}THREAT LEVEL: MEDIUM${NC}"
            echo -e "${YELLOW}Score: ${THREAT_SCORE}/${MAX_SCORE}${NC}"
            ;;
        "LOW")
            echo -e "${YELLOW}THREAT LEVEL: LOW${NC}"
            echo -e "${YELLOW}Score: ${THREAT_SCORE}/${MAX_SCORE}${NC}"
            ;;
        *)
            echo -e "${GREEN}THREAT LEVEL: CLEAN${NC}"
            echo -e "${GREEN}Score: ${THREAT_SCORE}/${MAX_SCORE}${NC}"
            ;;
    esac

    if [ ${#FINDINGS[@]} -gt 0 ]; then
        echo -e "\n${CYAN}Findings:${NC}"
        for finding in "${FINDINGS[@]}"; do
            echo -e "  - ${finding}"
        done
    fi

    echo -e "\n${CYAN}Detection Details:${NC}"
    echo -e "  Obfuscation Patterns: ${OBFUSCATION_PATTERNS}"
    echo -e "  Cryptominer Detected: ${CRYPTOMINER_DETECTED}"
    echo -e "  Backdoors Found: ${BACKDOORS_FOUND}"
    echo -e "  Spam Keywords: ${SPAM_KEYWORDS}"
    echo -e "  Hidden Links: ${HIDDEN_LINKS}"
    echo -e "  Defacement Detected: ${DEFACEMENT_DETECTED}"
    echo -e "  Suspicious TLDs: ${SUSPICIOUS_TLDS}"
    echo -e "  Blacklisted Patterns: ${BLACKLISTED_PATTERNS}"

    # Recommendation
    echo -e "\n${CYAN}RECOMMENDATION:${NC}"
    case "$THREAT_LEVEL" in
        "HIGH")
            echo -e "${RED}  CRITICAL: Multiple malware indicators detected. Site appears compromised.${NC}"
            echo -e "${RED}  Investigate immediately and consider taking the site offline.${NC}"
            ;;
        "MEDIUM")
            echo -e "${YELLOW}  WARNING: Suspicious patterns found. Manual review recommended.${NC}"
            echo -e "${YELLOW}  Check source code and server logs for further investigation.${NC}"
            ;;
        "LOW")
            echo -e "${YELLOW}  NOTICE: Minor concerns detected. Review may be beneficial.${NC}"
            echo -e "${YELLOW}  Some findings may be false positives from legitimate code.${NC}"
            ;;
        *)
            echo -e "${GREEN}  Site appears clean. No significant malware indicators detected.${NC}"
            echo -e "${GREEN}  Continue regular security monitoring.${NC}"
            ;;
    esac

    echo -e "\n${PURPLE}======================================================${NC}"

    # JSON output for application parsing
    local findings_json=""
    for finding in "${FINDINGS[@]}"; do
        [ -n "$findings_json" ] && findings_json="${findings_json},"
        findings_json="${findings_json}\"$(echo "$finding" | sed 's/"/\\"/g')\""
    done

    # Build recommendation text
    local recommendation_text=""
    case "$THREAT_LEVEL" in
        HIGH) recommendation_text="CRITICAL: Site appears compromised - investigate immediately";;
        MEDIUM) recommendation_text="Suspicious patterns found - manual review recommended";;
        LOW) recommendation_text="Minor concerns detected - review may be beneficial";;
        *) recommendation_text="Site appears clean - continue regular monitoring";;
    esac

    echo ""
    echo "--- JSON OUTPUT ---"
    cat << EOF
{
  "targetUrl": "${TARGET_URL}",
  "threatLevel": "${THREAT_LEVEL}",
  "threatScore": ${THREAT_SCORE},
  "maxScore": ${MAX_SCORE},
  "findings": [${findings_json}],
  "details": {
    "obfuscationPatterns": ${OBFUSCATION_PATTERNS},
    "cryptominerDetected": ${CRYPTOMINER_DETECTED},
    "backdoorsFound": ${BACKDOORS_FOUND},
    "spamKeywords": ${SPAM_KEYWORDS},
    "hiddenLinks": ${HIDDEN_LINKS},
    "defacementDetected": ${DEFACEMENT_DETECTED},
    "suspiciousTLDs": ${SUSPICIOUS_TLDS},
    "blacklistedPatterns": ${BLACKLISTED_PATTERNS}
  },
  "recommendation": "${recommendation_text}"
}
EOF
}

# Main execution
if ! fetch_content; then
    echo -e "${RED}Unable to scan target - could not retrieve content${NC}"
    echo ""
    echo "--- JSON OUTPUT ---"
    cat << EOF
{
  "targetUrl": "${TARGET_URL}",
  "threatLevel": "UNKNOWN",
  "threatScore": 0,
  "maxScore": ${MAX_SCORE},
  "findings": ["Unable to fetch target content"],
  "details": {
    "obfuscationPatterns": 0,
    "cryptominerDetected": false,
    "backdoorsFound": false,
    "spamKeywords": 0,
    "hiddenLinks": 0,
    "defacementDetected": false,
    "suspiciousTLDs": 0,
    "blacklistedPatterns": 0
  },
  "recommendation": "Unable to scan - check target URL and network connectivity"
}
EOF
    exit 1
fi

detect_malicious_javascript

if [ "$MODE" = "full" ]; then
    detect_backdoors
    detect_seo_spam
    detect_defacement
    detect_suspicious_resources
    detect_blacklisted_patterns
elif [ "$MODE" = "quick" ]; then
    echo -e "\n${GRAY}Quick mode: Skipping backdoor probing, SEO spam, defacement, and detailed checks${NC}"
    # Still run lightweight content-based checks
    detect_suspicious_resources
fi

generate_summary
