#!/bin/bash

# WAF/Firewall Detection Script
# Detects Web Application Firewalls protecting target websites
# Part of Security Scanner Desktop - Reconnaissance Features
# Usage: ./waf-detection-test.sh [TARGET_URL] [MODE] [CLEAN_OUTPUT]

set -e

if [[ "$1" == "--help" || "$1" == "-h" ]]; then
    echo "WAF/FIREWALL DETECTION SCANNER"
    echo "======================================================"
    echo "Detects Web Application Firewalls protecting websites"
    echo ""
    echo "USAGE: $0 <TARGET_URL> [MODE] [CLEAN_OUTPUT]"
    echo ""
    echo "DETECTION METHODS:"
    echo "  - HTTP Header Analysis"
    echo "  - Cookie Pattern Analysis"
    echo "  - Response Behavior Analysis"
    echo "  - Block Page Signature Detection"
    echo ""
    echo "SUPPORTED WAF VENDORS (40+):"
    echo "  Cloud: Cloudflare, AWS WAF, Azure WAF, Google Cloud Armor"
    echo "  Enterprise: Imperva, F5 BIG-IP, Akamai, Barracuda, Fortinet"
    echo "  Open-Source: ModSecurity, NAXSI"
    echo "  CMS: Wordfence, Sucuri, SiteLock"
    exit 0
fi

TARGET_URL="${1:-}"
MODE="${2:-full}"
CLEAN_OUTPUT="${3:-false}"

if [ "$CLEAN_OUTPUT" = "true" ]; then
    RED='' GREEN='' YELLOW='' BLUE='' PURPLE='' CYAN='' GRAY='' NC=''
else
    RED='\033[0;31m' GREEN='\033[0;32m' YELLOW='\033[1;33m' BLUE='\033[0;34m'
    PURPLE='\033[0;35m' CYAN='\033[0;36m' GRAY='\033[0;90m' NC='\033[0m'
fi

# Detection results
WAF_DETECTED=false
WAF_VENDOR=""
WAF_CONFIDENCE="none"
RATE_LIMITED=false
DETECTION_METHODS=()

# Probe results
PROBE_SQL="unknown"
PROBE_XSS="unknown"
PROBE_TRAVERSAL="unknown"
PROBE_RCE="unknown"

if [ -z "$TARGET_URL" ]; then
    echo -e "${RED}Error: Target URL is required${NC}"
    echo "Usage: $0 <TARGET_URL> [MODE] [CLEAN_OUTPUT]"
    exit 1
fi

[[ ! "$TARGET_URL" =~ ^https?:// ]] && TARGET_URL="https://$TARGET_URL"
TARGET_URL="${TARGET_URL%/}"

# Authentication support
AUTH_TOKEN="${AUTH_TOKEN:-}"
AUTH_COOKIE="${AUTH_COOKIE:-}"

auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

echo -e "${PURPLE}WAF/FIREWALL DETECTION SCANNER${NC}"
echo -e "${PURPLE}======================================================${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${NC}"
echo -e "${BLUE}Mode: ${MODE}${NC}"
echo -e "${BLUE}Timestamp: $(date)${NC}"
[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication${NC}"

echo ""

# Function to set WAF detection
detect_waf() {
    local vendor="$1"
    local method="$2"
    local confidence="$3"

    WAF_DETECTED=true
    WAF_VENDOR="$vendor"

    # Upgrade confidence if higher
    case "$WAF_CONFIDENCE" in
        "none") WAF_CONFIDENCE="$confidence" ;;
        "low") [ "$confidence" != "low" ] && WAF_CONFIDENCE="$confidence" ;;
        "medium") [ "$confidence" = "high" ] && WAF_CONFIDENCE="$confidence" ;;
    esac

    DETECTION_METHODS+=("$method")
    echo -e "${YELLOW}  WAF Detected: ${vendor} (${confidence} confidence via ${method})${NC}"
}

# Phase 1: HTTP Header Analysis
analyze_headers() {
    echo -e "\n${YELLOW}PHASE 1: HTTP HEADER ANALYSIS${NC}"

    local headers=$(auth_curl -sI "$TARGET_URL" --max-time 15 2>/dev/null || echo "")

    if [ -z "$headers" ]; then
        echo -e "${RED}  Failed to retrieve headers${NC}"
        return
    fi

    # Cloudflare
    if echo "$headers" | grep -qi "cf-ray\|cf-cache-status\|server:.*cloudflare"; then
        detect_waf "Cloudflare" "HTTP Header" "high"
    fi

    # Sucuri
    if echo "$headers" | grep -qi "x-sucuri-id\|x-sucuri-cache\|server:.*sucuri"; then
        detect_waf "Sucuri" "HTTP Header" "high"
    fi

    # Imperva/Incapsula
    if echo "$headers" | grep -qi "x-cdn:.*incapsula\|x-iinfo\|incap_ses"; then
        detect_waf "Imperva (Incapsula)" "HTTP Header" "high"
    fi

    # Akamai
    if echo "$headers" | grep -qi "x-akamai-transformed\|akamai"; then
        detect_waf "Akamai" "HTTP Header" "high"
    fi

    # AWS CloudFront/WAF
    if echo "$headers" | grep -qi "x-amz-cf-id\|x-amz-cf-pop\|x-amzn-waf"; then
        detect_waf "AWS CloudFront/WAF" "HTTP Header" "high"
    fi

    # F5 BIG-IP
    if echo "$headers" | grep -qi "server:.*bigip\|x-cnection\|x-wa-info"; then
        detect_waf "F5 BIG-IP" "HTTP Header" "high"
    fi

    # ModSecurity
    if echo "$headers" | grep -qi "server:.*mod_security\|x-mod-security\|modsecurity"; then
        detect_waf "ModSecurity" "HTTP Header" "high"
    fi

    # Barracuda
    if echo "$headers" | grep -qi "server:.*barracuda\|barra_counter_session"; then
        detect_waf "Barracuda WAF" "HTTP Header" "high"
    fi

    # Fortinet FortiWeb
    if echo "$headers" | grep -qi "fortiwafsid\|fgwaft_"; then
        detect_waf "Fortinet FortiWeb" "HTTP Header" "high"
    fi

    # Citrix NetScaler
    if echo "$headers" | grep -qi "via:.*netscaler\|x-citrix"; then
        detect_waf "Citrix NetScaler" "HTTP Header" "high"
    fi

    # Azure Application Gateway/WAF
    if echo "$headers" | grep -qi "x-azure-ref\|x-ms-request-id\|x-fd-"; then
        detect_waf "Azure WAF" "HTTP Header" "medium"
    fi

    # Google Cloud Armor
    if echo "$headers" | grep -qi "x-goog-\|x-cloud-trace"; then
        detect_waf "Google Cloud Armor" "HTTP Header" "low"
    fi

    # Fastly
    if echo "$headers" | grep -qi "x-fastly\|fastly-"; then
        detect_waf "Fastly" "HTTP Header" "medium"
    fi

    # StackPath
    if echo "$headers" | grep -qi "x-sp-\|stackpath"; then
        detect_waf "StackPath" "HTTP Header" "high"
    fi

    # Radware AppWall
    if echo "$headers" | grep -qi "x-sl-compstate\|rdwr"; then
        detect_waf "Radware AppWall" "HTTP Header" "high"
    fi

    # DenyAll
    if echo "$headers" | grep -qi "sessioncookie=\|denyall"; then
        detect_waf "DenyAll WAF" "HTTP Header" "high"
    fi

    # SonicWall
    if echo "$headers" | grep -qi "sonicwall\|snwl"; then
        detect_waf "SonicWall" "HTTP Header" "high"
    fi

    # Check for generic security headers that might indicate WAF
    if echo "$headers" | grep -qi "x-content-security-policy\|x-webkit-csp"; then
        echo -e "${GRAY}  Security headers present (may indicate WAF or security configuration)${NC}"
    fi

    [ "$WAF_DETECTED" = false ] && echo -e "${GREEN}  No WAF detected via headers${NC}" || true
}

# Phase 2: Cookie Pattern Analysis
analyze_cookies() {
    echo -e "\n${YELLOW}PHASE 2: COOKIE PATTERN ANALYSIS${NC}"

    local cookies=$(auth_curl -sI "$TARGET_URL" --max-time 15 2>/dev/null | grep -i "set-cookie" || echo "")

    if [ -z "$cookies" ]; then
        echo -e "${GRAY}  No cookies detected${NC}"
        return
    fi

    # Cloudflare cookies
    if echo "$cookies" | grep -qi "__cf_bm\|cf_clearance\|__cfduid\|__cfruid"; then
        detect_waf "Cloudflare" "Cookie Pattern" "high"
    fi

    # Imperva/Incapsula cookies
    if echo "$cookies" | grep -qi "incap_ses_\|visid_incap_\|nlbi_\|reese84"; then
        detect_waf "Imperva (Incapsula)" "Cookie Pattern" "high"
    fi

    # F5 BIG-IP cookies
    if echo "$cookies" | grep -qiE "bigipserver[a-z0-9_]*=\|ts[a-f0-9]{8}="; then
        detect_waf "F5 BIG-IP" "Cookie Pattern" "high"
    fi

    # AWS ALB cookies
    if echo "$cookies" | grep -qi "awsalb\|awsalbcors"; then
        detect_waf "AWS ALB" "Cookie Pattern" "medium"
    fi

    # Sucuri cookies
    if echo "$cookies" | grep -qi "sucuri_\|sucuri-"; then
        detect_waf "Sucuri" "Cookie Pattern" "high"
    fi

    # Barracuda cookies
    if echo "$cookies" | grep -qi "barra_counter_session"; then
        detect_waf "Barracuda WAF" "Cookie Pattern" "high"
    fi

    # DataDome
    if echo "$cookies" | grep -qi "datadome\|dd_"; then
        detect_waf "DataDome" "Cookie Pattern" "high"
    fi

    # PerimeterX
    if echo "$cookies" | grep -qi "_px\|pxhd"; then
        detect_waf "PerimeterX" "Cookie Pattern" "high"
    fi

    # Distil Networks (now Imperva)
    if echo "$cookies" | grep -qi "distil_\|d_id"; then
        detect_waf "Distil Networks" "Cookie Pattern" "high"
    fi

    # Shape Security (now F5)
    if echo "$cookies" | grep -qi "shape_"; then
        detect_waf "Shape Security" "Cookie Pattern" "high"
    fi

    # Reblaze
    if echo "$cookies" | grep -qi "rbzid\|reblaze"; then
        detect_waf "Reblaze" "Cookie Pattern" "high"
    fi

    # WordPress Wordfence
    if echo "$cookies" | grep -qi "wfwaf-\|wordfence"; then
        detect_waf "Wordfence (WordPress)" "Cookie Pattern" "high"
    fi

    [ "$WAF_DETECTED" = false ] && echo -e "${GREEN}  No WAF detected via cookies${NC}" || true
}

# Phase 3: Response Behavior Analysis (Attack Probe Tests)
analyze_response_behavior() {
    echo -e "\n${YELLOW}PHASE 3: RESPONSE BEHAVIOR ANALYSIS${NC}"

    local baseline_status=$(auth_curl -so /dev/null -w "%{http_code}" "$TARGET_URL" --max-time 15 2>/dev/null || echo "000")
    echo -e "${GRAY}  Baseline response: HTTP ${baseline_status}${NC}"

    # Test SQL injection payload
    local sqli_payload="' OR '1'='1"
    local sqli_encoded=$(echo "$sqli_payload" | sed 's/ /%20/g' | sed "s/'/%27/g")
    local sqli_status=$(auth_curl -so /dev/null -w "%{http_code}" "${TARGET_URL}/?id=${sqli_encoded}" --max-time 15 2>/dev/null || echo "000")
    PROBE_SQL="$sqli_status"

    if [ "$sqli_status" = "403" ] || [ "$sqli_status" = "406" ] || [ "$sqli_status" = "429" ] || [ "$sqli_status" = "503" ]; then
        echo -e "${YELLOW}  SQL injection probe blocked (HTTP ${sqli_status})${NC}"
        [ "$WAF_DETECTED" = false ] && detect_waf "Unknown WAF" "Behavior Analysis" "medium"
    else
        echo -e "${GRAY}  SQL injection probe: HTTP ${sqli_status}${NC}"
    fi

    # Test XSS payload
    local xss_payload="<script>alert('XSS')</script>"
    local xss_encoded=$(echo "$xss_payload" | sed 's/</%3C/g' | sed 's/>/%3E/g' | sed "s/'/%27/g")
    local xss_status=$(auth_curl -so /dev/null -w "%{http_code}" "${TARGET_URL}/?q=${xss_encoded}" --max-time 15 2>/dev/null || echo "000")
    PROBE_XSS="$xss_status"

    if [ "$xss_status" = "403" ] || [ "$xss_status" = "406" ] || [ "$xss_status" = "429" ] || [ "$xss_status" = "503" ]; then
        echo -e "${YELLOW}  XSS probe blocked (HTTP ${xss_status})${NC}"
        [ "$WAF_DETECTED" = false ] && detect_waf "Unknown WAF" "Behavior Analysis" "medium"
    else
        echo -e "${GRAY}  XSS probe: HTTP ${xss_status}${NC}"
    fi

    # Test path traversal payload
    local traversal_payload="../../../etc/passwd"
    local traversal_encoded=$(echo "$traversal_payload" | sed 's/\./%2e/g' | sed 's/\//%2f/g')
    local traversal_status=$(auth_curl -so /dev/null -w "%{http_code}" "${TARGET_URL}/?file=${traversal_encoded}" --max-time 15 2>/dev/null || echo "000")
    PROBE_TRAVERSAL="$traversal_status"

    if [ "$traversal_status" = "403" ] || [ "$traversal_status" = "406" ] || [ "$traversal_status" = "429" ] || [ "$traversal_status" = "503" ]; then
        echo -e "${YELLOW}  Path traversal probe blocked (HTTP ${traversal_status})${NC}"
        [ "$WAF_DETECTED" = false ] && detect_waf "Unknown WAF" "Behavior Analysis" "medium"
    else
        echo -e "${GRAY}  Path traversal probe: HTTP ${traversal_status}${NC}"
    fi

    # Test RCE payload (if full mode)
    if [ "$MODE" = "full" ]; then
        local rce_payload="; cat /etc/passwd"
        local rce_encoded=$(echo "$rce_payload" | sed 's/;/%3b/g' | sed 's/ /%20/g' | sed 's/\//%2f/g')
        local rce_status=$(auth_curl -so /dev/null -w "%{http_code}" "${TARGET_URL}/?cmd=${rce_encoded}" --max-time 15 2>/dev/null || echo "000")
        PROBE_RCE="$rce_status"

        if [ "$rce_status" = "403" ] || [ "$rce_status" = "406" ] || [ "$rce_status" = "429" ] || [ "$rce_status" = "503" ]; then
            echo -e "${YELLOW}  RCE probe blocked (HTTP ${rce_status})${NC}"
            [ "$WAF_DETECTED" = false ] && detect_waf "Unknown WAF" "Behavior Analysis" "medium"
        else
            echo -e "${GRAY}  RCE probe: HTTP ${rce_status}${NC}"
        fi
    fi

    # Test rate limiting
    echo -e "${GRAY}  Testing rate limiting...${NC}"
    local rate_limit_count=0
    for i in {1..10}; do
        local status=$(auth_curl -so /dev/null -w "%{http_code}" "$TARGET_URL" --max-time 5 2>/dev/null || echo "000")
        [ "$status" = "429" ] && rate_limit_count=$((rate_limit_count + 1))
    done

    if [ $rate_limit_count -ge 3 ]; then
        RATE_LIMITED=true
        echo -e "${YELLOW}  Rate limiting detected (${rate_limit_count}/10 requests returned 429)${NC}"
    else
        echo -e "${GRAY}  No rate limiting detected${NC}"
    fi
}

# Phase 4: Block Page Signature Detection
analyze_block_pages() {
    echo -e "\n${YELLOW}PHASE 4: BLOCK PAGE SIGNATURE DETECTION${NC}"

    # Send a payload that should trigger most WAFs
    local trigger_payload="<script>alert('XSS')</script>' OR '1'='1' --"
    local trigger_encoded=$(echo "$trigger_payload" | sed 's/</%3C/g' | sed 's/>/%3E/g' | sed "s/'/%27/g" | sed 's/ /%20/g')

    local response=$(auth_curl -s "${TARGET_URL}/?attack=${trigger_encoded}" --max-time 15 2>/dev/null || echo "")

    if [ -z "$response" ]; then
        echo -e "${GRAY}  No response received for trigger payload${NC}"
        return
    fi

    # Cloudflare block page
    if echo "$response" | grep -qi "cloudflare\|ray id\|cf-error-details\|attention required"; then
        detect_waf "Cloudflare" "Block Page" "high"
    fi

    # Sucuri block page
    if echo "$response" | grep -qi "sucuri\|cloudproxy\|access denied.*sucuri"; then
        detect_waf "Sucuri" "Block Page" "high"
    fi

    # Imperva/Incapsula block page
    if echo "$response" | grep -qi "incapsula\|incident\|_incap_\|request blocked"; then
        detect_waf "Imperva (Incapsula)" "Block Page" "high"
    fi

    # Akamai block page
    if echo "$response" | grep -qi "akamai\|reference.*#[0-9a-f]"; then
        detect_waf "Akamai" "Block Page" "high"
    fi

    # ModSecurity block page
    if echo "$response" | grep -qi "modsecurity\|mod_security\|not acceptable\|security rule"; then
        detect_waf "ModSecurity" "Block Page" "high"
    fi

    # F5 BIG-IP ASM block page
    if echo "$response" | grep -qi "the requested url was rejected\|big-ip\|f5 networks\|asm"; then
        detect_waf "F5 BIG-IP ASM" "Block Page" "high"
    fi

    # AWS WAF block page
    if echo "$response" | grep -qi "request blocked\|aws\|web application firewall"; then
        detect_waf "AWS WAF" "Block Page" "medium"
    fi

    # Barracuda block page
    if echo "$response" | grep -qi "barracuda\|barra_\|access denied.*firewall"; then
        detect_waf "Barracuda WAF" "Block Page" "high"
    fi

    # Fortinet FortiWeb block page
    if echo "$response" | grep -qi "fortiweb\|fortinet\|fortigate"; then
        detect_waf "Fortinet FortiWeb" "Block Page" "high"
    fi

    # Citrix NetScaler AppFirewall block page
    if echo "$response" | grep -qi "netscaler\|citrix\|ns_af"; then
        detect_waf "Citrix NetScaler AppFirewall" "Block Page" "high"
    fi

    # WordPress Wordfence block page
    if echo "$response" | grep -qi "wordfence\|wf-block\|blocked by wordfence"; then
        detect_waf "Wordfence (WordPress)" "Block Page" "high"
    fi

    # SiteLock block page
    if echo "$response" | grep -qi "sitelock\|trueshield"; then
        detect_waf "SiteLock" "Block Page" "high"
    fi

    # Comodo WAF block page
    if echo "$response" | grep -qi "comodo\|cwatch"; then
        detect_waf "Comodo WAF" "Block Page" "high"
    fi

    # Generic block page indicators
    if echo "$response" | grep -qiE "access.denied|request.blocked|forbidden|security.violation|your.request.has.been.blocked|suspicious.activity|firewall|waf"; then
        [ "$WAF_DETECTED" = false ] && detect_waf "Unknown WAF" "Block Page" "low"
    fi

    [ "$WAF_DETECTED" = false ] && echo -e "${GREEN}  No WAF block pages detected${NC}" || true
}

# Generate summary
generate_summary() {
    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}WAF DETECTION SUMMARY${NC}"
    echo -e "${PURPLE}======================================================${NC}"

    echo -e "\n${BLUE}Target: ${TARGET_URL}${NC}"
    echo -e "${BLUE}Scan completed: $(date)${NC}\n"

    if [ "$WAF_DETECTED" = true ]; then
        echo -e "${YELLOW}WAF DETECTED: ${WAF_VENDOR}${NC}"
        echo -e "${YELLOW}Confidence: ${WAF_CONFIDENCE}${NC}"

        if [ ${#DETECTION_METHODS[@]} -gt 0 ]; then
            echo -e "\n${CYAN}Detection Methods:${NC}"
            # Deduplicate detection methods
            local seen_methods=""
            for method in "${DETECTION_METHODS[@]}"; do
                if [[ "$seen_methods" != *"$method"* ]]; then
                    echo -e "  - ${method}"
                    seen_methods="${seen_methods}|${method}"
                fi
            done
        fi

        echo -e "\n${CYAN}Attack Probe Results:${NC}"
        echo -e "  SQL Injection: HTTP ${PROBE_SQL}"
        echo -e "  XSS: HTTP ${PROBE_XSS}"
        echo -e "  Path Traversal: HTTP ${PROBE_TRAVERSAL}"
        [ "$MODE" = "full" ] && echo -e "  RCE: HTTP ${PROBE_RCE}"

        echo -e "\n${CYAN}Rate Limiting: ${NC}${RATE_LIMITED}"

        echo -e "\n${YELLOW}RECOMMENDATION:${NC}"
        echo -e "  WAF detected - some vulnerability tests may be blocked or filtered."
        echo -e "  Consider testing from authorized IP ranges or coordinating with the security team."
    else
        echo -e "${GREEN}NO WAF DETECTED${NC}"
        echo -e "\n${GREEN}Target appears to have direct access without WAF protection.${NC}"

        echo -e "\n${CYAN}Attack Probe Results:${NC}"
        echo -e "  SQL Injection: HTTP ${PROBE_SQL}"
        echo -e "  XSS: HTTP ${PROBE_XSS}"
        echo -e "  Path Traversal: HTTP ${PROBE_TRAVERSAL}"
        [ "$MODE" = "full" ] && echo -e "  RCE: HTTP ${PROBE_RCE}"

        echo -e "\n${CYAN}Rate Limiting: ${NC}${RATE_LIMITED}"

        echo -e "\n${YELLOW}RECOMMENDATION:${NC}"
        echo -e "  No WAF detected - proceed with full vulnerability testing."
        echo -e "  Consider recommending WAF implementation for production environments."
    fi

    echo -e "\n${PURPLE}======================================================${NC}"

    # JSON output for application parsing
    echo ""
    echo "--- JSON OUTPUT ---"
    cat << EOF
{
  "detected": ${WAF_DETECTED},
  "vendor": "${WAF_VENDOR}",
  "confidence": "${WAF_CONFIDENCE}",
  "rateLimited": ${RATE_LIMITED},
  "probeResults": {
    "sqlInjection": "${PROBE_SQL}",
    "xss": "${PROBE_XSS}",
    "pathTraversal": "${PROBE_TRAVERSAL}",
    "rce": "${PROBE_RCE}"
  },
  "recommendation": "$([ "$WAF_DETECTED" = true ] && echo "WAF detected - some vulnerability tests may be blocked or filtered" || echo "No WAF detected - proceed with full vulnerability testing")"
}
EOF
}

# Main execution
analyze_headers
analyze_cookies

if [ "$MODE" = "full" ]; then
    analyze_response_behavior
    analyze_block_pages
elif [ "$MODE" = "quick" ]; then
    echo -e "\n${GRAY}Skipping behavior analysis and block page detection in quick mode${NC}"
fi

generate_summary
