<#
.SYNOPSIS
    WAF/Firewall Detection Script (PowerShell)

.DESCRIPTION
    Detects Web Application Firewalls protecting target websites
    Part of Security Scanner Desktop - Reconnaissance Features

.PARAMETER TargetUrl
    Target web application URL

.PARAMETER Mode
    Scan mode: quick or full (default: full)

.PARAMETER CleanOutput
    Disable colored output

.EXAMPLE
    .\waf-detection-test.ps1 -TargetUrl "https://example.com" -Mode quick
    .\waf-detection-test.ps1 -TargetUrl "https://example.com" -Mode full
#>

[CmdletBinding()]
param(
    [Parameter(Mandatory=$false)]
    [string]$TargetUrl,

    [Parameter(Mandatory=$false)]
    [ValidateSet('quick', 'full')]
    [string]$Mode = 'full',

    [Parameter(Mandatory=$false)]
    [switch]$CleanOutput,

    [Parameter(Mandatory=$false)]
    [switch]$Help
)

if ($Help) {
    Write-Host "WAF/FIREWALL DETECTION SCANNER"
    Write-Host "======================================================"
    Write-Host "Detects Web Application Firewalls protecting websites"
    Write-Host ""
    Write-Host "USAGE: .\waf-detection-test.ps1 -TargetUrl <URL> [-Mode <quick|full>] [-CleanOutput]"
    Write-Host ""
    Write-Host "DETECTION METHODS:"
    Write-Host "  - HTTP Header Analysis"
    Write-Host "  - Cookie Pattern Analysis"
    Write-Host "  - Response Behavior Analysis"
    Write-Host "  - Block Page Signature Detection"
    Write-Host ""
    Write-Host "SUPPORTED WAF VENDORS (40+):"
    Write-Host "  Cloud: Cloudflare, AWS WAF, Azure WAF, Google Cloud Armor"
    Write-Host "  Enterprise: Imperva, F5 BIG-IP, Akamai, Barracuda, Fortinet"
    Write-Host "  Open-Source: ModSecurity, NAXSI"
    Write-Host "  CMS: Wordfence, Sucuri, SiteLock"
    exit 0
}

if ([string]::IsNullOrEmpty($TargetUrl)) {
    Write-Host "Error: Target URL is required" -ForegroundColor Red
    Write-Host "Usage: .\waf-detection-test.ps1 -TargetUrl <URL> [-Mode <quick|full>] [-CleanOutput]"
    exit 1
}

# Normalize URL
if ($TargetUrl -notmatch "^https?://") {
    $TargetUrl = "https://$TargetUrl"
}
$TargetUrl = $TargetUrl.TrimEnd('/')

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [int]$TimeoutSec = 15)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

# Detection results
$script:WafDetected = $false
$script:WafVendor = ""
$script:WafConfidence = "none"
$script:RateLimited = $false
$script:DetectionMethods = @()

# Probe results
$script:ProbeSql = "unknown"
$script:ProbeXss = "unknown"
$script:ProbeTraversal = "unknown"
$script:ProbeRce = "unknown"

# Header output
Write-Host ""
Write-Host "WAF/FIREWALL DETECTION SCANNER" -ForegroundColor Magenta
Write-Host "======================================================" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl" -ForegroundColor Blue
Write-Host "Mode: $Mode" -ForegroundColor Blue
Write-Host "Timestamp: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -ForegroundColor Blue
if ($AuthToken) { Write-Host "🔐 Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "🔐 Cookie authentication enabled" -ForegroundColor Green }
if (-not $AuthToken -and -not $AuthCookie) { Write-Host "ℹ️ Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

Write-Host ""

# Function to set WAF detection
function Set-WafDetection {
    param(
        [string]$Vendor,
        [string]$Method,
        [string]$Confidence
    )

    $script:WafDetected = $true
    $script:WafVendor = $Vendor

    # Upgrade confidence if higher
    switch ($script:WafConfidence) {
        "none" { $script:WafConfidence = $Confidence }
        "low" { if ($Confidence -ne "low") { $script:WafConfidence = $Confidence } }
        "medium" { if ($Confidence -eq "high") { $script:WafConfidence = $Confidence } }
    }

    $script:DetectionMethods += $Method
    Write-Host "  WAF Detected: $Vendor ($Confidence confidence via $Method)" -ForegroundColor Yellow
}

# Phase 1: HTTP Header Analysis
function Test-HttpHeaders {
    Write-Host ""
    Write-Host "PHASE 1: HTTP HEADER ANALYSIS" -ForegroundColor Yellow

    try {
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method Head -TimeoutSec 15
        $headers = $response.Headers
        $rawHeaders = $response.RawContent
    } catch {
        Write-Host "  Failed to retrieve headers" -ForegroundColor Red
        return
    }

    # Cloudflare
    if ($headers['cf-ray'] -or $headers['cf-cache-status'] -or $rawHeaders -match "server:.*cloudflare") {
        Set-WafDetection -Vendor "Cloudflare" -Method "HTTP Header" -Confidence "high"
    }

    # Sucuri
    if ($headers['x-sucuri-id'] -or $headers['x-sucuri-cache'] -or $rawHeaders -match "server:.*sucuri") {
        Set-WafDetection -Vendor "Sucuri" -Method "HTTP Header" -Confidence "high"
    }

    # Imperva/Incapsula
    if ($rawHeaders -match "x-cdn:.*incapsula" -or $headers['x-iinfo'] -or $rawHeaders -match "incap_ses") {
        Set-WafDetection -Vendor "Imperva (Incapsula)" -Method "HTTP Header" -Confidence "high"
    }

    # Akamai
    if ($headers['x-akamai-transformed'] -or $rawHeaders -match "akamai") {
        Set-WafDetection -Vendor "Akamai" -Method "HTTP Header" -Confidence "high"
    }

    # AWS CloudFront/WAF
    if ($headers['x-amz-cf-id'] -or $headers['x-amz-cf-pop'] -or $headers['x-amzn-waf']) {
        Set-WafDetection -Vendor "AWS CloudFront/WAF" -Method "HTTP Header" -Confidence "high"
    }

    # F5 BIG-IP
    if ($rawHeaders -match "server:.*bigip" -or $headers['x-cnection'] -or $headers['x-wa-info']) {
        Set-WafDetection -Vendor "F5 BIG-IP" -Method "HTTP Header" -Confidence "high"
    }

    # ModSecurity
    if ($rawHeaders -match "server:.*mod_security" -or $headers['x-mod-security'] -or $rawHeaders -match "modsecurity") {
        Set-WafDetection -Vendor "ModSecurity" -Method "HTTP Header" -Confidence "high"
    }

    # Barracuda
    if ($rawHeaders -match "server:.*barracuda" -or $rawHeaders -match "barra_counter_session") {
        Set-WafDetection -Vendor "Barracuda WAF" -Method "HTTP Header" -Confidence "high"
    }

    # Fortinet FortiWeb
    if ($rawHeaders -match "fortiwafsid" -or $rawHeaders -match "fgwaft_") {
        Set-WafDetection -Vendor "Fortinet FortiWeb" -Method "HTTP Header" -Confidence "high"
    }

    # Citrix NetScaler
    if ($rawHeaders -match "via:.*netscaler" -or $headers['x-citrix']) {
        Set-WafDetection -Vendor "Citrix NetScaler" -Method "HTTP Header" -Confidence "high"
    }

    # Azure Application Gateway/WAF
    if ($headers['x-azure-ref'] -or $headers['x-ms-request-id'] -or $rawHeaders -match "x-fd-") {
        Set-WafDetection -Vendor "Azure WAF" -Method "HTTP Header" -Confidence "medium"
    }

    # Google Cloud Armor
    if ($rawHeaders -match "x-goog-" -or $rawHeaders -match "x-cloud-trace") {
        Set-WafDetection -Vendor "Google Cloud Armor" -Method "HTTP Header" -Confidence "low"
    }

    # Fastly
    if ($headers['x-fastly'] -or $rawHeaders -match "fastly-") {
        Set-WafDetection -Vendor "Fastly" -Method "HTTP Header" -Confidence "medium"
    }

    # StackPath
    if ($rawHeaders -match "x-sp-" -or $rawHeaders -match "stackpath") {
        Set-WafDetection -Vendor "StackPath" -Method "HTTP Header" -Confidence "high"
    }

    # Radware AppWall
    if ($headers['x-sl-compstate'] -or $rawHeaders -match "rdwr") {
        Set-WafDetection -Vendor "Radware AppWall" -Method "HTTP Header" -Confidence "high"
    }

    # DenyAll
    if ($rawHeaders -match "sessioncookie=" -or $rawHeaders -match "denyall") {
        Set-WafDetection -Vendor "DenyAll WAF" -Method "HTTP Header" -Confidence "high"
    }

    # SonicWall
    if ($rawHeaders -match "sonicwall" -or $rawHeaders -match "snwl") {
        Set-WafDetection -Vendor "SonicWall" -Method "HTTP Header" -Confidence "high"
    }

    # Check for generic security headers
    if ($rawHeaders -match "x-content-security-policy" -or $rawHeaders -match "x-webkit-csp") {
        Write-Host "  Security headers present (may indicate WAF or security configuration)" -ForegroundColor Gray
    }

    if (-not $script:WafDetected) {
        Write-Host "  No WAF detected via headers" -ForegroundColor Green
    }
}

# Phase 2: Cookie Pattern Analysis
function Test-CookiePatterns {
    Write-Host ""
    Write-Host "PHASE 2: COOKIE PATTERN ANALYSIS" -ForegroundColor Yellow

    try {
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method Get -TimeoutSec 15
        $cookies = $response.Headers['Set-Cookie']

        if (-not $cookies) {
            Write-Host "  No cookies detected" -ForegroundColor Gray
            return
        }

        $cookieString = $cookies -join " "
    } catch {
        Write-Host "  Failed to retrieve cookies" -ForegroundColor Red
        return
    }

    # Cloudflare cookies
    if ($cookieString -match "__cf_bm|cf_clearance|__cfduid|__cfruid") {
        Set-WafDetection -Vendor "Cloudflare" -Method "Cookie Pattern" -Confidence "high"
    }

    # Imperva/Incapsula cookies
    if ($cookieString -match "incap_ses_|visid_incap_|nlbi_|reese84") {
        Set-WafDetection -Vendor "Imperva (Incapsula)" -Method "Cookie Pattern" -Confidence "high"
    }

    # F5 BIG-IP cookies
    if ($cookieString -match "bigipserver[a-z0-9_]*=|ts[a-f0-9]{8}=") {
        Set-WafDetection -Vendor "F5 BIG-IP" -Method "Cookie Pattern" -Confidence "high"
    }

    # AWS ALB cookies
    if ($cookieString -match "awsalb|awsalbcors") {
        Set-WafDetection -Vendor "AWS ALB" -Method "Cookie Pattern" -Confidence "medium"
    }

    # Sucuri cookies
    if ($cookieString -match "sucuri_|sucuri-") {
        Set-WafDetection -Vendor "Sucuri" -Method "Cookie Pattern" -Confidence "high"
    }

    # Barracuda cookies
    if ($cookieString -match "barra_counter_session") {
        Set-WafDetection -Vendor "Barracuda WAF" -Method "Cookie Pattern" -Confidence "high"
    }

    # DataDome
    if ($cookieString -match "datadome|dd_") {
        Set-WafDetection -Vendor "DataDome" -Method "Cookie Pattern" -Confidence "high"
    }

    # PerimeterX
    if ($cookieString -match "_px|pxhd") {
        Set-WafDetection -Vendor "PerimeterX" -Method "Cookie Pattern" -Confidence "high"
    }

    # Distil Networks (now Imperva)
    if ($cookieString -match "distil_|d_id") {
        Set-WafDetection -Vendor "Distil Networks" -Method "Cookie Pattern" -Confidence "high"
    }

    # Shape Security (now F5)
    if ($cookieString -match "shape_") {
        Set-WafDetection -Vendor "Shape Security" -Method "Cookie Pattern" -Confidence "high"
    }

    # Reblaze
    if ($cookieString -match "rbzid|reblaze") {
        Set-WafDetection -Vendor "Reblaze" -Method "Cookie Pattern" -Confidence "high"
    }

    # WordPress Wordfence
    if ($cookieString -match "wfwaf-|wordfence") {
        Set-WafDetection -Vendor "Wordfence (WordPress)" -Method "Cookie Pattern" -Confidence "high"
    }

    if (-not $script:WafDetected) {
        Write-Host "  No WAF detected via cookies" -ForegroundColor Green
    }
}

# Phase 3: Response Behavior Analysis (Attack Probe Tests)
function Test-ResponseBehavior {
    Write-Host ""
    Write-Host "PHASE 3: RESPONSE BEHAVIOR ANALYSIS" -ForegroundColor Yellow

    # Get baseline
    try {
        $baseline = Invoke-AuthWebRequest -Uri $TargetUrl -Method Get -TimeoutSec 15
        $baselineStatus = $baseline.StatusCode
    } catch {
        $baselineStatus = 0
    }
    Write-Host "  Baseline response: HTTP $baselineStatus" -ForegroundColor Gray

    # Test SQL injection payload
    $sqliPayload = "' OR '1'='1"
    $sqliEncoded = [System.Web.HttpUtility]::UrlEncode($sqliPayload)
    try {
        $sqliResponse = Invoke-AuthWebRequest -Uri "$TargetUrl/?id=$sqliEncoded" -Method Get -TimeoutSec 15
        $script:ProbeSql = $sqliResponse.StatusCode.ToString()
    } catch [System.Net.WebException] {
        $script:ProbeSql = [int]$_.Exception.Response.StatusCode
    } catch {
        $script:ProbeSql = "error"
    }

    if ($script:ProbeSql -in @("403", "406", "429", "503")) {
        Write-Host "  SQL injection probe blocked (HTTP $($script:ProbeSql))" -ForegroundColor Yellow
        if (-not $script:WafDetected) {
            Set-WafDetection -Vendor "Unknown WAF" -Method "Behavior Analysis" -Confidence "medium"
        }
    } else {
        Write-Host "  SQL injection probe: HTTP $($script:ProbeSql)" -ForegroundColor Gray
    }

    # Test XSS payload
    $xssPayload = "<script>alert('XSS')</script>"
    $xssEncoded = [System.Web.HttpUtility]::UrlEncode($xssPayload)
    try {
        $xssResponse = Invoke-AuthWebRequest -Uri "$TargetUrl/?q=$xssEncoded" -Method Get -TimeoutSec 15
        $script:ProbeXss = $xssResponse.StatusCode.ToString()
    } catch [System.Net.WebException] {
        $script:ProbeXss = [int]$_.Exception.Response.StatusCode
    } catch {
        $script:ProbeXss = "error"
    }

    if ($script:ProbeXss -in @("403", "406", "429", "503")) {
        Write-Host "  XSS probe blocked (HTTP $($script:ProbeXss))" -ForegroundColor Yellow
        if (-not $script:WafDetected) {
            Set-WafDetection -Vendor "Unknown WAF" -Method "Behavior Analysis" -Confidence "medium"
        }
    } else {
        Write-Host "  XSS probe: HTTP $($script:ProbeXss)" -ForegroundColor Gray
    }

    # Test path traversal payload
    $traversalPayload = "../../../etc/passwd"
    $traversalEncoded = [System.Web.HttpUtility]::UrlEncode($traversalPayload)
    try {
        $traversalResponse = Invoke-AuthWebRequest -Uri "$TargetUrl/?file=$traversalEncoded" -Method Get -TimeoutSec 15
        $script:ProbeTraversal = $traversalResponse.StatusCode.ToString()
    } catch [System.Net.WebException] {
        $script:ProbeTraversal = [int]$_.Exception.Response.StatusCode
    } catch {
        $script:ProbeTraversal = "error"
    }

    if ($script:ProbeTraversal -in @("403", "406", "429", "503")) {
        Write-Host "  Path traversal probe blocked (HTTP $($script:ProbeTraversal))" -ForegroundColor Yellow
        if (-not $script:WafDetected) {
            Set-WafDetection -Vendor "Unknown WAF" -Method "Behavior Analysis" -Confidence "medium"
        }
    } else {
        Write-Host "  Path traversal probe: HTTP $($script:ProbeTraversal)" -ForegroundColor Gray
    }

    # Test RCE payload (if full mode)
    if ($Mode -eq "full") {
        $rcePayload = "; cat /etc/passwd"
        $rceEncoded = [System.Web.HttpUtility]::UrlEncode($rcePayload)
        try {
            $rceResponse = Invoke-AuthWebRequest -Uri "$TargetUrl/?cmd=$rceEncoded" -Method Get -TimeoutSec 15
            $script:ProbeRce = $rceResponse.StatusCode.ToString()
        } catch [System.Net.WebException] {
            $script:ProbeRce = [int]$_.Exception.Response.StatusCode
        } catch {
            $script:ProbeRce = "error"
        }

        if ($script:ProbeRce -in @("403", "406", "429", "503")) {
            Write-Host "  RCE probe blocked (HTTP $($script:ProbeRce))" -ForegroundColor Yellow
            if (-not $script:WafDetected) {
                Set-WafDetection -Vendor "Unknown WAF" -Method "Behavior Analysis" -Confidence "medium"
            }
        } else {
            Write-Host "  RCE probe: HTTP $($script:ProbeRce)" -ForegroundColor Gray
        }
    }

    # Test rate limiting
    Write-Host "  Testing rate limiting..." -ForegroundColor Gray
    $rateLimitCount = 0
    for ($i = 1; $i -le 10; $i++) {
        try {
            $r = Invoke-AuthWebRequest -Uri $TargetUrl -Method Get -TimeoutSec 5
            if ($r.StatusCode -eq 429) { $rateLimitCount++ }
        } catch [System.Net.WebException] {
            if ([int]$_.Exception.Response.StatusCode -eq 429) { $rateLimitCount++ }
        } catch {}
    }

    if ($rateLimitCount -ge 3) {
        $script:RateLimited = $true
        Write-Host "  Rate limiting detected ($rateLimitCount/10 requests returned 429)" -ForegroundColor Yellow
    } else {
        Write-Host "  No rate limiting detected" -ForegroundColor Gray
    }
}

# Phase 4: Block Page Signature Detection
function Test-BlockPages {
    Write-Host ""
    Write-Host "PHASE 4: BLOCK PAGE SIGNATURE DETECTION" -ForegroundColor Yellow

    # Send a payload that should trigger most WAFs
    $triggerPayload = "<script>alert('XSS')</script>' OR '1'='1' --"
    $triggerEncoded = [System.Web.HttpUtility]::UrlEncode($triggerPayload)

    try {
        $response = Invoke-AuthWebRequest -Uri "$TargetUrl/?attack=$triggerEncoded" -Method Get -TimeoutSec 15
        $content = $response.Content
    } catch [System.Net.WebException] {
        try {
            $reader = New-Object System.IO.StreamReader($_.Exception.Response.GetResponseStream())
            $content = $reader.ReadToEnd()
            $reader.Close()
        } catch {
            Write-Host "  No response received for trigger payload" -ForegroundColor Gray
            return
        }
    } catch {
        Write-Host "  No response received for trigger payload" -ForegroundColor Gray
        return
    }

    if ([string]::IsNullOrEmpty($content)) {
        Write-Host "  No response received for trigger payload" -ForegroundColor Gray
        return
    }

    # Cloudflare block page
    if ($content -match "cloudflare|ray id|cf-error-details|attention required") {
        Set-WafDetection -Vendor "Cloudflare" -Method "Block Page" -Confidence "high"
    }

    # Sucuri block page
    if ($content -match "sucuri|cloudproxy|access denied.*sucuri") {
        Set-WafDetection -Vendor "Sucuri" -Method "Block Page" -Confidence "high"
    }

    # Imperva/Incapsula block page
    if ($content -match "incapsula|incident|_incap_|request blocked") {
        Set-WafDetection -Vendor "Imperva (Incapsula)" -Method "Block Page" -Confidence "high"
    }

    # Akamai block page
    if ($content -match "akamai|reference.*#[0-9a-f]") {
        Set-WafDetection -Vendor "Akamai" -Method "Block Page" -Confidence "high"
    }

    # ModSecurity block page
    if ($content -match "modsecurity|mod_security|not acceptable|security rule") {
        Set-WafDetection -Vendor "ModSecurity" -Method "Block Page" -Confidence "high"
    }

    # F5 BIG-IP ASM block page
    if ($content -match "the requested url was rejected|big-ip|f5 networks|asm") {
        Set-WafDetection -Vendor "F5 BIG-IP ASM" -Method "Block Page" -Confidence "high"
    }

    # AWS WAF block page
    if ($content -match "request blocked|aws|web application firewall") {
        Set-WafDetection -Vendor "AWS WAF" -Method "Block Page" -Confidence "medium"
    }

    # Barracuda block page
    if ($content -match "barracuda|barra_|access denied.*firewall") {
        Set-WafDetection -Vendor "Barracuda WAF" -Method "Block Page" -Confidence "high"
    }

    # Fortinet FortiWeb block page
    if ($content -match "fortiweb|fortinet|fortigate") {
        Set-WafDetection -Vendor "Fortinet FortiWeb" -Method "Block Page" -Confidence "high"
    }

    # Citrix NetScaler AppFirewall block page
    if ($content -match "netscaler|citrix|ns_af") {
        Set-WafDetection -Vendor "Citrix NetScaler AppFirewall" -Method "Block Page" -Confidence "high"
    }

    # WordPress Wordfence block page
    if ($content -match "wordfence|wf-block|blocked by wordfence") {
        Set-WafDetection -Vendor "Wordfence (WordPress)" -Method "Block Page" -Confidence "high"
    }

    # SiteLock block page
    if ($content -match "sitelock|trueshield") {
        Set-WafDetection -Vendor "SiteLock" -Method "Block Page" -Confidence "high"
    }

    # Comodo WAF block page
    if ($content -match "comodo|cwatch") {
        Set-WafDetection -Vendor "Comodo WAF" -Method "Block Page" -Confidence "high"
    }

    # Generic block page indicators
    if ($content -match "access.denied|request.blocked|forbidden|security.violation|your.request.has.been.blocked|suspicious.activity|firewall|waf") {
        if (-not $script:WafDetected) {
            Set-WafDetection -Vendor "Unknown WAF" -Method "Block Page" -Confidence "low"
        }
    }

    if (-not $script:WafDetected) {
        Write-Host "  No WAF block pages detected" -ForegroundColor Green
    }
}

# Generate summary
function Show-Summary {
    Write-Host ""
    Write-Host "======================================================" -ForegroundColor Magenta
    Write-Host "WAF DETECTION SUMMARY" -ForegroundColor Magenta
    Write-Host "======================================================" -ForegroundColor Magenta

    Write-Host ""
    Write-Host "Target: $TargetUrl" -ForegroundColor Blue
    Write-Host "Scan completed: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -ForegroundColor Blue
    Write-Host ""

    if ($script:WafDetected) {
        Write-Host "WAF DETECTED: $($script:WafVendor)" -ForegroundColor Yellow
        Write-Host "Confidence: $($script:WafConfidence)" -ForegroundColor Yellow

        if ($script:DetectionMethods.Count -gt 0) {
            Write-Host ""
            Write-Host "Detection Methods:" -ForegroundColor Cyan
            # Deduplicate detection methods
            $uniqueMethods = $script:DetectionMethods | Select-Object -Unique
            foreach ($method in $uniqueMethods) {
                Write-Host "  - $method"
            }
        }

        Write-Host ""
        Write-Host "Attack Probe Results:" -ForegroundColor Cyan
        Write-Host "  SQL Injection: HTTP $($script:ProbeSql)"
        Write-Host "  XSS: HTTP $($script:ProbeXss)"
        Write-Host "  Path Traversal: HTTP $($script:ProbeTraversal)"
        if ($Mode -eq "full") {
            Write-Host "  RCE: HTTP $($script:ProbeRce)"
        }

        Write-Host ""
        Write-Host "Rate Limiting: $($script:RateLimited)" -ForegroundColor Cyan

        Write-Host ""
        Write-Host "RECOMMENDATION:" -ForegroundColor Yellow
        Write-Host "  WAF detected - some vulnerability tests may be blocked or filtered."
        Write-Host "  Consider testing from authorized IP ranges or coordinating with the security team."
    } else {
        Write-Host "NO WAF DETECTED" -ForegroundColor Green
        Write-Host ""
        Write-Host "Target appears to have direct access without WAF protection." -ForegroundColor Green

        Write-Host ""
        Write-Host "Attack Probe Results:" -ForegroundColor Cyan
        Write-Host "  SQL Injection: HTTP $($script:ProbeSql)"
        Write-Host "  XSS: HTTP $($script:ProbeXss)"
        Write-Host "  Path Traversal: HTTP $($script:ProbeTraversal)"
        if ($Mode -eq "full") {
            Write-Host "  RCE: HTTP $($script:ProbeRce)"
        }

        Write-Host ""
        Write-Host "Rate Limiting: $($script:RateLimited)" -ForegroundColor Cyan

        Write-Host ""
        Write-Host "RECOMMENDATION:" -ForegroundColor Yellow
        Write-Host "  No WAF detected - proceed with full vulnerability testing."
        Write-Host "  Consider recommending WAF implementation for production environments."
    }

    Write-Host ""
    Write-Host "======================================================" -ForegroundColor Magenta

    # JSON output for application parsing
    Write-Host ""
    Write-Host "--- JSON OUTPUT ---"

    $recommendation = if ($script:WafDetected) {
        "WAF detected - some vulnerability tests may be blocked or filtered"
    } else {
        "No WAF detected - proceed with full vulnerability testing"
    }

    $jsonOutput = @{
        detected = $script:WafDetected
        vendor = $script:WafVendor
        confidence = $script:WafConfidence
        rateLimited = $script:RateLimited
        probeResults = @{
            sqlInjection = $script:ProbeSql.ToString()
            xss = $script:ProbeXss.ToString()
            pathTraversal = $script:ProbeTraversal.ToString()
            rce = $script:ProbeRce.ToString()
        }
        recommendation = $recommendation
    } | ConvertTo-Json -Compress

    Write-Host $jsonOutput
}

# Main execution
try {
    Add-Type -AssemblyName System.Web -ErrorAction SilentlyContinue

    Test-HttpHeaders
    Test-CookiePatterns

    if ($Mode -eq "full") {
        Test-ResponseBehavior
        Test-BlockPages
    } elseif ($Mode -eq "quick") {
        Write-Host ""
        Write-Host "Skipping behavior analysis and block page detection in quick mode" -ForegroundColor Gray
    }

    Show-Summary

} catch {
    Write-Host "Script error: $($_.Exception.Message)" -ForegroundColor Red
    exit 1
}
