#!/bin/bash

# Generic Web Application Security Testing Script.
# Works with any website - no specific API endpoints or authentication required
# Usage: ./generic-security-test.sh [TARGET_URL] [--zap-only] [--quick]

set -e

# Configuration
TARGET_URL="${1:-}"
MODE="${2:-full}"  # full, zap-only, quick
CLEAN_OUTPUT="${3:-false}"  # clean output for desktop apps
ZAP_PORT="${ZAP_PORT:-8090}"
ZAP_HOST="${ZAP_HOST:-127.0.0.1}"
TIMEOUT="${TIMEOUT:-300}"
# Authentication for authenticated scanning (passed via AUTH_COOKIE or AUTH_TOKEN env vars)
AUTH_COOKIE="${AUTH_COOKIE:-}"
AUTH_TOKEN="${AUTH_TOKEN:-}"

# Colors - disable for clean output
if [ "$CLEAN_OUTPUT" = "true" ]; then
    RED=''
    GREEN=''
    YELLOW=''
    BLUE=''
    PURPLE=''
    CYAN=''
    GRAY=''
    NC=''
else
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    BLUE='\033[0;34m'
    PURPLE='\033[0;35m'
    CYAN='\033[0;36m'
    GRAY='\033[0;90m'
    NC='\033[0m'
fi

# Test tracking
TESTS_PASSED=0
TESTS_FAILED=0
TESTS_TOTAL=0
HIGH_RISK_FINDINGS=0
MEDIUM_RISK_FINDINGS=0
LOW_RISK_FINDINGS=0
INFORMATIONAL_FINDINGS=0

# Findings tracking arrays
HIGH_RISK_LIST=()
MEDIUM_RISK_LIST=()
LOW_RISK_LIST=()
INFO_LIST=()
AUTH_ERRORS=0  # Track 401/403 responses

# URL encoding helper for safe API calls
url_encode() {
    python3 -c "import urllib.parse; print(urllib.parse.quote('''$1''', safe=''))" 2>/dev/null || echo "$1"
}

# Validate input
if [ -z "$TARGET_URL" ]; then
    echo -e "${RED}❌ Error: Target URL is required${NC}"
    echo -e "${BLUE}Usage: $0 <TARGET_URL> [mode]${NC}"
    echo -e "${BLUE}Modes: ${NC}"
    echo -e "${BLUE}  quick   - Fast core tests (headers, CORS, CSRF, path traversal, SQLi, SSL) ~2-5 min${NC}"
    echo -e "${BLUE}  full    - DEEP PENETRATION TESTING - all 11 phases including ZAP (15-60+ min, default)${NC}"
    echo -e "${BLUE}  zap-only - ZAP deep vulnerability scan only${NC}"
    echo -e "${RED}💡 BEST FOR LOCAL TESTING: No bot protection, unlimited depth scanning${NC}"
    echo -e "${BLUE}Example: $0 http://localhost:3000 full${NC}"
    exit 1
fi

# Ensure URL has protocol
if [[ ! "$TARGET_URL" =~ ^https?:// ]]; then
    TARGET_URL="https://$TARGET_URL"
fi

echo -e "${PURPLE}🔐 GENERIC WEB APPLICATION SECURITY SCANNER${NC}"
echo -e "${PURPLE}==============================================${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${NC}"
echo -e "${BLUE}Mode: ${MODE}${NC}"
echo -e "${BLUE}Timestamp: $(date)${NC}"
echo -e "${BLUE}Output: Terminal display only${NC}"
if [ -n "$AUTH_TOKEN" ]; then
    echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
fi
if [ -n "$AUTH_COOKIE" ]; then
    echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
fi

# Auth status
if [ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ]; then
    echo -e "${YELLOW}ℹ️ Running without authentication (unauthenticated scan)${NC}"
fi

# Helper function for authenticated curl
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

echo -e ""

# Build curl options for authentication
CURL_AUTH_OPTS=""
if [ -n "$AUTH_TOKEN" ]; then
    CURL_AUTH_OPTS="-H \"Authorization: Bearer $AUTH_TOKEN\""
elif [ -n "$AUTH_COOKIE" ]; then
    CURL_AUTH_OPTS="--cookie \"$AUTH_COOKIE\""
fi

# Helper function for authenticated curl requests (supports cookie and bearer token)
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl --connect-timeout 10 --max-time 30 -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --connect-timeout 10 --max-time 30 --cookie "$AUTH_COOKIE" "$@"
    else
        curl --connect-timeout 10 --max-time 30 "$@"
    fi
}

# Error tracking - POSIX compatible
ERROR_COUNT=0
ERROR_SEEN=""  # Simple string to track seen errors

# Function to log results with error tracking
log_result() {
    echo "$1"
}

# Function to handle errors with detailed reporting - POSIX compatible
handle_error() {
    local error_msg="$1"
    local location="$2"
    local condition="$3"
    local fix_suggestion="$4"
    
    # Create unique error key to avoid duplicates
    local error_key="${location}:${error_msg}"
    
    # Check if we've seen this error before using simple string matching
    if echo "$ERROR_SEEN" | grep -q "$error_key"; then
        return 0  # Already reported this error
    fi
    
    # Add to seen errors list
    ERROR_SEEN="${ERROR_SEEN}|${error_key}"
    ERROR_COUNT=$((ERROR_COUNT + 1))
    
    echo -e "${RED}❌ ERROR #${ERROR_COUNT}: ${error_msg}${NC}"
    echo -e "${BLUE}   📍 LOCATION: ${location}${NC}"
    echo -e "${BLUE}   🔍 CONDITION: ${condition}${NC}"
    if [ -n "$fix_suggestion" ]; then
        echo -e "${YELLOW}   🔧 SUGGESTION: ${fix_suggestion}${NC}"
    fi
    echo ""
    
    log_result "ERROR #${ERROR_COUNT}: ${error_msg} at ${location}"
}

# Function to check if a command exists
check_command() {
    local cmd_name="$1"
    local description="$2"
    local location="$3"

    if ! command -v "$cmd_name" >/dev/null 2>&1; then
        handle_error "Command not found: $description" "$location" "Command '$cmd_name' is not installed" "Check if required tools are installed and accessible"
        return 1
    fi
    return 0
}

# Function to get detailed finding explanations
get_finding_details() {
    local finding_type="$1"
    
    case "$finding_type" in
        "HSTS")
            echo "🔒 MISSING: HTTP Strict Transport Security (HSTS)"
            echo "   RISK: Medium - Allows protocol downgrade attacks"
            echo "   WHAT: Forces browsers to use HTTPS, prevents man-in-the-middle attacks"
            echo "   FIX: Add header: Strict-Transport-Security: max-age=31536000; includeSubDomains"
            echo "   EXAMPLE: In Apache: Header always set Strict-Transport-Security \"max-age=31536000; includeSubDomains\""
            ;;
        "X-Frame-Options")
            echo "🔒 MISSING: X-Frame-Options Header"
            echo "   RISK: Medium - Vulnerable to clickjacking attacks"
            echo "   WHAT: Prevents page from being embedded in frames/iframes"
            echo "   FIX: Add header: X-Frame-Options: DENY or X-Frame-Options: SAMEORIGIN"
            echo "   EXAMPLE: In nginx: add_header X-Frame-Options \"SAMEORIGIN\" always;"
            ;;
        "X-Content-Type-Options")
            echo "🔒 MISSING: X-Content-Type-Options Header"
            echo "   RISK: Low - MIME type sniffing vulnerabilities"
            echo "   WHAT: Prevents browsers from MIME-sniffing responses"
            echo "   FIX: Add header: X-Content-Type-Options: nosniff"
            echo "   EXAMPLE: In Apache: Header always set X-Content-Type-Options \"nosniff\""
            ;;
        "X-XSS-Protection")
            echo "🔒 MISSING: X-XSS-Protection Header"
            echo "   RISK: Low - Legacy XSS protection (mostly deprecated)"
            echo "   WHAT: Enables browser XSS filtering (use CSP instead)"
            echo "   FIX: Add header: X-XSS-Protection: 1; mode=block"
            echo "   BETTER: Implement Content-Security-Policy instead"
            ;;
        "Content-Security-Policy")
            echo "🔒 MISSING: Content Security Policy (CSP)"
            echo "   RISK: Medium - No protection against XSS/injection attacks"
            echo "   WHAT: Controls which resources browser can load (scripts, styles, etc.)"
            echo "   FIX: Add header: Content-Security-Policy: default-src 'self'"
            echo "   EXAMPLE: Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'"
            ;;
        "Referrer-Policy")
            echo "🔒 MISSING: Referrer-Policy Header"
            echo "   RISK: Low - Information leakage via referrer"
            echo "   WHAT: Controls what referrer information is sent with requests"
            echo "   FIX: Add header: Referrer-Policy: strict-origin-when-cross-origin"
            echo "   EXAMPLE: In nginx: add_header Referrer-Policy \"strict-origin-when-cross-origin\" always;"
            ;;
        "CORS-Wildcard")
            echo "🚨 FOUND: CORS Wildcard Origin (*)"
            echo "   RISK: High - Allows any website to access your API"
            echo "   WHAT: Access-Control-Allow-Origin: * allows all domains"
            echo "   FIX: Specify exact origins: Access-Control-Allow-Origin: https://yourdomain.com"
            echo "   NEVER: Use '*' with credentials (Access-Control-Allow-Credentials: true)"
            ;;
        "XSS-Reflected")
            echo "🚨 FOUND: Reflected XSS Vulnerability"
            echo "   RISK: High - Remote code execution in user browsers"
            echo "   WHAT: User input reflected in HTML without proper escaping"
            echo "   FIX: Sanitize ALL user input, use output encoding, implement CSP"
            echo "   EXAMPLE: HTML encode: &lt;script&gt; instead of <script>"
            ;;
        "SQL-Injection")
            echo "🚨 FOUND: SQL Injection Vulnerability"
            echo "   RISK: High - Database compromise, data theft"
            echo "   WHAT: User input directly concatenated into SQL queries"
            echo "   FIX: Use parameterized queries/prepared statements ALWAYS"
            echo "   EXAMPLE: SELECT * FROM users WHERE id = ? (NOT WHERE id = '$user_input')"
            ;;
        "Info-Robots")
            echo "ℹ️ INFO: robots.txt File Accessible"
            echo "   RISK: Info - May reveal sensitive paths/directories"
            echo "   WHAT: robots.txt tells search engines what NOT to crawl"
            echo "   REVIEW: Check if it reveals admin panels, backup directories, etc."
            echo "   ACTION: Ensure no sensitive paths are listed in robots.txt"
            ;;
        "Info-Git")
            echo "🚨 FOUND: .git Directory Accessible"
            echo "   RISK: High - Source code disclosure"
            echo "   WHAT: Git repository accessible via web browser"
            echo "   FIX: Block access to .git in web server config"
            echo "   APACHE: <DirectoryMatch \"^\\.git\"> Require all denied </DirectoryMatch>"
            ;;
        "Info-Env")
            echo "🚨 FOUND: .env File Accessible"
            echo "   RISK: High - Environment variables/secrets exposed"
            echo "   WHAT: Environment configuration file accessible"
            echo "   FIX: Block access to .env files in web server"
            echo "   NGINX: location ~ /\\.env { deny all; }"
            ;;
    esac
}

# Function to run a test with detailed reporting
run_test() {
    local test_name="$1"
    local test_input="$2"
    local success_pattern="$3"
    local failure_is_good="${4:-false}"  # Some tests should fail for security
    local finding_type="${5:-}"  # For detailed explanations

    TESTS_TOTAL=$((TESTS_TOTAL + 1))

    echo -n "Testing: $test_name... "

    # Use direct string matching instead of eval to prevent command injection
    result="$test_input"
    
    # Check result
    local test_passed=false
    if [ "$failure_is_good" = "true" ]; then
        # For security tests where failure indicates good security
        if ! echo "$result" | grep -qi "$success_pattern"; then
            test_passed=true
        fi
    else
        # Normal tests where success pattern indicates pass
        if echo "$result" | grep -qi "$success_pattern"; then
            test_passed=true
        fi
    fi
    
    if [ "$test_passed" = "true" ]; then
        echo -e "${GREEN}PASS${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        log_result "✅ PASS: $test_name"
    else
        echo -e "${RED}FAIL${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        log_result "❌ FAIL: $test_name"
        
        # Add detailed explanation if finding type provided
        if [ -n "$finding_type" ]; then
            echo -e "${YELLOW}$(get_finding_details "$finding_type" | head -1)${NC}"
            get_finding_details "$finding_type" | tail -n +2
            echo ""
        else
            log_result "   Result: $result"
        fi
    fi
}

# Function to check if ZAP is running
check_zap_status() {
    # Try to get ZAP version - this confirms ZAP is fully ready
    local response=$(curl -s --connect-timeout 5 --max-time 10 "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/version/" 2>/dev/null)
    
    # Check if we got a valid JSON response with version info
    if echo "$response" | grep -q '"version"'; then
        return 0  # ZAP is ready
    else
        return 1  # ZAP not ready or not responding
    fi
}

# Function to start ZAP if needed
start_zap_if_needed() {
    if ! check_zap_status; then
        echo -e "${YELLOW}🕷️ Starting OWASP ZAP...${NC}"
        echo -e "${CYAN}💡 Note: First startup on macOS can take 3-5 minutes${NC}"
        
        # Try to start ZAP with visible output for better debugging
        if [ -x "/Applications/ZAP.app/Contents/Java/zap.sh" ]; then
            echo -e "${BLUE}🚀 Launching ZAP daemon on port $ZAP_PORT...${NC}"
            /Applications/ZAP.app/Contents/Java/zap.sh -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
            ZAP_PID=$!
        elif command -v zap.sh >/dev/null 2>&1; then
            echo -e "${BLUE}🚀 Launching ZAP daemon on port $ZAP_PORT...${NC}"
            zap.sh -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
            ZAP_PID=$!
        elif command -v owasp-zap >/dev/null 2>&1; then
            echo -e "${BLUE}🚀 Launching ZAP daemon on port $ZAP_PORT...${NC}"
            owasp-zap -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
            ZAP_PID=$!
        else
            echo -e "${RED}❌ OWASP ZAP not found. Please install ZAP or run manually on port $ZAP_PORT${NC}"
            return 1
        fi
        
        # Show startup progress
        echo -e "${YELLOW}🔄 ZAP is initializing (this may take several minutes on Apple Silicon Macs)...${NC}"
        echo -n "Progress: "
        
        # Extended timeout for macOS (especially Apple Silicon) - up to 5 minutes
        local max_attempts=150  # 150 × 2s = 5 minutes
        local attempt=0
        
        while [ $attempt -lt $max_attempts ]; do
            if check_zap_status; then
                echo -e "\n${GREEN}✅ ZAP daemon is ready! (took $((attempt * 2)) seconds)${NC}"
                return 0
            fi
            
            # Show progress every 10 attempts (20 seconds)
            if [ $((attempt % 10)) -eq 0 ]; then
                local elapsed=$((attempt * 2))
                echo -n "${elapsed}s "
                
                # Show what ZAP is doing (last few lines of log)
                if [ -f "/tmp/zap_startup.log" ]; then
                    local last_line=$(tail -1 /tmp/zap_startup.log 2>/dev/null | grep -o "INFO.*" | head -c 60)
                    if [ -n "$last_line" ]; then
                        echo -e "\n${CYAN}📝 Status: $last_line...${NC}"
                        echo -n "Progress: "
                    fi
                fi
            else
                echo -n "."
            fi
            
            # Check if ZAP process is still running
            if ! kill -0 $ZAP_PID 2>/dev/null; then
                echo -e "\n${RED}❌ ZAP process died during startup${NC}"
                echo -e "${YELLOW}📋 Startup log (last 20 lines):${NC}"
                tail -20 /tmp/zap_startup.log 2>/dev/null || echo "No log available"
                return 1
            fi
            
            attempt=$((attempt + 1))
            sleep 2
        done
        
        echo -e "\n${RED}❌ ZAP failed to start within timeout (${max_attempts} attempts, $((max_attempts * 2)) seconds)${NC}"
        echo -e "${YELLOW}📋 ZAP startup log (last 30 lines):${NC}"
        tail -30 /tmp/zap_startup.log 2>/dev/null || echo "No log available"
        echo -e "${BLUE}💡 Try running ZAP manually: zap.sh -daemon -port $ZAP_PORT -config api.disablekey=true${NC}"
        return 1
    else
        echo -e "${GREEN}✅ ZAP is already running${NC}"
    fi
}

# Function to run basic security header tests
run_security_headers_test() {
    local phase_display=${PHASE_NUM:-1}
    echo -e "\n${YELLOW}🛡️ PHASE $phase_display: SECURITY HEADERS ANALYSIS${NC}"
    
    # Test for important security headers with error handling
    echo -e "${BLUE}📡 Fetching HTTP headers from target (following redirects)...${NC}"
    # Get ALL headers from redirect chain for security header analysis
    all_headers=$(auth_curl -sIL "$TARGET_URL" 2>/dev/null)
    
    # For security headers, we need to check ALL responses in the redirect chain
    # because headers like HSTS can appear in redirect responses
    headers="$all_headers"
    
    if [ $? -ne 0 ] || [ -z "$headers" ]; then
        handle_error "Failed to fetch HTTP headers from target" "run_security_headers_test():header_fetch" "curl -sI '$TARGET_URL' failed or returned empty response" "Check if target URL is accessible, verify network connectivity"
        return 1
    fi
    
    echo -e "${BLUE}✅ Successfully retrieved headers from target${NC}"
    
    # Check for HSTS (only for HTTPS sites - HTTP doesn't need HSTS)
    if [[ "$TARGET_URL" == https://* ]]; then
        run_test "HSTS Header Present" "$headers" "Strict-Transport-Security" "false" "HSTS"
        if ! echo "$headers" | grep -qi "Strict-Transport-Security"; then
            LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
            LOW_RISK_LIST+=("Missing HSTS Header")
        fi
    else
        echo -n "Testing: HSTS Header Present... "
        echo -e "${BLUE}SKIP${NC}"
        echo -e "${GRAY}ℹ️  SKIP: HSTS not applicable to HTTP sites (use HTTPS for HSTS)${NC}"
        echo ""
    fi
    
    # Check for X-Frame-Options
    run_test "X-Frame-Options Header" "$headers" "X-Frame-Options" "false" "X-Frame-Options"
    if ! echo "$headers" | grep -qi "X-Frame-Options"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing X-Frame-Options Header")
    fi
    
    # Check for X-Content-Type-Options
    run_test "X-Content-Type-Options Header" "$headers" "X-Content-Type-Options" "false" "X-Content-Type-Options"
    if ! echo "$headers" | grep -qi "X-Content-Type-Options"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing X-Content-Type-Options Header")
    fi
    
    # Check for X-XSS-Protection
    run_test "X-XSS-Protection Header" "$headers" "X-XSS-Protection" "false" "X-XSS-Protection"
    if ! echo "$headers" | grep -qi "X-XSS-Protection"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing X-XSS-Protection Header")
    fi
    
    # Check for Content-Security-Policy
    run_test "Content-Security-Policy Header" "$headers" "Content-Security-Policy" "false" "Content-Security-Policy"
    if ! echo "$headers" | grep -qi "Content-Security-Policy"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing Content-Security-Policy Header")
    fi
    
    # Check for Referrer-Policy
    run_test "Referrer-Policy Header" "$headers" "Referrer-Policy" "false" "Referrer-Policy"
    if ! echo "$headers" | grep -qi "Referrer-Policy"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing Referrer-Policy Header")
    fi
    
    # Check server header (should be minimal)
    if echo "$headers" | grep -i "server:" | grep -qE "(Apache|nginx|IIS|lighttpd)" 2>/dev/null; then
        echo -e "${YELLOW}⚠️ Server header reveals server type${NC}"
        log_result "⚠️ WARNING: Server header reveals server type"
    fi
    
    # Context note about header importance
    echo -e "\n${CYAN}ℹ️  NOTE: Well-configured sites (like GitHub) have all security headers.${NC}"
    echo -e "${CYAN}   Missing headers indicate opportunities to improve security posture.${NC}"
}

# Function to run CORS tests
run_cors_tests() {
    local phase_display=${PHASE_NUM:-2}
    echo -e "\n${YELLOW}🌐 PHASE $phase_display: CORS SECURITY TESTS${NC}"
    
    # Test CORS with evil origin - check both OPTIONS preflight AND main GET response
    evil_origin="https://evil-site.com"
    
    # Test OPTIONS preflight request
    cors_preflight_response=$(auth_curl -s -H "Origin: $evil_origin" \
        -H "Access-Control-Request-Method: POST" \
        -X OPTIONS "$TARGET_URL" 2>/dev/null || echo "")

    # Test main GET request with evil origin
    cors_main_response=$(auth_curl -s -I -H "Origin: $evil_origin" "$TARGET_URL" 2>/dev/null || echo "")
    
    # Check if CORS allows any origin (bad) - check both responses
    cors_wildcard_found=false
    if echo "$cors_preflight_response" | grep -qi "access-control-allow-origin:.*\*"; then
        cors_wildcard_found=true
    elif echo "$cors_main_response" | grep -qi "access-control-allow-origin:.*\*"; then
        cors_wildcard_found=true
    fi
    
    if [ "$cors_wildcard_found" = "true" ]; then
        echo -e "${RED}FAIL: CORS Wildcard Protection${NC}"
        echo -e "${YELLOW}🚨 FOUND: CORS Wildcard Origin (*)${NC}"
        echo "   RISK: High - Allows any website to access your API"
        echo "   WHAT: Access-Control-Allow-Origin: * allows all domains"
        echo "   FIX: Specify exact origins: Access-Control-Allow-Origin: https://yourdomain.com"
        echo "   NEVER: Use '*' with credentials (Access-Control-Allow-Credentials: true)"
        echo "   DETECTED IN: $(echo "$cors_main_response" | grep -i "access-control-allow-origin" || echo "preflight response")"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("CORS Wildcard Origin (*)")
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN}PASS: CORS Wildcard Protection${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        log_result "✅ PASS: CORS Wildcard Protection"
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test if evil origin is accepted (bad) - check both responses
    evil_origin_accepted=false
    if echo "$cors_preflight_response" | grep -q "$evil_origin"; then
        evil_origin_accepted=true
    elif echo "$cors_main_response" | grep -q "$evil_origin"; then
        evil_origin_accepted=true
    fi
    
    if [ "$evil_origin_accepted" = "true" ]; then
        echo -e "${RED}FAIL: Evil Origin Rejection${NC}"
        echo -e "${YELLOW}🚨 FOUND: Evil Origin Accepted${NC}"
        echo "   RISK: High - Malicious websites can access your resources"
        echo "   WHAT: Server accepts requests from unauthorized origin: $evil_origin"
        echo "   FIX: Implement proper origin whitelist checking"
        echo "   EXAMPLE: Only allow trusted domains like https://yourdomain.com"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Evil Origin Accepted")
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN}PASS: Evil Origin Rejection${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        log_result "✅ PASS: Evil Origin Rejection"
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test common endpoints for CORS issues
    for endpoint in "/api" "/admin" "/login" "/auth"; do
        cors_test=$(auth_curl -s -H "Origin: $evil_origin" -X OPTIONS "${TARGET_URL}${endpoint}" 2>/dev/null || echo "")
        if echo "$cors_test" | grep -q "Access-Control-Allow-Origin"; then
            log_result "⚠️ CORS headers found on: ${endpoint}"
        fi
    done
}

# Function to run XSS tests
run_xss_tests() {
    local phase_display=${PHASE_NUM:-3}
    echo -e "\n${YELLOW}⚡ PHASE $phase_display: XSS VULNERABILITY TESTS${NC}"
    
    # Common XSS payloads
    xss_payloads=(
        "<script>alert('XSS')</script>"
        "javascript:alert('XSS')"
        "'\"><script>alert('XSS')</script>"
        "<img src=x onerror=alert('XSS')>"
        "<svg onload=alert('XSS')>"
    )
    
    # Test common parameters
    common_params=("q" "search" "query" "name" "message" "comment" "text" "input")
    
    for param in "${common_params[@]}"; do
        for payload in "${xss_payloads[@]}"; do
            # URL encode payload
            encoded_payload=$(echo "$payload" | sed 's/ /%20/g' | sed 's/</%3C/g' | sed 's/>/%3E/g')
            
            # Test GET parameter
            response=$(auth_curl -s "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")

            # Check if payload is reflected unescaped
            if echo "$response" | grep -q "$payload"; then
                echo -e "${RED}🚨 Potential XSS in parameter: $param${NC}"
                echo -e "${YELLOW}🚨 FOUND: Reflected XSS Vulnerability in parameter '$param'${NC}"
                echo "   RISK: High - Remote code execution in user browsers"
                echo "   WHAT: User input '$param' reflected in HTML without proper escaping"
                echo "   PAYLOAD: $payload"
                echo "   FIX: Sanitize ALL user input, use output encoding, implement CSP"
                echo "   EXAMPLE: HTML encode: &lt;script&gt; instead of <script>"
                echo "   CSP: Content-Security-Policy: default-src 'self'; script-src 'self'"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("XSS Vulnerability in parameter '$param'")
                break
            fi
        done
    done
    
    run_test "XSS Protection Test" "Completed XSS parameter testing" "Completed"
}

# Function to run SQL injection tests
run_sql_injection_tests() {
    local phase_display=${PHASE_NUM:-4}
    echo -e "\n${YELLOW}💉 PHASE $phase_display: SQL INJECTION TESTS${NC}"
    
    # Common SQL injection payloads
    sql_payloads=(
        "' OR '1'='1"
        "'; DROP TABLE users; --"
        "' UNION SELECT null, null, null --"
        "admin'--"
        "' OR 1=1 --"
        "' WAITFOR DELAY '00:00:05' --"
    )
    
    # Common vulnerable parameters
    sql_params=("id" "user" "username" "email" "search" "category" "page" "limit")
    
    for param in "${sql_params[@]}"; do
        for payload in "${sql_payloads[@]}"; do
            # URL encode payload
            encoded_payload=$(echo "$payload" | sed "s/ /%20/g" | sed "s/'/%27/g")
            
            # Test parameter
            response=$(auth_curl -s "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")

            # Check for specific SQL error messages (more accurate detection)
            if echo "$response" | grep -qiE "(sql syntax|mysql_.*error|postgresql.*error|ora-[0-9]+|mssql.*error|sqlite.*error|you have an error in your sql|syntax error.*sql|duplicate entry|table.*doesn't exist|column.*unknown|invalid.*query)"; then
                echo -e "${RED}🚨 Potential SQL injection in parameter: $param${NC}"
                echo -e "${YELLOW}🚨 FOUND: SQL Injection Vulnerability in parameter '$param'${NC}"
                echo "   RISK: High - Database compromise, data theft possible"
                echo "   WHAT: User input '$param' causes SQL error - likely injectable"
                echo "   PAYLOAD: $payload"
                echo "   FIX: Use parameterized queries/prepared statements ALWAYS"
                echo "   EXAMPLE: SELECT * FROM users WHERE id = ? (NOT WHERE id = '\$user_input')"
                echo "   NEVER: Concatenate user input directly into SQL queries"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("SQL Injection in parameter '$param'")
                break
            fi
        done
    done
    
    run_test "SQL Injection Protection Test" "Completed SQL injection testing" "Completed"
}

# Function to run information disclosure tests
run_info_disclosure_tests() {
    local phase_display=${PHASE_NUM:-5}
    echo -e "\n${YELLOW}📋 PHASE $phase_display: INFORMATION DISCLOSURE TESTS${NC}"
    
    # High-confidence sensitive files/directories (reduced false positives)
    sensitive_paths=(
        "/.git"
        "/.env"
        "/phpmyadmin"
        "/admin"
    )
    
    for path in "${sensitive_paths[@]}"; do
        # Get both status code and content for better detection
        response=$(auth_curl -s "${TARGET_URL}${path}" 2>/dev/null || echo "ERROR")
        status_code=$(auth_curl -s -o /dev/null -w "%{http_code}" "${TARGET_URL}${path}" 2>/dev/null || echo "000")
        
        # Enhanced detection logic
        is_real_finding=false
        
        if [ "$status_code" = "200" ]; then
            case "$path" in
                "/.git")
                    # .git directory should contain git-specific content
                    if echo "$response" | grep -qi "objects\|refs\|HEAD"; then
                        is_real_finding=true
                    fi
                    ;;
                "/.env")
                    # .env files should contain environment variables
                    if echo "$response" | grep -qE "^[A-Z_]+=|APP_|DB_|API_"; then
                        is_real_finding=true
                    fi
                    ;;
                "/phpmyadmin")
                    # phpMyAdmin should have specific login interface elements
                    if echo "$response" | grep -qi "pma_username\|pma_password\|phpmyadmin.*login\|server.*choice" && \
                       ! echo "$response" | grep -qi "github\|repository"; then
                        is_real_finding=true
                    fi
                    ;;
                "/admin")
                    # Admin interface should have login form AND not be a generic site page
                    if echo "$response" | grep -qi "admin.*login\|admin.*password\|admin.*dashboard" && \
                       echo "$response" | grep -qi "<input.*password\|<form.*login" && \
                       ! echo "$response" | grep -qi "github\|repository"; then
                        is_real_finding=true
                    fi
                    ;;
            esac
        fi
        
        if [ "$is_real_finding" = "true" ]; then
            echo -e "${YELLOW}⚠️ Accessible: $path (HTTP $status_code)${NC}"
            
            # Provide detailed explanations based on path type
            case "$path" in
                "/.git")
                    get_finding_details "Info-Git"
                    echo ""
                    HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                    HIGH_RISK_LIST+=(".git directory accessible")
                    ;;
                "/.env")
                    get_finding_details "Info-Env"
                    echo ""
                    HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                    HIGH_RISK_LIST+=(".env file accessible")
                    ;;
                "/admin"|"/phpmyadmin")
                    echo "🚨 FOUND: Admin Interface Accessible"
                    echo "   RISK: Medium - Admin interface exposed to public"
                    echo "   WHAT: Administrative interface accessible without authentication"
                    echo "   FIX: Restrict access by IP, implement proper authentication"
                    echo "   EXAMPLE: Allow only from trusted IPs (10.0.0.0/8, office IPs)"
                    echo ""
                    MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                    MEDIUM_RISK_LIST+=("Admin interface accessible: $path")
                    ;;
                *)
                    MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                    MEDIUM_RISK_LIST+=("Sensitive content accessible: $path")
                    ;;
            esac
        elif [ "$status_code" = "403" ]; then
            echo -e "${CYAN}🔒 Forbidden: $path (HTTP $status_code)${NC}"
            log_result "✅ PASS: Path properly protected: $path"
        fi
    done
    
    run_test "Information Disclosure Test" "Completed path enumeration" "Completed"
}

# Function to run ZAP scan
run_zap_scan() {
    local phase_display=${PHASE_NUM:-6}
    echo -e "\n${YELLOW}🕷️ PHASE $phase_display: OWASP ZAP VULNERABILITY SCAN${NC}"
    
    if ! start_zap_if_needed; then
        echo -e "${RED}❌ Cannot run ZAP scan - ZAP not available${NC}"
        return 1
    fi
    
    # Use reliable bash-based ZAP scan (no Python dependencies)
    echo -e "${CYAN}🔍 Running comprehensive ZAP vulnerability scan via REST API${NC}"
    echo -e "${BLUE}📋 Features: Spider crawling, passive analysis, active scanning, terminal results${NC}"
    run_basic_zap_scan
}

# Function to run basic ZAP scan (original functionality)
run_basic_zap_scan() {
    echo "🔍 Starting basic vulnerability scan..."
    
    # Set up ZAP context
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/context/action/newContext/?contextName=GenericTest" >/dev/null 2>&1
    
    # Add target to context (comprehensive coverage for local testing)
    ENCODED_URL=$(url_encode "$TARGET_URL")
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/context/action/includeInContext/?contextName=GenericTest&regex=${ENCODED_URL}.*" >/dev/null 2>&1
    
    # Configure ZAP for DEEP PENETRATION TESTING (comprehensive security analysis)
    echo "🔥 Configuring ZAP for MAXIMUM THOROUGHNESS deep penetration testing..."
    echo -e "${RED}💀 EXTREME MODE: Maximum attack strength, extended timeouts, comprehensive coverage${NC}"
    echo -e "${BLUE}💡 This will take significantly longer but find more vulnerabilities${NC}"
    
    # SPIDER CONFIGURATION - MAXIMUM THOROUGHNESS (Deep Penetration Testing)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxDepth/?Integer=10" >/dev/null 2>&1        # Max 10 levels deep (was 6)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxChildren/?Integer=100" >/dev/null 2>&1    # Max 100 URLs per node (was 50)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxDuration/?Integer=900" >/dev/null 2>&1     # Max 15 minutes spider (was 5)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxParseSizeBytes/?Integer=52428800" >/dev/null 2>&1  # 50MB per page (was 10MB)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionRequestWaitTime/?Integer=200" >/dev/null 2>&1  # Slower but thorough (was 100ms)
    
    # Add URL exclusions to prevent noise (Next.js, static files, etc.)
    # Note: Using core exclusions instead of spider-specific ones for broader coverage
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*\.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot|pdf)(\\\?.*)?$" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*/_next/static/.*" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*\\\?ts=[0-9]+.*" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*\.(git|svn).*" >/dev/null 2>&1
    
    # Enable essential spidering techniques (balanced approach)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionProcessForm/?Boolean=true" >/dev/null 2>&1   # Process forms
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionPostForm/?Boolean=false" >/dev/null 2>&1     # Don't auto-submit forms (reduces noise)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseComments/?Boolean=true" >/dev/null 2>&1 # Parse HTML comments
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseRobotsTxt/?Boolean=true" >/dev/null 2>&1 # Check robots.txt
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseSitemapXml/?Boolean=false" >/dev/null 2>&1 # Skip sitemap (can be huge)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseSVNEntries/?Boolean=false" >/dev/null 2>&1 # Skip .svn (rarely needed)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseGit/?Boolean=false" >/dev/null 2>&1      # Skip .git (can generate many URLs)
    
    # ACTIVE SCAN CONFIGURATION - MAXIMUM THOROUGHNESS (Deep Penetration Testing)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionDelayInMs/?Integer=300" >/dev/null 2>&1        # Slower, more thorough (was 100ms)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionThreadPerHost/?Integer=5" >/dev/null 2>&1      # 5 concurrent threads (was 3)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionMaxRuleDurationInMins/?Integer=10" >/dev/null 2>&1  # Max 10 min per rule (was 2)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionMaxScanDurationInMins/?Integer=60" >/dev/null 2>&1  # Max 60 min total (was 10)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionMaxResultsToList/?Integer=500" >/dev/null 2>&1   # More results (was 100)
    
    # Enable MAXIMUM vulnerability detection (thoroughness over speed)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionAttackStrength/?String=HIGH" >/dev/null 2>&1   # HIGH attack strength (was MEDIUM)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionAlertThreshold/?String=LOW" >/dev/null 2>&1    # LOW threshold for maximum detection (was MEDIUM)
    
    # Spider the target (comprehensive discovery)
    echo "🕸️ Deep spidering - discovering ALL pages, forms, and endpoints..."
    spider_response=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/scan/?url=${TARGET_URL}" 2>/dev/null)
    
    # Enhanced error checking for spider scan
    if [ $? -ne 0 ]; then
        handle_error "Failed to connect to ZAP for spider scan" "run_basic_zap_scan():spider_start" "curl command failed to reach ZAP API" "Ensure ZAP is running on ${ZAP_HOST}:${ZAP_PORT}"
        return 1
    fi
    
    # Check if jq is available for JSON parsing
    if ! command -v jq >/dev/null 2>&1; then
        handle_error "jq not found - required for ZAP result parsing" "run_basic_zap_scan():json_parsing" "jq command not available in PATH" "Install jq: brew install jq (macOS) or apt-get install jq (Linux)"
        return 1
    fi
    
    spider_id=$(echo "$spider_response" | jq -r '.scan // empty' 2>/dev/null)
    
    if [ -z "$spider_id" ] || [ "$spider_id" = "empty" ] || [ "$spider_id" = "null" ]; then
        handle_error "Failed to start spider scan - invalid response from ZAP" "run_basic_zap_scan():spider_start" "ZAP returned: $spider_response" "Check ZAP logs, ensure target URL is accessible, verify ZAP API is working"
        return 1
    fi
    
    echo -e "${BLUE}🕸️ Started spider scan (ID: $spider_id)${NC}"
    
    # Debug mode - show ZAP URLs being tested
    if [ "${DEBUG:-}" = "1" ]; then
        echo -e "${GRAY}🔧 Debug: Testing ${TARGET_URL}${NC}"
        echo -e "${GRAY}🔧 Debug: ZAP running on ${ZAP_HOST}:${ZAP_PORT}${NC}"
    fi
    
    # Wait for spider to complete (unlimited - deep penetration testing)
    spider_elapsed=0
    echo -e "${BLUE}💡 Comprehensive discovery - will find ALL pages, forms, and hidden content${NC}"
    while true; do
        spider_status=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/status/?scanId=${spider_id}" | jq -r '.status // "0"' 2>/dev/null)
        if [ "$spider_status" = "100" ]; then
            break
        fi
        
        # Show detailed progress every 30 seconds
        if [ $((spider_elapsed % 30)) -eq 0 ] && [ $spider_elapsed -gt 0 ]; then
            urls_found=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | jq -r '.results | length // 0' 2>/dev/null)
            echo -e "\n${BLUE}🕸️ Deep spider progress: ${spider_status}% - Found ${urls_found} URLs (${spider_elapsed}s elapsed)${NC}"
        fi
        
        echo -n "."
        sleep 2
        spider_elapsed=$((spider_elapsed + 2))
    done
    
    # Get final spider results
    urls_found=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | jq -r '.results | length // 0' 2>/dev/null)
    echo -e " ${GREEN}✅ Deep spidering completed - ${urls_found} URLs discovered (${spider_elapsed}s)${NC}"
    
    # Show sample URLs for debugging (first 10)
    if [ "$urls_found" -gt 100 ]; then
        echo -e "${YELLOW}🔍 Sample URLs found (showing first 10 of ${urls_found}):${NC}"
        curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | \
        jq -r '.results[0:10][]?' 2>/dev/null | while read url; do
            echo -e "${GRAY}  • $url${NC}"
        done
        echo -e "${YELLOW}  ... and $((urls_found - 10)) more${NC}"
    fi
    
    # Export spider URLs for other scripts (if ZAP_URLS_FILE env var is set)
    if [ -n "${ZAP_URLS_FILE:-}" ]; then
        echo -e "${CYAN}📤 Exporting spider URLs to ${ZAP_URLS_FILE}${NC}"
        curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | \
            jq -r '.results[]?' > "$ZAP_URLS_FILE" 2>/dev/null
        local exported_count=$(wc -l < "$ZAP_URLS_FILE" 2>/dev/null | tr -d ' ')
        echo -e "${GREEN}📤 Exported ${exported_count} URLs for SSRF/file-upload testing${NC}"
    fi

    # Passive scan (automatic)
    echo "🔍 Running passive security scan..."
    sleep 5
    
    # Active scan with timeout and progress (skip if quick mode)
    if [ "$MODE" = "quick" ]; then
        echo -e "${BLUE}⚡ Skipping active scan in quick mode${NC}"
        echo -e "${BLUE}💡 Use 'full' mode for comprehensive vulnerability scanning${NC}"
    else
        echo "🔥 Running active vulnerability scan..."
        echo -e "${RED}💀 PENETRATION MODE: All 50+ security rules, HIGH attack strength, LOW threshold${NC}"
        echo -e "${RED}🚨 This can take 15+ minutes depending on the size of the website but will find ALL possible vulnerabilities${NC}"
        
        # Start comprehensive active scan (already configured above)
        active_scan_response=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/scan/?url=${TARGET_URL}" 2>/dev/null)
        
        # Enhanced error checking for active scan
        if [ $? -ne 0 ]; then
            handle_error "Failed to connect to ZAP for active scan" "run_basic_zap_scan():active_scan_start" "curl command failed to reach ZAP API" "Ensure ZAP is running and responsive on ${ZAP_HOST}:${ZAP_PORT}"
            return 1
        fi
        
        active_scan_id=$(echo "$active_scan_response" | jq -r '.scan // empty' 2>/dev/null)
        
        if [ -z "$active_scan_id" ] || [ "$active_scan_id" = "empty" ] || [ "$active_scan_id" = "null" ]; then
            handle_error "Failed to start active scan - invalid response from ZAP" "run_basic_zap_scan():active_scan_start" "ZAP returned: $active_scan_response" "Check if spider found any URLs to scan, verify ZAP active scan configuration"
            return 1
        fi
        
        echo -e "${BLUE}🔥 Started active scan (ID: $active_scan_id)${NC}"
        
        # Wait for comprehensive active scan to complete (unlimited - deep penetration)
        active_elapsed=0
        progress_interval=60  # Show detailed progress every minute
        last_progress=0
        
        echo -e "${BLUE}🔥 Comprehensive vulnerability analysis in progress...${NC}"
        echo -e "${BLUE}💡 Testing: XSS, SQLi, Command injection, Path traversal, XXE, CSRF, etc.${NC}"
        
        while true; do
            scan_status=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/view/status/?scanId=${active_scan_id}" | jq -r '.status // "0"' 2>/dev/null)
            if [ "$scan_status" = "100" ]; then
                break
            fi
            
            # Show detailed progress every minute
            if [ $((active_elapsed - last_progress)) -ge $progress_interval ]; then
                # Get current vulnerability counts using jq
                current_alerts=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/" | jq -r '.alerts | length // 0' 2>/dev/null)
                echo -e "\n${RED}🔍 Smart scan progress: ${scan_status}% - Found ${current_alerts} issues (${active_elapsed}s)${NC}"
                echo -e "${BLUE}💀 Testing: XSS, SQLi, Command injection, Path traversal, XXE, CSRF...${NC}"
                last_progress=$active_elapsed
            fi
            
            echo -n "."
            sleep 5
            active_elapsed=$((active_elapsed + 5))
        done
        
        # Get final results
        final_alerts=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/" | jq -r '.alerts | length // 0' 2>/dev/null)
        echo -e " ${GREEN}✅ Smart penetration scan completed - ${final_alerts} security issues found (${active_elapsed}s)${NC}"
    fi
    
    # Get comprehensive vulnerability findings for deep penetration testing
    echo -e "\n${PURPLE}🔍 DEEP PENETRATION TESTING RESULTS${NC}"
    echo -e "${PURPLE}===================================${NC}"
    
    alerts=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/")
    
    # Populate finding lists with actual ZAP results (deduplicated and filtered) - Fixed subshell issue
    # Use temporary files to avoid subshell array population issues
    # Enhanced filtering: consolidate CSP issues and skip duplicate noise
    high_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "High") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$high_findings" ]; then
        # Convert to array using word splitting (POSIX compatible)
        HIGH_RISK_LIST=()
        IFS=$'\n'
        for finding in $high_findings; do
            [ -n "$finding" ] && HIGH_RISK_LIST+=("$finding")
        done
        unset IFS
    fi

    medium_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "Medium") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$medium_findings" ]; then
        MEDIUM_RISK_LIST=()
        csp_count=0
        cookie_count=0
        IFS=$'\n'
        for finding in $medium_findings; do
            # Consolidate multiple CSP findings into one
            if echo "$finding" | grep -qi "content security policy\|CSP:"; then
                csp_count=$((csp_count + 1))
                if [ $csp_count -eq 1 ]; then
                    MEDIUM_RISK_LIST+=("CSP Configuration Issues")
                fi
                continue
            fi
            # Consolidate cookie-related findings
            if echo "$finding" | grep -qi "cookie"; then
                cookie_count=$((cookie_count + 1))
                if [ $cookie_count -eq 1 ]; then
                    MEDIUM_RISK_LIST+=("Cookie Security Issues")
                fi
                continue
            fi
            [ -n "$finding" ] && MEDIUM_RISK_LIST+=("$finding")
        done
        unset IFS
    fi

    low_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "Low") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$low_findings" ]; then
        LOW_RISK_LIST=()
        header_count=0
        IFS=$'\n'
        for finding in $low_findings; do
            # Consolidate multiple header findings
            if echo "$finding" | grep -qi "header\|x-content-type\|x-frame\|x-xss"; then
                header_count=$((header_count + 1))
                if [ $header_count -eq 1 ]; then
                    LOW_RISK_LIST+=("Security Header Issues")
                fi
                continue
            fi
            [ -n "$finding" ] && LOW_RISK_LIST+=("$finding")
        done
        unset IFS
    fi
    
    # Count risk levels using actual unique findings (not raw ZAP counts)
    HIGH_RISK_FINDINGS=${#HIGH_RISK_LIST[@]}
    MEDIUM_RISK_FINDINGS=${#MEDIUM_RISK_LIST[@]}
    LOW_RISK_FINDINGS=${#LOW_RISK_LIST[@]}
    
    # Filter noise from informational findings - Fixed subshell issue
    # Enhanced filtering to reduce ZAP noise
    info_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "Informational") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$info_findings" ]; then
        INFO_LIST=()
        IFS=$'\n'
        for finding in $info_findings; do
            # Skip noise findings that aren't valuable for security assessment
            skip_finding=false
            case "$finding" in
                "User Agent Fuzzer"*) skip_finding=true ;;
                "Modern Web Application"*) skip_finding=true ;;
                "Retrieved from Cache"*) skip_finding=true ;;
                "Storable and Cacheable Content"*) skip_finding=true ;;
                "Non-Storable Content"*) skip_finding=true ;;
                "Information Disclosure - Suspicious Comments"*) skip_finding=true ;;
                "Timestamp Disclosure"*) skip_finding=true ;;
                "Re-examine Cache-control Directives"*) skip_finding=true ;;
            esac
            if [ "$skip_finding" = "false" ] && [ -n "$finding" ]; then
                INFO_LIST+=("$finding")
            fi
        done
        unset IFS
    fi
    
    # Get informational findings count
    INFORMATIONAL_FINDINGS=${#INFO_LIST[@]}
    
    # Show detailed vulnerability breakdown with accurate counts and explanations
    if [ $HIGH_RISK_FINDINGS -gt 0 ]; then
        echo -e "${RED}🚨 HIGH RISK: ${HIGH_RISK_FINDINGS}${NC}"
    else
        echo -e "${GREEN}🚨 HIGH RISK: 0 ✅ (Excellent - No critical vulnerabilities found)${NC}"
    fi
    
    if [ $MEDIUM_RISK_FINDINGS -gt 0 ]; then
        echo -e "${YELLOW}⚠️  MEDIUM RISK: ${MEDIUM_RISK_FINDINGS}${NC}"
    else
        echo -e "${GREEN}⚠️  MEDIUM RISK: 0 ✅ (Good - No medium-risk vulnerabilities found)${NC}"
    fi
    
    if [ $LOW_RISK_FINDINGS -gt 0 ]; then
        echo -e "${BLUE}ℹ️  LOW RISK: ${LOW_RISK_FINDINGS}${NC}"
    else
        echo -e "${GREEN}ℹ️  LOW RISK: 0 ✅ (Good - No low-risk vulnerabilities found)${NC}"
    fi
    
    if [ $INFORMATIONAL_FINDINGS -gt 0 ]; then
        echo -e "${GRAY}📋 INFORMATIONAL: ${INFORMATIONAL_FINDINGS}${NC}"
    else
        echo -e "${GRAY}📋 INFORMATIONAL: 0 (No informational findings after noise filtering)${NC}"
    fi
    
    # Show top vulnerability types found
    if [ "$HIGH_RISK_FINDINGS" -gt 0 ] || [ "$MEDIUM_RISK_FINDINGS" -gt 0 ]; then
        echo -e "\n${RED}🔥 TOP VULNERABILITY CLASSES DETECTED:${NC}"
        
        # Extract and show unique vulnerability types using jq
        echo "$alerts" | jq -r '
            [.alerts[].name] 
            | group_by(.) 
            | map({name: .[0], count: length}) 
            | sort_by(.count) 
            | reverse 
            | .[0:10][] 
            | "  • \(.name) (\(.count) instances)"
        ' 2>/dev/null | while read line; do
            echo -e "${YELLOW}${line}${NC}"
        done || echo -e "${YELLOW}  • Unable to parse vulnerability details${NC}"
    fi
    
    echo -e "\n${GREEN}✅ Deep penetration testing completed${NC}"
    echo -e "${GREEN}📊 Total Security Issues Found: $((HIGH_RISK_FINDINGS + MEDIUM_RISK_FINDINGS + LOW_RISK_FINDINGS + INFORMATIONAL_FINDINGS))${NC}"
}

# Function to run CSRF protection tests
run_csrf_tests() {
    local phase_display=${PHASE_NUM:-7}
    echo -e "\n${YELLOW}🛡️ PHASE $phase_display: CSRF PROTECTION TESTING${NC}"

    # Test main page for forms with CSRF tokens
    main_page=$(auth_curl -s "$TARGET_URL" 2>/dev/null || echo "")

    # Check if forms exist
    if echo "$main_page" | grep -qi "<form"; then
        # Look for CSRF tokens in forms
        if echo "$main_page" | grep -qiE "csrf|_token|authenticity_token|__RequestVerificationToken"; then
            echo -e "${GREEN}✅ CSRF tokens detected in forms${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
            log_result "✅ PASS: CSRF tokens present in forms"
        else
            echo -e "${RED}❌ No CSRF tokens found in forms${NC}"
            echo -e "${YELLOW}🚨 POTENTIAL: Missing CSRF Protection${NC}"
            echo "   RISK: Medium - Forms vulnerable to Cross-Site Request Forgery"
            echo "   WHAT: Forms without CSRF tokens can be submitted by malicious websites"
            echo "   FIX: Add CSRF tokens to all forms and validate on server-side"
            echo ""
            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
            MEDIUM_RISK_LIST+=("Missing CSRF Protection in Forms")
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
    else
        echo -e "${BLUE}ℹ️  No forms detected on main page${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))

    # Check SameSite cookie attributes
    cookie_response=$(auth_curl -s -I "$TARGET_URL" 2>/dev/null || echo "")
    if echo "$cookie_response" | grep -qi "set-cookie:"; then
        if echo "$cookie_response" | grep -qi "samesite=strict\|samesite=lax"; then
            echo -e "${GREEN}✅ SameSite cookie attributes detected${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
            log_result "✅ PASS: SameSite cookie attributes present"
        else
            echo -e "${YELLOW}⚠️ SameSite cookie attributes missing${NC}"
            echo "   RISK: Low - Cookies vulnerable to CSRF attacks"
            echo "   FIX: Add SameSite=Strict or SameSite=Lax to all cookies"
            LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
            LOW_RISK_LIST+=("Missing SameSite Cookie Attributes")
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
    else
        echo -e "${BLUE}ℹ️  No cookies set by server${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
}

# Function to run path traversal tests
run_path_traversal_tests() {
    local phase_display=${PHASE_NUM:-8}
    echo -e "\n${YELLOW}📁 PHASE $phase_display: PATH TRAVERSAL DETECTION${NC}"

    # Path traversal payloads
    traversal_payloads=(
        "../../../etc/passwd"
        "..%5C..%5C..%5Cwindows%5Csystem32%5Cdrivers%5Cetc%5Chosts"
        "....//....//....//etc/passwd"
        "%2e%2e%2f%2e%2e%2f%2e%2e%2fetc%2fpasswd"
    )

    # Common vulnerable parameters
    traversal_params=("file" "path" "page" "include" "template" "doc" "view")

    local vulnerabilities_found=0

    for param in "${traversal_params[@]}"; do
        for payload in "${traversal_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed 's/\//%2F/g')
            response=$(auth_curl -s "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")

            # Check for Unix/Linux system files
            if echo "$response" | grep -qE "root:.*:/bin/(bash|sh)|daemon:.*:/sbin"; then
                echo -e "${RED}🚨 Path traversal vulnerability in parameter: $param${NC}"
                echo "VULN_DETAILS: PATH_TRAVERSAL | Parameter: $param | Payload: $payload"
                echo "   RISK: High - Unauthorized file system access detected"
                echo "   EVIDENCE: Unix passwd file content found"
                echo "   FIX: Validate and sanitize file path parameters, use whitelist approach"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Path Traversal in parameter '$param'")
                vulnerabilities_found=1
                break
            fi

            # Check for Windows system files
            if echo "$response" | grep -qE "127\.0\.0\.1.*localhost|# Copyright.*Microsoft"; then
                echo -e "${RED}🚨 Path traversal vulnerability in parameter: $param${NC}"
                echo "VULN_DETAILS: PATH_TRAVERSAL | Parameter: $param | Payload: $payload"
                echo "   RISK: High - Unauthorized file system access detected"
                echo "   EVIDENCE: Windows hosts file content found"
                echo "   FIX: Validate and sanitize file path parameters, use whitelist approach"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Path Traversal in parameter '$param'")
                vulnerabilities_found=1
                break
            fi
        done
        if [ $vulnerabilities_found -eq 1 ]; then
            break
        fi
    done

    run_test "Path Traversal Protection Test" "Completed path traversal testing" "Completed"
}

# Function to run open redirect tests
run_open_redirect_tests() {
    local phase_display=${PHASE_NUM:-9}
    echo -e "\n${YELLOW}🔀 PHASE $phase_display: OPEN REDIRECT TESTING${NC}"

    # Redirect payloads
    redirect_payloads=(
        "https://evil.com"
        "http://malicious-site.com"
        "//evil.com"
        "https://google.com"
    )

    # Common redirect parameters
    redirect_params=("url" "redirect" "return" "next" "goto" "continue" "target" "destination")

    local vulnerabilities_found=0

    for param in "${redirect_params[@]}"; do
        for payload in "${redirect_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed 's/:/%3A/g' | sed 's/\//%2F/g')

            # Get response headers to check for redirects
            response=$(auth_curl -s -I "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")

            # Check for redirect status codes and location header
            if echo "$response" | grep -qE "HTTP/[0-9]\.[0-9] 30[12]"; then
                location_header=$(echo "$response" | grep -i "location:" | head -1)
                if echo "$location_header" | grep -qF "$payload"; then
                    echo -e "${RED}🚨 Open redirect vulnerability in parameter: $param${NC}"
                    echo "VULN_DETAILS: OPEN_REDIRECT | Parameter: $param | Redirect: $payload"
                    echo "   RISK: Medium - Unvalidated redirect to external domain"
                    echo "   EVIDENCE: Location header contains external URL"
                    echo "   FIX: Validate redirect URLs against whitelist, use relative URLs"
                    echo ""
                    MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                    MEDIUM_RISK_LIST+=("Open Redirect in parameter '$param'")
                    vulnerabilities_found=1
                    break
                fi
            fi
        done
        if [ $vulnerabilities_found -eq 1 ]; then
            break
        fi
    done

    run_test "Open Redirect Protection Test" "Completed redirect testing" "Completed"
}

# Function to run SSL/TLS tests
run_ssl_tests() {
    local phase_display=${PHASE_NUM:-10}
    echo -e "\n${YELLOW}🔒 PHASE $phase_display: SSL/TLS CONFIGURATION TESTING${NC}"

    # Only test HTTPS URLs
    if [[ "$TARGET_URL" =~ ^https:// ]]; then
        # Extract domain from URL
        domain=$(echo "$TARGET_URL" | sed 's|https\?://||' | cut -d'/' -f1 | cut -d':' -f1)

        # Test SSL certificate with OpenSSL
        if command -v openssl >/dev/null 2>&1; then
            # Test certificate validity with timeout
            cert_info=$(echo "" | gtimeout 10 openssl s_client -servername "$domain" -connect "$domain:443" 2>/dev/null || echo "" | timeout 10 openssl s_client -servername "$domain" -connect "$domain:443" 2>/dev/null || echo | openssl s_client -servername "$domain" -connect "$domain:443" 2>/dev/null | head -20)

            if echo "$cert_info" | grep -q "Verify return code: 0 (ok)"; then
                echo -e "${GREEN}✅ SSL certificate is valid${NC}"
                TESTS_PASSED=$((TESTS_PASSED + 1))
                log_result "✅ PASS: SSL certificate valid"
            else
                echo -e "${YELLOW}⚠️ SSL certificate issues detected${NC}"
                verify_error=$(echo "$cert_info" | grep "Verify return code:" | head -1)
                echo "   ISSUE: $verify_error"
                echo "   RISK: Medium - SSL certificate validation problems"
                echo "   FIX: Ensure valid, trusted SSL certificate is properly configured"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("SSL Certificate Validation Issues")
                TESTS_FAILED=$((TESTS_FAILED + 1))
            fi
            TESTS_TOTAL=$((TESTS_TOTAL + 1))

            # Check TLS version support
            if echo "$cert_info" | grep -q "Protocol.*TLSv1\.[23]"; then
                echo -e "${GREEN}✅ Modern TLS version supported (TLS 1.2+)${NC}"
                TESTS_PASSED=$((TESTS_PASSED + 1))
                log_result "✅ PASS: Modern TLS version"
            else
                echo -e "${YELLOW}⚠️ Weak TLS version detected${NC}"
                protocol=$(echo "$cert_info" | grep "Protocol:" | head -1)
                echo "   DETECTED: $protocol"
                echo "   RISK: Medium - Weak TLS version vulnerable to attacks"
                echo "   FIX: Disable TLS 1.0/1.1, enable only TLS 1.2/1.3"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("Weak TLS Version Support")
                TESTS_FAILED=$((TESTS_FAILED + 1))
            fi
            TESTS_TOTAL=$((TESTS_TOTAL + 1))

            # Check for weak ciphers
            if echo "$cert_info" | grep -qiE "cipher.*rc4|cipher.*des|cipher.*md5"; then
                echo -e "${RED}❌ Weak cipher suites detected${NC}"
                weak_cipher=$(echo "$cert_info" | grep -iE "cipher.*rc4|cipher.*des|cipher.*md5" | head -1)
                echo "   DETECTED: $weak_cipher"
                echo "   RISK: High - Weak encryption vulnerable to attacks"
                echo "   FIX: Disable weak ciphers (RC4, DES, MD5), use strong AES ciphers"
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Weak Cipher Suites")
                TESTS_FAILED=$((TESTS_FAILED + 1))
            else
                echo -e "${GREEN}✅ Strong cipher suites in use${NC}"
                TESTS_PASSED=$((TESTS_PASSED + 1))
                log_result "✅ PASS: Strong cipher suites"
            fi
            TESTS_TOTAL=$((TESTS_TOTAL + 1))
        else
            echo -e "${BLUE}ℹ️  OpenSSL not available, skipping detailed SSL tests${NC}"
        fi
    else
        echo -e "${BLUE}ℹ️  Target uses HTTP, SSL/TLS tests not applicable${NC}"
        echo -e "${YELLOW}⚠️ Consider upgrading to HTTPS for secure communication${NC}"
    fi
}

# Function to run command injection tests
run_command_injection_tests() {
    local phase_display=${PHASE_NUM:-11}
    echo -e "\n${YELLOW}💻 PHASE $phase_display: COMMAND INJECTION TESTING${NC}"

    # Command injection payloads
    command_payloads=(
        "; whoami"
        "&& id"
        "| cat /etc/passwd"
        "; ping -c 1 127.0.0.1"
        "&& sleep 5"
    )

    # Common vulnerable parameters
    command_params=("cmd" "exec" "system" "ping" "host" "ip" "domain" "command")

    local vulnerabilities_found=0

    for param in "${command_params[@]}"; do
        for payload in "${command_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed 's/ /%20/g' | sed 's/;/%3B/g' | sed 's/|/%7C/g')

            # Time the request for sleep-based detection
            start_time=$(date +%s)
            response=$(auth_curl -s --max-time 15 "${TARGET_URL}?${param}=test${encoded_payload}" 2>/dev/null || echo "")
            end_time=$(date +%s)
            response_time=$((end_time - start_time))

            # Check for command output in response
            if echo "$response" | grep -qiE "uid=|gid=|root:|daemon:|PING.*127\.0\.0\.1"; then
                echo -e "${RED}🚨 Command injection vulnerability in parameter: $param${NC}"
                echo "VULN_DETAILS: COMMAND_INJECTION | Parameter: $param | Payload: $payload"
                echo "   RISK: High - Server command execution detected"
                echo "   EVIDENCE: Command output found in response"
                echo "   FIX: Never pass user input directly to system commands, use parameterization"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Command Injection in parameter '$param'")
                vulnerabilities_found=1
                break
            fi

            # Check for time-based injection (sleep command)
            if [[ "$payload" == *"sleep 5"* ]] && [ $response_time -ge 5 ]; then
                echo -e "${RED}🚨 Time-based command injection in parameter: $param${NC}"
                echo "VULN_DETAILS: COMMAND_INJECTION_BLIND | Parameter: $param | Delay: ${response_time}s"
                echo "   RISK: High - Blind command execution via timing attack"
                echo "   EVIDENCE: Response delayed by $response_time seconds"
                echo "   FIX: Never pass user input directly to system commands, use parameterization"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Time-based Command Injection in parameter '$param'")
                vulnerabilities_found=1
                break
            fi
        done
        if [ $vulnerabilities_found -eq 1 ]; then
            break
        fi
    done

    run_test "Command Injection Protection Test" "Completed command injection testing" "Completed"
}

# Function to generate summary report
generate_summary() {
    echo -e "\n${PURPLE}📊 SECURITY ASSESSMENT SUMMARY${NC}"
    echo -e "${PURPLE}================================${NC}"
    
    # Calculate scores
    local success_rate=0
    if [ $TESTS_TOTAL -gt 0 ]; then
        success_rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    fi
    
    # Calculate security score (higher is better, starts at 100)
    local score=100
    score=$((score - HIGH_RISK_FINDINGS * 20))
    score=$((score - MEDIUM_RISK_FINDINGS * 10))
    score=$((score - LOW_RISK_FINDINGS * 5))
    if [ $score -lt 0 ]; then score=0; fi

    # Display summary
    echo -e "${BLUE}Target: $TARGET_URL${NC}"
    echo -e "${BLUE}Scan Date: $(date)${NC}"
    echo -e "${BLUE}Test Success Rate: $success_rate% ($TESTS_PASSED/$TESTS_TOTAL)${NC}"
    echo -e "${CYAN}Overall Security Score: ${NC}"
    if [ $score -ge 80 ]; then
        echo -e "  ${GREEN}${score}/100 - GOOD${NC}"
    elif [ $score -ge 60 ]; then
        echo -e "  ${YELLOW}${score}/100 - NEEDS IMPROVEMENT${NC}"
    else
        echo -e "  ${RED}${score}/100 - CRITICAL${NC}"
    fi
    
    if [ $HIGH_RISK_FINDINGS -gt 0 ] || [ $MEDIUM_RISK_FINDINGS -gt 0 ] || [ $LOW_RISK_FINDINGS -gt 0 ] || [ $INFORMATIONAL_FINDINGS -gt 0 ]; then
        echo -e "\n${YELLOW}🔍 DETAILED VULNERABILITY FINDINGS:${NC}"
        
        # High Risk Findings
        if [ $HIGH_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${RED}🔴 HIGH RISK FINDINGS ($HIGH_RISK_FINDINGS):${NC}"
            for finding in "${HIGH_RISK_LIST[@]}"; do
                echo -e "${RED}   • $finding${NC}"
                # Add detailed info for Vulnerable JS Library
                if [[ "$finding" == "Vulnerable JS Library" ]]; then
                    if command -v jq >/dev/null 2>&1; then
                        vuln_details=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/" 2>/dev/null | jq -r '.alerts[] | select(.risk == "High" and .name == "Vulnerable JS Library") | .other' 2>/dev/null | head -1)
                        if [[ "$vuln_details" =~ nextjs.*version.*([0-9]+\.[0-9]+\.[0-9]+) ]]; then
                            echo -e "${RED}      Library: Next.js ${BASH_REMATCH[1]}${NC}"
                            echo -e "${RED}      Multiple CVEs detected (CVE-2024-47831, CVE-2024-56332, etc.)${NC}"
                            echo -e "${RED}      Fix: Upgrade to Next.js 14.2.15+ or latest${NC}"
                        fi
                    fi
                fi
            done
        fi
        
        # Medium Risk Findings
        if [ $MEDIUM_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${YELLOW}🟡 MEDIUM RISK FINDINGS ($MEDIUM_RISK_FINDINGS):${NC}"
            for finding in "${MEDIUM_RISK_LIST[@]}"; do
                echo -e "${YELLOW}   • $finding${NC}"
            done
        fi
        
        # Low Risk Findings
        if [ $LOW_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${GREEN}🟢 LOW RISK FINDINGS ($LOW_RISK_FINDINGS):${NC}"
            for finding in "${LOW_RISK_LIST[@]}"; do
                echo -e "${GREEN}   • $finding${NC}"
            done
        fi
        
        # Informational Findings (only show if there are any after filtering)
        if [ $INFORMATIONAL_FINDINGS -gt 0 ]; then
            echo -e "\n${CYAN}ℹ️  INFORMATIONAL FINDINGS ($INFORMATIONAL_FINDINGS):${NC}"
            for finding in "${INFO_LIST[@]}"; do
                echo -e "${CYAN}   • $finding${NC}"
            done
        fi
    else
        echo -e "\n${GREEN}✅ No security findings detected!${NC}"
    fi
    
    # Show authentication error summary if any 401/403 responses
    if [ "${AUTH_ERRORS:-0}" -gt 0 ]; then
        echo -e "\n${YELLOW}🔐 AUTHENTICATION WARNING${NC}"
        echo -e "${YELLOW}=========================${NC}"
        echo -e "${YELLOW}$AUTH_ERRORS endpoints returned 401/403 - authentication may have failed${NC}"
        echo -e "${YELLOW}💡 Check if AUTH_COOKIE or AUTH_TOKEN environment variables are set correctly${NC}"
        echo ""
    fi

    # Show error summary if any errors occurred
    if [ $ERROR_COUNT -gt 0 ]; then
        echo -e "\n${RED}🚨 ERROR SUMMARY${NC}"
        echo -e "${RED}===============${NC}"
        echo -e "${RED}Total Errors Encountered: $ERROR_COUNT${NC}"
        echo -e "${YELLOW}💡 Review error details above for troubleshooting guidance${NC}"
        echo ""
    fi

    echo -e "\n${GREEN}✅ Security assessment completed - results shown above${NC}"
    
    # Exit code based on risk level and errors
    if [ $ERROR_COUNT -gt 0 ]; then
        echo -e "${YELLOW}⚠️ Exiting with error code due to $ERROR_COUNT errors encountered${NC}"
        exit 3  # Error exit code
    elif [ "$risk_level" = "HIGH" ]; then
        exit 2  # High risk exit code
    elif [ "$risk_level" = "MEDIUM" ]; then
        exit 1  # Medium risk exit code
    else
        exit 0  # Success exit code
    fi
}

# Cleanup function
cleanup() {
    # Remove temporary ZAP log file
    rm -f /tmp/zap_startup.log 2>/dev/null
}

# Set up cleanup on script exit
trap cleanup EXIT

# Main execution
echo "GENERIC WEB APPLICATION SECURITY SCAN"
echo "Target: $TARGET_URL"
echo "Started: $(date)"
echo "========================================"

# Run tests based on mode
case $MODE in
    "zap-only")
        # ZAP-only mode: single phase
        PHASE_NUM=1
        run_zap_scan
        ;;
    "quick")
        # Quick mode: core tests without ZAP (fast)
        PHASE_NUM=1
        run_security_headers_test
        PHASE_NUM=2
        run_cors_tests
        PHASE_NUM=3
        run_csrf_tests
        PHASE_NUM=4
        run_path_traversal_tests
        PHASE_NUM=5
        run_sql_injection_tests
        PHASE_NUM=6
        run_info_disclosure_tests
        PHASE_NUM=7
        run_ssl_tests
        ;;
    *)
        # Full mode: comprehensive testing with all phases
        PHASE_NUM=1
        run_security_headers_test
        PHASE_NUM=2
        run_cors_tests
        PHASE_NUM=3
        run_xss_tests
        PHASE_NUM=4
        run_sql_injection_tests
        PHASE_NUM=5
        run_info_disclosure_tests
        PHASE_NUM=6
        run_csrf_tests
        PHASE_NUM=7
        run_path_traversal_tests
        PHASE_NUM=8
        run_open_redirect_tests
        PHASE_NUM=9
        run_ssl_tests
        PHASE_NUM=10
        run_command_injection_tests
        PHASE_NUM=11
        run_zap_scan
        ;;
esac

# Generate final summary
generate_summary
