<#
.SYNOPSIS
    Generic Web Application Security Testing Script (PowerShell)
    
.DESCRIPTION
    Works with any website - no specific API endpoints or authentication required
    Advanced OWASP-based testing
    
.PARAMETER TargetUrl
    Target web application URL
    
.PARAMETER Mode
    Scan mode: Quick, Full, or ZapOnly
    
.EXAMPLE
    .\generic-security-test.ps1 -TargetUrl "http://localhost:3000" -Mode Quick
    .\generic-security-test.ps1 -TargetUrl "http://localhost:3000" -Mode Full
#>

[CmdletBinding()]
param(
    [Parameter(Mandatory=$true)]
    [string]$TargetUrl,

    [Parameter(Mandatory=$false)]
    [ValidateSet('Quick', 'Full', 'ZapOnly')]
    [string]$Mode = 'Full'
)

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [object]$Body, [string]$ContentType, [hashtable]$Headers, [int]$TimeoutSec = 30)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "Stop" }
    if ($Body) { $params.Body = $Body }
    if ($ContentType) { $params.ContentType = $ContentType }
    $authHeaders = @{}
    if ($Headers) { $Headers.Keys | ForEach-Object { $authHeaders[$_] = $Headers[$_] } }
    if ($AuthToken) { $authHeaders["Authorization"] = "Bearer $AuthToken" }
    if ($authHeaders.Count -gt 0) { $params.Headers = $authHeaders }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

$script:AuthErrors = 0  # Track 401/403 responses

# Global variables
$script:TestsPassed = 0
$script:TestsFailed = 0
$script:HighRiskFindings = @()
$script:MediumRiskFindings = @()
$script:LowRiskFindings = @()
$script:InfoFindings = @()

# Color output functions
function Write-ColorOutput {
    param([string]$Message, [string]$Color = 'White')
    Write-Host $Message -ForegroundColor $Color
}

function Write-Header {
    param([string]$Text)
    Write-Host ""; Write-Host $Text -ForegroundColor Cyan
}

function Write-Success {
    param([string]$Text)
    Write-Host "✅ $Text" -ForegroundColor Green
}

function Write-Failure {
    param([string]$Text)
    Write-Host "❌ $Text" -ForegroundColor Red
}

# Phase 1: Security Headers
function Test-SecurityHeaders {
    Write-Header "PHASE 1: SECURITY HEADERS ANALYSIS"
    
    try {
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method Get -TimeoutSec 30 -UseBasicParsing -ErrorAction Stop
        $headers = $response.Headers
        
        # HSTS
        if ($TargetUrl -like "https://*") {
            if ($headers['Strict-Transport-Security']) {
                Write-Success "HSTS Header Present"
                $script:TestsPassed++
            } else {
                Write-Failure "HSTS Header Missing"
                $script:TestsFailed++
                $script:LowRiskFindings += "Missing HSTS Header"
            }
        }
        
        # X-Frame-Options
        if ($headers['X-Frame-Options']) {
            Write-Success "X-Frame-Options Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-Frame-Options Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-Frame-Options"
        }
        
        # X-Content-Type-Options
        if ($headers['X-Content-Type-Options']) {
            Write-Success "X-Content-Type-Options Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-Content-Type-Options Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-Content-Type-Options"
        }
        
        # X-XSS-Protection
        if ($headers['X-XSS-Protection']) {
            Write-Success "X-XSS-Protection Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-XSS-Protection Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-XSS-Protection"
        }
        
        # CSP
        if ($headers['Content-Security-Policy']) {
            Write-Success "Content-Security-Policy Present"
            $script:TestsPassed++
        } else {
            Write-Failure "Content-Security-Policy Missing"
            $script:TestsFailed++
            $script:MediumRiskFindings += "Missing CSP"
        }
        
        # Referrer-Policy
        if ($headers['Referrer-Policy']) {
            Write-Success "Referrer-Policy Present"
            $script:TestsPassed++
        } else {
            Write-Failure "Referrer-Policy Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing Referrer-Policy"
        }
        
    } catch {
        Write-Failure "Failed to fetch headers: $($_.Exception.Message)"
    }
}

# Phase 2: CORS Tests
function Test-CORS {
    Write-Header "PHASE 2: CORS SECURITY TESTS"
    
    $evilOrigin = "https://evil-site.com"
    
    try {
        $hdrs = @{ 'Origin' = $evilOrigin }
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Headers $hdrs -Method Get -TimeoutSec 30 -UseBasicParsing -ErrorAction Stop
        
        $corsHeader = $response.Headers['Access-Control-Allow-Origin']
        if ($corsHeader -eq '*') {
            Write-Failure "CORS Wildcard Detected"
            $script:TestsFailed++
            $script:HighRiskFindings += "CORS Wildcard Origin (*)"
        } elseif ($corsHeader -eq $evilOrigin) {
            Write-Failure "Evil Origin Accepted"
            $script:TestsFailed++
            $script:HighRiskFindings += "Evil Origin Accepted"
        } else {
            Write-Success "CORS Properly Configured"
            $script:TestsPassed++
        }
    } catch {
        Write-Success "CORS Restricted (good)"
        $script:TestsPassed++
    }
}

# Phase 3: XSS Tests
function Test-XSS {
    Write-Header "PHASE 3: XSS VULNERABILITY TESTS"
    
    $xssPayloads = @(
        "<script>alert('XSS')</script>",
        "javascript:alert('XSS')",
        "'`"><script>alert('XSS')</script>",
        "<img src=x onerror=alert('XSS')>"
    )
    $params = @('q', 'search', 'query', 'name', 'message', 'text')
    $xssFound = $false
    
    foreach ($param in $params) {
        foreach ($payload in $xssPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                
                if ($response.Content -like "*$payload*") {
                    Write-Failure "XSS Vulnerability in parameter: $param"
                    Write-ColorOutput "  Payload: $payload" -Color Yellow
                    $script:HighRiskFindings += "XSS in parameter '$param'"
                    $xssFound = $true
                    break
                }
            } catch { }
        }
        if ($xssFound) { break }
    }
    
    if (-not $xssFound) {
        Write-Success "XSS Protection OK"
        $script:TestsPassed++
    } else {
        $script:TestsFailed++
    }
}

# Phase 4: SQL Injection Tests
function Test-SQLInjection {
    Write-Header "PHASE 4: SQL INJECTION TESTS"
    
    $sqlPayloads = @(
        "' OR '1'='1",
        "'; DROP TABLE users; --",
        "' UNION SELECT null, null --",
        "admin'--"
    )
    $params = @('id', 'user', 'username', 'email', 'search')
    $sqliFound = $false
    
    foreach ($param in $params) {
        foreach ($payload in $sqlPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                
                $pattern = 'sql syntax|mysql.*error|postgresql.*error|ora-[0-9]+|sqlite.*error|syntax error.*sql'
                if ($response.Content -match $pattern) {
                    Write-Failure "SQL Injection Vulnerability in parameter: $param"
                    Write-ColorOutput "  Payload: $payload" -Color Yellow
                    $script:HighRiskFindings += "SQL Injection in parameter '$param'"
                    $sqliFound = $true
                    break
                }
            } catch { }
        }
        if ($sqliFound) { break }
    }
    
    if (-not $sqliFound) {
        Write-Success "SQL Injection Protection OK"
        $script:TestsPassed++
    } else {
        $script:TestsFailed++
    }
}

# Phase 5: Information Disclosure
function Test-InformationDisclosure {
    Write-Header "PHASE 5: INFORMATION DISCLOSURE TESTS"
    
    $sensitivePaths = @('/.git', '/.env', '/robots.txt', '/admin', '/phpmyadmin', '/config.php', '/web.config')
    
    foreach ($path in $sensitivePaths) {
        try {
            $testUrl = $TargetUrl + $path
            $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -UseBasicParsing -ErrorAction Stop
            
            if ($response.StatusCode -eq 200) {
                Write-Failure "Accessible: $path"
                if ($path -in @('/.git', '/.env', '/config.php', '/web.config')) {
                    $script:HighRiskFindings += "$path accessible"
                } elseif ($path -in @('/admin', '/phpmyadmin')) {
                    $script:MediumRiskFindings += "$path accessible"
                } else {
                    $script:InfoFindings += "$path accessible"
                }
            }
        } catch {
            # Path not accessible - good
        }
    }
    
    Write-Success "Information Disclosure Check Complete"
    $script:TestsPassed++
}

# Generate Summary
function Show-Summary {
    Write-Host ""
    Write-Header "SECURITY ASSESSMENT SUMMARY"
    Write-Host ("=" * 50)
    
    $totalFindings = $script:HighRiskFindings.Count + $script:MediumRiskFindings.Count + $script:LowRiskFindings.Count
    
    Write-ColorOutput "Target: $TargetUrl" -Color Blue
    Write-ColorOutput "Scan Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -Color Blue
    Write-ColorOutput "Tests Passed: $script:TestsPassed" -Color Green
    Write-ColorOutput "Tests Failed: $script:TestsFailed" -Color Red
    
    Write-Host ""
    
    # Risk Level
    $riskLevel = "LOW"
    $riskColor = "Green"
    if ($script:HighRiskFindings.Count -gt 0) {
        $riskLevel = "HIGH"
        $riskColor = "Red"
    } elseif ($script:MediumRiskFindings.Count -gt 3) {
        $riskLevel = "MEDIUM"
        $riskColor = "Yellow"
    }
    
    Write-Host "Overall Risk Level: " -NoNewline
    Write-Host $riskLevel -ForegroundColor $riskColor
    
    if ($totalFindings -gt 0) {
        Write-Host ""; Write-ColorOutput "VULNERABILITY FINDINGS:" -Color Yellow
        
        if ($script:HighRiskFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "HIGH RISK ($($script:HighRiskFindings.Count)):" -Color Red
            foreach ($finding in $script:HighRiskFindings) {
                Write-ColorOutput "  • $finding" -Color Red
            }
        }
        
        if ($script:MediumRiskFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "MEDIUM RISK ($($script:MediumRiskFindings.Count)):" -Color Yellow
            foreach ($finding in $script:MediumRiskFindings) {
                Write-ColorOutput "  • $finding" -Color Yellow
            }
        }
        
        if ($script:LowRiskFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "LOW RISK ($($script:LowRiskFindings.Count)):" -Color Gray
            foreach ($finding in $script:LowRiskFindings) {
                Write-ColorOutput "  • $finding" -Color Gray
            }
        }
        
        if ($script:InfoFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "INFORMATIONAL ($($script:InfoFindings.Count)):" -Color Cyan
            foreach ($finding in $script:InfoFindings) {
                Write-ColorOutput "  • $finding" -Color Cyan
            }
        }
    } else {
        Write-Host ""; Write-Success "No security findings detected!"
    }

    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "⚠️  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    Write-Host ""; Write-Host ("=" * 50)
}

# Main execution
try {
    Add-Type -AssemblyName System.Web -ErrorAction SilentlyContinue
    
    Write-Host ""
    Write-ColorOutput "GENERIC WEB SECURITY SCANNER (PowerShell)" -Color Magenta
    Write-Host ("=" * 50)
    Write-ColorOutput "Target: $TargetUrl" -Color Cyan
    Write-ColorOutput "Mode: $Mode" -Color Cyan
    if ($AuthToken) { Write-Host "🔐 Bearer token authentication enabled" -ForegroundColor Green }
    if ($AuthCookie -and -not $AuthToken) { Write-Host "🔐 Cookie authentication enabled" -ForegroundColor Green }
    if (-not $AuthToken -and -not $AuthCookie) { Write-Host "ℹ️ Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

    Write-Host ("=" * 50)

    # Run tests based on mode
    if ($Mode -ne 'ZapOnly') {
        Test-SecurityHeaders
        Test-CORS
        Test-XSS
        Test-SQLInjection
        Test-InformationDisclosure
    }
    
    # Show summary
    Show-Summary
    
    Write-Host ""
    Write-Success "Security scan completed!"
    Write-Host ""
    
} catch {
    Write-Failure "Script error: $($_.Exception.Message)"
    exit 1
}
