# File Upload Security Testing Script (PowerShell) - CWE-434, CWE-22
param([string]$TargetUrl, [ValidateSet("quick","full")][string]$Mode="full", [string]$UploadEndpoint="/upload", [switch]$Help)

if ($Help) { Write-Host "📁 File Upload Security Scanner"; exit 0 }
if ([string]::IsNullOrEmpty($TargetUrl)) { Write-Host "Error: TargetUrl required" -ForegroundColor Red; exit 1 }
if ($TargetUrl -notmatch "^https?://") { $TargetUrl = "https://$TargetUrl" }
$TargetUrl = $TargetUrl.TrimEnd('/')

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated REST requests (file uploads)
function Invoke-AuthRestMethod {
    param([string]$Uri, [string]$Method = "POST", [hashtable]$Form, [int]$TimeoutSec = 15)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; ErrorAction = "SilentlyContinue" }
    if ($Form) { $params.Form = $Form }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-RestMethod @params
}

$script:TestsPassed=0; $script:TestsFailed=0; $script:TestsTotal=0; $script:HighRiskFindings=0; $script:HighRiskList=@()

Write-Host "`n📁 FILE UPLOAD SECURITY SCANNER" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl$UploadEndpoint | Mode: $Mode" -ForegroundColor Blue
if ($AuthToken) { Write-Host "🔐 Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "🔐 Cookie authentication enabled" -ForegroundColor Green }

# Auth verification (need Invoke-AuthWebRequest for this, add helper)
function Invoke-AuthWebRequest {
    param([string]$Uri, [int]$TimeoutSec = 5)
    $params = @{ Uri = $Uri; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) { try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {} }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

if (-not $AuthToken -and -not $AuthCookie) { Write-Host "ℹ️ Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

$script:AuthErrors=0  # Track 401/403 responses

Write-Host ""

$TempDir = New-Item -ItemType Directory -Path ([System.IO.Path]::GetTempPath() + [System.Guid]::NewGuid().ToString()) -Force

function New-TestFiles {
    '<?php echo "VULN"; ?>' | Out-File -FilePath "$TempDir\test.php" -Encoding ascii
    '<?php echo "VULN"; ?>' | Out-File -FilePath "$TempDir\test.php.jpg" -Encoding ascii
    '<script>alert(1)</script>' | Out-File -FilePath "$TempDir\test.html" -Encoding ascii
}

function Test-UnrestrictedUpload {
    Write-Host "`n📤 PHASE 1: UNRESTRICTED UPLOAD" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $files = @("test.php", "test.html")
    foreach ($file in $files) {
        try {
            $filePath = "$TempDir\$file"
            $form = @{ file = Get-Item -Path $filePath }
            $r = Invoke-AuthRestMethod -Uri "$TargetUrl$UploadEndpoint" -Method Post -Form $form -TimeoutSec 15 -ErrorAction SilentlyContinue
            if ($r -match "success|uploaded|saved") { $vuln = $true; Write-Host "  ⚠️ Accepted: $file" -ForegroundColor Red }
        } catch {}
    }
    if ($vuln) {
        Write-Host "❌ FAIL: Unrestricted upload" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Unrestricted Upload"
    } else { Write-Host "✅ PASS: Blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Test-ExtensionBypass {
    Write-Host "`n🔄 PHASE 2: EXTENSION BYPASS" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $bypasses = @("test.php.jpg", "test.pHp", "test.php5")
    foreach ($bypass in $bypasses) {
        Copy-Item "$TempDir\test.php" "$TempDir\$bypass" -Force
        try {
            $form = @{ file = Get-Item -Path "$TempDir\$bypass" }
            $r = Invoke-AuthRestMethod -Uri "$TargetUrl$UploadEndpoint" -Method Post -Form $form -TimeoutSec 15 -ErrorAction SilentlyContinue
            if ($r -match "success|uploaded") { $vuln = $true; Write-Host "  ⚠️ Bypass: $bypass" -ForegroundColor Red }
        } catch {}
    }
    if ($vuln) {
        Write-Host "❌ FAIL: Extension bypass" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Extension Bypass"
    } else { Write-Host "✅ PASS: Blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Show-Summary {
    Write-Host "`n======================================================" -ForegroundColor Magenta
    Write-Host "📊 FILE UPLOAD SUMMARY" -ForegroundColor Magenta
    $rate = if($script:TestsTotal -gt 0){[Math]::Round(($script:TestsPassed/$script:TestsTotal)*100)}else{0}
    Write-Host "Tests: $($script:TestsTotal) | Passed: $($script:TestsPassed) | Failed: $($script:TestsFailed) | Rate: $rate%"
    if ($script:HighRiskList.Count -gt 0) { $script:HighRiskList | ForEach-Object { Write-Host "  🚨 $_" -ForegroundColor Red } }

    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "⚠️  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    $score = 100 - ($script:HighRiskFindings * 25); if($score -lt 0){$score=0}
    Write-Host "Score: $score/100" -ForegroundColor $(if($score -ge 80){"Green"}else{"Red"})
    Remove-Item -Path $TempDir -Recurse -Force -ErrorAction SilentlyContinue
}

New-TestFiles
Test-UnrestrictedUpload
Test-ExtensionBypass
Show-Summary
