#!/bin/bash

#######################################################################
# Dependency Security Testing Script
# Tests for outdated dependencies and known CVEs
# Covers CWE-1104 (Use of Unmaintained Third Party Components)
# and CWE-1035 (Using Components with Known Vulnerabilities)
#######################################################################

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Test tracking variables
TESTS_PASSED=0
TESTS_FAILED=0
TESTS_TOTAL=0
declare -a HIGH_RISK_FINDINGS
declare -a MEDIUM_RISK_FINDINGS
declare -a LOW_RISK_FINDINGS
declare -a CRITICAL_CVES
declare -a HIGH_CVES
declare -a MEDIUM_CVES
declare -a LOW_CVES

# Parse command line arguments (positional, matching main.js invocation)
PROJECT_PATH="${1:-}"
MODE="${2:-quick}"
CLEAN_OUTPUT="${3:-false}"
SCAN_TYPE="auto"

# Output functions
print_success() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[+] $1"
    else
        echo -e "${GREEN}[+] $1${NC}"
    fi
}

print_failure() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[-] $1"
    else
        echo -e "${RED}[-] $1${NC}"
    fi
}

print_info() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[*] $1"
    else
        echo -e "${BLUE}[*] $1${NC}"
    fi
}

print_warning() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[!] $1"
    else
        echo -e "${YELLOW}[!] $1${NC}"
    fi
}

print_phase() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo ""
        echo "=== $1 ==="
    else
        echo ""
        echo -e "${CYAN}=== $1 ===${NC}"
    fi
}

show_help() {
    cat << EOF

Dependency Security Testing Script
===================================

Usage: $0 <project_path> [options]

Options:
    -h, --help      Show this help message
    -c, --clean     Suppress colored output
    -m, --mode      Test mode: 'quick' or 'full' (default: quick)
    -t, --type      Scan type: 'auto', 'npm', 'pip', 'composer', 'maven', 'gradle', 'ruby' (default: auto)

Examples:
    $0 /path/to/project
    $0 /path/to/project -m full
    $0 /path/to/project -t npm

Test Phases:
    Phase 1: Dependency Discovery
    Phase 2: Known Vulnerability Scanning (CWE-1035)
    Phase 3: Outdated Dependency Detection (CWE-1104)

Supported Package Managers:
    - npm/yarn (Node.js)
    - pip (Python)
    - composer (PHP)
    - maven/gradle (Java)
    - bundler (Ruby)
    - cargo (Rust)
    - go modules (Go)

Requirements:
    - Respective package managers installed for each ecosystem
    - Optional: OWASP Dependency-Check, Snyk CLI, Retire.js

EOF
}

get_finding_details() {
    local finding_type="$1"
    case "$finding_type" in
        "known_vuln")
            echo "CWE-1035|HIGH|Using Components with Known Vulnerabilities|Remote code execution, data breach, privilege escalation|Update to patched version immediately"
            ;;
        "outdated")
            echo "CWE-1104|MEDIUM|Use of Unmaintained Third Party Components|Missing security patches, compatibility issues|Update to latest stable version"
            ;;
        "critical_cve")
            echo "CWE-1035|CRITICAL|Critical CVE in dependency|Complete system compromise|Immediate patching required"
            ;;
        "high_cve")
            echo "CWE-1035|HIGH|High severity CVE in dependency|Significant security risk|Patch within 24-48 hours"
            ;;
        "medium_cve")
            echo "CWE-1035|MEDIUM|Medium severity CVE in dependency|Moderate security risk|Patch within 1-2 weeks"
            ;;
        "low_cve")
            echo "CWE-1035|LOW|Low severity CVE in dependency|Minor security risk|Patch in next release cycle"
            ;;
        *)
            echo "Unknown|INFO|Unknown vulnerability|Unknown|Further investigation required"
            ;;
    esac
}

detect_project_type() {
    local path="$1"
    local types=()
    
    # Node.js
    if [ -f "$path/package.json" ]; then
        types+=("npm")
    fi
    
    # Python
    if [ -f "$path/requirements.txt" ] || [ -f "$path/Pipfile" ] || [ -f "$path/pyproject.toml" ] || [ -f "$path/setup.py" ]; then
        types+=("pip")
    fi
    
    # PHP
    if [ -f "$path/composer.json" ]; then
        types+=("composer")
    fi
    
    # Java Maven
    if [ -f "$path/pom.xml" ]; then
        types+=("maven")
    fi
    
    # Java Gradle
    if [ -f "$path/build.gradle" ] || [ -f "$path/build.gradle.kts" ]; then
        types+=("gradle")
    fi
    
    # Ruby
    if [ -f "$path/Gemfile" ]; then
        types+=("ruby")
    fi
    
    # Rust
    if [ -f "$path/Cargo.toml" ]; then
        types+=("cargo")
    fi
    
    # Go
    if [ -f "$path/go.mod" ]; then
        types+=("go")
    fi
    
    # .NET
    if ls "$path"/*.csproj 1>/dev/null 2>&1 || [ -f "$path/packages.config" ]; then
        types+=("dotnet")
    fi
    
    echo "${types[@]}"
}

#######################################################################
# Phase 1: Dependency Discovery
#######################################################################
discover_dependencies() {
    print_phase "Phase 1: Dependency Discovery"
    
    if [ "$SCAN_TYPE" = "auto" ]; then
        detected_types=$(detect_project_type "$PROJECT_PATH")
        if [ -z "$detected_types" ]; then
            print_warning "No recognized package manager files found"
            return 1
        fi
        print_info "Detected project types: $detected_types"
    else
        detected_types="$SCAN_TYPE"
        print_info "Using specified type: $SCAN_TYPE"
    fi
    
    for project_type in $detected_types; do
        ((TESTS_TOTAL++))
        print_info "Scanning $project_type dependencies..."
        
        case "$project_type" in
            "npm")
                if [ -f "$PROJECT_PATH/package-lock.json" ] || [ -f "$PROJECT_PATH/yarn.lock" ]; then
                    dep_count=$(cd "$PROJECT_PATH" && npm ls --all --json 2>/dev/null | grep -c '"version"' || echo "0")
                    print_success "Found approximately $dep_count npm dependencies (including transitive)"
                    ((TESTS_PASSED++))
                else
                    print_warning "No package-lock.json or yarn.lock found - run 'npm install' first"
                    ((TESTS_FAILED++))
                fi
                ;;
            "pip")
                if [ -f "$PROJECT_PATH/requirements.txt" ]; then
                    dep_count=$(wc -l < "$PROJECT_PATH/requirements.txt" | tr -d ' ')
                    print_success "Found $dep_count direct Python dependencies"
                    ((TESTS_PASSED++))
                elif [ -f "$PROJECT_PATH/Pipfile.lock" ]; then
                    dep_count=$(grep -c '"version"' "$PROJECT_PATH/Pipfile.lock" 2>/dev/null || echo "0")
                    print_success "Found approximately $dep_count Python dependencies"
                    ((TESTS_PASSED++))
                else
                    print_warning "No requirements.txt or Pipfile.lock found"
                    ((TESTS_FAILED++))
                fi
                ;;
            "composer")
                if [ -f "$PROJECT_PATH/composer.lock" ]; then
                    dep_count=$(grep -c '"name"' "$PROJECT_PATH/composer.lock" 2>/dev/null || echo "0")
                    print_success "Found approximately $dep_count PHP dependencies"
                    ((TESTS_PASSED++))
                else
                    print_warning "No composer.lock found - run 'composer install' first"
                    ((TESTS_FAILED++))
                fi
                ;;
            "maven")
                if command -v mvn &> /dev/null; then
                    dep_count=$(cd "$PROJECT_PATH" && mvn dependency:list -DoutputFile=/dev/stdout 2>/dev/null | grep -c ":" || echo "0")
                    print_success "Found approximately $dep_count Maven dependencies"
                    ((TESTS_PASSED++))
                else
                    print_warning "Maven not installed"
                    ((TESTS_FAILED++))
                fi
                ;;
            "gradle")
                if command -v gradle &> /dev/null || [ -f "$PROJECT_PATH/gradlew" ]; then
                    print_success "Gradle project detected"
                    ((TESTS_PASSED++))
                else
                    print_warning "Gradle not available"
                    ((TESTS_FAILED++))
                fi
                ;;
            "ruby")
                if [ -f "$PROJECT_PATH/Gemfile.lock" ]; then
                    dep_count=$(grep -c "^    [a-z]" "$PROJECT_PATH/Gemfile.lock" 2>/dev/null || echo "0")
                    print_success "Found approximately $dep_count Ruby gems"
                    ((TESTS_PASSED++))
                else
                    print_warning "No Gemfile.lock found - run 'bundle install' first"
                    ((TESTS_FAILED++))
                fi
                ;;
            "cargo")
                if [ -f "$PROJECT_PATH/Cargo.lock" ]; then
                    dep_count=$(grep -c '^\[\[package\]\]' "$PROJECT_PATH/Cargo.lock" 2>/dev/null || echo "0")
                    print_success "Found approximately $dep_count Rust crates"
                    ((TESTS_PASSED++))
                else
                    print_warning "No Cargo.lock found - run 'cargo build' first"
                    ((TESTS_FAILED++))
                fi
                ;;
            "go")
                if [ -f "$PROJECT_PATH/go.sum" ]; then
                    dep_count=$(wc -l < "$PROJECT_PATH/go.sum" | tr -d ' ')
                    print_success "Found approximately $((dep_count / 2)) Go modules"
                    ((TESTS_PASSED++))
                else
                    print_warning "No go.sum found - run 'go mod tidy' first"
                    ((TESTS_FAILED++))
                fi
                ;;
            *)
                print_warning "Unknown project type: $project_type"
                ((TESTS_FAILED++))
                ;;
        esac
    done
    
    echo "$detected_types"
}

#######################################################################
# Phase 2: Known Vulnerability Scanning (CWE-1035)
#######################################################################
scan_vulnerabilities() {
    print_phase "Phase 2: Known Vulnerability Scanning (CWE-1035)"
    
    local project_types="$1"
    
    for project_type in $project_types; do
        case "$project_type" in
            "npm")
                scan_npm_vulnerabilities
                ;;
            "pip")
                scan_pip_vulnerabilities
                ;;
            "composer")
                scan_composer_vulnerabilities
                ;;
            "ruby")
                scan_ruby_vulnerabilities
                ;;
            "cargo")
                scan_cargo_vulnerabilities
                ;;
            "go")
                scan_go_vulnerabilities
                ;;
            "maven"|"gradle")
                scan_java_vulnerabilities "$project_type"
                ;;
        esac
    done
    
    # Try OWASP Dependency-Check if available (full mode)
    if [ "$MODE" = "full" ] && command -v dependency-check &> /dev/null; then
        scan_with_owasp_dc
    fi
    
    # Try Snyk if available (full mode)
    if [ "$MODE" = "full" ] && command -v snyk &> /dev/null; then
        scan_with_snyk
    fi
}

scan_npm_vulnerabilities() {
    ((TESTS_TOTAL++))
    print_info "Running npm audit..."
    
    if ! command -v npm &> /dev/null; then
        print_warning "npm not installed - skipping npm vulnerability scan"
        ((TESTS_PASSED++))
        return
    fi
    
    # Run npm audit
    audit_output=$(cd "$PROJECT_PATH" && npm audit --json 2>/dev/null)
    
    if [ $? -eq 0 ] && [ -z "$audit_output" ]; then
        print_success "No npm vulnerabilities found"
        ((TESTS_PASSED++))
        return
    fi
    
    # Parse vulnerabilities by severity
    critical=$(echo "$audit_output" | grep -o '"critical":[0-9]*' | head -1 | grep -o '[0-9]*' || echo "0")
    high=$(echo "$audit_output" | grep -o '"high":[0-9]*' | head -1 | grep -o '[0-9]*' || echo "0")
    moderate=$(echo "$audit_output" | grep -o '"moderate":[0-9]*' | head -1 | grep -o '[0-9]*' || echo "0")
    low=$(echo "$audit_output" | grep -o '"low":[0-9]*' | head -1 | grep -o '[0-9]*' || echo "0")
    
    total=$((critical + high + moderate + low))
    
    if [ "$total" -gt 0 ]; then
        print_warning "Found $total npm vulnerabilities"
        
        if [ "$critical" -gt 0 ]; then
            print_failure "  CRITICAL: $critical"
            HIGH_RISK_FINDINGS+=("$critical critical npm vulnerabilities - CWE-1035")
        fi
        if [ "$high" -gt 0 ]; then
            print_failure "  HIGH: $high"
            HIGH_RISK_FINDINGS+=("$high high severity npm vulnerabilities - CWE-1035")
        fi
        if [ "$moderate" -gt 0 ]; then
            print_warning "  MODERATE: $moderate"
            MEDIUM_RISK_FINDINGS+=("$moderate moderate npm vulnerabilities - CWE-1035")
        fi
        if [ "$low" -gt 0 ]; then
            print_info "  LOW: $low"
            LOW_RISK_FINDINGS+=("$low low severity npm vulnerabilities - CWE-1035")
        fi
        
        ((TESTS_FAILED++))
    else
        print_success "No npm vulnerabilities found"
        ((TESTS_PASSED++))
    fi
    
    # Check for Retire.js (additional scanner)
    if [ "$MODE" = "full" ] && command -v retire &> /dev/null; then
        ((TESTS_TOTAL++))
        print_info "Running Retire.js scan..."
        retire_output=$(cd "$PROJECT_PATH" && retire --path . --outputformat json 2>/dev/null)
        
        if echo "$retire_output" | grep -q "vulnerabilities"; then
            print_warning "Retire.js found additional vulnerabilities"
            ((TESTS_FAILED++))
        else
            print_success "Retire.js scan clean"
            ((TESTS_PASSED++))
        fi
    fi
}

scan_pip_vulnerabilities() {
    ((TESTS_TOTAL++))
    print_info "Scanning Python dependencies for vulnerabilities..."
    
    # Try pip-audit first
    if command -v pip-audit &> /dev/null; then
        print_info "Running pip-audit..."
        audit_output=$(cd "$PROJECT_PATH" && pip-audit -r requirements.txt --format json 2>/dev/null || pip-audit --format json 2>/dev/null)
        
        vuln_count=$(echo "$audit_output" | grep -c '"id"' || echo "0")
        
        if [ "$vuln_count" -gt 0 ]; then
            print_warning "pip-audit found $vuln_count vulnerabilities"
            
            # Count by severity
            critical=$(echo "$audit_output" | grep -ci '"critical"' || echo "0")
            high=$(echo "$audit_output" | grep -ci '"high"' || echo "0")
            
            if [ "$critical" -gt 0 ] || [ "$high" -gt 0 ]; then
                HIGH_RISK_FINDINGS+=("$vuln_count Python package vulnerabilities - CWE-1035")
            else
                MEDIUM_RISK_FINDINGS+=("$vuln_count Python package vulnerabilities - CWE-1035")
            fi
            ((TESTS_FAILED++))
        else
            print_success "No Python vulnerabilities found by pip-audit"
            ((TESTS_PASSED++))
        fi
        return
    fi
    
    # Fallback to safety check
    if command -v safety &> /dev/null; then
        print_info "Running safety check..."
        safety_output=$(cd "$PROJECT_PATH" && safety check -r requirements.txt --json 2>/dev/null)
        
        if echo "$safety_output" | grep -q "vulnerability"; then
            vuln_count=$(echo "$safety_output" | grep -c "vulnerability_id" || echo "0")
            print_warning "Safety found $vuln_count vulnerabilities"
            MEDIUM_RISK_FINDINGS+=("$vuln_count Python package vulnerabilities - CWE-1035")
            ((TESTS_FAILED++))
        else
            print_success "No Python vulnerabilities found by safety"
            ((TESTS_PASSED++))
        fi
        return
    fi
    
    print_warning "No Python vulnerability scanner available (install pip-audit or safety)"
    ((TESTS_PASSED++))
}

scan_composer_vulnerabilities() {
    ((TESTS_TOTAL++))
    print_info "Scanning PHP dependencies for vulnerabilities..."
    
    if ! command -v composer &> /dev/null; then
        print_warning "Composer not installed - skipping PHP vulnerability scan"
        ((TESTS_PASSED++))
        return
    fi
    
    audit_output=$(cd "$PROJECT_PATH" && composer audit --format=json 2>/dev/null)
    
    if echo "$audit_output" | grep -q '"advisories":\[\]' || [ -z "$audit_output" ]; then
        print_success "No PHP vulnerabilities found"
        ((TESTS_PASSED++))
    else
        vuln_count=$(echo "$audit_output" | grep -c '"advisoryId"' || echo "0")
        if [ "$vuln_count" -gt 0 ]; then
            print_warning "Composer audit found $vuln_count vulnerabilities"
            MEDIUM_RISK_FINDINGS+=("$vuln_count PHP package vulnerabilities - CWE-1035")
            ((TESTS_FAILED++))
        else
            print_success "No PHP vulnerabilities found"
            ((TESTS_PASSED++))
        fi
    fi
}

scan_ruby_vulnerabilities() {
    ((TESTS_TOTAL++))
    print_info "Scanning Ruby dependencies for vulnerabilities..."
    
    # Try bundler-audit
    if command -v bundle-audit &> /dev/null; then
        audit_output=$(cd "$PROJECT_PATH" && bundle-audit check 2>/dev/null)
        
        if echo "$audit_output" | grep -qi "vulnerabilities found\|insecure"; then
            vuln_count=$(echo "$audit_output" | grep -c "CVE-\|GHSA-" || echo "0")
            print_warning "bundle-audit found $vuln_count vulnerabilities"
            MEDIUM_RISK_FINDINGS+=("$vuln_count Ruby gem vulnerabilities - CWE-1035")
            ((TESTS_FAILED++))
        else
            print_success "No Ruby vulnerabilities found"
            ((TESTS_PASSED++))
        fi
        return
    fi
    
    print_warning "bundle-audit not installed - skipping Ruby vulnerability scan"
    ((TESTS_PASSED++))
}

scan_cargo_vulnerabilities() {
    ((TESTS_TOTAL++))
    print_info "Scanning Rust dependencies for vulnerabilities..."
    
    if command -v cargo-audit &> /dev/null; then
        audit_output=$(cd "$PROJECT_PATH" && cargo audit --json 2>/dev/null)
        
        vuln_count=$(echo "$audit_output" | grep -c '"id"' || echo "0")
        
        if [ "$vuln_count" -gt 0 ]; then
            print_warning "cargo-audit found $vuln_count vulnerabilities"
            MEDIUM_RISK_FINDINGS+=("$vuln_count Rust crate vulnerabilities - CWE-1035")
            ((TESTS_FAILED++))
        else
            print_success "No Rust vulnerabilities found"
            ((TESTS_PASSED++))
        fi
        return
    fi
    
    print_warning "cargo-audit not installed - skipping Rust vulnerability scan"
    ((TESTS_PASSED++))
}

scan_go_vulnerabilities() {
    ((TESTS_TOTAL++))
    print_info "Scanning Go dependencies for vulnerabilities..."
    
    if command -v govulncheck &> /dev/null; then
        vuln_output=$(cd "$PROJECT_PATH" && govulncheck ./... 2>/dev/null)
        
        if echo "$vuln_output" | grep -qi "vulnerability\|GO-"; then
            vuln_count=$(echo "$vuln_output" | grep -c "GO-" || echo "0")
            print_warning "govulncheck found $vuln_count vulnerabilities"
            MEDIUM_RISK_FINDINGS+=("$vuln_count Go module vulnerabilities - CWE-1035")
            ((TESTS_FAILED++))
        else
            print_success "No Go vulnerabilities found"
            ((TESTS_PASSED++))
        fi
        return
    fi
    
    print_warning "govulncheck not installed - skipping Go vulnerability scan"
    ((TESTS_PASSED++))
}

scan_java_vulnerabilities() {
    local build_tool="$1"
    ((TESTS_TOTAL++))
    print_info "Scanning Java dependencies for vulnerabilities..."
    
    # OWASP Dependency-Check is the best option for Java
    if command -v dependency-check &> /dev/null; then
        print_info "Running OWASP Dependency-Check..."
        # This is handled in scan_with_owasp_dc
        ((TESTS_PASSED++))
        return
    fi
    
    print_warning "OWASP Dependency-Check not installed - limited Java vulnerability scanning"
    ((TESTS_PASSED++))
}

scan_with_owasp_dc() {
    ((TESTS_TOTAL++))
    print_info "Running OWASP Dependency-Check..."
    
    # Create temp directory for report
    report_dir=$(mktemp -d)
    
    dependency-check --project "Security Scan" --scan "$PROJECT_PATH" --format JSON --out "$report_dir" --nvdApiKey "" 2>/dev/null
    
    if [ -f "$report_dir/dependency-check-report.json" ]; then
        critical=$(grep -c '"severity" : "CRITICAL"' "$report_dir/dependency-check-report.json" 2>/dev/null || echo "0")
        high=$(grep -c '"severity" : "HIGH"' "$report_dir/dependency-check-report.json" 2>/dev/null || echo "0")
        medium=$(grep -c '"severity" : "MEDIUM"' "$report_dir/dependency-check-report.json" 2>/dev/null || echo "0")
        low=$(grep -c '"severity" : "LOW"' "$report_dir/dependency-check-report.json" 2>/dev/null || echo "0")
        
        total=$((critical + high + medium + low))
        
        if [ "$total" -gt 0 ]; then
            print_warning "OWASP DC found $total vulnerabilities"
            if [ "$critical" -gt 0 ]; then
                HIGH_RISK_FINDINGS+=("$critical critical vulnerabilities (OWASP DC) - CWE-1035")
            fi
            if [ "$high" -gt 0 ]; then
                HIGH_RISK_FINDINGS+=("$high high severity vulnerabilities (OWASP DC) - CWE-1035")
            fi
            ((TESTS_FAILED++))
        else
            print_success "OWASP Dependency-Check found no vulnerabilities"
            ((TESTS_PASSED++))
        fi
    else
        print_warning "OWASP Dependency-Check report not generated"
        ((TESTS_PASSED++))
    fi
    
    rm -rf "$report_dir"
}

scan_with_snyk() {
    ((TESTS_TOTAL++))
    print_info "Running Snyk vulnerability scan..."
    
    snyk_output=$(cd "$PROJECT_PATH" && snyk test --json 2>/dev/null)
    
    if [ $? -eq 0 ]; then
        print_success "Snyk found no vulnerabilities"
        ((TESTS_PASSED++))
    else
        vuln_count=$(echo "$snyk_output" | grep -c '"id"' || echo "0")
        
        if [ "$vuln_count" -gt 0 ]; then
            critical=$(echo "$snyk_output" | grep -c '"severity": "critical"' || echo "0")
            high=$(echo "$snyk_output" | grep -c '"severity": "high"' || echo "0")
            
            print_warning "Snyk found $vuln_count vulnerabilities"
            
            if [ "$critical" -gt 0 ] || [ "$high" -gt 0 ]; then
                HIGH_RISK_FINDINGS+=("$vuln_count vulnerabilities found by Snyk - CWE-1035")
            else
                MEDIUM_RISK_FINDINGS+=("$vuln_count vulnerabilities found by Snyk - CWE-1035")
            fi
            ((TESTS_FAILED++))
        else
            print_success "Snyk scan completed"
            ((TESTS_PASSED++))
        fi
    fi
}

#######################################################################
# Phase 3: Outdated Dependency Detection (CWE-1104)
#######################################################################
check_outdated_dependencies() {
    print_phase "Phase 3: Outdated Dependency Detection (CWE-1104)"
    
    local project_types="$1"
    
    for project_type in $project_types; do
        case "$project_type" in
            "npm")
                check_npm_outdated
                ;;
            "pip")
                check_pip_outdated
                ;;
            "composer")
                check_composer_outdated
                ;;
            "ruby")
                check_ruby_outdated
                ;;
            "cargo")
                check_cargo_outdated
                ;;
            "go")
                check_go_outdated
                ;;
        esac
    done
}

check_npm_outdated() {
    ((TESTS_TOTAL++))
    print_info "Checking for outdated npm packages..."
    
    if ! command -v npm &> /dev/null; then
        print_warning "npm not installed - skipping outdated check"
        ((TESTS_PASSED++))
        return
    fi
    
    outdated_output=$(cd "$PROJECT_PATH" && npm outdated --json 2>/dev/null)
    
    if [ -z "$outdated_output" ] || [ "$outdated_output" = "{}" ]; then
        print_success "All npm packages are up to date"
        ((TESTS_PASSED++))
    else
        outdated_count=$(echo "$outdated_output" | grep -c '"wanted"' || echo "0")
        major_updates=$(echo "$outdated_output" | grep -c '"latest"' || echo "0")
        
        if [ "$outdated_count" -gt 0 ]; then
            print_warning "Found $outdated_count outdated npm packages"
            
            if [ "$major_updates" -gt 10 ]; then
                MEDIUM_RISK_FINDINGS+=("$outdated_count outdated npm packages - CWE-1104")
            else
                LOW_RISK_FINDINGS+=("$outdated_count outdated npm packages - CWE-1104")
            fi
            ((TESTS_FAILED++))
        else
            print_success "All npm packages are up to date"
            ((TESTS_PASSED++))
        fi
    fi
}

check_pip_outdated() {
    ((TESTS_TOTAL++))
    print_info "Checking for outdated Python packages..."
    
    if ! command -v pip &> /dev/null; then
        print_warning "pip not installed - skipping outdated check"
        ((TESTS_PASSED++))
        return
    fi
    
    outdated_output=$(pip list --outdated --format=json 2>/dev/null)
    outdated_count=$(echo "$outdated_output" | grep -c '"name"' || echo "0")
    
    if [ "$outdated_count" -gt 0 ]; then
        print_warning "Found $outdated_count outdated Python packages"
        LOW_RISK_FINDINGS+=("$outdated_count outdated Python packages - CWE-1104")
        ((TESTS_FAILED++))
    else
        print_success "All Python packages are up to date"
        ((TESTS_PASSED++))
    fi
}

check_composer_outdated() {
    ((TESTS_TOTAL++))
    print_info "Checking for outdated PHP packages..."
    
    if ! command -v composer &> /dev/null; then
        print_warning "Composer not installed - skipping outdated check"
        ((TESTS_PASSED++))
        return
    fi
    
    outdated_output=$(cd "$PROJECT_PATH" && composer outdated --format=json 2>/dev/null)
    outdated_count=$(echo "$outdated_output" | grep -c '"name"' || echo "0")
    
    if [ "$outdated_count" -gt 0 ]; then
        print_warning "Found $outdated_count outdated PHP packages"
        LOW_RISK_FINDINGS+=("$outdated_count outdated PHP packages - CWE-1104")
        ((TESTS_FAILED++))
    else
        print_success "All PHP packages are up to date"
        ((TESTS_PASSED++))
    fi
}

check_ruby_outdated() {
    ((TESTS_TOTAL++))
    print_info "Checking for outdated Ruby gems..."
    
    if ! command -v bundle &> /dev/null; then
        print_warning "Bundler not installed - skipping outdated check"
        ((TESTS_PASSED++))
        return
    fi
    
    outdated_output=$(cd "$PROJECT_PATH" && bundle outdated 2>/dev/null)
    outdated_count=$(echo "$outdated_output" | grep -c "newer version" || echo "0")
    
    if [ "$outdated_count" -gt 0 ]; then
        print_warning "Found $outdated_count outdated Ruby gems"
        LOW_RISK_FINDINGS+=("$outdated_count outdated Ruby gems - CWE-1104")
        ((TESTS_FAILED++))
    else
        print_success "All Ruby gems are up to date"
        ((TESTS_PASSED++))
    fi
}

check_cargo_outdated() {
    ((TESTS_TOTAL++))
    print_info "Checking for outdated Rust crates..."
    
    if command -v cargo-outdated &> /dev/null; then
        outdated_output=$(cd "$PROJECT_PATH" && cargo outdated --format json 2>/dev/null)
        outdated_count=$(echo "$outdated_output" | grep -c '"name"' || echo "0")
        
        if [ "$outdated_count" -gt 0 ]; then
            print_warning "Found $outdated_count outdated Rust crates"
            LOW_RISK_FINDINGS+=("$outdated_count outdated Rust crates - CWE-1104")
            ((TESTS_FAILED++))
        else
            print_success "All Rust crates are up to date"
            ((TESTS_PASSED++))
        fi
    else
        print_warning "cargo-outdated not installed - skipping outdated check"
        ((TESTS_PASSED++))
    fi
}

check_go_outdated() {
    ((TESTS_TOTAL++))
    print_info "Checking for outdated Go modules..."
    
    if command -v go &> /dev/null; then
        outdated_output=$(cd "$PROJECT_PATH" && go list -u -m all 2>/dev/null | grep "\[" || echo "")
        outdated_count=$(echo "$outdated_output" | grep -c "\[" || echo "0")
        
        if [ "$outdated_count" -gt 0 ]; then
            print_warning "Found $outdated_count outdated Go modules"
            LOW_RISK_FINDINGS+=("$outdated_count outdated Go modules - CWE-1104")
            ((TESTS_FAILED++))
        else
            print_success "All Go modules are up to date"
            ((TESTS_PASSED++))
        fi
    else
        print_warning "Go not installed - skipping outdated check"
        ((TESTS_PASSED++))
    fi
}

#######################################################################
# Summary Report
#######################################################################
show_summary() {
    print_phase "Dependency Security Test Summary"
    
    echo ""
    echo "Project: $PROJECT_PATH"
    echo "Mode: $MODE"
    echo "Date: $(date '+%Y-%m-%d %H:%M:%S')"
    echo ""
    echo "Tests Passed: $TESTS_PASSED/$TESTS_TOTAL"
    echo "Tests Failed: $TESTS_FAILED/$TESTS_TOTAL"
    
    total_findings=$((${#HIGH_RISK_FINDINGS[@]} + ${#MEDIUM_RISK_FINDINGS[@]} + ${#LOW_RISK_FINDINGS[@]}))
    
    if [ $total_findings -gt 0 ]; then
        echo ""
        if [ "$CLEAN_OUTPUT" = "true" ]; then
            echo "--- Findings ---"
        else
            echo -e "${YELLOW}--- Findings ---${NC}"
        fi
        
        if [ ${#HIGH_RISK_FINDINGS[@]} -gt 0 ]; then
            echo ""
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "HIGH RISK:"
            else
                echo -e "${RED}HIGH RISK:${NC}"
            fi
            for finding in "${HIGH_RISK_FINDINGS[@]}"; do
                if [ "$CLEAN_OUTPUT" = "true" ]; then
                    echo "  - $finding"
                else
                    echo -e "${RED}  - $finding${NC}"
                fi
            done
        fi
        
        if [ ${#MEDIUM_RISK_FINDINGS[@]} -gt 0 ]; then
            echo ""
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "MEDIUM RISK:"
            else
                echo -e "${YELLOW}MEDIUM RISK:${NC}"
            fi
            for finding in "${MEDIUM_RISK_FINDINGS[@]}"; do
                if [ "$CLEAN_OUTPUT" = "true" ]; then
                    echo "  - $finding"
                else
                    echo -e "${YELLOW}  - $finding${NC}"
                fi
            done
        fi
        
        if [ ${#LOW_RISK_FINDINGS[@]} -gt 0 ]; then
            echo ""
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "LOW RISK:"
            else
                echo -e "${CYAN}LOW RISK:${NC}"
            fi
            for finding in "${LOW_RISK_FINDINGS[@]}"; do
                if [ "$CLEAN_OUTPUT" = "true" ]; then
                    echo "  - $finding"
                else
                    echo -e "${CYAN}  - $finding${NC}"
                fi
            done
        fi
    fi
    
    # Calculate security score
    max_score=100
    deductions=$((${#HIGH_RISK_FINDINGS[@]} * 20 + ${#MEDIUM_RISK_FINDINGS[@]} * 10 + ${#LOW_RISK_FINDINGS[@]} * 5))
    score=$((max_score - deductions))
    if [ $score -lt 0 ]; then
        score=0
    fi
    
    echo ""
    echo "--- Security Score ---"
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "Score: $score/100"
    else
        if [ $score -ge 80 ]; then
            echo -e "${GREEN}Score: $score/100${NC}"
        elif [ $score -ge 60 ]; then
            echo -e "${YELLOW}Score: $score/100${NC}"
        else
            echo -e "${RED}Score: $score/100${NC}"
        fi
    fi
    
    echo ""
    echo "--- CWEs Tested ---"
    echo "CWE-1035: Using Components with Known Vulnerabilities"
    echo "CWE-1104: Use of Unmaintained Third Party Components"
    
    echo ""
    echo "--- Recommendations ---"
    echo "1. Run 'npm audit fix' / 'pip-audit --fix' to auto-fix vulnerabilities"
    echo "2. Review and update dependencies regularly"
    echo "3. Use dependabot or renovate for automated updates"
    echo "4. Consider using lockfiles for reproducible builds"
}

#######################################################################
# Main Execution
#######################################################################

if [ -z "$PROJECT_PATH" ]; then
    echo "Error: Project path is required."
    echo "Usage: $0 <project_path> [options]"
    echo "Use -h or --help for more information."
    exit 1
fi

# Validate project path
if [ ! -d "$PROJECT_PATH" ]; then
    echo "Error: Project path does not exist: $PROJECT_PATH"
    exit 1
fi

# Convert to absolute path
PROJECT_PATH=$(cd "$PROJECT_PATH" && pwd)

if [ "$CLEAN_OUTPUT" = true ]; then
    echo ""
    echo "========================================"
    echo "Dependency Security Testing Script"
    echo "========================================"
else
    echo ""
    echo -e "${CYAN}========================================${NC}"
    echo -e "${CYAN}Dependency Security Testing Script${NC}"
    echo -e "${CYAN}========================================${NC}"
fi
echo "Project: $PROJECT_PATH"
echo "Mode: $MODE"
echo "Started: $(date '+%Y-%m-%d %H:%M:%S')"

# Run test phases
detected_types=$(discover_dependencies)

if [ -n "$detected_types" ]; then
    scan_vulnerabilities "$detected_types"
    check_outdated_dependencies "$detected_types"
fi

# Show summary
show_summary

if [ "$CLEAN_OUTPUT" = true ]; then
    echo ""
    echo "Dependency security testing completed."
else
    echo ""
    echo -e "${CYAN}Dependency security testing completed.${NC}"
fi
