#!/bin/bash

#######################################################################
# Cryptographic Security Testing Script
# Tests for weak TLS, certificate issues, weak algorithms, cleartext data
# Covers CWE-326, CWE-295, CWE-327, CWE-319
#######################################################################

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Test tracking variables
TESTS_PASSED=0
TESTS_FAILED=0
TESTS_TOTAL=0
declare -a HIGH_RISK_FINDINGS
declare -a MEDIUM_RISK_FINDINGS
declare -a LOW_RISK_FINDINGS

# Parse command line arguments (positional, matching main.js invocation)
TARGET_URL="${1:-}"
MODE="${2:-quick}"
CLEAN_OUTPUT="${3:-false}"

# Output functions
print_success() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[+] $1"
    else
        echo -e "${GREEN}[+] $1${NC}"
    fi
}

print_failure() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[-] $1"
    else
        echo -e "${RED}[-] $1${NC}"
    fi
}

print_info() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[*] $1"
    else
        echo -e "${BLUE}[*] $1${NC}"
    fi
}

print_warning() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "[!] $1"
    else
        echo -e "${YELLOW}[!] $1${NC}"
    fi
}

print_phase() {
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo ""
        echo "=== $1 ==="
    else
        echo ""
        echo -e "${CYAN}=== $1 ===${NC}"
    fi
}

show_help() {
    cat << EOF

Cryptographic Security Testing Script
=====================================

Usage: $0 <target_url> [options]

Options:
    -h, --help      Show this help message
    -c, --clean     Suppress colored output
    -m, --mode      Test mode: 'quick' or 'full' (default: quick)

Examples:
    $0 https://example.com
    $0 https://example.com -m full
    $0 https://example.com --clean

Test Phases:
    Phase 1: TLS Protocol Testing (CWE-326)
    Phase 2: Certificate Validation Testing (CWE-295)
    Phase 3: Weak Algorithm Detection (CWE-327)
    Phase 4: Cleartext Transmission Testing (CWE-319)

Requirements:
    - curl with SSL support
    - openssl
    - nmap (optional, for full mode)

EOF
}

get_finding_details() {
    local finding_type="$1"
    case "$finding_type" in
        "weak_tls")
            echo "CWE-326|HIGH|Inadequate Encryption Strength - weak TLS protocols enabled|Data interception, MITM attacks|Disable TLS 1.0/1.1, enable only TLS 1.2+"
            ;;
        "weak_cipher")
            echo "CWE-326|HIGH|Weak cipher suites enabled|Data decryption, cryptanalysis attacks|Disable weak ciphers (RC4, DES, 3DES, export ciphers)"
            ;;
        "cert_expired")
            echo "CWE-295|HIGH|Expired or invalid SSL certificate|MITM attacks, trust issues|Renew certificate before expiration"
            ;;
        "cert_mismatch")
            echo "CWE-295|MEDIUM|Certificate hostname mismatch|MITM attacks, phishing|Use certificate matching the domain"
            ;;
        "self_signed")
            echo "CWE-295|MEDIUM|Self-signed certificate in use|Trust issues, MITM attacks|Use CA-signed certificate"
            ;;
        "weak_hash")
            echo "CWE-327|MEDIUM|Weak hashing algorithm (MD5/SHA1)|Collision attacks, forgery|Use SHA-256 or stronger"
            ;;
        "weak_key")
            echo "CWE-326|HIGH|Weak key size detected|Brute force attacks|Use RSA 2048+ or ECC 256+"
            ;;
        "cleartext")
            echo "CWE-319|HIGH|Cleartext transmission of sensitive data|Data interception|Enforce HTTPS, use TLS everywhere"
            ;;
        "hsts_missing")
            echo "CWE-319|MEDIUM|HTTP Strict Transport Security not enabled|Downgrade attacks|Enable HSTS header"
            ;;
        *)
            echo "Unknown|INFO|Unknown vulnerability|Unknown|Further investigation required"
            ;;
    esac
}

# Extract hostname and port from URL
parse_url() {
    local url="$1"
    # Remove protocol
    local host_port="${url#*://}"
    # Remove path
    host_port="${host_port%%/*}"
    # Extract host and port
    if [[ "$host_port" == *":"* ]]; then
        HOST="${host_port%%:*}"
        PORT="${host_port##*:}"
    else
        HOST="$host_port"
        if [[ "$url" == https://* ]]; then
            PORT=443
        else
            PORT=80
        fi
    fi
}

#######################################################################
# Phase 1: TLS Protocol Testing (CWE-326)
#######################################################################
test_tls_protocols() {
    print_phase "Phase 1: TLS Protocol Testing (CWE-326)"
    
    parse_url "$TARGET_URL"
    
    # Test for SSLv2
    ((TESTS_TOTAL++))
    print_info "Testing for SSLv2 support..."
    if timeout 10 openssl s_client -ssl2 -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
        print_failure "SSLv2 is supported (CRITICAL)"
        ((TESTS_FAILED++))
        HIGH_RISK_FINDINGS+=("SSLv2 enabled - CWE-326")
    else
        print_success "SSLv2 is not supported"
        ((TESTS_PASSED++))
    fi
    
    # Test for SSLv3
    ((TESTS_TOTAL++))
    print_info "Testing for SSLv3 support (POODLE vulnerability)..."
    if timeout 10 openssl s_client -ssl3 -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
        print_failure "SSLv3 is supported (vulnerable to POODLE)"
        ((TESTS_FAILED++))
        HIGH_RISK_FINDINGS+=("SSLv3 enabled (POODLE vulnerable) - CWE-326")
    else
        print_success "SSLv3 is not supported"
        ((TESTS_PASSED++))
    fi
    
    # Test for TLS 1.0
    ((TESTS_TOTAL++))
    print_info "Testing for TLS 1.0 support..."
    if timeout 10 openssl s_client -tls1 -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
        print_warning "TLS 1.0 is supported (deprecated)"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("TLS 1.0 enabled (deprecated) - CWE-326")
    else
        print_success "TLS 1.0 is not supported"
        ((TESTS_PASSED++))
    fi
    
    # Test for TLS 1.1
    ((TESTS_TOTAL++))
    print_info "Testing for TLS 1.1 support..."
    if timeout 10 openssl s_client -tls1_1 -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
        print_warning "TLS 1.1 is supported (deprecated)"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("TLS 1.1 enabled (deprecated) - CWE-326")
    else
        print_success "TLS 1.1 is not supported"
        ((TESTS_PASSED++))
    fi
    
    # Test for TLS 1.2 (should be supported)
    ((TESTS_TOTAL++))
    print_info "Testing for TLS 1.2 support..."
    if timeout 10 openssl s_client -tls1_2 -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
        print_success "TLS 1.2 is supported"
        ((TESTS_PASSED++))
    else
        print_warning "TLS 1.2 is not supported"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("TLS 1.2 not supported - CWE-326")
    fi
    
    # Test for TLS 1.3 (preferred)
    ((TESTS_TOTAL++))
    print_info "Testing for TLS 1.3 support..."
    if timeout 10 openssl s_client -tls1_3 -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
        print_success "TLS 1.3 is supported (recommended)"
        ((TESTS_PASSED++))
    else
        print_info "TLS 1.3 is not supported (recommended but not critical)"
        ((TESTS_PASSED++))
        LOW_RISK_FINDINGS+=("TLS 1.3 not supported - consider enabling")
    fi
    
    # Test for weak cipher suites
    if [ "$MODE" = "full" ]; then
        print_info "Testing for weak cipher suites..."
        
        weak_ciphers=("RC4" "DES-CBC3" "DES-CBC" "EXP" "NULL" "ADH" "AECDH")
        
        for cipher in "${weak_ciphers[@]}"; do
            ((TESTS_TOTAL++))
            if timeout 10 openssl s_client -cipher "$cipher" -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
                print_failure "Weak cipher enabled: $cipher"
                ((TESTS_FAILED++))
                HIGH_RISK_FINDINGS+=("Weak cipher suite enabled: $cipher - CWE-326")
            else
                print_success "Weak cipher disabled: $cipher"
                ((TESTS_PASSED++))
            fi
        done
    fi
}

#######################################################################
# Phase 2: Certificate Validation Testing (CWE-295)
#######################################################################
test_certificate() {
    print_phase "Phase 2: Certificate Validation Testing (CWE-295)"
    
    parse_url "$TARGET_URL"
    
    # Get certificate details
    print_info "Retrieving certificate information..."
    cert_info=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -dates -subject -issuer -checkend 0 2>/dev/null)
    
    # Check certificate expiration
    ((TESTS_TOTAL++))
    print_info "Checking certificate expiration..."
    if echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -checkend 0 2>/dev/null; then
        print_success "Certificate is valid (not expired)"
        ((TESTS_PASSED++))
    else
        print_failure "Certificate is expired or will expire within 0 days"
        ((TESTS_FAILED++))
        HIGH_RISK_FINDINGS+=("Expired SSL certificate - CWE-295")
    fi
    
    # Check if certificate expires soon (30 days)
    ((TESTS_TOTAL++))
    print_info "Checking certificate expiration (30 day warning)..."
    if echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -checkend 2592000 2>/dev/null; then
        print_success "Certificate valid for more than 30 days"
        ((TESTS_PASSED++))
    else
        print_warning "Certificate expires within 30 days"
        ((TESTS_FAILED++))
        LOW_RISK_FINDINGS+=("Certificate expiring soon - CWE-295")
    fi
    
    # Check for self-signed certificate
    ((TESTS_TOTAL++))
    print_info "Checking for self-signed certificate..."
    cert_subject=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -subject 2>/dev/null)
    cert_issuer=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -issuer 2>/dev/null)
    
    if [ "$cert_subject" = "$cert_issuer" ] || [[ "$cert_issuer" == *"$HOST"* ]]; then
        print_warning "Self-signed certificate detected"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("Self-signed certificate in use - CWE-295")
    else
        print_success "Certificate is CA-signed"
        ((TESTS_PASSED++))
    fi
    
    # Check certificate hostname match
    ((TESTS_TOTAL++))
    print_info "Checking certificate hostname match..."
    cert_cn=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -subject 2>/dev/null | sed 's/.*CN = //' | sed 's/,.*//')
    cert_san=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -ext subjectAltName 2>/dev/null)
    
    if [[ "$cert_cn" == *"$HOST"* ]] || [[ "$cert_san" == *"$HOST"* ]]; then
        print_success "Certificate hostname matches"
        ((TESTS_PASSED++))
    else
        print_warning "Certificate hostname mismatch (CN: $cert_cn)"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("Certificate hostname mismatch - CWE-295")
    fi
    
    # Check key size
    ((TESTS_TOTAL++))
    print_info "Checking certificate key size..."
    key_size=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -text 2>/dev/null | grep "Public-Key:" | grep -oP '\d+')
    
    if [ -n "$key_size" ]; then
        if [ "$key_size" -lt 2048 ]; then
            print_failure "Weak key size: $key_size bits (minimum 2048 recommended)"
            ((TESTS_FAILED++))
            HIGH_RISK_FINDINGS+=("Weak certificate key size: ${key_size} bits - CWE-326")
        else
            print_success "Key size is adequate: $key_size bits"
            ((TESTS_PASSED++))
        fi
    else
        print_info "Could not determine key size"
        ((TESTS_PASSED++))
    fi
    
    # Check signature algorithm
    if [ "$MODE" = "full" ]; then
        ((TESTS_TOTAL++))
        print_info "Checking certificate signature algorithm..."
        sig_alg=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -text 2>/dev/null | grep "Signature Algorithm:" | head -1)
        
        if [[ "$sig_alg" == *"md5"* ]] || [[ "$sig_alg" == *"sha1"* ]]; then
            print_warning "Weak signature algorithm detected: $sig_alg"
            ((TESTS_FAILED++))
            MEDIUM_RISK_FINDINGS+=("Weak certificate signature algorithm - CWE-327")
        else
            print_success "Strong signature algorithm in use"
            ((TESTS_PASSED++))
        fi
    fi
}

#######################################################################
# Phase 3: Weak Algorithm Detection (CWE-327)
#######################################################################
test_weak_algorithms() {
    print_phase "Phase 3: Weak Algorithm Detection (CWE-327)"
    
    parse_url "$TARGET_URL"
    
    # Test for MD5 in certificate chain
    ((TESTS_TOTAL++))
    print_info "Checking for MD5 in certificate chain..."
    md5_check=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -text 2>/dev/null | grep -i "md5")
    
    if [ -n "$md5_check" ]; then
        print_failure "MD5 algorithm found in certificate"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("MD5 algorithm in certificate chain - CWE-327")
    else
        print_success "No MD5 in certificate chain"
        ((TESTS_PASSED++))
    fi
    
    # Test for SHA-1 in certificate
    ((TESTS_TOTAL++))
    print_info "Checking for SHA-1 in certificate..."
    sha1_check=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -text 2>/dev/null | grep -i "sha1WithRSA\|sha1WithECDSA")
    
    if [ -n "$sha1_check" ]; then
        print_warning "SHA-1 algorithm found in certificate (deprecated)"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("SHA-1 signature in certificate - CWE-327")
    else
        print_success "No deprecated SHA-1 signatures"
        ((TESTS_PASSED++))
    fi
    
    # Check for export-grade cryptography
    if [ "$MODE" = "full" ]; then
        ((TESTS_TOTAL++))
        print_info "Testing for export-grade cryptography (FREAK)..."
        if timeout 10 openssl s_client -cipher EXPORT -connect "$HOST:$PORT" 2>&1 | grep -q "CONNECTED"; then
            print_failure "Export-grade cryptography supported (FREAK vulnerability)"
            ((TESTS_FAILED++))
            HIGH_RISK_FINDINGS+=("Export-grade crypto enabled (FREAK) - CWE-327")
        else
            print_success "Export-grade cryptography not supported"
            ((TESTS_PASSED++))
        fi
        
        # Check for Logjam vulnerability (weak DH)
        ((TESTS_TOTAL++))
        print_info "Testing for weak Diffie-Hellman (Logjam)..."
        dh_params=$(echo | timeout 10 openssl s_client -connect "$HOST:$PORT" -servername "$HOST" 2>/dev/null | openssl x509 -noout -text 2>/dev/null | grep -A2 "DH Parameters")
        
        if [[ "$dh_params" == *"512"* ]] || [[ "$dh_params" == *"768"* ]] || [[ "$dh_params" == *"1024"* ]]; then
            print_warning "Weak Diffie-Hellman parameters detected"
            ((TESTS_FAILED++))
            MEDIUM_RISK_FINDINGS+=("Weak DH parameters (Logjam risk) - CWE-327")
        else
            print_success "No weak DH parameters detected"
            ((TESTS_PASSED++))
        fi
    fi
    
    # Test HTTP response for weak hashing indicators
    ((TESTS_TOTAL++))
    print_info "Checking for weak hashing in application responses..."
    response=$(auth_curl -sk -D - "$TARGET_URL" 2>/dev/null | head -50)
    
    weak_hash_found=false
    if echo "$response" | grep -qiE "md5|sha1"; then
        # Check if it's in a security-sensitive context
        if echo "$response" | grep -qiE "password.*md5|hash.*md5|md5.*hash|session.*md5"; then
            print_warning "Potential MD5 usage detected in response"
            weak_hash_found=true
        fi
        if echo "$response" | grep -qiE "password.*sha1|hash.*sha1|sha1.*hash"; then
            print_warning "Potential SHA-1 usage detected in response"
            weak_hash_found=true
        fi
    fi
    
    if [ "$weak_hash_found" = true ]; then
        ((TESTS_FAILED++))
        LOW_RISK_FINDINGS+=("Potential weak hashing algorithm usage - CWE-327")
    else
        print_success "No obvious weak hashing in responses"
        ((TESTS_PASSED++))
    fi
}

#######################################################################
# Phase 4: Cleartext Transmission Testing (CWE-319)
#######################################################################
test_cleartext() {
    print_phase "Phase 4: Cleartext Transmission Testing (CWE-319)"
    
    parse_url "$TARGET_URL"
    
    # Test for HTTP to HTTPS redirect
    ((TESTS_TOTAL++))
    print_info "Testing HTTP to HTTPS redirect..."
    http_url="http://$HOST"
    redirect_response=$(curl -sI -o /dev/null -w "%{http_code}:%{redirect_url}" "$http_url" 2>/dev/null)
    http_code="${redirect_response%%:*}"
    redirect_url="${redirect_response#*:}"
    
    if [[ "$http_code" == "301" || "$http_code" == "302" || "$http_code" == "307" || "$http_code" == "308" ]] && [[ "$redirect_url" == https://* ]]; then
        print_success "HTTP redirects to HTTPS (code: $http_code)"
        ((TESTS_PASSED++))
    elif [[ "$http_code" == "200" ]]; then
        print_failure "HTTP accessible without redirect (no HTTPS enforcement)"
        ((TESTS_FAILED++))
        HIGH_RISK_FINDINGS+=("No HTTP to HTTPS redirect - CWE-319")
    else
        print_info "HTTP not accessible or blocked (code: $http_code)"
        ((TESTS_PASSED++))
    fi
    
    # Test for HSTS header
    ((TESTS_TOTAL++))
    print_info "Testing for HTTP Strict Transport Security (HSTS)..."
    hsts_header=$(auth_curl -sI "$TARGET_URL" 2>/dev/null | grep -i "strict-transport-security")
    
    if [ -n "$hsts_header" ]; then
        print_success "HSTS header present: $hsts_header"
        ((TESTS_PASSED++))
        
        # Check HSTS max-age
        if [ "$MODE" = "full" ]; then
            ((TESTS_TOTAL++))
            max_age=$(echo "$hsts_header" | grep -oP 'max-age=\K\d+')
            if [ -n "$max_age" ] && [ "$max_age" -ge 31536000 ]; then
                print_success "HSTS max-age is adequate: $max_age seconds"
                ((TESTS_PASSED++))
            elif [ -n "$max_age" ]; then
                print_warning "HSTS max-age may be too short: $max_age seconds (recommend 31536000+)"
                ((TESTS_FAILED++))
                LOW_RISK_FINDINGS+=("Short HSTS max-age - CWE-319")
            fi
            
            # Check for includeSubDomains
            ((TESTS_TOTAL++))
            if echo "$hsts_header" | grep -qi "includeSubDomains"; then
                print_success "HSTS includes subdomains"
                ((TESTS_PASSED++))
            else
                print_info "HSTS does not include subdomains"
                ((TESTS_PASSED++))
                LOW_RISK_FINDINGS+=("HSTS does not include subdomains")
            fi
            
            # Check for preload
            ((TESTS_TOTAL++))
            if echo "$hsts_header" | grep -qi "preload"; then
                print_success "HSTS preload directive present"
                ((TESTS_PASSED++))
            else
                print_info "HSTS preload not enabled"
                ((TESTS_PASSED++))
            fi
        fi
    else
        print_failure "HSTS header not present"
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("HTTP Strict Transport Security not enabled - CWE-319")
    fi
    
    # Test for mixed content indicators
    ((TESTS_TOTAL++))
    print_info "Checking for mixed content indicators..."
    page_content=$(auth_curl -sk "$TARGET_URL" 2>/dev/null)
    
    mixed_content_found=false
    # Check for HTTP resources in HTTPS page
    if [[ "$TARGET_URL" == https://* ]]; then
        http_resources=$(echo "$page_content" | grep -oP 'src="http://[^"]+"|href="http://[^"]+' | head -5)
        if [ -n "$http_resources" ]; then
            print_warning "Mixed content detected (HTTP resources on HTTPS page)"
            mixed_content_found=true
        fi
    fi
    
    if [ "$mixed_content_found" = true ]; then
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("Mixed content (HTTP resources on HTTPS) - CWE-319")
    else
        print_success "No obvious mixed content issues"
        ((TESTS_PASSED++))
    fi
    
    # Test for secure cookie flags
    ((TESTS_TOTAL++))
    print_info "Testing for secure cookie flags..."
    cookies=$(auth_curl -sI "$TARGET_URL" 2>/dev/null | grep -i "set-cookie")
    
    insecure_cookies=false
    if [ -n "$cookies" ]; then
        if ! echo "$cookies" | grep -qi "secure"; then
            print_warning "Cookies without Secure flag detected"
            insecure_cookies=true
        fi
        if ! echo "$cookies" | grep -qi "httponly"; then
            print_warning "Cookies without HttpOnly flag detected"
            insecure_cookies=true
        fi
    fi
    
    if [ "$insecure_cookies" = true ]; then
        ((TESTS_FAILED++))
        MEDIUM_RISK_FINDINGS+=("Insecure cookie flags - CWE-319")
    else
        if [ -n "$cookies" ]; then
            print_success "Cookies have appropriate security flags"
        else
            print_info "No cookies set in response"
        fi
        ((TESTS_PASSED++))
    fi
    
    # Full mode: Test sensitive endpoints for HTTPS
    if [ "$MODE" = "full" ]; then
        print_info "Testing sensitive endpoints for HTTPS enforcement..."
        sensitive_paths=("/login" "/api/login" "/admin" "/account" "/profile" "/checkout" "/payment")
        
        for path in "${sensitive_paths[@]}"; do
            ((TESTS_TOTAL++))
            http_response=$(curl -sI -o /dev/null -w "%{http_code}" "http://$HOST$path" 2>/dev/null)
            
            if [ "$http_response" = "200" ]; then
                print_failure "Sensitive endpoint accessible over HTTP: $path"
                ((TESTS_FAILED++))
                HIGH_RISK_FINDINGS+=("Sensitive endpoint $path accessible over HTTP - CWE-319")
            else
                print_success "Endpoint protected: $path"
                ((TESTS_PASSED++))
            fi
        done
    fi
}

#######################################################################
# Summary Report
#######################################################################
show_summary() {
    print_phase "Cryptographic Security Test Summary"
    
    echo ""
    echo "Target: $TARGET_URL"
    echo "Mode: $MODE"
    echo "Date: $(date '+%Y-%m-%d %H:%M:%S')"
    echo ""
    echo "Tests Passed: $TESTS_PASSED/$TESTS_TOTAL"
    echo "Tests Failed: $TESTS_FAILED/$TESTS_TOTAL"
    
    total_findings=$((${#HIGH_RISK_FINDINGS[@]} + ${#MEDIUM_RISK_FINDINGS[@]} + ${#LOW_RISK_FINDINGS[@]}))
    
    if [ $total_findings -gt 0 ]; then
        echo ""
        if [ "$CLEAN_OUTPUT" = "true" ]; then
            echo "--- Findings ---"
        else
            echo -e "${YELLOW}--- Findings ---${NC}"
        fi
        
        if [ ${#HIGH_RISK_FINDINGS[@]} -gt 0 ]; then
            echo ""
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "HIGH RISK:"
            else
                echo -e "${RED}HIGH RISK:${NC}"
            fi
            for finding in "${HIGH_RISK_FINDINGS[@]}"; do
                if [ "$CLEAN_OUTPUT" = "true" ]; then
                    echo "  - $finding"
                else
                    echo -e "${RED}  - $finding${NC}"
                fi
            done
        fi
        
        if [ ${#MEDIUM_RISK_FINDINGS[@]} -gt 0 ]; then
            echo ""
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "MEDIUM RISK:"
            else
                echo -e "${YELLOW}MEDIUM RISK:${NC}"
            fi
            for finding in "${MEDIUM_RISK_FINDINGS[@]}"; do
                if [ "$CLEAN_OUTPUT" = "true" ]; then
                    echo "  - $finding"
                else
                    echo -e "${YELLOW}  - $finding${NC}"
                fi
            done
        fi
        
        if [ ${#LOW_RISK_FINDINGS[@]} -gt 0 ]; then
            echo ""
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "LOW RISK:"
            else
                echo -e "${CYAN}LOW RISK:${NC}"
            fi
            for finding in "${LOW_RISK_FINDINGS[@]}"; do
                if [ "$CLEAN_OUTPUT" = "true" ]; then
                    echo "  - $finding"
                else
                    echo -e "${CYAN}  - $finding${NC}"
                fi
            done
        fi
    fi
    
    # Warn about authentication errors
    if [ $AUTH_ERRORS -gt 0 ]; then
        echo ""
        if [ "$CLEAN_OUTPUT" = "true" ]; then
            echo "[!] WARNING: $AUTH_ERRORS requests returned 401/403 Unauthorized"
        else
            echo -e "${YELLOW}⚠️  WARNING: $AUTH_ERRORS requests returned 401/403 Unauthorized${NC}"
        fi
        if [ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ]; then
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints."
            else
                echo -e "${YELLOW}   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints.${NC}"
            fi
        else
            if [ "$CLEAN_OUTPUT" = "true" ]; then
                echo "   Provided credentials may be invalid or expired."
            else
                echo -e "${YELLOW}   Provided credentials may be invalid or expired.${NC}"
            fi
        fi
    fi

    # Calculate security score
    max_score=100
    deductions=$((${#HIGH_RISK_FINDINGS[@]} * 20 + ${#MEDIUM_RISK_FINDINGS[@]} * 10 + ${#LOW_RISK_FINDINGS[@]} * 5))
    score=$((max_score - deductions))
    if [ $score -lt 0 ]; then
        score=0
    fi

    echo ""
    echo "--- Security Score ---"
    if [ "$CLEAN_OUTPUT" = "true" ]; then
        echo "Score: $score/100"
    else
        if [ $score -ge 80 ]; then
            echo -e "${GREEN}Score: $score/100${NC}"
        elif [ $score -ge 60 ]; then
            echo -e "${YELLOW}Score: $score/100${NC}"
        else
            echo -e "${RED}Score: $score/100${NC}"
        fi
    fi
    
    echo ""
    echo "--- CWEs Tested ---"
    echo "CWE-326: Inadequate Encryption Strength"
    echo "CWE-295: Improper Certificate Validation"
    echo "CWE-327: Use of a Broken or Risky Cryptographic Algorithm"
    echo "CWE-319: Cleartext Transmission of Sensitive Information"
}

#######################################################################
# Main Execution
#######################################################################

if [ -z "$TARGET_URL" ]; then
    echo "Error: Target URL is required."
    echo "Usage: $0 <target_url> [options]"
    echo "Use -h or --help for more information."
    exit 1
fi

# Ensure URL has protocol
if [[ ! "$TARGET_URL" == http://* ]] && [[ ! "$TARGET_URL" == https://* ]]; then
    TARGET_URL="https://$TARGET_URL"
fi

# Remove trailing slash
TARGET_URL="${TARGET_URL%/}"

# Authentication support
AUTH_TOKEN="${AUTH_TOKEN:-}"
AUTH_COOKIE="${AUTH_COOKIE:-}"

auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

if [ "$CLEAN_OUTPUT" = true ]; then
    echo ""
    echo "========================================"
    echo "Cryptographic Security Testing Script"
    echo "========================================"
else
    echo ""
    echo -e "${CYAN}========================================${NC}"
    echo -e "${CYAN}Cryptographic Security Testing Script${NC}"
    echo -e "${CYAN}========================================${NC}"
fi
echo "Target: $TARGET_URL"
echo "Mode: $MODE"
echo "Started: $(date '+%Y-%m-%d %H:%M:%S')"
[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication (unauthenticated scan)${NC}"

AUTH_ERRORS=0  # Track 401/403 responses


# Check for required tools
if ! command -v openssl &> /dev/null; then
    print_failure "openssl is required but not installed"
    exit 1
fi

if ! command -v curl &> /dev/null; then
    print_failure "curl is required but not installed"
    exit 1
fi

# Run test phases
test_tls_protocols
test_certificate
test_weak_algorithms
test_cleartext

# Show summary
show_summary

if [ "$CLEAN_OUTPUT" = true ]; then
    echo ""
    echo "Cryptographic security testing completed."
else
    echo ""
    echo -e "${CYAN}Cryptographic security testing completed.${NC}"
fi
