<#
.SYNOPSIS
    Cryptographic Security Testing Script for Windows/PowerShell
    Tests for weak TLS, certificate issues, weak algorithms, cleartext data.

.DESCRIPTION
    This script performs comprehensive cryptographic security testing against web applications.
    It covers CWE-326, CWE-295, CWE-327, CWE-319 vulnerabilities.

.PARAMETER TargetUrl
    The base URL of the target application to test.

.PARAMETER Mode
    Test mode: 'quick' for essential tests only, 'full' for comprehensive testing.

.PARAMETER CleanOutput
    Suppress colored output for logging purposes.

.PARAMETER Help
    Display help information.

.EXAMPLE
    .\crypto-security-test.ps1 -TargetUrl "https://example.com"
    .\crypto-security-test.ps1 -TargetUrl "https://example.com" -Mode full
#>

param(
    [Parameter(Position=0)]
    [string]$TargetUrl,
    
    [Parameter()]
    [ValidateSet('quick', 'full')]
    [string]$Mode = 'quick',
    
    [Parameter()]
    [switch]$CleanOutput,
    
    [Parameter()]
    [switch]$Help
)

# Color output functions
function Write-Success { param([string]$Message) if (-not $CleanOutput) { Write-Host "[+] $Message" -ForegroundColor Green } else { Write-Host "[+] $Message" } }
function Write-Failure { param([string]$Message) if (-not $CleanOutput) { Write-Host "[-] $Message" -ForegroundColor Red } else { Write-Host "[-] $Message" } }
function Write-Info { param([string]$Message) if (-not $CleanOutput) { Write-Host "[*] $Message" -ForegroundColor Blue } else { Write-Host "[*] $Message" } }
function Write-Warning { param([string]$Message) if (-not $CleanOutput) { Write-Host "[!] $Message" -ForegroundColor Yellow } else { Write-Host "[!] $Message" } }
function Write-Phase { param([string]$Message) if (-not $CleanOutput) { Write-Host "`n=== $Message ===" -ForegroundColor Cyan } else { Write-Host "`n=== $Message ===" } }

# Test tracking variables
$script:TESTS_PASSED = 0
$script:TESTS_FAILED = 0
$script:TESTS_TOTAL = 0
$script:HIGH_RISK_FINDINGS = @()
$script:MEDIUM_RISK_FINDINGS = @()
$script:LOW_RISK_FINDINGS = @()

function Show-Help {
    Write-Host @"

Cryptographic Security Testing Script
=====================================

Usage: .\crypto-security-test.ps1 -TargetUrl <URL> [options]

Options:
    -TargetUrl      Target URL to test (required)
    -Mode           Test mode: 'quick' or 'full' (default: quick)
    -CleanOutput    Suppress colored output
    -Help           Show this help message

Examples:
    .\crypto-security-test.ps1 -TargetUrl "https://example.com"
    .\crypto-security-test.ps1 -TargetUrl "https://example.com" -Mode full

Test Phases:
    Phase 1: TLS Protocol Testing (CWE-326)
    Phase 2: Certificate Validation Testing (CWE-295)
    Phase 3: Weak Algorithm Detection (CWE-327)
    Phase 4: Cleartext Transmission Testing (CWE-319)

"@
}

function Get-FindingDetails {
    param([string]$FindingType)
    
    switch ($FindingType) {
        "weak_tls" {
            return @{
                CWE = "CWE-326"
                Risk = "HIGH"
                Description = "Inadequate Encryption Strength - weak TLS protocols enabled"
                Impact = "Data interception, MITM attacks"
                Remediation = "Disable TLS 1.0/1.1, enable only TLS 1.2+"
            }
        }
        "weak_cipher" {
            return @{
                CWE = "CWE-326"
                Risk = "HIGH"
                Description = "Weak cipher suites enabled"
                Impact = "Data decryption, cryptanalysis attacks"
                Remediation = "Disable weak ciphers (RC4, DES, 3DES, export ciphers)"
            }
        }
        "cert_expired" {
            return @{
                CWE = "CWE-295"
                Risk = "HIGH"
                Description = "Expired or invalid SSL certificate"
                Impact = "MITM attacks, trust issues"
                Remediation = "Renew certificate before expiration"
            }
        }
        "cert_mismatch" {
            return @{
                CWE = "CWE-295"
                Risk = "MEDIUM"
                Description = "Certificate hostname mismatch"
                Impact = "MITM attacks, phishing"
                Remediation = "Use certificate matching the domain"
            }
        }
        "cleartext" {
            return @{
                CWE = "CWE-319"
                Risk = "HIGH"
                Description = "Cleartext transmission of sensitive data"
                Impact = "Data interception"
                Remediation = "Enforce HTTPS, use TLS everywhere"
            }
        }
        default {
            return @{
                CWE = "Unknown"
                Risk = "INFO"
                Description = "Unknown vulnerability"
                Impact = "Unknown"
                Remediation = "Further investigation required"
            }
        }
    }
}

function Parse-Url {
    param([string]$Url)
    
    $uri = [System.Uri]$Url
    return @{
        Host = $uri.Host
        Port = if ($uri.Port -eq -1) { if ($uri.Scheme -eq "https") { 443 } else { 80 } } else { $uri.Port }
        Scheme = $uri.Scheme
    }
}

function Test-TlsProtocols {
    Write-Phase "Phase 1: TLS Protocol Testing (CWE-326)"
    
    $urlParts = Parse-Url -Url $TargetUrl
    $host = $urlParts.Host
    $port = $urlParts.Port
    
    # Test TLS versions using .NET
    $tlsVersions = @(
        @{ Name = "SSL 3.0"; Protocol = [System.Net.SecurityProtocolType]::Ssl3; Deprecated = $true; Risk = "HIGH" },
        @{ Name = "TLS 1.0"; Protocol = [System.Net.SecurityProtocolType]::Tls; Deprecated = $true; Risk = "MEDIUM" },
        @{ Name = "TLS 1.1"; Protocol = [System.Net.SecurityProtocolType]::Tls11; Deprecated = $true; Risk = "MEDIUM" },
        @{ Name = "TLS 1.2"; Protocol = [System.Net.SecurityProtocolType]::Tls12; Deprecated = $false; Risk = "NONE" },
        @{ Name = "TLS 1.3"; Protocol = [System.Net.SecurityProtocolType]::Tls13; Deprecated = $false; Risk = "NONE" }
    )
    
    foreach ($tlsVersion in $tlsVersions) {
        $script:TESTS_TOTAL++
        Write-Info "Testing for $($tlsVersion.Name) support..."
        
        try {
            # Save current protocol
            $originalProtocol = [System.Net.ServicePointManager]::SecurityProtocol
            
            # Set specific protocol
            [System.Net.ServicePointManager]::SecurityProtocol = $tlsVersion.Protocol
            
            $request = [System.Net.WebRequest]::Create($TargetUrl)
            $request.Timeout = 10000
            $response = $request.GetResponse()
            $response.Close()
            
            # Protocol is supported
            if ($tlsVersion.Deprecated) {
                Write-Warning "$($tlsVersion.Name) is supported (deprecated)"
                $script:TESTS_FAILED++
                if ($tlsVersion.Risk -eq "HIGH") {
                    $script:HIGH_RISK_FINDINGS += "$($tlsVersion.Name) enabled - CWE-326"
                } else {
                    $script:MEDIUM_RISK_FINDINGS += "$($tlsVersion.Name) enabled (deprecated) - CWE-326"
                }
            } else {
                Write-Success "$($tlsVersion.Name) is supported"
                $script:TESTS_PASSED++
            }
            
            # Restore original protocol
            [System.Net.ServicePointManager]::SecurityProtocol = $originalProtocol
        } catch {
            if ($tlsVersion.Deprecated) {
                Write-Success "$($tlsVersion.Name) is not supported"
                $script:TESTS_PASSED++
            } else {
                if ($tlsVersion.Name -eq "TLS 1.3") {
                    Write-Info "$($tlsVersion.Name) is not supported (recommended but not critical)"
                    $script:TESTS_PASSED++
                    $script:LOW_RISK_FINDINGS += "TLS 1.3 not supported - consider enabling"
                } else {
                    Write-Warning "$($tlsVersion.Name) is not supported"
                    $script:TESTS_FAILED++
                    $script:MEDIUM_RISK_FINDINGS += "$($tlsVersion.Name) not supported - CWE-326"
                }
            }
            
            # Restore original protocol
            [System.Net.ServicePointManager]::SecurityProtocol = $originalProtocol
        }
    }
    
    # Reset to default protocols
    [System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]::Tls12 -bor [System.Net.SecurityProtocolType]::Tls13
}

function Test-Certificate {
    Write-Phase "Phase 2: Certificate Validation Testing (CWE-295)"
    
    $urlParts = Parse-Url -Url $TargetUrl
    $host = $urlParts.Host
    $port = $urlParts.Port
    
    try {
        # Create TCP connection and get certificate
        $tcpClient = New-Object System.Net.Sockets.TcpClient
        $tcpClient.Connect($host, $port)
        
        $sslStream = New-Object System.Net.Security.SslStream($tcpClient.GetStream(), $false, {
            param($sender, $certificate, $chain, $sslPolicyErrors)
            return $true  # Accept all for testing purposes
        })
        
        $sslStream.AuthenticateAsClient($host)
        $cert = $sslStream.RemoteCertificate
        $cert2 = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2($cert)
        
        # Check certificate expiration
        $script:TESTS_TOTAL++
        Write-Info "Checking certificate expiration..."
        $now = Get-Date
        
        if ($cert2.NotAfter -lt $now) {
            Write-Failure "Certificate is expired (expired: $($cert2.NotAfter))"
            $script:TESTS_FAILED++
            $script:HIGH_RISK_FINDINGS += "Expired SSL certificate - CWE-295"
        } else {
            Write-Success "Certificate is valid (expires: $($cert2.NotAfter))"
            $script:TESTS_PASSED++
        }
        
        # Check if certificate expires soon (30 days)
        $script:TESTS_TOTAL++
        Write-Info "Checking certificate expiration (30 day warning)..."
        $thirtyDaysFromNow = $now.AddDays(30)
        
        if ($cert2.NotAfter -lt $thirtyDaysFromNow) {
            Write-Warning "Certificate expires within 30 days"
            $script:TESTS_FAILED++
            $script:LOW_RISK_FINDINGS += "Certificate expiring soon - CWE-295"
        } else {
            Write-Success "Certificate valid for more than 30 days"
            $script:TESTS_PASSED++
        }
        
        # Check for self-signed certificate
        $script:TESTS_TOTAL++
        Write-Info "Checking for self-signed certificate..."
        
        if ($cert2.Subject -eq $cert2.Issuer) {
            Write-Warning "Self-signed certificate detected"
            $script:TESTS_FAILED++
            $script:MEDIUM_RISK_FINDINGS += "Self-signed certificate in use - CWE-295"
        } else {
            Write-Success "Certificate is CA-signed"
            $script:TESTS_PASSED++
        }
        
        # Check certificate hostname match
        $script:TESTS_TOTAL++
        Write-Info "Checking certificate hostname match..."
        
        $cn = ($cert2.Subject -split "," | Where-Object { $_ -match "CN=" }) -replace "CN=", "" | ForEach-Object { $_.Trim() }
        
        # Also check SAN
        $san = $cert2.Extensions | Where-Object { $_.Oid.FriendlyName -eq "Subject Alternative Name" }
        $sanNames = if ($san) { $san.Format($false) } else { "" }
        
        if ($cn -like "*$host*" -or $sanNames -like "*$host*") {
            Write-Success "Certificate hostname matches"
            $script:TESTS_PASSED++
        } else {
            Write-Warning "Certificate hostname mismatch (CN: $cn)"
            $script:TESTS_FAILED++
            $script:MEDIUM_RISK_FINDINGS += "Certificate hostname mismatch - CWE-295"
        }
        
        # Check key size
        $script:TESTS_TOTAL++
        Write-Info "Checking certificate key size..."
        
        $publicKey = $cert2.PublicKey.Key
        $keySize = if ($publicKey) { $publicKey.KeySize } else { 0 }
        
        if ($keySize -gt 0) {
            if ($keySize -lt 2048) {
                Write-Failure "Weak key size: $keySize bits (minimum 2048 recommended)"
                $script:TESTS_FAILED++
                $script:HIGH_RISK_FINDINGS += "Weak certificate key size: $keySize bits - CWE-326"
            } else {
                Write-Success "Key size is adequate: $keySize bits"
                $script:TESTS_PASSED++
            }
        } else {
            Write-Info "Could not determine key size"
            $script:TESTS_PASSED++
        }
        
        # Check signature algorithm
        if ($Mode -eq 'full') {
            $script:TESTS_TOTAL++
            Write-Info "Checking certificate signature algorithm..."
            
            $sigAlg = $cert2.SignatureAlgorithm.FriendlyName
            
            if ($sigAlg -match "MD5|SHA1") {
                Write-Warning "Weak signature algorithm detected: $sigAlg"
                $script:TESTS_FAILED++
                $script:MEDIUM_RISK_FINDINGS += "Weak certificate signature algorithm - CWE-327"
            } else {
                Write-Success "Strong signature algorithm in use: $sigAlg"
                $script:TESTS_PASSED++
            }
        }
        
        $sslStream.Close()
        $tcpClient.Close()
        
    } catch {
        Write-Warning "Could not retrieve certificate: $($_.Exception.Message)"
        $script:TESTS_TOTAL++
        $script:TESTS_PASSED++
    }
}

function Test-WeakAlgorithms {
    Write-Phase "Phase 3: Weak Algorithm Detection (CWE-327)"
    
    $urlParts = Parse-Url -Url $TargetUrl
    
    # Test HTTP response for weak hashing indicators
    $script:TESTS_TOTAL++
    Write-Info "Checking for weak hashing in application responses..."
    
    try {
        $response = Invoke-WebRequest -Uri $TargetUrl -TimeoutSec 10 -ErrorAction SilentlyContinue
        $content = $response.Content
        
        $weakHashFound = $false
        
        if ($content -match "md5|sha1") {
            if ($content -match "password.*md5|hash.*md5|md5.*hash|session.*md5") {
                Write-Warning "Potential MD5 usage detected in response"
                $weakHashFound = $true
            }
            if ($content -match "password.*sha1|hash.*sha1|sha1.*hash") {
                Write-Warning "Potential SHA-1 usage detected in response"
                $weakHashFound = $true
            }
        }
        
        if ($weakHashFound) {
            $script:TESTS_FAILED++
            $script:LOW_RISK_FINDINGS += "Potential weak hashing algorithm usage - CWE-327"
        } else {
            Write-Success "No obvious weak hashing in responses"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Info "Could not check for weak hashing: $($_.Exception.Message)"
        $script:TESTS_PASSED++
    }
    
    # Check for weak algorithms in headers
    if ($Mode -eq 'full') {
        $script:TESTS_TOTAL++
        Write-Info "Checking for weak algorithms in security headers..."
        
        try {
            $response = Invoke-WebRequest -Uri $TargetUrl -Method Head -TimeoutSec 10 -ErrorAction SilentlyContinue
            $headers = $response.Headers
            
            $weakAlgFound = $false
            
            # Check Content-Security-Policy for sha1
            if ($headers.'Content-Security-Policy' -match "sha1") {
                Write-Warning "SHA-1 found in Content-Security-Policy"
                $weakAlgFound = $true
            }
            
            if ($weakAlgFound) {
                $script:TESTS_FAILED++
                $script:LOW_RISK_FINDINGS += "Weak algorithm in security headers - CWE-327"
            } else {
                Write-Success "No weak algorithms in security headers"
                $script:TESTS_PASSED++
            }
        } catch {
            Write-Info "Could not check security headers"
            $script:TESTS_PASSED++
        }
    }
}

function Test-Cleartext {
    Write-Phase "Phase 4: Cleartext Transmission Testing (CWE-319)"
    
    $urlParts = Parse-Url -Url $TargetUrl
    $host = $urlParts.Host
    
    # Test for HTTP to HTTPS redirect
    $script:TESTS_TOTAL++
    Write-Info "Testing HTTP to HTTPS redirect..."
    
    $httpUrl = "http://$host"
    
    try {
        # Disable automatic redirect following
        $response = Invoke-WebRequest -Uri $httpUrl -MaximumRedirection 0 -TimeoutSec 10 -ErrorAction SilentlyContinue
        
        # If we get here, no redirect
        Write-Failure "HTTP accessible without redirect (no HTTPS enforcement)"
        $script:TESTS_FAILED++
        $script:HIGH_RISK_FINDINGS += "No HTTP to HTTPS redirect - CWE-319"
    } catch {
        $statusCode = $_.Exception.Response.StatusCode.value__
        
        if ($statusCode -ge 301 -and $statusCode -le 308) {
            $location = $_.Exception.Response.Headers.Location
            if ($location -and $location.ToString() -match "^https://") {
                Write-Success "HTTP redirects to HTTPS (code: $statusCode)"
                $script:TESTS_PASSED++
            } else {
                Write-Warning "HTTP redirects but not to HTTPS"
                $script:TESTS_FAILED++
                $script:MEDIUM_RISK_FINDINGS += "HTTP redirect not to HTTPS - CWE-319"
            }
        } else {
            Write-Info "HTTP not accessible or blocked (code: $statusCode)"
            $script:TESTS_PASSED++
        }
    }
    
    # Test for HSTS header
    $script:TESTS_TOTAL++
    Write-Info "Testing for HTTP Strict Transport Security (HSTS)..."
    
    try {
        $response = Invoke-WebRequest -Uri $TargetUrl -Method Head -TimeoutSec 10 -ErrorAction SilentlyContinue
        $hstsHeader = $response.Headers.'Strict-Transport-Security'
        
        if ($hstsHeader) {
            Write-Success "HSTS header present: $hstsHeader"
            $script:TESTS_PASSED++
            
            if ($Mode -eq 'full') {
                # Check HSTS max-age
                $script:TESTS_TOTAL++
                if ($hstsHeader -match "max-age=(\d+)") {
                    $maxAge = [int]$Matches[1]
                    if ($maxAge -ge 31536000) {
                        Write-Success "HSTS max-age is adequate: $maxAge seconds"
                        $script:TESTS_PASSED++
                    } else {
                        Write-Warning "HSTS max-age may be too short: $maxAge seconds (recommend 31536000+)"
                        $script:TESTS_FAILED++
                        $script:LOW_RISK_FINDINGS += "Short HSTS max-age - CWE-319"
                    }
                }
                
                # Check for includeSubDomains
                $script:TESTS_TOTAL++
                if ($hstsHeader -match "includeSubDomains") {
                    Write-Success "HSTS includes subdomains"
                    $script:TESTS_PASSED++
                } else {
                    Write-Info "HSTS does not include subdomains"
                    $script:TESTS_PASSED++
                    $script:LOW_RISK_FINDINGS += "HSTS does not include subdomains"
                }
                
                # Check for preload
                $script:TESTS_TOTAL++
                if ($hstsHeader -match "preload") {
                    Write-Success "HSTS preload directive present"
                    $script:TESTS_PASSED++
                } else {
                    Write-Info "HSTS preload not enabled"
                    $script:TESTS_PASSED++
                }
            }
        } else {
            Write-Failure "HSTS header not present"
            $script:TESTS_FAILED++
            $script:MEDIUM_RISK_FINDINGS += "HTTP Strict Transport Security not enabled - CWE-319"
        }
    } catch {
        Write-Info "Could not check HSTS header: $($_.Exception.Message)"
        $script:TESTS_PASSED++
    }
    
    # Test for mixed content indicators
    $script:TESTS_TOTAL++
    Write-Info "Checking for mixed content indicators..."
    
    try {
        $response = Invoke-WebRequest -Uri $TargetUrl -TimeoutSec 10 -ErrorAction SilentlyContinue
        $content = $response.Content
        
        $mixedContentFound = $false
        
        if ($TargetUrl -match "^https://") {
            # Check for HTTP resources in HTTPS page
            if ($content -match 'src="http://[^"]+"|href="http://[^"]+') {
                Write-Warning "Mixed content detected (HTTP resources on HTTPS page)"
                $mixedContentFound = $true
            }
        }
        
        if ($mixedContentFound) {
            $script:TESTS_FAILED++
            $script:MEDIUM_RISK_FINDINGS += "Mixed content (HTTP resources on HTTPS) - CWE-319"
        } else {
            Write-Success "No obvious mixed content issues"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Info "Could not check for mixed content"
        $script:TESTS_PASSED++
    }
    
    # Test for secure cookie flags
    $script:TESTS_TOTAL++
    Write-Info "Testing for secure cookie flags..."
    
    try {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $response = Invoke-WebRequest -Uri $TargetUrl -WebSession $session -TimeoutSec 10 -ErrorAction SilentlyContinue
        
        $cookies = $session.Cookies.GetCookies($TargetUrl)
        $insecureCookies = $false
        
        foreach ($cookie in $cookies) {
            if (-not $cookie.Secure) {
                Write-Warning "Cookie without Secure flag: $($cookie.Name)"
                $insecureCookies = $true
            }
            if (-not $cookie.HttpOnly) {
                Write-Warning "Cookie without HttpOnly flag: $($cookie.Name)"
                $insecureCookies = $true
            }
        }
        
        if ($insecureCookies) {
            $script:TESTS_FAILED++
            $script:MEDIUM_RISK_FINDINGS += "Insecure cookie flags - CWE-319"
        } else {
            if ($cookies.Count -gt 0) {
                Write-Success "Cookies have appropriate security flags"
            } else {
                Write-Info "No cookies set in response"
            }
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Info "Could not check cookie flags"
        $script:TESTS_PASSED++
    }
    
    # Full mode: Test sensitive endpoints for HTTPS
    if ($Mode -eq 'full') {
        Write-Info "Testing sensitive endpoints for HTTPS enforcement..."
        $sensitivePaths = @("/login", "/api/login", "/admin", "/account", "/profile", "/checkout", "/payment")
        
        foreach ($path in $sensitivePaths) {
            $script:TESTS_TOTAL++
            $httpEndpoint = "http://$host$path"
            
            try {
                $response = Invoke-WebRequest -Uri $httpEndpoint -MaximumRedirection 0 -TimeoutSec 5 -ErrorAction SilentlyContinue
                
                if ($response.StatusCode -eq 200) {
                    Write-Failure "Sensitive endpoint accessible over HTTP: $path"
                    $script:TESTS_FAILED++
                    $script:HIGH_RISK_FINDINGS += "Sensitive endpoint $path accessible over HTTP - CWE-319"
                } else {
                    Write-Success "Endpoint protected: $path"
                    $script:TESTS_PASSED++
                }
            } catch {
                Write-Success "Endpoint protected: $path"
                $script:TESTS_PASSED++
            }
        }
    }
}

function Show-Summary {
    Write-Phase "Cryptographic Security Test Summary"
    
    $totalFindings = $script:HIGH_RISK_FINDINGS.Count + $script:MEDIUM_RISK_FINDINGS.Count + $script:LOW_RISK_FINDINGS.Count
    
    Write-Host "`nTarget: $TargetUrl"
    Write-Host "Mode: $Mode"
    Write-Host "Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')"
    Write-Host ""
    Write-Host "Tests Passed: $($script:TESTS_PASSED)/$($script:TESTS_TOTAL)"
    Write-Host "Tests Failed: $($script:TESTS_FAILED)/$($script:TESTS_TOTAL)"
    
    if ($totalFindings -gt 0) {
        Write-Host "`n--- Findings ---" -ForegroundColor Yellow
        
        if ($script:HIGH_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nHIGH RISK:" -ForegroundColor Red
            foreach ($finding in $script:HIGH_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Red
            }
        }
        
        if ($script:MEDIUM_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nMEDIUM RISK:" -ForegroundColor Yellow
            foreach ($finding in $script:MEDIUM_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Yellow
            }
        }
        
        if ($script:LOW_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nLOW RISK:" -ForegroundColor Cyan
            foreach ($finding in $script:LOW_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Cyan
            }
        }
    }
    
    # Calculate security score
    $maxScore = 100
    $deductions = ($script:HIGH_RISK_FINDINGS.Count * 20) + ($script:MEDIUM_RISK_FINDINGS.Count * 10) + ($script:LOW_RISK_FINDINGS.Count * 5)
    $score = [Math]::Max(0, $maxScore - $deductions)
    
    Write-Host "`n--- Security Score ---"
    $scoreColor = if ($score -ge 80) { "Green" } elseif ($score -ge 60) { "Yellow" } else { "Red" }
    Write-Host "Score: $score/100" -ForegroundColor $scoreColor
    
    Write-Host "`n--- CWEs Tested ---"
    Write-Host "CWE-326: Inadequate Encryption Strength"
    Write-Host "CWE-295: Improper Certificate Validation"
    Write-Host "CWE-327: Use of a Broken or Risky Cryptographic Algorithm"
    Write-Host "CWE-319: Cleartext Transmission of Sensitive Information"
}

# Main execution
if ($Help) {
    Show-Help
    exit 0
}

if (-not $TargetUrl) {
    Write-Host "Error: Target URL is required." -ForegroundColor Red
    Write-Host "Usage: .\crypto-security-test.ps1 -TargetUrl <URL>"
    Write-Host "Use -Help for more information."
    exit 1
}

# Ensure URL has protocol
if ($TargetUrl -notmatch "^https?://") {
    $TargetUrl = "https://$TargetUrl"
}

# Normalize URL
$TargetUrl = $TargetUrl.TrimEnd('/')

Write-Host "`n========================================" -ForegroundColor Cyan
Write-Host "Cryptographic Security Testing Script" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Target: $TargetUrl"
Write-Host "Mode: $Mode"
Write-Host "Started: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')"

# Run test phases
Test-TlsProtocols
Test-Certificate
Test-WeakAlgorithms
Test-Cleartext

# Show summary
Show-Summary

Write-Host "`nCryptographic security testing completed." -ForegroundColor Cyan
