#!/bin/bash
# Business Logic Testing Script - CWE-362, CWE-770, CWE-840, CWE-841, CWE-208
# Usage: ./business-logic-test.sh [TARGET_URL] [MODE]

set -e
[[ "$1" == "-h" ]] && { echo "⚙️ Business Logic Scanner - Race conditions, Rate limiting, Workflow bypass"; exit 0; }

TARGET_URL="${1:-}"; MODE="${2:-full}"; CLEAN_OUTPUT="${3:-false}"
AUTH_COOKIE="${AUTH_COOKIE:-}"
AUTH_TOKEN="${AUTH_TOKEN:-}"
[ -z "$TARGET_URL" ] && { echo "❌ Error: Target URL required"; exit 1; }
[[ ! "$TARGET_URL" =~ ^https?:// ]] && TARGET_URL="https://$TARGET_URL"
TARGET_URL="${TARGET_URL%/}"

[[ "$CLEAN_OUTPUT" = "true" ]] && { RED=''; GREEN=''; YELLOW=''; PURPLE=''; NC=''; } || \
{ RED='\033[0;31m'; GREEN='\033[0;32m'; YELLOW='\033[1;33m'; PURPLE='\033[0;35m'; NC='\033[0m'; }

TESTS_PASSED=0; TESTS_FAILED=0; TESTS_TOTAL=0; HIGH_RISK_FINDINGS=0; MEDIUM_RISK_FINDINGS=0; HIGH_RISK_LIST=(); MEDIUM_RISK_LIST=()
AUTH_ERRORS=0  # Track 401/403 responses

# Helper function for authenticated curl requests (supports cookie and bearer token)
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

echo -e "${PURPLE}⚙️ BUSINESS LOGIC SCANNER${NC}"
echo -e "Target: ${TARGET_URL} | Mode: ${MODE}"

[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication (unauthenticated scan)${NC}"

echo -e ""

test_race_condition() {
    echo -e "\n${YELLOW}🏃 PHASE 1: RACE CONDITION TESTING (CWE-362)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    echo -e "Sending 10 concurrent requests to ${TARGET_URL}..."

    # Test concurrent requests to provided URL
    local pids=()
    local results_file=$(mktemp)
    trap 'rm -f "$results_file"' RETURN

    for i in {1..10}; do
        (auth_curl -s -o /dev/null -w "%{http_code}" "${TARGET_URL}" \
            -X POST -d "amount=100&to=user2" --max-time 5 2>/dev/null >> "$results_file") &
        pids+=($!)
    done

    for pid in "${pids[@]}"; do wait $pid 2>/dev/null; done

    local success_count=$(grep -c "200" "$results_file" 2>/dev/null | tr -d '\n' || echo 0)
    local auth_errors=$(grep -cE "401|403" "$results_file" 2>/dev/null | tr -d '\n' || echo 0)
    [ -z "$success_count" ] && success_count=0
    [ -z "$auth_errors" ] && auth_errors=0
    AUTH_ERRORS=$((AUTH_ERRORS + auth_errors))
    rm -f "$results_file"

    if [ "$success_count" -gt 5 ]; then
        echo -e "${YELLOW}  ℹ️ Multiple requests accepted (${success_count}/10) - verify manually for race condition${NC}"
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Potential Race Condition")
    fi

    echo -e "${GREEN}✅ PASS: Race condition test completed (manual verification recommended)${NC}"
    TESTS_PASSED=$((TESTS_PASSED + 1))
}

test_rate_limiting() {
    echo -e "\n${YELLOW}🚦 PHASE 2: RATE LIMITING (CWE-770)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    echo -e "Testing rate limiting on ${TARGET_URL}..."

    local blocked=false
    for i in {1..20}; do
        local response=$(auth_curl -si -X POST "${TARGET_URL}" \
            -d "test=data" --max-time 5 2>/dev/null || echo "")
        local status=$(echo "$response" | head -1 | grep -oE '[0-9]{3}' | head -1)

        # Track auth errors
        if [ "$status" = "401" ] || [ "$status" = "403" ]; then
            AUTH_ERRORS=$((AUTH_ERRORS + 1))
            continue
        fi

        if [ "$status" = "429" ] || echo "$response" | grep -qiE "rate limit|too many|blocked"; then
            blocked=true; break
        fi
    done

    if [ "$blocked" = false ]; then
        echo -e "${YELLOW}  ℹ️ No rate limiting detected on provided URL${NC}"
        vuln=true
    fi

    [ "$vuln" = true ] && { echo -e "${YELLOW}⚠️ Rate limiting may be missing (verify manually)${NC}"; MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1)); MEDIUM_RISK_LIST+=("Possible Missing Rate Limiting (CWE-770)"); TESTS_PASSED=$((TESTS_PASSED + 1)); } || \
    { echo -e "${GREEN}✅ PASS: Rate limiting present${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

test_workflow_bypass() {
    echo -e "\n${YELLOW}🔄 PHASE 3: WORKFLOW BYPASS (CWE-841)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    echo -e "Testing workflow bypass on ${TARGET_URL}..."

    # Test direct access to provided URL with completion payload
    local response=$(auth_curl -si -X POST "${TARGET_URL}" \
        -H "Content-Type: application/json" \
        -d '{"confirmed":true,"step":"complete"}' --max-time 10 2>/dev/null || echo "")

    # Check for auth errors
    if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
        AUTH_ERRORS=$((AUTH_ERRORS + 1))
    fi

    if echo "$response" | grep -qiE "success|completed|order.*created|payment.*processed"; then
        echo -e "${YELLOW}  ℹ️ Endpoint accepts completion payload - verify workflow enforcement${NC}"
        vuln=true
    fi

    [ "$vuln" = true ] && { echo -e "${YELLOW}⚠️ Possible workflow bypass (verify manually)${NC}"; MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1)); MEDIUM_RISK_LIST+=("Possible Workflow Bypass (CWE-841)"); TESTS_PASSED=$((TESTS_PASSED + 1)); } || \
    { echo -e "${GREEN}✅ PASS: Workflow appears protected${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

test_parameter_tampering() {
    echo -e "\n${YELLOW}💰 PHASE 4: PARAMETER TAMPERING${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false

    echo -e "Testing parameter tampering on ${TARGET_URL}..."

    # Test price/amount manipulation
    local tamper_payloads=(
        '{"price":-100}'
        '{"amount":0.001}'
        '{"quantity":-1}'
        '{"discount":100}'
        '{"total":0}'
    )

    for payload in "${tamper_payloads[@]}"; do
        local response=$(auth_curl -si -X POST "${TARGET_URL}" \
            -H "Content-Type: application/json" \
            -d "$payload" --max-time 10 2>/dev/null || echo "")

        # Check for auth errors
        if echo "$response" | head -1 | grep -qE "HTTP/[0-9.]+ (401|403)"; then
            AUTH_ERRORS=$((AUTH_ERRORS + 1))
            continue
        fi

        if echo "$response" | grep -qiE "success|order.*created|payment.*0"; then
            echo -e "${RED}  ⚠️ Parameter tampering accepted: $payload${NC}"
            vuln=true
        fi
    done

    [ "$vuln" = true ] && { echo -e "${RED}❌ FAIL: Parameter tampering${NC}"; TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("Parameter Tampering"); } || \
    { echo -e "${GREEN}✅ PASS: Parameters validated${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

test_timing_attack() {
    echo -e "\n${YELLOW}⏱️ PHASE 5: TIMING ATTACK (CWE-208)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1))

    echo -e "Testing timing differences on ${TARGET_URL}..."

    local times=()
    local valid_user="admin"
    local invalid_user="nonexistent12345"

    # Measure response times with different inputs
    for i in {1..5}; do
        local start=$(date +%s%N)
        auth_curl -si -X POST "${TARGET_URL}" -d "username=${valid_user}&password=wrong" --max-time 10 > /dev/null 2>&1
        local end=$(date +%s%N)
        times+=($((($end - $start) / 1000000)))
    done

    local valid_avg=0
    for t in "${times[@]}"; do valid_avg=$((valid_avg + t)); done
    valid_avg=$((valid_avg / ${#times[@]}))

    times=()
    for i in {1..5}; do
        local start=$(date +%s%N)
        auth_curl -si -X POST "${TARGET_URL}" -d "username=${invalid_user}&password=wrong" --max-time 10 > /dev/null 2>&1
        local end=$(date +%s%N)
        times+=($((($end - $start) / 1000000)))
    done

    local invalid_avg=0
    for t in "${times[@]}"; do invalid_avg=$((invalid_avg + t)); done
    invalid_avg=$((invalid_avg / ${#times[@]}))

    local diff=$((valid_avg - invalid_avg))
    [ $diff -lt 0 ] && diff=$((-diff))

    echo -e "  Input 1 avg: ${valid_avg}ms, Input 2 avg: ${invalid_avg}ms, Diff: ${diff}ms"

    if [ $diff -gt 100 ]; then
        echo -e "${YELLOW}⚠️ Significant timing difference detected${NC}"
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Potential Timing Attack")
    fi

    echo -e "${GREEN}✅ PASS: Timing test completed${NC}"
    TESTS_PASSED=$((TESTS_PASSED + 1))
}

generate_summary() {
    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}📊 BUSINESS LOGIC SUMMARY${NC}"
    local rate=0; [ $TESTS_TOTAL -gt 0 ] && rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    echo -e "Tests: ${TESTS_TOTAL} | ${GREEN}Passed: ${TESTS_PASSED}${NC} | ${RED}Failed: ${TESTS_FAILED}${NC} | Rate: ${rate}%"
    echo -e "${RED}High Risk: ${HIGH_RISK_FINDINGS}${NC} | ${YELLOW}Medium Risk: ${MEDIUM_RISK_FINDINGS}${NC}"
    [ ${#HIGH_RISK_LIST[@]} -gt 0 ] && { echo -e "${RED}🚨 HIGH RISK:${NC}"; for f in "${HIGH_RISK_LIST[@]}"; do echo -e "  ${RED}• ${f}${NC}"; done; }
    [ ${#MEDIUM_RISK_LIST[@]} -gt 0 ] && { echo -e "${YELLOW}⚠️ MEDIUM RISK:${NC}"; for f in "${MEDIUM_RISK_LIST[@]}"; do echo -e "  ${YELLOW}• ${f}${NC}"; done; }

    # Warn about authentication errors
    if [ $AUTH_ERRORS -gt 0 ]; then
        echo -e "\n${YELLOW}⚠️  WARNING: ${AUTH_ERRORS} requests returned 401/403 Unauthorized${NC}"
        if [ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ]; then
            echo -e "${YELLOW}   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints.${NC}"
        else
            echo -e "${YELLOW}   Provided credentials may be invalid or expired.${NC}"
        fi
    fi

    local score=$((100 - HIGH_RISK_FINDINGS * 20 - MEDIUM_RISK_FINDINGS * 10)); [ $score -lt 0 ] && score=0
    echo -e "Score: $score/100"
    echo -e "${PURPLE}======================================================${NC}"
}

test_race_condition
test_rate_limiting
[ "$MODE" = "full" ] && { test_workflow_bypass; test_parameter_tampering; test_timing_attack; }
generate_summary
