<#
.SYNOPSIS
    Business Logic Security Testing Script for Windows/PowerShell
    Tests for race conditions, rate limiting, workflow bypass, parameter tampering, and timing attacks.

.DESCRIPTION
    This script performs comprehensive business logic security testing against web applications.
    It covers CWE-362, CWE-770, CWE-841, and CWE-208 vulnerabilities.

.PARAMETER TargetUrl
    The base URL of the target application to test.

.PARAMETER Mode
    Test mode: 'quick' for essential tests only, 'full' for comprehensive testing.

.PARAMETER CleanOutput
    Suppress colored output for logging purposes.

.PARAMETER Help
    Display help information.

.EXAMPLE
    .\business-logic-test.ps1 -TargetUrl "http://testphp.vulnweb.com"
    .\business-logic-test.ps1 -TargetUrl "http://example.com" -Mode full
#>

param(
    [Parameter(Position=0)]
    [string]$TargetUrl,
    
    [Parameter()]
    [ValidateSet('quick', 'full')]
    [string]$Mode = 'quick',
    
    [Parameter()]
    [switch]$CleanOutput,
    
    [Parameter()]
    [switch]$Help
)

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [object]$Body, [string]$ContentType, [hashtable]$Headers, [int]$TimeoutSec = 10)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    if ($Body) { $params.Body = $Body }
    if ($ContentType) { $params.ContentType = $ContentType }
    $authHeaders = @{}
    if ($Headers) { $Headers.Keys | ForEach-Object { $authHeaders[$_] = $Headers[$_] } }
    if ($AuthToken) { $authHeaders["Authorization"] = "Bearer $AuthToken" }
    if ($authHeaders.Count -gt 0) { $params.Headers = $authHeaders }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

$script:AuthErrors = 0  # Track 401/403 responses

# Color output functions
function Write-Success { param([string]$Message) if (-not $CleanOutput) { Write-Host "[+] $Message" -ForegroundColor Green } else { Write-Host "[+] $Message" } }
function Write-Failure { param([string]$Message) if (-not $CleanOutput) { Write-Host "[-] $Message" -ForegroundColor Red } else { Write-Host "[-] $Message" } }
function Write-Info { param([string]$Message) if (-not $CleanOutput) { Write-Host "[*] $Message" -ForegroundColor Blue } else { Write-Host "[*] $Message" } }
function Write-Warning { param([string]$Message) if (-not $CleanOutput) { Write-Host "[!] $Message" -ForegroundColor Yellow } else { Write-Host "[!] $Message" } }
function Write-Phase { param([string]$Message) if (-not $CleanOutput) { Write-Host "`n=== $Message ===" -ForegroundColor Cyan } else { Write-Host "`n=== $Message ===" } }

# Test tracking variables
$script:TESTS_PASSED = 0
$script:TESTS_FAILED = 0
$script:TESTS_TOTAL = 0
$script:HIGH_RISK_FINDINGS = @()
$script:MEDIUM_RISK_FINDINGS = @()
$script:LOW_RISK_FINDINGS = @()

function Show-Help {
    Write-Host @"

Business Logic Security Testing Script
======================================

Usage: .\business-logic-test.ps1 -TargetUrl <URL> [options]

Options:
    -TargetUrl      Target URL to test (required)
    -Mode           Test mode: 'quick' or 'full' (default: quick)
    -CleanOutput    Suppress colored output
    -Help           Show this help message

Examples:
    .\business-logic-test.ps1 -TargetUrl "http://testphp.vulnweb.com"
    .\business-logic-test.ps1 -TargetUrl "http://example.com" -Mode full

Test Phases:
    Phase 1: Race Condition Testing (CWE-362)
    Phase 2: Rate Limiting Testing (CWE-770)
    Phase 3: Workflow Bypass Testing (CWE-841)
    Phase 4: Parameter Tampering Testing
    Phase 5: Timing Attack Testing (CWE-208)

"@
}

function Get-FindingDetails {
    param([string]$FindingType)
    
    switch ($FindingType) {
        "race_condition" {
            return @{
                CWE = "CWE-362"
                Risk = "HIGH"
                Description = "Race Condition vulnerability allows attackers to exploit time-of-check to time-of-use (TOCTOU) flaws"
                Impact = "Financial fraud, privilege escalation, data corruption"
                Remediation = "Implement proper locking mechanisms, atomic operations, and transaction isolation"
            }
        }
        "rate_limiting" {
            return @{
                CWE = "CWE-770"
                Risk = "MEDIUM"
                Description = "Allocation of Resources Without Limits allows resource exhaustion attacks"
                Impact = "Denial of service, brute force attacks, account lockout bypass"
                Remediation = "Implement rate limiting, throttling, and resource quotas"
            }
        }
        "workflow_bypass" {
            return @{
                CWE = "CWE-841"
                Risk = "HIGH"
                Description = "Improper Enforcement of Behavioral Workflow allows skipping critical steps"
                Impact = "Authentication bypass, payment bypass, verification bypass"
                Remediation = "Enforce server-side workflow validation and state management"
            }
        }
        "timing_attack" {
            return @{
                CWE = "CWE-208"
                Risk = "MEDIUM"
                Description = "Observable Timing Discrepancy allows information leakage through response times"
                Impact = "Username enumeration, password guessing, cryptographic key recovery"
                Remediation = "Use constant-time comparison functions and uniform response times"
            }
        }
        default {
            return @{
                CWE = "Unknown"
                Risk = "INFO"
                Description = "Unknown vulnerability type"
                Impact = "Unknown"
                Remediation = "Further investigation required"
            }
        }
    }
}

function Test-RaceConditions {
    Write-Phase "Phase 1: Race Condition Testing (CWE-362)"

    $script:TESTS_TOTAL++
    Write-Info "Testing race condition on $TargetUrl..."

    try {
        # Send concurrent requests to detect race conditions
        $jobs = @()
        $results = @()

        for ($i = 0; $i -lt 5; $i++) {
            $jobs += Start-Job -ScriptBlock {
                param($url)
                try {
                    $response = Invoke-WebRequest -Uri $url -Method POST -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                    return @{ StatusCode = $response.StatusCode; Success = $true }
                } catch {
                    return @{ StatusCode = $_.Exception.Response.StatusCode.value__; Success = $false; Error = $_.Exception.Message }
                }
            } -ArgumentList $TargetUrl
        }

        # Wait for jobs with timeout
        $null = Wait-Job -Job $jobs -Timeout 15

        foreach ($job in $jobs) {
            $result = Receive-Job -Job $job -ErrorAction SilentlyContinue
            if ($result) { $results += $result }
            Remove-Job -Job $job -Force -ErrorAction SilentlyContinue
        }

        # Track auth errors
        $authErrorCount = ($results | Where-Object { $_.StatusCode -eq 401 -or $_.StatusCode -eq 403 }).Count
        $script:AuthErrors += $authErrorCount

        # Analyze results for race condition indicators
        $successCount = ($results | Where-Object { $_.StatusCode -eq 200 -or $_.StatusCode -eq 201 }).Count

        if ($successCount -gt 1) {
            Write-Warning "Potential race condition: Multiple successful requests ($successCount/5)"
            $script:TESTS_FAILED++
            $details = Get-FindingDetails "race_condition"
            $script:HIGH_RISK_FINDINGS += "Race condition vulnerability - $($details.CWE)"
        } else {
            Write-Success "No obvious race condition detected"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Info "Error during race condition test: $($_.Exception.Message)"
        $script:TESTS_PASSED++
    }
}

function Test-RateLimiting {
    Write-Phase "Phase 2: Rate Limiting Testing (CWE-770)"

    $script:TESTS_TOTAL++
    Write-Info "Testing rate limiting on $TargetUrl..."

    try {
        $successfulRequests = 0
        $rateLimited = $false
        $requestCount = if ($Mode -eq 'full') { 20 } else { 10 }

        for ($i = 0; $i -lt $requestCount; $i++) {
            try {
                $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method POST -TimeoutSec 5 -ErrorAction Stop
                if ($response.StatusCode -eq 401 -or $response.StatusCode -eq 403) {
                    $script:AuthErrors++
                    continue
                }
                $successfulRequests++
            } catch {
                $statusCode = $_.Exception.Response.StatusCode.value__
                if ($statusCode -eq 401 -or $statusCode -eq 403) {
                    $script:AuthErrors++
                    continue
                }
                if ($statusCode -eq 429 -or $statusCode -eq 503) {
                    $rateLimited = $true
                    break
                }
                # Check for rate limit headers
                $headers = $_.Exception.Response.Headers
                if ($headers -and ($headers['X-RateLimit-Remaining'] -eq '0' -or $headers['Retry-After'])) {
                    $rateLimited = $true
                    break
                }
            }
        }

        if ($rateLimited) {
            Write-Success "Rate limiting detected after $successfulRequests requests"
            $script:TESTS_PASSED++
        } elseif ($successfulRequests -ge ($requestCount - 2)) {
            Write-Warning "No rate limiting detected ($successfulRequests successful requests)"
            $details = Get-FindingDetails "rate_limiting"
            $script:MEDIUM_RISK_FINDINGS += "Possible missing rate limiting - $($details.CWE)"
            $script:TESTS_PASSED++
        } else {
            Write-Success "Endpoint protected or not accessible"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Info "Could not test rate limiting: $($_.Exception.Message)"
        $script:TESTS_PASSED++
    }
}

function Test-WorkflowBypass {
    Write-Phase "Phase 3: Workflow Bypass Testing (CWE-841)"

    $script:TESTS_TOTAL++
    Write-Info "Testing workflow bypass on $TargetUrl..."

    try {
        # Try to send completion payload directly
        $body = '{"confirmed":true,"step":"complete"}'
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method POST -Body $body -ContentType "application/json" -TimeoutSec 10 -ErrorAction SilentlyContinue

        if ($response.StatusCode -eq 401 -or $response.StatusCode -eq 403) {
            $script:AuthErrors++
            Write-Success "Endpoint requires authentication"
            $script:TESTS_PASSED++
            return
        }

        if ($response.StatusCode -eq 200) {
            # Check if response indicates successful bypass
            $content = $response.Content.ToLower()
            if ($content -match "success|completed|order.*created|payment.*processed") {
                Write-Warning "Potential workflow bypass: Completion payload accepted"
                $details = Get-FindingDetails "workflow_bypass"
                $script:MEDIUM_RISK_FINDINGS += "Possible workflow bypass - $($details.CWE)"
            }
        }
        Write-Success "Workflow bypass test completed"
        $script:TESTS_PASSED++
    } catch {
        $statusCode = $_.Exception.Response.StatusCode.value__
        if ($statusCode -eq 401 -or $statusCode -eq 403) {
            $script:AuthErrors++
            Write-Success "Endpoint requires authentication"
        } else {
            Write-Info "Could not fully test workflow: $($_.Exception.Message)"
        }
        $script:TESTS_PASSED++
    }
}

function Test-ParameterTampering {
    Write-Phase "Phase 4: Parameter Tampering Testing"

    $script:TESTS_TOTAL++
    Write-Info "Testing parameter tampering on $TargetUrl..."

    $tamperPayloads = @(
        '{"price":-100}',
        '{"amount":0.001}',
        '{"quantity":-1}',
        '{"discount":100}',
        '{"total":0}'
    )

    $tamperFound = $false
    foreach ($payload in $tamperPayloads) {
        try {
            $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method POST -Body $payload -ContentType "application/json" -TimeoutSec 5 -ErrorAction SilentlyContinue

            if ($response.StatusCode -eq 401 -or $response.StatusCode -eq 403) {
                $script:AuthErrors++
                continue
            }

            if ($response.StatusCode -eq 200) {
                $content = $response.Content.ToLower()
                if ($content -match "success|order.*created|payment.*0") {
                    Write-Warning "Parameter tampering may be possible: $payload"
                    $tamperFound = $true
                }
            }
        } catch {
            if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) {
                $script:AuthErrors++
            }
        }
    }

    if ($tamperFound) {
        $script:TESTS_FAILED++
        $script:HIGH_RISK_FINDINGS += "Parameter tampering vulnerability (price/quantity manipulation)"
    } else {
        Write-Success "No obvious parameter tampering vulnerabilities detected"
        $script:TESTS_PASSED++
    }
}

function Test-TimingAttacks {
    Write-Phase "Phase 5: Timing Attack Testing (CWE-208)"

    $script:TESTS_TOTAL++
    Write-Info "Testing for timing differences on $TargetUrl..."

    # Test timing differences with different inputs
    $timingData = @()
    $testInputs = @("admin", "nonexistent12345xyz")

    foreach ($input in $testInputs) {
        $times = @()
        for ($i = 0; $i -lt 3; $i++) {
            try {
                $stopwatch = [System.Diagnostics.Stopwatch]::StartNew()
                $body = @{ username = $input; password = "wrongpassword123" } | ConvertTo-Json
                $null = Invoke-AuthWebRequest -Uri "$TargetUrl" -Method POST -Body $body -ContentType "application/json" -TimeoutSec 10 -ErrorAction SilentlyContinue
                $stopwatch.Stop()
                $times += $stopwatch.ElapsedMilliseconds
            } catch {
                $stopwatch.Stop()
                $times += $stopwatch.ElapsedMilliseconds
            }
        }

        $avgTime = ($times | Measure-Object -Average).Average
        $timingData += @{ Input = $input; AvgTime = $avgTime }
    }

    # Analyze timing differences
    if ($timingData.Count -gt 1) {
        $avgTimes = $timingData | ForEach-Object { $_.AvgTime }
        $minTime = ($avgTimes | Measure-Object -Minimum).Minimum
        $maxTime = ($avgTimes | Measure-Object -Maximum).Maximum
        $timeDiff = $maxTime - $minTime

        Write-Info "Input 1 avg: $([Math]::Round($timingData[0].AvgTime))ms, Input 2 avg: $([Math]::Round($timingData[1].AvgTime))ms, Diff: $([Math]::Round($timeDiff))ms"

        if ($timeDiff -gt 100) {
            Write-Warning "Significant timing difference detected: $([Math]::Round($timeDiff))ms"
            $details = Get-FindingDetails "timing_attack"
            $script:MEDIUM_RISK_FINDINGS += "Potential timing attack - $($details.CWE)"
        } else {
            Write-Success "No significant timing differences detected"
        }
        $script:TESTS_PASSED++
    } else {
        $script:TESTS_PASSED++
    }
}

function Show-Summary {
    Write-Phase "Business Logic Security Test Summary"
    
    $totalFindings = $script:HIGH_RISK_FINDINGS.Count + $script:MEDIUM_RISK_FINDINGS.Count + $script:LOW_RISK_FINDINGS.Count
    
    Write-Host "`nTarget: $TargetUrl"
    Write-Host "Mode: $Mode"
    Write-Host "Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')"
    Write-Host ""
    Write-Host "Tests Passed: $($script:TESTS_PASSED)/$($script:TESTS_TOTAL)"
    Write-Host "Tests Failed: $($script:TESTS_FAILED)/$($script:TESTS_TOTAL)"
    
    if ($totalFindings -gt 0) {
        Write-Host "`n--- Findings ---" -ForegroundColor Yellow
        
        if ($script:HIGH_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nHIGH RISK:" -ForegroundColor Red
            foreach ($finding in $script:HIGH_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Red
            }
        }
        
        if ($script:MEDIUM_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nMEDIUM RISK:" -ForegroundColor Yellow
            foreach ($finding in $script:MEDIUM_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Yellow
            }
        }
        
        if ($script:LOW_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nLOW RISK:" -ForegroundColor Cyan
            foreach ($finding in $script:LOW_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Cyan
            }
        }
    }
    
    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "⚠️  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    # Calculate security score
    $maxScore = 100
    $deductions = ($script:HIGH_RISK_FINDINGS.Count * 20) + ($script:MEDIUM_RISK_FINDINGS.Count * 10) + ($script:LOW_RISK_FINDINGS.Count * 5)
    $score = [Math]::Max(0, $maxScore - $deductions)

    Write-Host "`n--- Security Score ---"
    $scoreColor = if ($score -ge 80) { "Green" } elseif ($score -ge 60) { "Yellow" } else { "Red" }
    Write-Host "Score: $score/100" -ForegroundColor $scoreColor
    
    Write-Host "`n--- CWEs Tested ---"
    Write-Host "CWE-362: Concurrent Execution using Shared Resource with Improper Synchronization"
    Write-Host "CWE-770: Allocation of Resources Without Limits or Throttling"
    Write-Host "CWE-841: Improper Enforcement of Behavioral Workflow"
    Write-Host "CWE-208: Observable Timing Discrepancy"
}

# Main execution
if ($Help) {
    Show-Help
    exit 0
}

if (-not $TargetUrl) {
    Write-Host "Error: Target URL is required." -ForegroundColor Red
    Write-Host "Usage: .\business-logic-test.ps1 -TargetUrl <URL>"
    Write-Host "Use -Help for more information."
    exit 1
}

# Normalize URL
$TargetUrl = $TargetUrl.TrimEnd('/')

Write-Host "`n========================================" -ForegroundColor Cyan
Write-Host "Business Logic Security Testing Script" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Target: $TargetUrl"
Write-Host "Mode: $Mode"
Write-Host "Started: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')"
if ($AuthToken) { Write-Host "🔐 Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "🔐 Cookie authentication enabled" -ForegroundColor Green }
if (-not $AuthToken -and -not $AuthCookie) { Write-Host "ℹ️ Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

# Run test phases
Test-RaceConditions
Test-RateLimiting
Test-WorkflowBypass
Test-ParameterTampering
Test-TimingAttacks

# Show summary
Show-Summary

Write-Host "`nBusiness logic security testing completed." -ForegroundColor Cyan
