# Authentication & Session Security Testing Script (PowerShell)
# Comprehensive testing for authentication and session management vulnerabilities
# Covers: CWE-384, CWE-521, CWE-307, CWE-200, CWE-330, CWE-613, CWE-287, CWE-306
# Usage: .\auth-security-test.ps1 -TargetUrl <URL> [-Mode <quick|full>] [-CleanOutput]

param(
    [Parameter(Position=0)]
    [string]$TargetUrl,
    
    [Parameter(Position=1)]
    [ValidateSet("quick", "full")]
    [string]$Mode = "full",
    
    [switch]$CleanOutput,
    [switch]$Help,
    
    # Configurable endpoints
    [string]$LoginEndpoint = "/login",
    [string]$LogoutEndpoint = "/logout",
    [string]$RegisterEndpoint = "/register",
    [string]$ResetEndpoint = "/reset-password",
    [string]$UsernameField = "username",
    [string]$PasswordField = "password",
    [string]$ValidUsername = "admin",
    [string]$ValidPassword = "admin123"
)

# Show help
if ($Help) {
    Write-Host "🔐 AUTHENTICATION & SESSION SECURITY SCANNER" -ForegroundColor Magenta
    Write-Host "======================================================" -ForegroundColor Magenta
    Write-Host "Comprehensive security testing for authentication mechanisms" -ForegroundColor Blue
    Write-Host "Tests session management, password policies, and auth bypass vectors`n" -ForegroundColor Gray
    
    Write-Host "USAGE:" -ForegroundColor Yellow
    Write-Host "  .\auth-security-test.ps1 -TargetUrl <URL> [-Mode <quick|full>] [-CleanOutput]`n"
    
    Write-Host "PARAMETERS:" -ForegroundColor Yellow
    Write-Host "  -TargetUrl        Target web application URL (required)"
    Write-Host "  -Mode             Testing mode: quick or full (default: full)"
    Write-Host "  -CleanOutput      Disable colored output"
    Write-Host "  -LoginEndpoint    Login endpoint path (default: /login)"
    Write-Host "  -ValidUsername    Valid username for testing (default: admin)"
    Write-Host "  -ValidPassword    Valid password for testing (default: admin123)`n"
    
    Write-Host "EXAMPLES:" -ForegroundColor Yellow
    Write-Host "  .\auth-security-test.ps1 -TargetUrl http://localhost:3000 -Mode full"
    Write-Host "  .\auth-security-test.ps1 http://example.com quick`n"
    
    Write-Host "VULNERABILITY COVERAGE:" -ForegroundColor Yellow
    Write-Host "  • Session Fixation (CWE-384)"
    Write-Host "  • Weak Password Policy (CWE-521)"
    Write-Host "  • Brute Force / No Rate Limiting (CWE-307)"
    Write-Host "  • User Enumeration (CWE-200)"
    Write-Host "  • Weak Session Tokens (CWE-330)"
    Write-Host "  • Missing Session Timeout (CWE-613)"
    Write-Host "  • Authentication Bypass (CWE-287)"
    Write-Host "  • Missing Cookie Security Flags (CWE-1004)`n"
    exit 0
}

# Validate input
if ([string]::IsNullOrEmpty($TargetUrl)) {
    Write-Host "❌ Error: Target URL is required" -ForegroundColor Red
    Write-Host "Usage: .\auth-security-test.ps1 -TargetUrl <URL> [-Mode <quick|full>]" -ForegroundColor Blue
    Write-Host "Example: .\auth-security-test.ps1 -TargetUrl http://localhost:3000 -Mode full" -ForegroundColor Blue
    exit 1
}

# Ensure URL has protocol
if ($TargetUrl -notmatch "^https?://") {
    $TargetUrl = "https://$TargetUrl"
}

# Remove trailing slash
$TargetUrl = $TargetUrl.TrimEnd('/')

# Test tracking
$script:TestsPassed = 0
$script:TestsFailed = 0
$script:TestsTotal = 0
$script:HighRiskFindings = 0
$script:MediumRiskFindings = 0
$script:LowRiskFindings = 0

# Findings lists
$script:HighRiskList = @()
$script:MediumRiskList = @()
$script:LowRiskList = @()
$script:InfoList = @()

# Color helper functions
function Write-Success($message) {
    if ($CleanOutput) { Write-Host "✅ PASS: $message" }
    else { Write-Host "✅ PASS: $message" -ForegroundColor Green }
}

function Write-Failure($message) {
    if ($CleanOutput) { Write-Host "❌ FAIL: $message" }
    else { Write-Host "❌ FAIL: $message" -ForegroundColor Red }
}

function Write-Warning($message) {
    if ($CleanOutput) { Write-Host "⚠️ $message" }
    else { Write-Host "⚠️ $message" -ForegroundColor Yellow }
}

function Write-Info($message) {
    if ($CleanOutput) { Write-Host "ℹ️ $message" }
    else { Write-Host "ℹ️ $message" -ForegroundColor Gray }
}

function Write-Phase($message) {
    if ($CleanOutput) { Write-Host "`n$message" }
    else { Write-Host "`n$message" -ForegroundColor Yellow }
}

function Write-Detail($message) {
    if ($CleanOutput) { Write-Host "   $message" }
    else { Write-Host "   $message" -ForegroundColor Cyan }
}

# Get finding details
function Get-FindingDetails {
    param([string]$FindingType)
    
    switch ($FindingType) {
        "SESSION-FIXATION" {
            Write-Host "🚨 FOUND: Session Fixation Vulnerability (CWE-384)" -ForegroundColor Red
            Write-Host "   RISK: High - Attackers can hijack user sessions"
            Write-Host "   WHAT: Session ID not regenerated after authentication"
            Write-Host "   FIX: Regenerate session ID upon successful login"
        }
        "WEAK-PASSWORD" {
            Write-Host "🚨 FOUND: Weak Password Policy (CWE-521)" -ForegroundColor Red
            Write-Host "   RISK: Medium - Easy to guess/brute force passwords"
            Write-Host "   WHAT: No minimum length, complexity, or common password checks"
            Write-Host "   FIX: Enforce min 8 chars, mixed case, numbers, special chars"
        }
        "NO-RATE-LIMIT" {
            Write-Host "🚨 FOUND: No Rate Limiting (CWE-307)" -ForegroundColor Red
            Write-Host "   RISK: High - Vulnerable to brute force attacks"
            Write-Host "   WHAT: No limit on login attempts"
            Write-Host "   FIX: Implement rate limiting, account lockout, CAPTCHA"
        }
        "USER-ENUM" {
            Write-Host "🚨 FOUND: User Enumeration (CWE-200)" -ForegroundColor Red
            Write-Host "   RISK: Medium - Attackers can discover valid usernames"
            Write-Host "   WHAT: Different responses for valid/invalid usernames"
            Write-Host "   FIX: Use generic error messages for all login failures"
        }
        "WEAK-TOKEN" {
            Write-Host "🚨 FOUND: Weak Session Token (CWE-330)" -ForegroundColor Red
            Write-Host "   RISK: High - Session tokens can be predicted"
            Write-Host "   WHAT: Predictable or low-entropy session identifiers"
            Write-Host "   FIX: Use cryptographically secure random token generation"
        }
        "NO-TIMEOUT" {
            Write-Host "🚨 FOUND: No Session Timeout (CWE-613)" -ForegroundColor Red
            Write-Host "   RISK: Medium - Sessions remain active indefinitely"
            Write-Host "   WHAT: No idle or absolute session timeout"
            Write-Host "   FIX: Implement 15-30 min idle timeout, 8-24hr absolute timeout"
        }
        "AUTH-BYPASS" {
            Write-Host "🚨 FOUND: Authentication Bypass (CWE-287)" -ForegroundColor Red
            Write-Host "   RISK: Critical - Access without valid credentials"
            Write-Host "   WHAT: Authentication can be bypassed"
            Write-Host "   FIX: Enforce authentication on all protected resources"
        }
        "COOKIE-SECURE" {
            Write-Host "🔒 MISSING: Secure Cookie Flag (CWE-614)" -ForegroundColor Yellow
            Write-Host "   RISK: Medium - Cookie transmitted over HTTP"
            Write-Host "   WHAT: Session cookie missing Secure flag"
            Write-Host "   FIX: Set Secure flag on all session cookies"
        }
        "COOKIE-HTTPONLY" {
            Write-Host "🔒 MISSING: HttpOnly Cookie Flag (CWE-1004)" -ForegroundColor Yellow
            Write-Host "   RISK: Medium - Cookie accessible via JavaScript"
            Write-Host "   WHAT: Session cookie missing HttpOnly flag"
            Write-Host "   FIX: Set HttpOnly flag to prevent XSS cookie theft"
        }
        "COOKIE-SAMESITE" {
            Write-Host "🔒 MISSING: SameSite Cookie Attribute" -ForegroundColor Yellow
            Write-Host "   RISK: Medium - Vulnerable to CSRF attacks"
            Write-Host "   WHAT: Session cookie missing SameSite attribute"
            Write-Host "   FIX: Set SameSite=Strict or SameSite=Lax"
        }
    }
}

# Extract cookies from response
function Get-SessionCookies {
    param($Response)
    
    $cookies = @()
    if ($Response.Headers -and $Response.Headers["Set-Cookie"]) {
        $cookieHeaders = $Response.Headers["Set-Cookie"]
        if ($cookieHeaders -is [array]) {
            $cookies = $cookieHeaders
        } else {
            $cookies = @($cookieHeaders)
        }
    }
    return $cookies
}

# Extract session ID from cookies
function Get-SessionId {
    param([array]$Cookies)
    
    $sessionNames = @("JSESSIONID", "PHPSESSID", "ASP.NET_SessionId", "sessionid", "session", "sid", "connect.sid", "token")
    
    foreach ($name in $sessionNames) {
        foreach ($cookie in $Cookies) {
            if ($cookie -match "$name=([^;]+)") {
                return $matches[1]
            }
        }
    }
    
    # Return first cookie value as fallback
    if ($Cookies.Count -gt 0 -and $Cookies[0] -match "=([^;]+)") {
        return $matches[1]
    }
    return $null
}

# PHASE 1: Cookie Security Flags Testing
function Test-CookieSecurity {
    Write-Phase "🍪 PHASE 1: COOKIE SECURITY FLAGS ANALYSIS"
    
    try {
        # Try to get cookies from login endpoint
        $response = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -UseBasicParsing -SessionVariable session -ErrorAction SilentlyContinue
        $cookies = Get-SessionCookies -Response $response
        
        if ($cookies.Count -eq 0) {
            Write-Info "No cookies set on initial request, attempting login..."
            
            $body = @{
                $UsernameField = $ValidUsername
                $PasswordField = $ValidPassword
            }
            
            $response = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $body -UseBasicParsing -SessionVariable session -ErrorAction SilentlyContinue
            $cookies = Get-SessionCookies -Response $response
        }
        
        if ($cookies.Count -eq 0) {
            Write-Info "No cookies detected - may use different auth mechanism (JWT, etc.)"
            $script:InfoList += "No traditional session cookies detected"
            return
        }
        
        Write-Detail "Found cookies:"
        $cookies | Select-Object -First 5 | ForEach-Object { Write-Host "  $_" -ForegroundColor Gray }
        Write-Host ""
        
        # Test Secure flag
        $script:TestsTotal++
        $hasSecure = $cookies | Where-Object { $_ -match "Secure" }
        if ($hasSecure) {
            Write-Success "Secure flag present"
            $script:TestsPassed++
        } else {
            Write-Failure "Secure flag missing"
            Get-FindingDetails "COOKIE-SECURE"
            Write-Host ""
            $script:TestsFailed++
            $script:MediumRiskFindings++
            $script:MediumRiskList += "Missing Secure cookie flag"
        }
        
        # Test HttpOnly flag
        $script:TestsTotal++
        $hasHttpOnly = $cookies | Where-Object { $_ -match "HttpOnly" }
        if ($hasHttpOnly) {
            Write-Success "HttpOnly flag present"
            $script:TestsPassed++
        } else {
            Write-Failure "HttpOnly flag missing"
            Get-FindingDetails "COOKIE-HTTPONLY"
            Write-Host ""
            $script:TestsFailed++
            $script:MediumRiskFindings++
            $script:MediumRiskList += "Missing HttpOnly cookie flag"
        }
        
        # Test SameSite attribute
        $script:TestsTotal++
        $hasSameSite = $cookies | Where-Object { $_ -match "SameSite" }
        if ($hasSameSite) {
            Write-Success "SameSite attribute present"
            $script:TestsPassed++
        } else {
            Write-Failure "SameSite attribute missing"
            Get-FindingDetails "COOKIE-SAMESITE"
            Write-Host ""
            $script:TestsFailed++
            $script:MediumRiskFindings++
            $script:MediumRiskList += "Missing SameSite cookie attribute"
        }
    }
    catch {
        Write-Info "Error testing cookie security: $_"
    }
}

# PHASE 2: Session Fixation Testing
function Test-SessionFixation {
    Write-Phase "🔄 PHASE 2: SESSION FIXATION TESTING"
    
    try {
        # Get pre-login session
        $preLoginResponse = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -UseBasicParsing -SessionVariable preSession -ErrorAction SilentlyContinue
        $preLoginCookies = Get-SessionCookies -Response $preLoginResponse
        $preLoginSessionId = Get-SessionId -Cookies $preLoginCookies
        
        if ([string]::IsNullOrEmpty($preLoginSessionId)) {
            Write-Info "No pre-login session ID detected"
            $script:InfoList += "No pre-login session ID - may be secure by design"
            return
        }
        
        Write-Detail "Pre-login session ID: $($preLoginSessionId.Substring(0, [Math]::Min(20, $preLoginSessionId.Length)))..."
        
        # Perform login
        $body = @{
            $UsernameField = $ValidUsername
            $PasswordField = $ValidPassword
        }
        
        $postLoginResponse = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $body -UseBasicParsing -WebSession $preSession -ErrorAction SilentlyContinue
        $postLoginCookies = Get-SessionCookies -Response $postLoginResponse
        $postLoginSessionId = Get-SessionId -Cookies $postLoginCookies
        
        if ([string]::IsNullOrEmpty($postLoginSessionId)) {
            $postLoginSessionId = $preLoginSessionId
        }
        
        Write-Detail "Post-login session ID: $($postLoginSessionId.Substring(0, [Math]::Min(20, $postLoginSessionId.Length)))..."
        
        # Compare session IDs
        $script:TestsTotal++
        if ($preLoginSessionId -eq $postLoginSessionId) {
            Write-Failure "Session ID not regenerated after login"
            Get-FindingDetails "SESSION-FIXATION"
            Write-Host ""
            $script:TestsFailed++
            $script:HighRiskFindings++
            $script:HighRiskList += "Session Fixation Vulnerability"
        } else {
            Write-Success "Session ID regenerated after login"
            $script:TestsPassed++
        }
    }
    catch {
        Write-Info "Error testing session fixation: $_"
    }
}

# PHASE 3: Session Token Entropy Testing
function Test-SessionTokenEntropy {
    Write-Phase "🎲 PHASE 3: SESSION TOKEN ENTROPY ANALYSIS"
    
    $tokens = @()
    
    Write-Detail "Collecting session tokens..."
    
    try {
        for ($i = 1; $i -le 5; $i++) {
            $response = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -UseBasicParsing -ErrorAction SilentlyContinue
            $cookies = Get-SessionCookies -Response $response
            $token = Get-SessionId -Cookies $cookies
            
            if ($token) {
                $tokens += $token
                Write-Host "  Token $i`: $($token.Substring(0, [Math]::Min(30, $token.Length)))... (length: $($token.Length))" -ForegroundColor Gray
            }
            Start-Sleep -Milliseconds 500
        }
        
        if ($tokens.Count -lt 2) {
            Write-Info "Not enough tokens collected for entropy analysis"
            return
        }
        
        $script:TestsTotal++
        
        # Check minimum length
        $minLength = ($tokens | ForEach-Object { $_.Length } | Measure-Object -Minimum).Minimum
        
        if ($minLength -lt 16) {
            Write-Failure "Session tokens too short ($minLength chars)"
            Get-FindingDetails "WEAK-TOKEN"
            Write-Host ""
            $script:TestsFailed++
            $script:HighRiskFindings++
            $script:HighRiskList += "Weak Session Token (too short)"
        } else {
            # Check for uniqueness
            $uniqueTokens = $tokens | Select-Object -Unique
            
            if ($uniqueTokens.Count -eq $tokens.Count) {
                Write-Success "Session tokens appear random and unique"
                $script:TestsPassed++
            } else {
                Write-Failure "Duplicate session tokens detected"
                Get-FindingDetails "WEAK-TOKEN"
                Write-Host ""
                $script:TestsFailed++
                $script:HighRiskFindings++
                $script:HighRiskList += "Duplicate Session Tokens"
            }
        }
    }
    catch {
        Write-Info "Error testing token entropy: $_"
    }
}

# PHASE 4: User Enumeration Testing
function Test-UserEnumeration {
    Write-Phase "👤 PHASE 4: USER ENUMERATION TESTING"
    
    try {
        # Test with valid username
        $validBody = @{
            $UsernameField = $ValidUsername
            $PasswordField = "wrongpassword123"
        }
        
        $validResponse = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $validBody -UseBasicParsing -ErrorAction SilentlyContinue
        $validStatus = $validResponse.StatusCode
        $validContent = $validResponse.Content
        
        # Test with invalid username
        $invalidBody = @{
            $UsernameField = "nonexistentuser12345"
            $PasswordField = "wrongpassword123"
        }
        
        $invalidResponse = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $invalidBody -UseBasicParsing -ErrorAction SilentlyContinue
        $invalidStatus = $invalidResponse.StatusCode
        $invalidContent = $invalidResponse.Content
        
        Write-Detail "Valid username response: HTTP $validStatus"
        Write-Detail "Invalid username response: HTTP $invalidStatus"
        
        $script:TestsTotal++
        $enumerationDetected = $false
        
        # Different status codes
        if ($validStatus -ne $invalidStatus) {
            $enumerationDetected = $true
            Write-Host "  Different HTTP status codes detected" -ForegroundColor Gray
        }
        
        # Different response length
        $lenDiff = [Math]::Abs($validContent.Length - $invalidContent.Length)
        if ($lenDiff -gt 50) {
            $enumerationDetected = $true
            Write-Host "  Significant response length difference: $($validContent.Length) vs $($invalidContent.Length)" -ForegroundColor Gray
        }
        
        # Check for specific enumeration messages
        if ($invalidContent -match "user not found|username not found|no such user|invalid username|user does not exist") {
            $enumerationDetected = $true
            Write-Host "  Specific 'user not found' message detected" -ForegroundColor Gray
        }
        
        if ($enumerationDetected) {
            Write-Failure "User enumeration possible"
            Get-FindingDetails "USER-ENUM"
            Write-Host ""
            $script:TestsFailed++
            $script:MediumRiskFindings++
            $script:MediumRiskList += "User Enumeration Vulnerability"
        } else {
            Write-Success "No obvious user enumeration detected"
            $script:TestsPassed++
        }
    }
    catch {
        Write-Info "Error testing user enumeration: $_"
    }
}

# PHASE 5: Brute Force / Rate Limiting Testing
function Test-RateLimiting {
    Write-Phase "🚫 PHASE 5: BRUTE FORCE / RATE LIMITING TESTING"
    
    $attempts = 10
    $successCount = 0
    $blocked = $false
    
    Write-Detail "Attempting $attempts rapid login requests..."
    
    try {
        for ($i = 1; $i -le $attempts; $i++) {
            $body = @{
                $UsernameField = $ValidUsername
                $PasswordField = "wrongpassword$i"
            }
            
            try {
                $response = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $body -UseBasicParsing -ErrorAction Stop
                $status = $response.StatusCode
                $content = $response.Content
                
                if ($status -eq 429 -or $content -match "rate limit|too many|blocked|try again later|captcha") {
                    $blocked = $true
                    Write-Host "  Attempt $i`: Rate limiting detected (HTTP $status)" -ForegroundColor Green
                    break
                } else {
                    Write-Host "  Attempt $i`: HTTP $status" -ForegroundColor Gray
                    $successCount++
                }
            }
            catch {
                if ($_.Exception.Response.StatusCode -eq 429) {
                    $blocked = $true
                    Write-Host "  Attempt $i`: HTTP 429 Too Many Requests" -ForegroundColor Green
                    break
                }
                Write-Host "  Attempt $i`: Error" -ForegroundColor Gray
                $successCount++
            }
        }
        
        $script:TestsTotal++
        
        if ($blocked) {
            Write-Success "Rate limiting detected after $successCount attempts"
            $script:TestsPassed++
        } else {
            Write-Failure "No rate limiting detected after $attempts attempts"
            Get-FindingDetails "NO-RATE-LIMIT"
            Write-Host ""
            $script:TestsFailed++
            $script:HighRiskFindings++
            $script:HighRiskList += "No Rate Limiting on Login"
        }
    }
    catch {
        Write-Info "Error testing rate limiting: $_"
    }
}

# PHASE 6: Weak Password Policy Testing
function Test-PasswordPolicy {
    Write-Phase "🔑 PHASE 6: WEAK PASSWORD POLICY TESTING"
    
    $weakPasswords = @("123456", "password", "admin", "test", "1234", "a", "abc")
    $weakAccepted = $false
    $acceptedPasswords = @()
    
    Write-Detail "Testing if weak passwords are accepted..."
    
    try {
        foreach ($pwd in $weakPasswords) {
            $timestamp = Get-Date -Format "yyyyMMddHHmmss"
            $body = @{
                $UsernameField = "testuser$timestamp"
                $PasswordField = $pwd
                "email" = "test$timestamp@test.com"
            }
            
            try {
                $response = Invoke-WebRequest -Uri "$TargetUrl$RegisterEndpoint" -Method POST -Body $body -UseBasicParsing -ErrorAction SilentlyContinue
                $status = $response.StatusCode
                $content = $response.Content
                
                if ($status -in @(200, 201, 302)) {
                    if ($content -notmatch "password.*weak|password.*short|password.*simple|password.*common|password.*required|password.*minimum") {
                        $weakAccepted = $true
                        $acceptedPasswords += $pwd
                        Write-Host "  Weak password '$pwd' may be accepted" -ForegroundColor Red
                    }
                }
            }
            catch {
                # Registration endpoint may not exist or may reject
            }
        }
        
        $script:TestsTotal++
        
        if ($weakAccepted) {
            Write-Failure "Weak password policy detected"
            Write-Host "  Accepted weak passwords: $($acceptedPasswords -join ', ')" -ForegroundColor Gray
            Get-FindingDetails "WEAK-PASSWORD"
            Write-Host ""
            $script:TestsFailed++
            $script:MediumRiskFindings++
            $script:MediumRiskList += "Weak Password Policy"
        } else {
            Write-Success "Password policy appears enforced (or registration unavailable)"
            $script:TestsPassed++
        }
    }
    catch {
        Write-Info "Error testing password policy: $_"
    }
}

# PHASE 7: Authentication Bypass Testing
function Test-AuthBypass {
    Write-Phase "🚪 PHASE 7: AUTHENTICATION BYPASS TESTING"
    
    $protectedEndpoints = @("/admin", "/dashboard", "/api/users", "/api/admin", "/profile", "/settings", "/account", "/user", "/panel", "/management")
    $bypassFound = $false
    $bypassedEndpoints = @()
    
    Write-Detail "Testing access to protected endpoints without authentication..."
    
    try {
        foreach ($endpoint in $protectedEndpoints) {
            try {
                $response = Invoke-WebRequest -Uri "$TargetUrl$endpoint" -UseBasicParsing -TimeoutSec 10 -ErrorAction SilentlyContinue
                $status = $response.StatusCode
                $content = $response.Content
                
                if ($status -eq 200) {
                    if ($content -notmatch "login|sign in|authenticate|unauthorized|forbidden|access denied") {
                        $bypassFound = $true
                        $bypassedEndpoints += $endpoint
                        Write-Host "  $endpoint`: HTTP $status - Accessible without auth!" -ForegroundColor Red
                    } else {
                        Write-Host "  $endpoint`: HTTP $status - Login page/redirect" -ForegroundColor Gray
                    }
                } else {
                    Write-Host "  $endpoint`: HTTP $status" -ForegroundColor Gray
                }
            }
            catch {
                $errorStatus = if ($_.Exception.Response) { $_.Exception.Response.StatusCode.value__ } else { "Error" }
                Write-Host "  $endpoint`: HTTP $errorStatus" -ForegroundColor Gray
            }
        }
        
        $script:TestsTotal++
        
        if ($bypassFound) {
            Write-Failure "Authentication bypass detected"
            Write-Host "  Bypassed endpoints: $($bypassedEndpoints -join ', ')" -ForegroundColor Gray
            Get-FindingDetails "AUTH-BYPASS"
            Write-Host ""
            $script:TestsFailed++
            $script:HighRiskFindings++
            $script:HighRiskList += "Authentication Bypass"
        } else {
            Write-Success "No obvious authentication bypass detected"
            $script:TestsPassed++
        }
        
        # Test SQL injection in login
        Write-Host "`n" -NoNewline
        Write-Detail "Testing SQL injection in authentication..."
        
        $sqliPayloads = @("admin'--", "' OR '1'='1", "admin' OR '1'='1'--", "' OR 1=1--", "admin'/*")
        $sqliBypass = $false
        
        $script:TestsTotal++
        
        foreach ($payload in $sqliPayloads) {
            $body = @{
                $UsernameField = $payload
                $PasswordField = "anything"
            }
            
            try {
                $response = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $body -UseBasicParsing -MaximumRedirection 0 -ErrorAction SilentlyContinue
                $status = $response.StatusCode
                $content = $response.Content
                
                if ($status -in @(302, 303)) {
                    $location = $response.Headers["Location"]
                    if ($location -match "dashboard|admin|home|profile|welcome") {
                        $sqliBypass = $true
                        Write-Host "  SQL injection payload succeeded: $payload" -ForegroundColor Red
                        break
                    }
                }
                
                if ($content -match "welcome|logged in|success|dashboard") {
                    $sqliBypass = $true
                    Write-Host "  SQL injection payload succeeded: $payload" -ForegroundColor Red
                    break
                }
            }
            catch {
                # Expected for failed attempts
            }
        }
        
        if ($sqliBypass) {
            Write-Failure "SQL injection authentication bypass possible"
            $script:TestsFailed++
            $script:HighRiskFindings++
            $script:HighRiskList += "SQL Injection Auth Bypass"
        } else {
            Write-Success "No SQL injection auth bypass detected"
            $script:TestsPassed++
        }
    }
    catch {
        Write-Info "Error testing auth bypass: $_"
    }
}

# PHASE 8: Session Timeout Testing
function Test-SessionTimeout {
    Write-Phase "⏰ PHASE 8: SESSION TIMEOUT ANALYSIS"
    
    Write-Detail "Analyzing session configuration..."
    
    try {
        $body = @{
            $UsernameField = $ValidUsername
            $PasswordField = $ValidPassword
        }
        
        $response = Invoke-WebRequest -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $body -UseBasicParsing -ErrorAction SilentlyContinue
        $cookies = Get-SessionCookies -Response $response
        
        if ($cookies.Count -eq 0) {
            Write-Info "Could not establish session for timeout testing"
            return
        }
        
        $script:TestsTotal++
        $hasExpiry = $false
        
        foreach ($cookie in $cookies) {
            if ($cookie -match "max-age=(\d+)") {
                $hasExpiry = $true
                $maxAge = [int]$matches[1]
                $hours = [Math]::Round($maxAge / 3600)
                Write-Detail "Session max-age: ${maxAge}s (~$hours hours)"
                
                if ($maxAge -gt 86400) {
                    Write-Warning "WARNING: Long session timeout ($hours hours)"
                    $script:MediumRiskFindings++
                    $script:MediumRiskList += "Long session timeout (${hours}h)"
                }
            }
            
            if ($cookie -match "expires=([^;]+)") {
                $hasExpiry = $true
                Write-Detail "Session expires: $($matches[1])"
            }
        }
        
        if ($hasExpiry) {
            Write-Success "Session has expiration configured"
            $script:TestsPassed++
        } else {
            Write-Failure "No session expiration detected"
            Get-FindingDetails "NO-TIMEOUT"
            Write-Host ""
            $script:TestsFailed++
            $script:MediumRiskFindings++
            $script:MediumRiskList += "No Session Timeout"
        }
    }
    catch {
        Write-Info "Error testing session timeout: $_"
    }
}

# Generate summary report
function Show-Summary {
    Write-Host "`n======================================================" -ForegroundColor Magenta
    Write-Host "📊 AUTHENTICATION SECURITY SCAN SUMMARY" -ForegroundColor Magenta
    Write-Host "======================================================" -ForegroundColor Magenta
    
    Write-Host "`nTarget: $TargetUrl" -ForegroundColor Blue
    Write-Host "Scan completed: $(Get-Date)" -ForegroundColor Blue
    Write-Host ""
    
    # Test statistics
    $passRate = if ($script:TestsTotal -gt 0) { [Math]::Round(($script:TestsPassed / $script:TestsTotal) * 100) } else { 0 }
    
    Write-Host "Test Results:" -ForegroundColor Cyan
    Write-Host "  Total Tests: $($script:TestsTotal)"
    Write-Host "  Passed: $($script:TestsPassed)" -ForegroundColor Green
    Write-Host "  Failed: $($script:TestsFailed)" -ForegroundColor Red
    Write-Host "  Pass Rate: $passRate%`n"
    
    # Risk summary
    Write-Host "Risk Summary:" -ForegroundColor Cyan
    Write-Host "  High Risk: $($script:HighRiskFindings)" -ForegroundColor Red
    Write-Host "  Medium Risk: $($script:MediumRiskFindings)" -ForegroundColor Yellow
    Write-Host "  Low Risk: $($script:LowRiskFindings)" -ForegroundColor Blue
    Write-Host ""
    
    # List findings
    if ($script:HighRiskList.Count -gt 0) {
        Write-Host "🚨 HIGH RISK FINDINGS:" -ForegroundColor Red
        foreach ($finding in $script:HighRiskList) {
            Write-Host "  • $finding" -ForegroundColor Red
        }
        Write-Host ""
    }
    
    if ($script:MediumRiskList.Count -gt 0) {
        Write-Host "⚠️ MEDIUM RISK FINDINGS:" -ForegroundColor Yellow
        foreach ($finding in $script:MediumRiskList) {
            Write-Host "  • $finding" -ForegroundColor Yellow
        }
        Write-Host ""
    }
    
    if ($script:LowRiskList.Count -gt 0) {
        Write-Host "ℹ️ LOW RISK FINDINGS:" -ForegroundColor Blue
        foreach ($finding in $script:LowRiskList) {
            Write-Host "  • $finding" -ForegroundColor Blue
        }
        Write-Host ""
    }
    
    # Overall security score
    $score = 100
    $score -= $script:HighRiskFindings * 20
    $score -= $script:MediumRiskFindings * 10
    $score -= $script:LowRiskFindings * 5
    if ($score -lt 0) { $score = 0 }
    
    Write-Host "Overall Security Score:" -ForegroundColor Cyan
    if ($score -ge 80) {
        Write-Host "  $score/100 - GOOD" -ForegroundColor Green
    } elseif ($score -ge 60) {
        Write-Host "  $score/100 - NEEDS IMPROVEMENT" -ForegroundColor Yellow
    } else {
        Write-Host "  $score/100 - CRITICAL" -ForegroundColor Red
    }
    
    Write-Host "`n======================================================" -ForegroundColor Magenta
}

# Main execution
Write-Host "🔐 AUTHENTICATION & SESSION SECURITY SCANNER" -ForegroundColor Magenta
Write-Host "======================================================" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl" -ForegroundColor Blue
Write-Host "Mode: $Mode" -ForegroundColor Blue
Write-Host "Login Endpoint: $LoginEndpoint" -ForegroundColor Blue
Write-Host "Timestamp: $(Get-Date)`n" -ForegroundColor Blue

# Run tests based on mode
Test-CookieSecurity
Test-SessionFixation
Test-SessionTokenEntropy
Test-UserEnumeration

if ($Mode -eq "full") {
    Test-RateLimiting
    Test-PasswordPolicy
    Test-AuthBypass
    Test-SessionTimeout
}

Show-Summary
