#!/bin/bash
# Access Control Testing Script - CWE-639, CWE-306, CWE-863, CWE-862, CWE-269, CWE-284
# Usage: ./access-control-test.sh [TARGET_URL] [MODE]

set -e
[[ "$1" == "-h" ]] && { echo "🔐 Access Control Scanner - IDOR, Admin Panel, Privilege Escalation"; exit 0; }

TARGET_URL="${1:-}"; MODE="${2:-full}"; CLEAN_OUTPUT="${3:-false}"
AUTH_COOKIE="${AUTH_COOKIE:-}"
AUTH_TOKEN="${AUTH_TOKEN:-}"
[ -z "$TARGET_URL" ] && { echo "❌ Error: Target URL required"; exit 1; }
[[ ! "$TARGET_URL" =~ ^https?:// ]] && TARGET_URL="https://$TARGET_URL"
TARGET_URL="${TARGET_URL%/}"

[[ "$CLEAN_OUTPUT" = "true" ]] && { RED=''; GREEN=''; YELLOW=''; PURPLE=''; NC=''; } || \
{ RED='\033[0;31m'; GREEN='\033[0;32m'; YELLOW='\033[1;33m'; PURPLE='\033[0;35m'; NC='\033[0m'; }

TESTS_PASSED=0; TESTS_FAILED=0; TESTS_TOTAL=0; HIGH_RISK_FINDINGS=0; HIGH_RISK_LIST=()
AUTH_ERRORS=0  # Track 401/403 responses

# Helper function for authenticated curl requests (supports cookie and bearer token)
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

echo -e "${PURPLE}🔐 ACCESS CONTROL SCANNER${NC}"
echo -e "Target: ${TARGET_URL} | Mode: ${MODE}"

[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication (unauthenticated scan)${NC}"

echo -e ""

# Baseline fingerprinting - detect catch-all responses (apps that return 200 for everything)
get_body_length() {
    echo -n "$1" | wc -c | tr -d ' '
}

capture_baselines() {
    # Fetch root page
    BASELINE_ROOT_BODY=$(auth_curl -s "${TARGET_URL}/" --max-time 10 2>/dev/null || echo "")
    BASELINE_ROOT_LEN=$(get_body_length "$BASELINE_ROOT_BODY")

    # Fetch a random non-existent path to get 404/catch-all baseline
    BASELINE_404_BODY=$(auth_curl -s "${TARGET_URL}/nonexistent_xyzzy_$(date +%s)" --max-time 10 2>/dev/null || echo "")
    BASELINE_404_LEN=$(get_body_length "$BASELINE_404_BODY")

    echo -e "${GRAY}  Baselines: root=${BASELINE_ROOT_LEN}b, 404=${BASELINE_404_LEN}b${NC}"
}

# Check if a response body matches the baseline (within 10%)
is_catch_all_response() {
    local body_len="$1"
    if [ "$BASELINE_ROOT_LEN" -gt 0 ]; then
        local diff=$((body_len - BASELINE_ROOT_LEN))
        diff=${diff#-}  # absolute value
        local threshold=$((BASELINE_ROOT_LEN / 10))
        [ "$threshold" -lt 50 ] && threshold=50
        if [ "$diff" -le "$threshold" ]; then
            return 0  # matches root
        fi
    fi
    if [ "$BASELINE_404_LEN" -gt 0 ]; then
        local diff=$((body_len - BASELINE_404_LEN))
        diff=${diff#-}  # absolute value
        local threshold=$((BASELINE_404_LEN / 10))
        [ "$threshold" -lt 50 ] && threshold=50
        if [ "$diff" -le "$threshold" ]; then
            return 0  # matches 404
        fi
    fi
    return 1  # unique response
}

capture_baselines

test_idor() {
    echo -e "\n${YELLOW}🆔 PHASE 1: IDOR TESTING (CWE-639)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false
    
    local idor_endpoints=("/api/user/1" "/api/user/2" "/api/users/1" "/api/users/2" "/user/1" "/user/2" "/profile/1" "/profile/2" "/order/1" "/order/2" "/document/1" "/document/2")
    
    for endpoint in "${idor_endpoints[@]}"; do
        local response=$(auth_curl -s "${TARGET_URL}${endpoint}" --max-time 10 2>/dev/null || echo "")
        local status=$(auth_curl -sI "${TARGET_URL}${endpoint}" --max-time 10 2>/dev/null | head -1 | grep -oE '[0-9]{3}' | head -1)
        
        if [ "$status" = "200" ] && echo "$response" | grep -qiE '"(user|email|name|id|data|order|document)"'; then
            echo -e "${RED}  ⚠️ IDOR: ${endpoint} accessible${NC}"
            vuln=true
        fi
    done
    
    [ "$vuln" = true ] && { echo -e "${RED}❌ FAIL: IDOR vulnerability${NC}"; TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("IDOR (CWE-639)"); } || \
    { echo -e "${GREEN}✅ PASS: IDOR not detected${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

test_admin_panel() {
    echo -e "\n${YELLOW}👑 PHASE 2: ADMIN PANEL ACCESS (CWE-306)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false
    
    local admin_paths=("/admin" "/administrator" "/manage" "/management" "/dashboard/admin" "/admin/dashboard" "/adminpanel" "/superuser" "/backend" "/control")

    # Detect server stack (informational only)
    local server_header=$(auth_curl -sI "${TARGET_URL}/" --max-time 10 2>/dev/null | grep -i "^Server:\|^X-Powered-By:" | head -2)
    if [ -n "$server_header" ]; then
        echo -e "${YELLOW}  Server stack detected: $(echo "$server_header" | tr '\r\n' ' ')${NC}"
    fi
    
    for path in "${admin_paths[@]}"; do
        local response=$(auth_curl -si "${TARGET_URL}${path}" --max-time 10 2>/dev/null || echo "")
        local status=$(echo "$response" | head -1 | grep -oE '[0-9]{3}' | head -1)

        if [ "$status" = "200" ] && ! echo "$response" | grep -qiE "login|sign in|authenticate|403|401|forbidden"; then
            # Extract body (after headers)
            local body=$(echo "$response" | sed -n '/^\r*$/,$p' | tail -n +2)
            local body_len=$(get_body_length "$body")

            # Skip if response matches root/404 baseline (catch-all redirect)
            if is_catch_all_response "$body_len"; then
                echo -e "${GRAY}  ${path}: HTTP 200 but same as root (catch-all, skipped)${NC}"
                continue
            fi

            # Also require admin-like content in response body
            if ! echo "$body" | grep -qiE "admin|management|settings|configuration|dashboard.*panel|user.*manage"; then
                echo -e "${GRAY}  ${path}: HTTP 200 but no admin content (skipped)${NC}"
                continue
            fi

            echo -e "${RED}  ⚠️ Admin panel accessible: ${path}${NC}"
            vuln=true
        fi
    done
    
    [ "$vuln" = true ] && { echo -e "${RED}❌ FAIL: Unprotected admin panel${NC}"; TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("Unprotected Admin (CWE-306)"); } || \
    { echo -e "${GREEN}✅ PASS: Admin panels protected${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

test_horizontal_bypass() {
    echo -e "\n${YELLOW}↔️ PHASE 3: HORIZONTAL ACCESS BYPASS (CWE-863)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false
    
    # Test accessing other users' resources
    local endpoints=("/api/user/profile" "/api/account" "/api/settings" "/user/data")
    local user_ids=("1" "2" "100" "admin" "test")
    
    for endpoint in "${endpoints[@]}"; do
        for uid in "${user_ids[@]}"; do
            local response=$(auth_curl -s "${TARGET_URL}${endpoint}?user_id=${uid}" --max-time 10 2>/dev/null || echo "")
            local response2=$(auth_curl -s -X POST "${TARGET_URL}${endpoint}" -d "user_id=${uid}" --max-time 10 2>/dev/null || echo "")
            
            if echo "$response$response2" | grep -qiE '"(email|password|token|secret|private)"'; then
                echo -e "${RED}  ⚠️ Horizontal bypass: ${endpoint}?user_id=${uid}${NC}"
                vuln=true
            fi
        done
    done
    
    [ "$vuln" = true ] && { echo -e "${RED}❌ FAIL: Horizontal access bypass${NC}"; TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("Horizontal Bypass (CWE-863)"); } || \
    { echo -e "${GREEN}✅ PASS: Horizontal access protected${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

test_vertical_bypass() {
    echo -e "\n${YELLOW}↕️ PHASE 4: VERTICAL ACCESS BYPASS (CWE-863)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false
    
    # Test accessing admin functions without admin role
    local admin_functions=("/api/admin/users" "/api/admin/settings" "/api/admin/config" "/admin/api/users" "/api/users/delete" "/api/user/promote" "/api/settings/global")
    
    for func in "${admin_functions[@]}"; do
        local response=$(auth_curl -s "${TARGET_URL}${func}" --max-time 10 2>/dev/null || echo "")
        local status=$(auth_curl -sI "${TARGET_URL}${func}" --max-time 10 2>/dev/null | head -1 | grep -oE '[0-9]{3}' | head -1)
        
        if [ "$status" = "200" ] && echo "$response" | grep -qiE '"(users|config|settings|success)"'; then
            echo -e "${RED}  ⚠️ Vertical bypass: ${func}${NC}"
            vuln=true
        fi
    done
    
    [ "$vuln" = true ] && { echo -e "${RED}❌ FAIL: Vertical access bypass${NC}"; TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("Vertical Bypass (CWE-863)"); } || \
    { echo -e "${GREEN}✅ PASS: Vertical access protected${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

test_privilege_escalation() {
    echo -e "\n${YELLOW}⬆️ PHASE 5: PRIVILEGE ESCALATION (CWE-269)${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1)); local vuln=false
    
    # Test role manipulation
    local role_payloads=(
        '{"role":"admin"}'
        '{"isAdmin":true}'
        '{"user_role":"administrator"}'
        '{"permissions":["admin","write","delete"]}'
    )
    
    for payload in "${role_payloads[@]}"; do
        local response=$(auth_curl -s -X POST "${TARGET_URL}/api/user/update" \
            -H "Content-Type: application/json" \
            -d "$payload" --max-time 10 2>/dev/null || echo "")
        
        if echo "$response" | grep -qiE "success|updated|admin.*true"; then
            echo -e "${RED}  ⚠️ Privilege escalation possible${NC}"
            vuln=true
        fi
    done
    
    [ "$vuln" = true ] && { echo -e "${RED}❌ FAIL: Privilege escalation${NC}"; TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1)); HIGH_RISK_LIST+=("Privilege Escalation (CWE-269)"); } || \
    { echo -e "${GREEN}✅ PASS: Privilege escalation blocked${NC}"; TESTS_PASSED=$((TESTS_PASSED + 1)); }
}

generate_summary() {
    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}📊 ACCESS CONTROL SUMMARY${NC}"
    local rate=0; [ $TESTS_TOTAL -gt 0 ] && rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    echo -e "Tests: ${TESTS_TOTAL} | ${GREEN}Passed: ${TESTS_PASSED}${NC} | ${RED}Failed: ${TESTS_FAILED}${NC} | Rate: ${rate}%"
    echo -e "${RED}High Risk: ${HIGH_RISK_FINDINGS}${NC}"
    [ ${#HIGH_RISK_LIST[@]} -gt 0 ] && { echo -e "${RED}🚨 FINDINGS:${NC}"; for f in "${HIGH_RISK_LIST[@]}"; do echo -e "  ${RED}• ${f}${NC}"; done; }

    # Warn about authentication errors and potential false negatives
    if [ $AUTH_ERRORS -gt 0 ]; then
        echo -e "\n${YELLOW}⚠️  WARNING: ${AUTH_ERRORS} requests returned 401/403 Unauthorized${NC}"
        if [ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ]; then
            echo -e "${YELLOW}   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints.${NC}"
        else
            echo -e "${YELLOW}   Provided credentials may be invalid or expired.${NC}"
        fi
    fi

    local score=$((100 - HIGH_RISK_FINDINGS * 20)); [ $score -lt 0 ] && score=0
    echo -e "Score: $score/100"
    echo -e "${PURPLE}======================================================${NC}"
}

test_idor
test_admin_panel
[ "$MODE" = "full" ] && { test_horizontal_bypass; test_vertical_bypass; test_privilege_escalation; }
generate_summary
