# Access Control Testing Script (PowerShell) - CWE-639, CWE-306, CWE-863, CWE-269
param([string]$TargetUrl, [ValidateSet("quick","full")][string]$Mode="full", [switch]$Help)

if ($Help) { Write-Host "🔐 Access Control Scanner"; exit 0 }
if ([string]::IsNullOrEmpty($TargetUrl)) { Write-Host "Error: TargetUrl required" -ForegroundColor Red; exit 1 }
if ($TargetUrl -notmatch "^https?://") { $TargetUrl = "https://$TargetUrl" }
$TargetUrl = $TargetUrl.TrimEnd('/')

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [object]$Body, [string]$ContentType, [int]$TimeoutSec = 10)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    if ($Body) { $params.Body = $Body }
    if ($ContentType) { $params.ContentType = $ContentType }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

$script:TestsPassed=0; $script:TestsFailed=0; $script:TestsTotal=0; $script:HighRiskFindings=0; $script:HighRiskList=@()
$script:AuthErrors=0  # Track 401/403 responses

Write-Host "`n🔐 ACCESS CONTROL SCANNER" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl | Mode: $Mode" -ForegroundColor Blue
if ($AuthToken) { Write-Host "🔐 Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "🔐 Cookie authentication enabled" -ForegroundColor Green }
if (-not $AuthToken -and -not $AuthCookie) { Write-Host "ℹ️ Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

Write-Host ""

function Test-IDOR {
    Write-Host "`n🆔 PHASE 1: IDOR TESTING" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $endpoints = @("/api/user/1", "/api/user/2", "/user/1", "/profile/1", "/order/1")
    foreach ($ep in $endpoints) {
        try {
            $r = Invoke-AuthWebRequest -Uri "$TargetUrl$ep" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
            if ($r.StatusCode -eq 200 -and $r.Content -match '"(user|email|name|data)"') {
                $vuln = $true; Write-Host "  ⚠️ IDOR: $ep" -ForegroundColor Red
            }
        } catch {}
    }
    if ($vuln) { Write-Host "❌ FAIL: IDOR" -ForegroundColor Red; $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "IDOR (CWE-639)" }
    else { Write-Host "✅ PASS: IDOR blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Test-AdminPanel {
    Write-Host "`n👑 PHASE 2: ADMIN PANEL" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $paths = @("/admin", "/administrator", "/wp-admin", "/backend", "/manage", "/dashboard/admin")
    foreach ($path in $paths) {
        try {
            $r = Invoke-AuthWebRequest -Uri "$TargetUrl$path" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
            if ($r.StatusCode -eq 200 -and $r.Content -notmatch "login|sign in|403|401") {
                $vuln = $true; Write-Host "  ⚠️ Admin accessible: $path" -ForegroundColor Red
            }
        } catch {}
    }
    if ($vuln) { Write-Host "❌ FAIL: Unprotected admin" -ForegroundColor Red; $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Unprotected Admin" }
    else { Write-Host "✅ PASS: Admin protected" -ForegroundColor Green; $script:TestsPassed++ }
}

function Test-PrivilegeEscalation {
    Write-Host "`n⬆️ PHASE 3: PRIVILEGE ESCALATION" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $payloads = @('{"role":"admin"}', '{"isAdmin":true}')
    foreach ($payload in $payloads) {
        try {
            $r = Invoke-AuthWebRequest -Uri "$TargetUrl/api/user/update" -Method Post -Body $payload -ContentType "application/json" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
            if ($r.Content -match "success|admin.*true") { $vuln = $true; Write-Host "  ⚠️ Priv escalation" -ForegroundColor Red; break }
        } catch {}
    }
    if ($vuln) { Write-Host "❌ FAIL: Privilege escalation" -ForegroundColor Red; $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Privilege Escalation" }
    else { Write-Host "✅ PASS: Blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Show-Summary {
    Write-Host "`n======================================================" -ForegroundColor Magenta
    Write-Host "📊 ACCESS CONTROL SUMMARY" -ForegroundColor Magenta
    $rate = if($script:TestsTotal -gt 0){[Math]::Round(($script:TestsPassed/$script:TestsTotal)*100)}else{0}
    Write-Host "Tests: $($script:TestsTotal) | Passed: $($script:TestsPassed) | Failed: $($script:TestsFailed) | Rate: $rate%"
    if ($script:HighRiskList.Count -gt 0) { $script:HighRiskList | ForEach-Object { Write-Host "  🚨 $_" -ForegroundColor Red } }

    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "⚠️  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    $score = 100 - ($script:HighRiskFindings * 20); if($score -lt 0){$score=0}
    Write-Host "Score: $score/100" -ForegroundColor $(if($score -ge 80){"Green"}else{"Red"})
}

Test-IDOR; Test-AdminPanel
if ($Mode -eq "full") { Test-PrivilegeEscalation }
Show-Summary
