# Security Scripts

Security testing scripts for the Security Scanner Pro application.

## Authentication Support

All scripts support authenticated scanning via environment variables:

```bash
# Cookie-based authentication
AUTH_COOKIE="session=abc123; token=xyz789"

# Bearer token authentication
AUTH_TOKEN="eyJhbGciOiJIUzI1NiIs..."
```

### How Authentication Works

1. Scripts check for `AUTH_TOKEN` first (Bearer token), then `AUTH_COOKIE`
2. If neither is set, scripts run in unauthenticated mode
3. All HTTP requests include the configured credentials

## URL Testing Behavior

Scripts test **only the provided target URL** - they do not crawl or probe additional paths.

| Script | Test Behavior |
|--------|---------------|
| `generic-security-test` | Tests TARGET_URL with query parameters |
| `injection-advanced-test` | Tests TARGET_URL with injection payloads |
| `ssrf-security-test` | Tests TARGET_URL with SSRF parameters |
| `business-logic-test` | Tests TARGET_URL for race conditions, rate limiting |
| `access-control-test` | Tests IDOR paths (`/user/1`, `/user/2`, etc.) - configurable |

### Exception: IDOR/Access Control Testing

The `access-control-test` script intentionally probes multiple user ID paths to detect Insecure Direct Object Reference (IDOR) vulnerabilities. This is required behavior for IDOR detection.

## Track & Warn: Handling 401/403 Responses

Scripts track authentication errors (401/403) during testing and display a warning in the summary:

```
⚠️  WARNING: 15 requests returned 401/403 Unauthorized
   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints.
```

### Why Track & Warn?

- **Mixed sites**: Many websites have both public and protected endpoints
- **No false positives**: Auth errors don't count as vulnerabilities
- **Clear feedback**: Users know when credentials are needed
- **No early exit**: Scanning continues to test all accessible endpoints

### What "False Negatives" Means

When scanning without credentials:
- Protected endpoints return 401/403
- Vulnerabilities on protected endpoints **cannot be detected**
- Results are incomplete for authenticated areas

To get complete results, provide valid credentials.

## Script Categories

| Category | Scripts | Description |
|----------|---------|-------------|
| **Generic** | `generic-security-test` | Headers, CORS, XSS, SQLi, path traversal |
| **Injection** | `injection-advanced-test` | XXE, NoSQL, SSTI, LDAP, prototype pollution |
| **SSRF** | `ssrf-security-test` | Internal SSRF, cloud metadata, protocol handlers |
| **Business Logic** | `business-logic-test` | Race conditions, rate limiting, workflow bypass |
| **Access Control** | `access-control-test` | IDOR, privilege escalation, role bypass |
| **Authentication** | `authentication-test` | Session handling, password policies |
| **JWT** | `jwt-security-test` | JWT vulnerabilities, algorithm confusion |
| **File Upload** | `file-upload-security-test` | Unrestricted upload, extension bypass |
| **Crypto** | `crypto-security-test` | TLS configuration, certificate validation |
| **Reconnaissance** | `waf-detection-test`, `web-malware-detection-test` | WAF fingerprinting, malware detection |

## Usage Examples

### Basic Scan (Unauthenticated)
```bash
./generic-security-test.sh https://example.com
```

### Authenticated Scan with Cookie
```bash
AUTH_COOKIE="session=abc123" ./generic-security-test.sh https://example.com
```

### Authenticated Scan with Bearer Token
```bash
AUTH_TOKEN="eyJhbGci..." ./injection-advanced-test.sh https://example.com/api/endpoint
```

### Quick Mode (Faster, Fewer Tests)
```bash
./generic-security-test.sh https://example.com quick
```

### Full Mode (Comprehensive)
```bash
./business-logic-test.sh https://example.com full
```

## Platform Support

Each script has both bash (`.sh`) and PowerShell (`.ps1`) versions:
- **macOS/Linux**: Use `.sh` scripts
- **Windows**: Use `.ps1` scripts

Both versions produce identical output and have identical authentication support.
