# Struts 2 Comprehensive Security Testing Script

## 🪟 Windows Setup (2 Simple Steps)

### Step 1: Install OWASP ZAP
Download and install from: **https://www.zaproxy.org/download/**

**Note:** Windows may show a security warning. Click "..." → "Keep" → "Show more" → "Keep anyway"

### Step 2: Run Automated Setup
```powershell
# Right-click setup-windows.ps1 → Run with PowerShell (as Administrator)
# OR in PowerShell as Admin:
.\setup-windows.ps1
```

**The setup script automatically installs:**
- WSL (Windows Subsystem for Linux)
- Java 17 (required by OWASP ZAP)
- jq (JSON parser)
- Updates test scripts with your network configuration

**Setup complete! Now you can run tests:**

### Quick Test (~5 min, no ZAP)
```powershell
# Navigate to the script directory first
cd C:\Users\YourUsername\Desktop\vulnerability-tests\security-scripts\struts2WithAuth

# Run quick test (PowerShell requires .\ prefix)
.\struts2vulnerability-quick-test.bat http://localhost:8080/myapp

# For Docker apps, use host.docker.internal
.\struts2vulnerability-quick-test.bat http://host.docker.internal:7070/myapp
```

### Full Test (~20-60 min, with ZAP)
```powershell
# Navigate to the script directory first
cd C:\Users\YourUsername\Desktop\vulnerability-tests\security-scripts\struts2WithAuth

# 1. Start ZAP (wait 30-60 seconds)
.\start-zap.bat

# 2. Run full test
.\struts2vulnerability-full-test.bat http://localhost:8080/myapp

# For Docker apps
.\struts2vulnerability-full-test.bat http://host.docker.internal:7070/myapp
```

**All files are in this directory - just run setup once, then use the bat files to test!**

---

## 🎯 Overview

A specialized security testing script designed specifically for **Apache Struts 2** applications with **authentication mechanisms**. This script extends generic penetration testing capabilities with Struts 2-specific vulnerability detection including OGNL injection, action method exposure, and authentication security testing.

### Key Features

- ✅ **Authentication Security Testing** - Login mechanisms, session management, auth bypass detection  
- ✅ **CSRF Token Handling** - Auto-extracts and tests CSRF tokens in authenticated actions  
- ✅ **OGNL Injection Detection** - Tests for 21+ critical CVEs (S2-001 through S2-062)  
- ✅ **Action Method Security** - Direct method invocation and namespace traversal testing  
- ✅ **Action Enumeration** - Discovers 31+ common action patterns and namespaces  
- ✅ **DevMode Detection** - Identifies development mode exposure  
- ✅ **Complete OWASP Coverage** - Security headers, CORS, XSS, SQLi, information disclosure  
- ✅ **ZAP Integration** - Deep penetration testing with OWASP ZAP  
- ✅ **Multiple Scan Modes** - Full, Quick, Auth-only, OGNL-scan, ZAP-only

## 📋 Testing Phases

### Original Phases (1-6)
1. **Security Headers Analysis**
2. **CORS Security Tests**
3. **XSS Vulnerability Tests**
4. **SQL Injection Tests**
5. **Information Disclosure**
6. **ZAP Vulnerability Scan**

### NEW Struts 2-Specific Phases (7-10)
7. **🔐 Authentication Security Testing**
8. **🎯 OGNL Injection Testing** (Critical for Struts 2)
9. **⚡ Action Method Security**
10. **🕵️ DevMode & Enumeration**

---

## 📊 Complete Test Coverage

**Total: 187+ individual security checks across 10 phases**

### Phase 1: Security Headers (6 tests)
- HSTS (HTTP Strict Transport Security)
- Content Security Policy (CSP)
- X-Frame-Options (clickjacking protection)
- X-Content-Type-Options (MIME sniffing)
- X-XSS-Protection
- Referrer-Policy

### Phase 2: CORS Security (2 tests)
- Wildcard origin detection
- Evil origin rejection testing

### Phase 3: XSS Vulnerabilities (40 tests)
- 5 XSS payloads × 8 common parameters
- Reflected XSS detection

### Phase 4: SQL Injection (48 tests)
- 6 SQLi payloads × 8 parameters
- Union-based, blind, time-based detection

### Phase 5: Information Disclosure (4 tests)
- `.git`, `.env` exposure
- Admin panel accessibility

### Phase 6: ZAP Deep Scan
- 50+ vulnerability rules
- Active + passive scanning

### Phase 7: Authentication Security (5 tests)
- Login form discovery
- **CSRF token auto-extraction & testing**
- SQL injection in authentication
- Session cookie security (HttpOnly, Secure)
- **Authenticated action CSRF validation**

### Phase 8: OGNL Injection (21 CVEs)
**S2-001, S2-003, S2-005, S2-007, S2-008, S2-009, S2-012, S2-013, S2-015, S2-016, S2-029, S2-032, S2-033, S2-037, S2-045, S2-046, S2-048, S2-053, S2-057, S2-059, S2-062**

- Content-Type injection
- Redirect exploitation
- Parameter injection
- Deep scan: 14 comprehensive payloads

### Phase 9: Action Method Security (30+ tests)
- Direct method invocation: `!execute`, `!input`, `!delete`, `!update`, `!admin`, `!list`
- Namespace traversal (3 patterns)

### Phase 10: DevMode & Enumeration (40+ tests)
- DevMode detection
- Version fingerprinting
- **Action enumeration: 31 patterns**
- **Namespace discovery: 5 patterns**

### OWASP Top 10 Coverage: 7/10 (70%)

**Covered:**
✅ A01: Broken Access Control
✅ A02: Cryptographic Failures
✅ A03: Injection
✅ A04: Insecure Design
✅ A05: Security Misconfiguration
✅ A07: Authentication Failures

**Not Covered (by design):**
❌ **A06: Vulnerable Components** - ZAP Phase 6 handles library scanning
❌ **A08: Software Integrity** - File upload vulnerabilities (not Struts 2 specific)
❌ **A10: SSRF** - Not Struts 2 framework-specific

**Verdict:** This script is **perfectly optimized for Struts 2 testing**. The 3 missing categories are either handled by ZAP or not relevant to Struts 2 framework testing.

## 🚀 Quick Start

```bash
# Basic scan
./struts2-comprehensive-security-test.sh http://localhost:8080/myapp full

# With authentication
./struts2-comprehensive-security-test.sh http://localhost:8080/myapp full \
  --auth-type=form \
  --login-url=/login.action \
  --username=admin \
  --password=admin123
```

## 📖 Usage Examples

### Example 1: Full Security Audit
```bash
./struts2-comprehensive-security-test.sh \
  http://localhost:8080/shop \
  full \
  --auth-type=form \
  --login-url=/customer/login.action \
  --username=testuser \
  --password=Test123! \
  --success-indicator="dashboard"
```

### Example 2: Quick CI/CD Check
```bash
./struts2-comprehensive-security-test.sh \
  http://staging.example.com/app \
  quick
```

### Example 3: Focus on OGNL Vulnerabilities with Deep Scan
```bash
./struts2-comprehensive-security-test.sh \
  http://localhost:8080/legacyapp \
  ognl-scan \
  --ognl-deep-scan \
  --action-enumeration \
  --struts-version=2.5
```

## ⚙️ Configuration Options

### Authentication Options
```bash
--auth-type=form|basic|none
--login-url=/login.action
--username=testuser
--password=password123
--success-indicator="dashboard"
--session-cookie=JSESSIONID
```

### Struts 2 Options
```bash
--struts-version=2.5
--ognl-deep-scan           # 21 CVEs (S2-001 to S2-062)
--action-enumeration       # Discover 31+ action patterns
--devmode-check            # Test for development mode
--namespace-traversal      # Test namespace security
--csrf-token-name=struts.token  # Custom CSRF token name
```

## 🛡️ Security Warning

⚠️ **AUTHORIZED TESTING ONLY**

This tool performs active security testing. Only use on systems you own or have explicit permission to test.

✅ DO: Test localhost and staging environments  
❌ DON'T: Test production or third-party systems without authorization  

## 📊 Output Format

```
🎯 STRUTS 2 COMPREHENSIVE SECURITY SCANNER
============================================

🔐 PHASE 7: AUTHENTICATION SECURITY TESTING
✅ Login successful
⚠️ Missing HttpOnly flag on session cookies

🎯 PHASE 8: STRUTS 2 OGNL INJECTION TESTING  
✅ No OGNL injection vulnerabilities detected

📊 SECURITY ASSESSMENT SUMMARY
Overall Risk Level: MEDIUM
Risk Score: 35/100

🔴 HIGH RISK FINDINGS (2)
🟡 MEDIUM RISK FINDINGS (3)
🟢 LOW RISK FINDINGS (5)
```

## 🐛 Troubleshooting

### "Cannot connect to target URL" Error

**Problem:** Script says "Cannot connect to target" but URL works in browser.

**Cause:** Windows Firewall blocks WSL from accessing localhost/app ports.

**Fix (Run in PowerShell as Administrator):**
```powershell
New-NetFirewallRule -DisplayName "WSL to Localhost" -Direction Inbound -LocalPort 8080 -Protocol TCP -Action Allow
```
Replace `8080` with your actual app port.

**Alternative:** Use the actual IP that works in browser:
```powershell
# If browser works with this IP, use it directly:
.\struts2vulnerability-full-test.bat http://172.29.32.1:8080/myapp
```

---

### Line Ending Error (Japanese RDP)

**ERROR YOU SEE:**
```
./struts2-comprehensive-security-test.sh: line 2: $'¥r': command not found
```

**FIX (One-time command):**
```powershell
wsl bash -c "tr -d '\r' < struts2-comprehensive-security-test.sh > temp.sh && mv temp.sh struts2-comprehensive-security-test.sh && chmod +x struts2-comprehensive-security-test.sh"
```

**Why:** Git on Japanese Windows converts line endings to CRLF. Bash needs LF only.


### ZAP Not Available

#### Windows Installation (Recommended)
```powershell
# Install via winget (easiest method)
winget install ZAP.ZAP
Or under https://www.zaproxy.org/download/

# After installation, start ZAP in GUI mode
Start-Process "C:\Program Files\ZAP\Zed Attack Proxy\ZAP.exe"

# Configure ZAP for API access:
# 1. In ZAP GUI: Tools > Options > API
# 2. Enable "API Enabled"
# 3. Uncheck "Require API Key" (for local testing only)
# 4. Set Address: 127.0.0.1
# 5. Set Port: 8090
# 6. Click OK

# ZAP is now ready for full mode scans
```

#### macOS Installation
```bash
# Install OWASP ZAP
brew install --cask owasp-zap

# Start ZAP daemon
zap.sh -daemon -host 127.0.0.1 -port 8090 -config api.disablekey=true
```

#### Linux Installation
```bash
# Ubuntu/Debian
apt-get install zaproxy

# Start ZAP daemon
zap.sh -daemon -host 127.0.0.1 -port 8090 -config api.disablekey=true
```

#### Or use quick mode (no ZAP needed)
```bash
./struts2-comprehensive-security-test.sh http://localhost:8080/app quick
```

### Authentication Not Working
```bash
# Add success/failure indicators
--success-indicator="Welcome"
--failure-indicator="Invalid"
```

## 📈 Comparison

| Feature | Generic | Struts 2 Script |
|---------|---------|-----------------|
| Security Headers | ✅ | ✅ |
| XSS/SQLi | ✅ | ✅ Enhanced |
| **Authentication** | ❌ | ✅ NEW |
| **OGNL Injection** | ❌ | ✅ NEW |
| **Action Methods** | ❌ | ✅ NEW |

---

# 日本語ドキュメント (Japanese Documentation)

## 🪟 Windowsセットアップ (2ステップ)

### ステップ1: OWASP ZAPをインストール
ダウンロードとインストール: **https://www.zaproxy.org/download/**

**注意:** Windowsがセキュリティ警告を表示する場合があります。「...」→「保存」→「詳細表示」→「強制的に保存」をクリック

### ステップ2: 自動セットアップを実行
```powershell
# setup-windows.ps1を右クリック→PowerShellで実行(管理者権限)
# またはPowerShellで管理者として実行:
.\setup-windows.ps1
```

**セットアップスクリプトが自動でインストール:**
- WSL (Windows Subsystem for Linux)
- Java 17 (ZAPが必要)
- jq (JSONパーサー)
- テストスクリプトのネットワーク設定を更新

**セットアップ完了！これでテストを実行できます:**

### クイックテスト (~5分、ZAP不要)
```powershell
# まずスクリプトディレクトリに移動
cd C:\Users\YourUsername\Desktop\vulnerability-tests\security-scripts\struts2WithAuth

# クイックテスト実行 (PowerShellでは .\ プレフィックスが必要)
.\struts2vulnerability-quick-test.bat http://localhost:8080/myapp

# Dockerアプリの場合、host.docker.internalを使用
.\struts2vulnerability-quick-test.bat http://host.docker.internal:7070/myapp
```

### フルテスト (~20-60分、ZAP付き)
```powershell
# まずスクリプトディレクトリに移動
cd C:\Users\YourUsername\Desktop\vulnerability-tests\security-scripts\struts2WithAuth

# 1. ZAPを起動 (30-60秒待つ)
.\start-zap.bat

# 2. フルテスト実行
.\struts2vulnerability-full-test.bat http://localhost:8080/myapp

# Dockerアプリの場合
.\struts2vulnerability-full-test.bat http://host.docker.internal:7070/myapp
```

**すべてのファイルはこのディレクトリにあります - セットアップを一度実行してからbatファイルでテストするだけです！**

---

## 🎯 概要

**Apache Struts 2** アプリケーション専用のセキュリティテストスクリプトです。認証機能のテストとStruts 2特有の脆弱性検出（OGNL インジェクション、アクションメソッド露出など）に対応しています。

### 主な機能

- ✅ **認証セキュリティテスト** - ログイン機構、セッション管理、認証バイパス検出  
- ✅ **OGNL インジェクション検出** - 重大なCVE（S2-045, S2-046, S2-048, S2-057, S2-059, S2-062）のテスト  
- ✅ **アクションメソッドセキュリティ** - 直接メソッド呼び出しと名前空間トラバーサルのテスト  
- ✅ **DevMode 検出** - 開発モードの露出を特定  
- ✅ **完全なOWASP対応** - セキュリティヘッダー、CORS、XSS、SQLi、情報漏洩  
- ✅ **ZAP統合** - OWASP ZAPによる深層ペネトレーションテスト  
- ✅ **複数のスキャンモード** - Full、Quick、Auth-only、OGNL-scan、ZAP-only  

## 📋 テストフェーズ

### 既存フェーズ（1-6）
1. **セキュリティヘッダー解析**
2. **CORSセキュリティテスト**
3. **XSS脆弱性テスト**
4. **SQLインジェクションテスト**
5. **情報漏洩テスト**
6. **ZAP脆弱性スキャン**

### 新規Struts 2専用フェーズ（7-10）
7. **🔐 認証セキュリティテスト**
8. **🎯 OGNL インジェクションテスト**（Struts 2の重大脆弱性）
9. **⚡ アクションメソッドセキュリティ**
10. **🕵️ DevMode & 列挙**

---

## 📊 完全なテストカバレッジ

**合計: 10フェーズで187+個のセキュリティチェック**

### フェーズ1: セキュリティヘッダー (6テスト)
- HSTS (HTTP Strict Transport Security)
- Content Security Policy (CSP)
- X-Frame-Options (クリックジャッキング保護)
- X-Content-Type-Options (MIMEスニッフィング)
- X-XSS-Protection
- Referrer-Policy

### フェーズ2: CORSセキュリティ (2テスト)
- ワイルドカードオリジン検出
- 悪意のあるオリジンの拒否テスト

### フェーズ3: XSS脆弱性 (40テスト)
- 5個のXSSペイロード × 8個の共通パラメータ
- 反射型XSS検出

### フェーズ4: SQLインジェクション (48テスト)
- 6個のSQLiペイロード × 8個のパラメータ
- Unionベース、ブラインド、タイムベース検出

### フェーズ5: 情報漏洩 (4テスト)
- `.git`, `.env`ファイルの露出
- 管理パネルのアクセス可能性

### フェーズ6: ZAP深層スキャン
- 50+脆弱性ルール
- アクティブ + パッシブスキャン

### フェーズ7: 認証セキュリティ (5テスト)
- ログインフォーム検出
- **CSRFトークン自動抽出とテスト**
- 認証におけるSQLインジェクション
- セッションクッキーセキュリティ (HttpOnly, Secure)
- **認証済みアクションCSRF検証**

### フェーズ8: OGNLインジェクション (21個のCVE)
**S2-001, S2-003, S2-005, S2-007, S2-008, S2-009, S2-012, S2-013, S2-015, S2-016, S2-029, S2-032, S2-033, S2-037, S2-045, S2-046, S2-048, S2-053, S2-057, S2-059, S2-062**

- Content-Typeインジェクション
- リダイレクトエクスプロイト
- パラメータインジェクション
- ディープスキャン: 14個の包括的ペイロード

### フェーズ9: アクションメソッドセキュリティ (30+テスト)
- 直接メソッド呼び出し: `!execute`, `!input`, `!delete`, `!update`, `!admin`, `!list`
- ネームスペーストラバーサル (3パターン)

### フェーズ10: DevMode & 列挙 (40+テスト)
- DevMode検出
- バージョンフィンガープリント
- **アクション列挙: 31パターン**
- **ネームスペース発見: 5パターン**

### OWASP Top 10 カバレッジ: 7/10 (70%)

**カバー済み:**
✅ A01: アクセス制御の不備
✅ A02: 暗号化の失敗
✅ A03: インジェクション
✅ A04: 安全でない設計
✅ A05: セキュリティの設定ミス
✅ A07: 認証の失敗

**カバーしていない（意図的）:**
❌ **A06: 脆弱なコンポーネント** - ZAPフェーズ6がライブラリスキャンを処理
❌ **A08: ソフトウェア整合性** - ファイルアップロード脆弱性（Struts 2固有ではない）
❌ **A10: SSRF** - Struts 2フレームワーク固有ではない

**結論:** このスクリプトは **Struts 2テストに完全に最適化**されています。カバーしていない3カテゴリは、ZAPが処理するか、Struts 2フレームワークテストに関連していません。

## 🚀 クイックスタート

```bash
# 基本的なスキャン
./struts2-comprehensive-security-test.sh http://localhost:8080/myapp full

# 認証ありのスキャン
./struts2-comprehensive-security-test.sh http://localhost:8080/myapp full \
  --auth-type=form \
  --login-url=/login.action \
  --username=admin \
  --password=admin123
```

## 📖 使用例

### 例1: 完全なセキュリティ監査
```bash
./struts2-comprehensive-security-test.sh \
  http://localhost:8080/shop \
  full \
  --auth-type=form \
  --login-url=/customer/login.action \
  --username=testuser \
  --password=Test123! \
  --success-indicator="dashboard"
```

### 例2: CI/CDパイプライン用の高速チェック
```bash
./struts2-comprehensive-security-test.sh \
  http://staging.example.com/app \
  quick
```

### 例3: OGNL脆弱性に焦点を当てたスキャン
```bash
./struts2-comprehensive-security-test.sh \
  http://localhost:8080/legacyapp \
  ognl-scan \
  --ognl-deep-scan \
  --struts-version=2.5
```

## ⚙️ 設定オプション

### 認証オプション
```bash
--auth-type=form|basic|none      # 認証方式
--login-url=/login.action         # ログインエンドポイント
--username=testuser               # テストユーザー名
--password=password123            # テストパスワード
--success-indicator="dashboard"   # ログイン成功の判定文字列
--session-cookie=JSESSIONID       # セッションクッキー名
```

### Struts 2 専用オプション
```bash
--struts-version=2.5              # 対象のStrutsバージョン
--ognl-deep-scan                  # 包括的なOGNLテストを有効化
--action-enumeration              # 全アクションマッピングを発見
--devmode-check                   # 開発モードをテスト（デフォルト: true）
--namespace-traversal             # 名前空間セキュリティをテスト（デフォルト: true）
```

## 🕵️ 検出される脆弱性

### 重大（即座の対応が必要）
- **OGNL インジェクション** - リモートコード実行（RCE）が可能
- **認証バイパス** - 保護されたリソースへの不正アクセス

### 高リスク
- **露出したアクションメソッド** - 直接メソッド呼び出し（`!execute`、`!delete`）
- **DevMode有効** - 本番環境での開発モード
- **セッション管理の問題** - HttpOnly、Secureフラグの欠落

### 中/低リスク
- **セキュリティヘッダーの欠落** - HSTS、CSP、X-Frame-Options
- **CORS設定ミス** - ワイルドカードオリジン
- **CSRFトークンの欠落** - クロスサイトリクエストフォージェリに脆弱

## 📊 スキャンモード

| モード | 所要時間 | 説明 | ZAP必要 |
|--------|---------|------|--------|
| **full** | 20-60分 | 全10フェーズの完全スキャン | ✅ 必要 |
| **quick** | 5分 | ZAPなしの高速テスト | ❌ 不要 |
| **auth-only** | 2分 | 認証テストのみ | ❌ 不要 |
| **ognl-scan** | 5-10分 | Struts 2特有の脆弱性 | ❌ 不要 |
| **zap-only** | 15-30分 | ZAP深層スキャンのみ | ✅ 必要 |

## 🛡️ セキュリティ警告

⚠️ **承認されたテストのみ実施してください**

このツールはアクティブなセキュリティテストを実行します。所有しているシステム、または明示的な許可を得たシステムでのみ使用してください。

✅ すべきこと:
- localhostおよびステージング環境でのテスト
- 書面による許可の取得

❌ してはいけないこと:
- 承認なしの本番環境テスト
- サードパーティのウェブサイトのテスト
- 不正なペネトレーションテスト

## 🐛 トラブルシューティング

### 日本語Windows/RDPでの改行エラー修正

エラー表示：
```
./struts2-comprehensive-security-test.sh: line 2: $'¥r': command not found
./struts2-comprehensive-security-test.sh: line 7: $'¥r': command not found
```

**修正方法（日本語RDPで実行）:**
```powershell
wsl bash -c "tr -d '\r' < struts2-comprehensive-security-test.sh > temp.sh && mv temp.sh struts2-comprehensive-security-test.sh && chmod +x struts2-comprehensive-security-test.sh"
```

**理由:**
- 日本語WindowsではGitがファイルをCRLF改行 (\r\n) に変換
- WSLのBashはLF改行 (\n) のみ必要
- 日本語Windowsでは \ が ¥ と表示されるため、エラーは $'¥r' と表示

このコマンドは一度だけ実行すればOK。次回からは通常にテストが実行できます。

---

### ZAPが利用できない

#### Windows インストール（推奨）
```powershell
# winget経由でインストール（最も簡単な方法）
winget install ZAP.ZAP
# または https://www.zaproxy.org/download/ からダウンロード

# インストール後、ZAPをGUIモードで起動
Start-Process "C:\Program Files\ZAP\Zed Attack Proxy\ZAP.exe"

# API アクセス用にZAPを設定:
# 1. ZAP GUI で: ツール > オプション > API
# 2. 「API Enabled」を有効にする
# 3. 「Require API Key」のチェックを外す（ローカルテストのみ）
# 4. アドレス: 127.0.0.1
# 5. ポート: 8090
# 6. OKをクリック

# これでフルモードスキャンの準備完了
```

#### macOS インストール
```bash
# OWASP ZAPをインストール
brew install --cask owasp-zap

# ZAPデーモンを起動
zap.sh -daemon -host 127.0.0.1 -port 8090 -config api.disablekey=true
```

#### Linux インストール
```bash
# Ubuntu/Debian
apt-get install zaproxy

# ZAPデーモンを起動
zap.sh -daemon -host 127.0.0.1 -port 8090 -config api.disablekey=true
```

#### またはZAPなしのquickモードを使用
```bash
./struts2-comprehensive-security-test.sh http://localhost:8080/app quick
```

### 認証が機能しない
```bash
# 成功/失敗の判定文字列を追加
--success-indicator="ようこそ"
--failure-indicator="エラー"
```

### jqが見つからない
```bash
# jqをインストール
brew install jq        # macOS
apt-get install jq     # Linux
```

## 📈 出力例

```
🎯 STRUTS 2 包括的セキュリティスキャナー
============================================

🔐 フェーズ 7: 認証セキュリティテスト
✅ ログイン成功
⚠️ セッションクッキーにHttpOnlyフラグがありません

🎯 フェーズ 8: STRUTS 2 OGNL インジェクションテスト  
✅ OGNL インジェクション脆弱性は検出されませんでした

📊 セキュリティ評価サマリー
総合リスクレベル: 中
リスクスコア: 35/100

🔴 高リスク発見事項 (2)
🟡 中リスク発見事項 (3)
🟢 低リスク発見事項 (5)
```

## 🔧 必要な環境

**基本:**
- bash
- curl
- grep/sed/awk

**上級機能（フルスキャン用）:**
- OWASP ZAP
- jq（JSON解析用）

```bash
# macOSにインストール
brew install --cask owasp-zap
brew install jq

# Linuxにインストール
apt-get install zaproxy jq
```

## 💡 重要な注意事項

1. **ZAPは必須ではありません** - `quick`、`auth-only`、`ognl-scan`モードはZAPなしで動作します
2. **認証テストは新機能** - 汎用スクリプトにはない機能です
3. **OGNL検出が重要** - Struts 2の最も深刻な脆弱性をテストします
4. **ローカルテストに最適** - localhost環境での使用を推奨

---

**Version**: 1.0.0  
**Last Updated**: 2025-10-15  
**最終更新**: 2025年10月15日  

For detailed documentation, run with `--help` or see examples above.  
詳細なドキュメントは`--help`オプションまたは上記の例を参照してください。
