<#
.SYNOPSIS
    Struts 2 Comprehensive Security Testing Script (PowerShell Native)
    
.DESCRIPTION
    Security vulnerability scanner for Apache Struts 2 applications.
    Pure PowerShell implementation - no bash/WSL required!
    
.PARAMETER TargetUrl
    Target application URL (e.g., http://localhost:8080/myapp)
    
.PARAMETER Mode
    Scan mode: Quick (native tests), Full (includes OWASP ZAP scan)
    
.EXAMPLE
    .\Struts2-SecurityTest.ps1 -TargetUrl "http://localhost:8080/myapp" -Mode Quick
    .\Struts2-SecurityTest.ps1 -TargetUrl "http://localhost:8080/myapp" -Mode Full
#>

[CmdletBinding()]
param(
    [Parameter(Mandatory=$true)]
    [string]$TargetUrl,
    
    [Parameter(Mandatory=$false)]
    [ValidateSet('Quick', 'Full')]
    [string]$Mode = 'Quick',
    
    [Parameter(Mandatory=$false)]
    [string]$ZapPath = "C:\Program Files\ZAP\Zed Attack Proxy\zap.bat"
)

# Global variables
$script:TestsPassed = 0
$script:TestsFailed = 0
$script:HighRiskFindings = @()
$script:MediumRiskFindings = @()
$script:LowRiskFindings = @()
$script:InfoFindings = @()
$script:RiskScore = 0.0  # Percentage-based score (0-100)

# Color output functions
function Write-ColorOutput {
    param(
        [string]$Message,
        [string]$Color = 'White'
    )
    Write-Host $Message -ForegroundColor $Color
}

function Write-Header {
    param([string]$Text)
    Write-Host ""
    Write-Host $Text -ForegroundColor Cyan
}

function Write-Success {
    param([string]$Text)
    Write-Host "checkmark $Text" -ForegroundColor Green
}

function Write-Failure {
    param([string]$Text)
    Write-Host "X $Text" -ForegroundColor Red
}

function Write-Warning2 {
    param([string]$Text)
    Write-Host "! $Text" -ForegroundColor Yellow
}

# Test connection
function Test-Connection2 {
    param([string]$Url)
    
    Write-ColorOutput "`nTesting connection to target..." -Color Cyan
    
    try {
        $response = Invoke-WebRequest -Uri $Url -Method Head -TimeoutSec 15 -ErrorAction Stop -UseBasicParsing
        Write-Success "Connection successful! (HTTP $($response.StatusCode))"
        return $true
    }
    catch {
        Write-Failure "Cannot connect to target URL"
        Write-Host ""
        Write-ColorOutput "Target: $Url" -Color Yellow
        Write-ColorOutput "Error: $($_.Exception.Message)" -Color Yellow
        Write-Host ""
        Write-ColorOutput "Troubleshooting:" -Color Cyan
        Write-ColorOutput "  1. Verify application is running" -Color Gray
        Write-ColorOutput "  2. Check URL is correct" -Color Gray
        Write-ColorOutput "  3. Test in browser first" -Color Gray
        return $false
    }
}

# Phase 1: Security Headers
function Test-SecurityHeaders {
    Write-Header "PHASE 1: SECURITY HEADERS ANALYSIS"
    
    try {
        $response = Invoke-WebRequest -Uri $TargetUrl -Method Get -TimeoutSec 30 -UseBasicParsing -ErrorAction Stop
        $headers = $response.Headers
        
        # Test HSTS (only for HTTPS)
        if ($TargetUrl -like "https://*") {
            if ($headers['Strict-Transport-Security']) {
                Write-Success "HSTS Header Present"
                $script:TestsPassed++
            } else {
                Write-Failure "HSTS Header Missing"
                $script:TestsFailed++
                $script:LowRiskFindings += "Missing HSTS Header"
            }
        } else {
            Write-Host "  HSTS Header (SKIP - HTTP only)" -ForegroundColor Blue
        }
        
        # X-Frame-Options
        if ($headers['X-Frame-Options']) {
            Write-Success "X-Frame-Options Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-Frame-Options Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-Frame-Options Header"
            $script:RiskScore += 3  # 3% weight
        }
        
        # X-Content-Type-Options
        if ($headers['X-Content-Type-Options']) {
            Write-Success "X-Content-Type-Options Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-Content-Type-Options Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-Content-Type-Options Header"
            $script:RiskScore += 3  # 3% weight
        }
        
        # X-XSS-Protection
        if ($headers['X-XSS-Protection']) {
            Write-Success "X-XSS-Protection Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-XSS-Protection Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-XSS-Protection Header"
            $script:RiskScore += 3  # 3% weight
        }
        
        # Content-Security-Policy
        if ($headers['Content-Security-Policy']) {
            Write-Success "Content-Security-Policy Present"
            $script:TestsPassed++
        } else {
            Write-Failure "Content-Security-Policy Missing"
            $script:TestsFailed++
            $script:MediumRiskFindings += "Missing Content-Security-Policy Header"
            $script:RiskScore += 5  # 5% weight - Important for injection prevention
        }
        
        # Referrer-Policy
        if ($headers['Referrer-Policy']) {
            Write-Success "Referrer-Policy Present"
            $script:TestsPassed++
        } else {
            Write-Failure "Referrer-Policy Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing Referrer-Policy Header"
            $script:RiskScore += 3  # 3% weight
        }
        
    } catch {
        Write-Failure "Failed to fetch headers: $($_.Exception.Message)"
    }
}

# Phase 2: CORS Tests
function Test-CORS {
    Write-Header "PHASE 2: CORS SECURITY TESTS"
    
    $evilOrigin = "https://evil-site.com"
    
    try {
        # Test CORS with evil origin
        $hdrs = @{
            'Origin' = $evilOrigin
        }
        $response = Invoke-WebRequest -Uri $TargetUrl -Headers $hdrs -Method Get -TimeoutSec 30 -UseBasicParsing -ErrorAction Stop
        
        # Check for wildcard
        $corsHeader = $response.Headers['Access-Control-Allow-Origin']
        if ($corsHeader -eq '*') {
            Write-Failure "CORS Wildcard Protection"
            Write-Warning2 "  RISK: High - Allows any website to access your API"
            $script:TestsFailed++
            $script:HighRiskFindings += "CORS Wildcard Origin (*)"
            $script:RiskScore += 5  # 5% weight
        } else {
            Write-Success "CORS Wildcard Protection"
            $script:TestsPassed++
        }
        
        # Check if evil origin accepted
        if ($corsHeader -eq $evilOrigin) {
            Write-Failure "Evil Origin Rejection"
            Write-Warning2 "  RISK: High - Evil origin accepted"
            $script:TestsFailed++
            $script:HighRiskFindings += "Evil Origin Accepted"
        } else {
            Write-Success "Evil Origin Rejection"
            $script:TestsPassed++
        }
        
    } catch {
        Write-Success "CORS Properly Restricted (or not configured)"
        $script:TestsPassed += 2
    }
}

# Phase 3: XSS Tests
function Test-XSS {
    Write-Header "PHASE 3: XSS VULNERABILITY TESTS"
    
    $xssPayloads = @(
        "<script>alert('XSS')</script>",
        "javascript:alert('XSS')",
        "'`"><script>alert('XSS')</script>",
        "<img src=x onerror=alert('XSS')>"
    )
    
    $params = @('q', 'search', 'query', 'name', 'message', 'text')
    $xssFound = $false
    
    foreach ($param in $params) {
        foreach ($payload in $xssPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                $response = Invoke-WebRequest -Uri $testUrl -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                
                if ($response.Content -like "*$payload*") {
                    Write-Failure "XSS Vulnerability in parameter: $param"
                    Write-Warning2 "  RISK: High - Reflected XSS detected"
                    Write-Warning2 "  Payload: $payload"
                    $script:HighRiskFindings += "XSS Vulnerability in parameter '$param'"
                    $script:RiskScore += 5  # 5% weight
                    $xssFound = $true
                    break
                }
            } catch {
                # Ignore errors
            }
        }
        if ($xssFound) { break }
    }
    
    if (-not $xssFound) {
        Write-Success "XSS Protection Test"
        $script:TestsPassed++
    } else {
        $script:TestsFailed++
    }
}

# Phase 4: SQL Injection Tests
function Test-SQLInjection {
    Write-Header "PHASE 4: SQL INJECTION TESTS"
    
    $sqlPayloads = @(
        "' OR '1'='1",
        "'; DROP TABLE users; --",
        "' UNION SELECT null, null --",
        "admin'--"
    )
    
    $params = @('id', 'user', 'username', 'email', 'search')
    $sqliFound = $false
    
    foreach ($param in $params) {
        foreach ($payload in $sqlPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                $response = Invoke-WebRequest -Uri $testUrl -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
                
                $pattern = 'sql syntax|mysql.*error|postgresql.*error|ora-[0-9]+|sqlite.*error|syntax error.*sql'
                if ($response.Content -match $pattern) {
                    Write-Failure "SQL Injection Vulnerability in parameter: $param"
                    Write-Warning2 "  RISK: High - SQL error detected"
                    Write-Warning2 "  Payload: $payload"
                    $script:HighRiskFindings += "SQL Injection in parameter '$param'"
                    $script:RiskScore += 5  # 5% weight
                    $sqliFound = $true
                    break
                }
            } catch {
                # Ignore errors
            }
        }
        if ($sqliFound) { break }
    }
    
    if (-not $sqliFound) {
        Write-Success "SQL Injection Protection Test"
        $script:TestsPassed++
    } else {
        $script:TestsFailed++
    }
}

# Phase 5: Information Disclosure
function Test-InformationDisclosure {
    Write-Header "PHASE 5: INFORMATION DISCLOSURE TESTS"
    
    $sensitivePaths = @('/.git', '/.env', '/admin', '/phpmyadmin')
    
    foreach ($path in $sensitivePaths) {
        try {
            $testUrl = $TargetUrl + $path
            $response = Invoke-WebRequest -Uri $testUrl -TimeoutSec 10 -UseBasicParsing -ErrorAction Stop
            
            if ($response.StatusCode -eq 200) {
                Write-Failure "Accessible: $path"
                Write-Warning2 "  RISK: Sensitive path exposed"
                if ($path -in @('/.git', '/.env')) {
                    $script:HighRiskFindings += "$path accessible"
                    $script:RiskScore += 3  # 3% weight - part of info disclosure
                } else {
                    $script:MediumRiskFindings += "$path accessible"
                    $script:RiskScore += 3  # 3% weight - part of info disclosure
                }
            }
        } catch {
            # Path not accessible - good!
        }
    }
    
    Write-Success "Information Disclosure Test Completed"
    $script:TestsPassed++
}

# Phase 6: Struts 2 DevMode Detection
function Test-StrutsDevMode {
    Write-Header "PHASE 6: STRUTS 2 DEVMODE & ENUMERATION"
    
    $devModeParams = @('debug=xml', 'debug=console', 'struts.devMode=true')
    
    foreach ($param in $devModeParams) {
        try {
            $testUrl = "${TargetUrl}?${param}"
            $response = Invoke-WebRequest -Uri $testUrl -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
            
            $pattern = 'devmode|debug.*mode|struts.*debug|action.*mapping'
            if ($response.Content -match $pattern) {
                Write-Failure "DevMode ENABLED - Debug information exposed!"
                Write-Warning2 "  RISK: High - Development mode in production"
                $script:HighRiskFindings += "Struts 2 DevMode Enabled"
                $script:RiskScore += 10  # 10% weight - Critical for Struts 2
                $script:TestsFailed++
                return
            }
        } catch {
            # Ignore
        }
    }
    
    Write-Success "DevMode Not Detected"
    $script:TestsPassed++
}

# Phase 7: Comprehensive OGNL Injection Tests (21 CVEs)
function Test-OGNLInjection {
    Write-Header "PHASE 7: STRUTS 2 OGNL INJECTION TESTING"
    Write-ColorOutput "Testing for Remote Code Execution vulnerabilities (21 CVEs)" -Color Red
    
    $ognlVulnsFound = 0
    
    # Test 1: Content-Type OGNL Injection (S2-045, S2-046)
    Write-Host ""
    Write-ColorOutput "Test 1: Content-Type OGNL Injection (S2-045, S2-046)" -Color Cyan
    $ognlPayload = "%{(#_='multipart/form-data').(#dm=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS)}"
    
    try {
        $hdrs = @{
            'Content-Type' = $ognlPayload
        }
        $response = Invoke-WebRequest -Uri $TargetUrl -Headers $hdrs -Method Post -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
        
        if ($response.Content -match 'ognl|exception|error.*ognl|struts.*problem') {
            Write-Failure "OGNL Injection via Content-Type (S2-045/046)"
            Write-Warning2 "  RISK: CRITICAL - Remote Code Execution possible"
            $script:HighRiskFindings += "OGNL Injection via Content-Type (S2-045/046)"
            $script:RiskScore += 15  # 15% weight - CRITICAL RCE vulnerability
            $script:TestsFailed++
            $ognlVulnsFound++
        } else {
            Write-Success "Content-Type OGNL injection not detected"
            $script:TestsPassed++
        }
    } catch {
        Write-Success "Content-Type OGNL injection not detected"
        $script:TestsPassed++
    }
    
    # Test 2: Redirect Parameter OGNL Injection (S2-057)
    Write-Host ""
    Write-ColorOutput "Test 2: Redirect Parameter OGNL Injection (S2-057)" -Color Cyan
    $redirectOgnl = "redirect:%{@java.lang.System@getProperty('user.name')}"
    $encodedRedirect = [System.Web.HttpUtility]::UrlEncode($redirectOgnl)
    
    try {
        $response = Invoke-WebRequest -Uri "${TargetUrl}?redirect=${encodedRedirect}" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
        
        if ($response.Content -match 'ognl|redirect.*error|invalid.*redirect') {
            Write-Failure "OGNL Injection via redirect parameter (S2-057)"
            Write-Warning2 "  RISK: CRITICAL - Remote Code Execution possible"
            $script:HighRiskFindings += "OGNL Injection via redirect parameter (S2-057)"
            $script:RiskScore += 10  # 10% weight - CRITICAL RCE (less common vector)
            $script:TestsFailed++
            $ognlVulnsFound++
        } else {
            Write-Success "Redirect OGNL injection not detected"
            $script:TestsPassed++
        }
    } catch {
        Write-Success "Redirect OGNL injection not detected"
        $script:TestsPassed++
    }
    
    # Test 3: URL Parameter OGNL Injection
    Write-Host ""
    Write-ColorOutput "Test 3: URL Parameter OGNL Injection" -Color Cyan
    $urlPayloads = @(
        "%{#context['xwork.MethodAccessor.denyMethodExecution']=false}",
        "%{@java.lang.System@getProperty('user.name')}",
        "`${@java.lang.System@getProperty('os.name')}"
    )
    
    $urlOgnlFound = $false
    foreach ($payload in $urlPayloads) {
        try {
            $encoded = [System.Web.HttpUtility]::UrlEncode($payload)
            $response = Invoke-WebRequest -Uri "${TargetUrl}?test=${encoded}" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
            
            if ($response.Content -match 'ognl.*exception|struts.*problem|evaluation.*error') {
                Write-Failure "OGNL evaluation detected in URL parameters"
                Write-Warning2 "  Payload: $payload"
                $script:HighRiskFindings += "OGNL Injection via URL parameters"
                $script:RiskScore += 10  # 10% weight - CRITICAL RCE via URL
                $script:TestsFailed++
                $ognlVulnsFound++
                $urlOgnlFound = $true
                break
            }
        } catch {
            # Continue testing
        }
    }
    
    if (-not $urlOgnlFound) {
        Write-Success "URL parameter OGNL injection not detected"
        $script:TestsPassed++
    }
    
    # Test 4: Deep OGNL Scan (Additional CVEs)
    Write-Host ""
    Write-ColorOutput "Test 4: Deep OGNL Scan (S2-001 through S2-062)" -Color Cyan
    Write-ColorOutput "Testing comprehensive CVE coverage..." -Color Yellow
    
    $deepOgnlPayloads = @(
        # S2-001, S2-003, S2-005
        "%{#a=(new java.lang.ProcessBuilder(new java.lang.String[]{'whoami'})).start()}",
        "%{#_memberAccess['allowStaticMethodAccess']=true}",
        # S2-007, S2-008, S2-009
        "age=1&name=%{#context['xwork.MethodAccessor.denyMethodExecution']=false}",
        # S2-012, S2-013
        "redirect:%{#a=(new java.lang.ProcessBuilder(new java.lang.String[]{'id'})).start()}",
        # S2-015, S2-016
        "*{#context['xwork.MethodAccessor.denyMethodExecution']=false}",
        # S2-029
        "%{(#_='multipart/form-data').(#_memberAccess=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS)}",
        # S2-032, S2-033
        "method:%{#_memberAccess=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS}",
        # S2-037
        "class.classLoader.resources.dirContext.docBase=.",
        # S2-053
        "%{(#dm=@ognl.OgnlContext@DEFAULT_MEMBER_ACCESS)}",
        # S2-059
        "%{(#context['xwork.MethodAccessor.denyMethodExecution']=false)}",
        # S2-062
        "class['classLoader']['resources']['dirContext']['docBase']"
    )
    
    $deepVulnsFound = 0
    $testEndpoints = @("", "/index.action", "/login.action", "/user.action", "/admin.action")
    
    foreach ($endpoint in $testEndpoints) {
        foreach ($payload in $deepOgnlPayloads) {
            try {
                $testUrl = "${TargetUrl}${endpoint}?test=${payload}"
                $response = Invoke-WebRequest -Uri $testUrl -TimeoutSec 5 -UseBasicParsing -ErrorAction SilentlyContinue
                
                if ($response.Content -match 'ognl.*exception|ognl.*error|expression.*evaluation|java\.lang|ProcessBuilder|memberAccess|xwork\.MethodAccessor') {
                    Write-Failure "Deep scan OGNL vulnerability at: ${endpoint}"
                    Write-Warning2 "  Payload matched OGNL execution pattern"
                    $script:HighRiskFindings += "OGNL vulnerability (deep scan) at ${endpoint}"
                    $script:RiskScore += 15  # 15% weight - CRITICAL RCE (comprehensive CVE coverage)
                    $script:TestsFailed++
                    $ognlVulnsFound++
                    $deepVulnsFound++
                    break
                }
            } catch {
                # Continue
            }
        }
        if ($deepVulnsFound -gt 0) { break }
    }
    
    if ($deepVulnsFound -eq 0) {
        Write-Success "No additional OGNL vulnerabilities in deep scan"
        $script:TestsPassed++
    }
    
    # Summary
    Write-Host ""
    if ($ognlVulnsFound -gt 0) {
        Write-ColorOutput "CRITICAL: $ognlVulnsFound OGNL injection vulnerabilities found!" -Color Red
        Write-ColorOutput "IMMEDIATE ACTION REQUIRED - Remote Code Execution possible" -Color Red
        Write-ColorOutput "Recommended: Upgrade to latest Struts 2 version immediately" -Color Yellow
    } else {
        Write-ColorOutput "No OGNL injection vulnerabilities detected" -Color Green
    }
}

# Phase 8: Action Method Exposure Testing
function Test-ActionMethods {
    Write-Header "PHASE 8: STRUTS 2 ACTION METHOD SECURITY"
    
    $actionMethodIssues = 0
    
    # Test direct method invocation
    Write-Host ""
    Write-ColorOutput "Test 1: Direct Action Method Invocation" -Color Cyan
    
    $actionMethods = @('execute', 'input', 'delete', 'update', 'admin', 'list')
    $baseActions = @('login', 'user', 'admin', 'action', 'default')
    
    foreach ($base in $baseActions) {
        foreach ($method in $actionMethods) {
            $methodUrl = "${TargetUrl}/${base}!${method}.action"
            try {
                $response = Invoke-WebRequest -Uri $methodUrl -TimeoutSec 5 -UseBasicParsing -ErrorAction SilentlyContinue
                
                if ($response.StatusCode -eq 200 -or $response.StatusCode -eq 302) {
                    Write-Failure "Exposed method: ${base}!${method}.action (HTTP $($response.StatusCode))"
                    $script:HighRiskFindings += "Exposed action method: ${base}!${method}.action"
                    $script:RiskScore += 2  # 2% weight each (10% total for DMI = 5 methods max)
                    $script:TestsFailed++
                    $actionMethodIssues++
                    break
                }
            } catch {
                # Method not accessible - good
            }
        }
    }
    
    if ($actionMethodIssues -eq 0) {
        Write-Success "No direct method invocation vulnerabilities detected"
        $script:TestsPassed++
    }
    
    # Test namespace traversal
    Write-Host ""
    Write-ColorOutput "Test 2: Namespace Traversal Testing" -Color Cyan
    
    $namespaces = @('/admin', '/api', '/secure', '/internal', '/management')
    $traversalIssues = 0
    
    foreach ($ns in $namespaces) {
        try {
            $response = Invoke-WebRequest -Uri "${TargetUrl}${ns}/.." -TimeoutSec 5 -UseBasicParsing -ErrorAction SilentlyContinue
            
            if ($response.StatusCode -eq 200 -and $response.Content -match 'admin|secure|internal') {
                Write-Failure "Namespace traversal possible: $ns"
                $script:MediumRiskFindings += "Namespace traversal at: $ns"
                $script:RiskScore += 1  # 1% weight each (5% total for namespace traversal)
                $script:TestsFailed++
                $traversalIssues++
            }
        } catch {
            # Good - traversal blocked
        }
    }
    
    if ($traversalIssues -eq 0) {
        Write-Success "No namespace traversal vulnerabilities detected"
        $script:TestsPassed++
    }
    
    Write-Host ""
    if ($actionMethodIssues -gt 0) {
        Write-ColorOutput "Found $actionMethodIssues exposed action methods" -Color Yellow
        Write-ColorOutput "Recommendation: Disable Dynamic Method Invocation (DMI)" -Color Yellow
    }
}

# Phase 9: OWASP ZAP Scan (Full Mode)
function Invoke-ZapScan {
    Write-Header "PHASE 9: OWASP ZAP COMPREHENSIVE SCAN"
    
    if (-not (Test-Path $ZapPath)) {
        Write-Warning2 "ZAP not found at: $ZapPath"
        Write-Warning2 "Skipping ZAP scan. Install ZAP or specify -ZapPath"
        Write-ColorOutput "  Download ZAP: https://www.zaproxy.org/download/" -Color Gray
        return
    }
    
    # Kill any running ZAP instances to avoid home directory conflicts
    Write-ColorOutput "Checking for running ZAP instances..." -Color Cyan
    
    # Find ZAP processes using WMI (works in PowerShell 5.1)
    $zapProcesses = Get-WmiObject Win32_Process -Filter "name = 'java.exe'" -ErrorAction SilentlyContinue | 
                    Where-Object { $_.CommandLine -like "*zap*.jar*" }
    
    if ($zapProcesses) {
        $count = @($zapProcesses).Count
        Write-ColorOutput "  Found $count running ZAP instance(s). Stopping them..." -Color Yellow
        foreach ($proc in $zapProcesses) {
            try {
                Stop-Process -Id $proc.ProcessId -Force -ErrorAction SilentlyContinue
                Write-ColorOutput "  Stopped ZAP process (PID: $($proc.ProcessId))" -Color Gray
            } catch {
                Write-ColorOutput "  Could not stop process $($proc.ProcessId)" -Color Gray
            }
        }
        Start-Sleep -Seconds 3
        Write-Success "Previous ZAP instances closed"
    } else {
        Write-ColorOutput "  No running ZAP instances found" -Color Gray
    }
    Write-Host ""
    
    Write-ColorOutput "Starting OWASP ZAP comprehensive scan (10-30 minutes)..." -Color Yellow
    Write-ColorOutput "ZAP will test: Advanced XSS, SQLi, Command Injection, Path Traversal," -Color Yellow
    Write-ColorOutput "               XXE, CSRF, Session Management, Cookie Security, and more..." -Color Yellow
    Write-Host ""
    
    try {
        # ZAP quick scan (console output only, no report files)
        # Use port 7777 to avoid conflicts with applications running on 8080
        $zapArgs = @(
            "-cmd",
            "-quickurl", $TargetUrl,
            "-quickprogress",
            "-port", "7777"
        )
        
        Write-ColorOutput "Executing ZAP scan..." -Color Cyan
        Write-Host ""
        
        # Execute ZAP (must run from ZAP directory)
        $zapDir = Split-Path $ZapPath -Parent
        
        Write-ColorOutput "Running ZAP baseline scan against: $TargetUrl" -Color Cyan
        Write-ColorOutput "Progress will be displayed below..." -Color Gray
        Write-Host ("=" * 60)
        
        $zapProcess = Start-Process -FilePath $ZapPath `
                                    -ArgumentList $zapArgs `
                                    -WorkingDirectory $zapDir `
                                    -Wait `
                                    -PassThru `
                                    -NoNewWindow
        
        $exitCode = $zapProcess.ExitCode
        
        Write-Host ("=" * 60)
        Write-Host ""
        
        if ($exitCode -eq 0) {
            Write-Success "ZAP scan completed successfully - No critical issues found"
            $script:TestsPassed++
        } elseif ($exitCode -eq 1) {
            Write-Warning2 "ZAP scan completed - Vulnerabilities detected (Exit Code: 1)"
            Write-ColorOutput "  ZAP found security issues during the scan" -Color Yellow
            $script:InfoFindings += "ZAP detected vulnerabilities (check console output above)"
        } elseif ($exitCode -eq 2) {
            Write-Failure "ZAP scan completed with warnings (Exit Code: 2)"
            Write-ColorOutput "  Some tests may have failed or been skipped" -Color Yellow
            $script:TestsFailed++
        } else {
            Write-Failure "ZAP scan error (Exit Code: $exitCode)"
            $script:TestsFailed++
        }
        
        Write-Host ""
        
        Write-ColorOutput "ZAP scan analysis:" -Color Cyan
        Write-ColorOutput "  - Tests 50+ vulnerability types" -Color Gray
        Write-ColorOutput "  - Advanced XSS (DOM, stored, reflected)" -Color Gray
        Write-ColorOutput "  - Advanced SQLi (blind, time-based, boolean)" -Color Gray
        Write-ColorOutput "  - Command injection, path traversal, XXE" -Color Gray
        Write-ColorOutput "  - CSRF, session management, cookie security" -Color Gray
        Write-ColorOutput "  - Authentication bypass, authorization issues" -Color Gray
        Write-ColorOutput "  - SSL/TLS configuration, information leakage" -Color Gray
        
    } catch {
        Write-Failure "ZAP scan failed: $($_.Exception.Message)"
        Write-ColorOutput "  Error type: $($_.Exception.GetType().Name)" -Color Gray
        $script:TestsFailed++
    }
}

# Generate Summary
function Show-Summary {
    Write-Host ""
    Write-Header "SECURITY ASSESSMENT SUMMARY"
    Write-Host ("=" * 50)
    
    $totalFindings = $script:HighRiskFindings.Count + $script:MediumRiskFindings.Count + $script:LowRiskFindings.Count
    
    # Use percentage-based risk score (0-100)
    # Each failed test adds its weight percentage to the score
    # Score is accumulated throughout the scan
    $riskScore = [Math]::Min([Math]::Round($script:RiskScore), 100)
    
    Write-ColorOutput "Target: $TargetUrl" -Color Blue
    Write-ColorOutput "Scan Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -Color Blue
    Write-ColorOutput "Tests Passed: $script:TestsPassed" -Color Green
    Write-ColorOutput "Tests Failed: $script:TestsFailed" -Color Red
    
    Write-Host ""
    
    # Risk Level
    $riskLevel = "LOW"
    $riskColor = "Green"
    if ($script:HighRiskFindings.Count -gt 0) {
        $riskLevel = "HIGH"
        $riskColor = "Red"
    } elseif ($script:MediumRiskFindings.Count -gt 3) {
        $riskLevel = "MEDIUM"
        $riskColor = "Yellow"
    }
    
    Write-Host "Overall Risk Level: " -NoNewline
    Write-Host $riskLevel -ForegroundColor $riskColor
    Write-ColorOutput "Risk Score: $riskScore/100 (lower is better)" -Color Blue
    
    # Findings
    if ($totalFindings -gt 0) {
        Write-Host ""
        Write-ColorOutput "VULNERABILITY FINDINGS:" -Color Yellow
        
        if ($script:HighRiskFindings.Count -gt 0) {
            Write-Host ""
            Write-ColorOutput "HIGH RISK ($($script:HighRiskFindings.Count)):" -Color Red
            foreach ($finding in $script:HighRiskFindings) {
                Write-ColorOutput "  - $finding" -Color Red
            }
        }
        
        if ($script:MediumRiskFindings.Count -gt 0) {
            Write-Host ""
            Write-ColorOutput "MEDIUM RISK ($($script:MediumRiskFindings.Count)):" -Color Yellow
            foreach ($finding in $script:MediumRiskFindings) {
                Write-ColorOutput "  - $finding" -Color Yellow
            }
        }
        
        if ($script:LowRiskFindings.Count -gt 0) {
            Write-Host ""
            Write-ColorOutput "LOW RISK ($($script:LowRiskFindings.Count)):" -Color Gray
            foreach ($finding in $script:LowRiskFindings) {
                Write-ColorOutput "  - $finding" -Color Gray
            }
        }
    } else {
        Write-Host ""
        Write-Success "No security findings detected!"
    }
    
    Write-Host ""
    Write-Host ("=" * 50)
}

# Main execution
try {
    Add-Type -AssemblyName System.Web -ErrorAction SilentlyContinue
    
    Write-Host ""
    Write-ColorOutput "STRUTS 2 SECURITY SCANNER (PowerShell)" -Color Magenta
    Write-Host ("=" * 50)
    Write-ColorOutput "Target: $TargetUrl" -Color Cyan
    Write-ColorOutput "Mode: $Mode" -Color Cyan
    Write-Host ("=" * 50)
    
    # Test connection first
    if (-not (Test-Connection2 -Url $TargetUrl)) {
        exit 1
    }
    
    # Run tests
    Test-SecurityHeaders
    Test-CORS
    Test-XSS
    Test-SQLInjection
    Test-InformationDisclosure
    Test-StrutsDevMode
    Test-OGNLInjection
    Test-ActionMethods
    
    # Run ZAP scan if Full mode
    if ($Mode -eq 'Full') {
        Invoke-ZapScan
    } else {
        Write-Host ""
        Write-ColorOutput "Quick mode complete (8 phases). Use -Mode Full to add OWASP ZAP deep scan (Phase 9)." -Color Blue
    }
    
    # Show summary
    Show-Summary
    
    Write-Host ""
    Write-Success "Security scan completed!"
    Write-Host ""
    
} catch {
    Write-Failure "Script error: $($_.Exception.Message)"
    exit 1
}
