﻿<#
.SYNOPSIS
    Web Malware Detection Script (PowerShell)

.DESCRIPTION
    Scans websites for malicious code, backdoors, cryptominers, and signs of compromise
    Part of Security Scanner Desktop - Reconnaissance Features

.PARAMETER TargetUrl
    Target web application URL

.PARAMETER Mode
    Scan mode: quick or full (default: full)

.PARAMETER CleanOutput
    Disable colored output

.EXAMPLE
    .\web-malware-detection-test.ps1 -TargetUrl "https://example.com" -Mode quick
    .\web-malware-detection-test.ps1 -TargetUrl "https://example.com" -Mode full
#>

[CmdletBinding()]
param(
    [Parameter(Mandatory=$false)]
    [string]$TargetUrl,

    [Parameter(Mandatory=$false)]
    [ValidateSet('quick', 'full')]
    [string]$Mode = 'full',

    [Parameter(Mandatory=$false)]
    [switch]$CleanOutput,

    [Parameter(Mandatory=$false)]
    [switch]$Help
)

if ($Help) {
    Write-Host "WEB MALWARE DETECTION SCANNER"
    Write-Host "======================================================"
    Write-Host "Scans websites for malicious code and signs of compromise"
    Write-Host ""
    Write-Host "USAGE: .\web-malware-detection-test.ps1 -TargetUrl <URL> [-Mode <quick|full>] [-CleanOutput]"
    Write-Host ""
    Write-Host "DETECTION CATEGORIES:"
    Write-Host "  1. Malicious JavaScript (obfuscation, cryptominers, keyloggers)"
    Write-Host "  2. PHP Backdoors / Web Shells"
    Write-Host "  3. SEO Spam Injection"
    Write-Host "  4. Defacement Detection"
    Write-Host "  5. Suspicious External Resources"
    Write-Host "  6. Blacklisted Patterns"
    Write-Host ""
    Write-Host "THREAT LEVELS:"
    Write-Host "  CLEAN (0-2 points): No significant threats"
    Write-Host "  LOW (3-5 points): Minor concerns"
    Write-Host "  MEDIUM (6-9 points): Suspicious patterns found"
    Write-Host "  HIGH (10+ points): Multiple malware indicators"
    exit 0
}

if ([string]::IsNullOrEmpty($TargetUrl)) {
    Write-Host "Error: Target URL is required" -ForegroundColor Red
    Write-Host "Usage: .\web-malware-detection-test.ps1 -TargetUrl <URL> [-Mode <quick|full>] [-CleanOutput]"
    exit 1
}

# Normalize URL
if ($TargetUrl -notmatch "^https?://") {
    $TargetUrl = "https://$TargetUrl"
}
$TargetUrl = $TargetUrl.TrimEnd('/')

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [string]$UserAgent, [int]$TimeoutSec = 30)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    if ($UserAgent) { $params.UserAgent = $UserAgent }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

# Detection results
$script:ThreatScore = 0
$script:MaxScore = 20
$script:ThreatLevel = "CLEAN"
$script:Findings = @()

# Detection details
$script:ObfuscationPatterns = 0
$script:CryptominerDetected = $false
$script:BackdoorsFound = $false
$script:SpamKeywords = 0
$script:HiddenLinks = 0
$script:DefacementDetected = $false
$script:SuspiciousTLDs = 0
$script:BlacklistedPatterns = 0

# Page content
$script:PageContent = ""

# Header output
Write-Host ""
Write-Host "WEB MALWARE DETECTION SCANNER" -ForegroundColor Magenta
Write-Host "======================================================" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl" -ForegroundColor Blue
Write-Host "Mode: $Mode" -ForegroundColor Blue
Write-Host "Timestamp: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -ForegroundColor Blue
if ($AuthToken) { Write-Host "[*] Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "[*] Cookie authentication enabled" -ForegroundColor Green }
if (-not $AuthToken -and -not $AuthCookie) { Write-Host "[i] Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

Write-Host ""

# Function to add finding and update score
function Add-Finding {
    param(
        [string]$Finding,
        [int]$Points
    )

    $script:Findings += $Finding
    $script:ThreatScore += $Points
    Write-Host "  FOUND: $Finding (+$Points points)" -ForegroundColor Yellow
}

# Fetch page content
function Get-PageContent {
    Write-Host ""
    Write-Host "Fetching target content..." -ForegroundColor Cyan

    try {
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method Get -TimeoutSec 30 -UserAgent "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36"
        $script:PageContent = $response.Content

        if ([string]::IsNullOrEmpty($script:PageContent)) {
            Write-Host "  Failed to fetch page content" -ForegroundColor Red
            return $false
        }

        $contentLength = $script:PageContent.Length
        Write-Host "  Fetched $contentLength bytes" -ForegroundColor Gray
        return $true
    } catch {
        Write-Host "  Failed to fetch page content: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

# Category 1: Malicious JavaScript Detection
function Test-MaliciousJavaScript {
    Write-Host ""
    Write-Host "CATEGORY 1: MALICIOUS JAVASCRIPT DETECTION" -ForegroundColor Yellow

    if ([string]::IsNullOrEmpty($script:PageContent)) {
        Write-Host "  Skipped - no content available" -ForegroundColor Gray
        return
    }

    # Check for obfuscation patterns
    $obfuscationPatterns = @(
        "eval\(",
        "unescape\(",
        "atob\(",
        "String\.fromCharCode",
        "document\.write\(unescape",
        "window\.atob",
        "decodeURIComponent",
        "\\x[0-9a-fA-F]{2}",
        "\\u[0-9a-fA-F]{4}"
    )

    $obfCount = 0
    foreach ($pattern in $obfuscationPatterns) {
        if ($script:PageContent -match $pattern) {
            $obfCount++
        }
    }

    $script:ObfuscationPatterns = $obfCount

    if ($obfCount -ge 5) {
        Add-Finding -Finding "High JavaScript obfuscation detected ($obfCount patterns)" -Points 3
    } elseif ($obfCount -ge 3) {
        Add-Finding -Finding "Moderate JavaScript obfuscation ($obfCount patterns)" -Points 2
        Write-Host "    Note: Some obfuscation may be legitimate minification" -ForegroundColor Gray
    } elseif ($obfCount -ge 1) {
        Write-Host "  Low obfuscation level ($obfCount patterns) - likely normal" -ForegroundColor Gray
    }

    # Check for cryptominer signatures
    $cryptominerPatterns = @(
        "CoinHive",
        "coinhive\.min\.js",
        "coin-hive\.com",
        "JSEcoin",
        "Client\.Anonymous",
        "CryptoNoter",
        "CryptoLoot",
        "cryptoloot\.pro",
        "deepMiner",
        "webminerpool",
        "minero\.cc",
        "authedmine",
        "crypto-loot\.com",
        "miner\.start",
        "WebMR",
        "webmr\.js"
    )

    foreach ($pattern in $cryptominerPatterns) {
        if ($script:PageContent -match $pattern) {
            $script:CryptominerDetected = $true
            Add-Finding -Finding "Cryptominer detected: $pattern" -Points 5
            break
        }
    }

    # Check for keylogger patterns
    $keyloggerPatterns = @(
        "onkeypress.*POST",
        "onkeyup.*send",
        "onkeydown.*ajax",
        "addEventListener.*keypress.*fetch",
        "document\.onkeypress",
        "keyCode.*XMLHttpRequest"
    )

    foreach ($pattern in $keyloggerPatterns) {
        if ($script:PageContent -match $pattern) {
            Add-Finding -Finding "Potential keylogger pattern detected" -Points 4
            break
        }
    }

    # Check for malicious redirects
    if ($script:PageContent -match "location\.(href|replace).*\.(tk|ml|ga|cf|gq)") {
        Add-Finding -Finding "Suspicious redirect to malicious TLD" -Points 3
    }

    if (-not $script:CryptominerDetected -and $obfCount -lt 3) {
        Write-Host "  No malicious JavaScript patterns detected" -ForegroundColor Green
    }
}

# Category 2: PHP Backdoor / Web Shell Detection
function Test-Backdoors {
    Write-Host ""
    Write-Host "CATEGORY 2: PHP BACKDOOR / WEB SHELL DETECTION" -ForegroundColor Yellow

    # Common web shell paths to check
    $shellPaths = @(
        "shell.php",
        "c99.php",
        "r57.php",
        "wso.php",
        "b374k.php",
        "cmd.php",
        "backdoor.php",
        "upload.php",
        "filemanager.php",
        "phpinfo.php",
        ".well-known/shell.php",
        "wp-content/uploads/shell.php",
        "images/shell.php"
    )

    $foundShells = 0

    foreach ($path in $shellPaths) {
        try {
            $testUrl = "$TargetUrl/$path"
            $response = Invoke-AuthWebRequest -Uri $testUrl -Method Head -TimeoutSec 10
            if ($response.StatusCode -eq 200) {
                $script:BackdoorsFound = $true
                $foundShells++
                Write-Host "  Potential web shell found: $testUrl (HTTP $($response.StatusCode))" -ForegroundColor Red
            }
        } catch {
            # Ignore errors (404, 403, etc.)
        }
    }

    if ($foundShells -gt 0) {
        Add-Finding -Finding "Potential backdoor/shell files accessible ($foundShells found)" -Points 5
    }

    # Check for backdoor patterns in page content
    if (-not [string]::IsNullOrEmpty($script:PageContent)) {
        $backdoorPatterns = @(
            'eval\s*\(\s*\$_(POST|GET|REQUEST)',
            'system\s*\(\s*\$_(POST|GET|REQUEST)',
            'exec\s*\(\s*\$_(POST|GET|REQUEST)',
            'passthru\s*\(\s*\$_(POST|GET|REQUEST)',
            'shell_exec\s*\(\s*\$_(POST|GET|REQUEST)',
            'assert\s*\(\s*\$_(POST|GET|REQUEST)',
            'preg_replace.*/e',
            'base64_decode\s*\(\s*\$_(POST|GET|REQUEST)'
        )

        foreach ($pattern in $backdoorPatterns) {
            if ($script:PageContent -match $pattern) {
                $script:BackdoorsFound = $true
                Add-Finding -Finding "PHP backdoor code pattern detected" -Points 5
                break
            }
        }
    }

    if (-not $script:BackdoorsFound) {
        Write-Host "  No backdoor/shell patterns detected" -ForegroundColor Green
    }
}

# Category 3: SEO Spam Detection
function Test-SeoSpam {
    Write-Host ""
    Write-Host "CATEGORY 3: SEO SPAM DETECTION" -ForegroundColor Yellow

    if ([string]::IsNullOrEmpty($script:PageContent)) {
        Write-Host "  Skipped - no content available" -ForegroundColor Gray
        return
    }

    # Check for hidden links
    $hiddenCount = 0
    if ($script:PageContent -match 'display:\s*none.*<a\s|visibility:\s*hidden.*<a\s|font-size:\s*0.*<a\s') {
        $hiddenCount++
    }

    if ($script:PageContent -match '<div[^>]*style="[^"]*display:\s*none[^"]*"[^>]*>.*<a\s') {
        $hiddenCount++
    }

    $script:HiddenLinks = $hiddenCount

    if ($hiddenCount -gt 0) {
        Add-Finding -Finding "Hidden links detected ($hiddenCount instances)" -Points 2
    }

    # Check for spam keywords
    $spamKeywords = @(
        "buy viagra",
        "cheap cialis",
        "pharmacy online",
        "canadian pharmacy",
        "payday loan",
        "online casino",
        "free casino",
        "poker online",
        "replica watch",
        "louis vuitton",
        "cheap oakley",
        "jordan shoes",
        "ugg boots"
    )

    $spamCount = 0
    foreach ($keyword in $spamKeywords) {
        if ($script:PageContent -match [regex]::Escape($keyword)) {
            $spamCount++
        }
    }

    $script:SpamKeywords = $spamCount

    if ($spamCount -ge 3) {
        Add-Finding -Finding "SEO spam keywords detected ($spamCount found)" -Points 3
    } elseif ($spamCount -ge 1) {
        Add-Finding -Finding "Possible spam keywords ($spamCount found)" -Points 1
    }

    # Check for hidden iframes
    if ($script:PageContent -match '<iframe[^>]*style="[^"]*width:\s*0|height:\s*0|display:\s*none') {
        Add-Finding -Finding "Hidden iframe detected" -Points 3
    }

    if ($spamCount -eq 0 -and $hiddenCount -eq 0) {
        Write-Host "  No SEO spam detected" -ForegroundColor Green
    }
}

# Category 4: Defacement Detection
function Test-Defacement {
    Write-Host ""
    Write-Host "CATEGORY 4: DEFACEMENT DETECTION" -ForegroundColor Yellow

    if ([string]::IsNullOrEmpty($script:PageContent)) {
        Write-Host "  Skipped - no content available" -ForegroundColor Gray
        return
    }

    # Defacement signatures
    $defacementPatterns = @(
        "hacked by",
        "owned by",
        "pwned by",
        "defaced by",
        "greetz to",
        "h4ck3d",
        "0wn3d",
        "d3f4c3d",
        "Team hackers",
        "cyber army",
        "hacking team",
        "security breach",
        "anonymous collective"
    )

    foreach ($pattern in $defacementPatterns) {
        if ($script:PageContent -match [regex]::Escape($pattern)) {
            $script:DefacementDetected = $true
            Add-Finding -Finding "Defacement signature detected: '$pattern'" -Points 5
            break
        }
    }

    # Check for hacktivist group markers
    $hacktivistGroups = @(
        "Anonymous",
        "LulzSec",
        "Lizard Squad",
        "Syrian Electronic Army",
        "Ghost Squad",
        "AnonGhost"
    )

    foreach ($group in $hacktivistGroups) {
        if ($script:PageContent -match "hacked.*$group|$group.*hack") {
            $script:DefacementDetected = $true
            Add-Finding -Finding "Hacktivist group marker: $group" -Points 4
            break
        }
    }

    if (-not $script:DefacementDetected) {
        Write-Host "  No defacement signatures detected" -ForegroundColor Green
    }
}

# Category 5: Suspicious External Resources
function Test-SuspiciousResources {
    Write-Host ""
    Write-Host "CATEGORY 5: SUSPICIOUS EXTERNAL RESOURCES" -ForegroundColor Yellow

    if ([string]::IsNullOrEmpty($script:PageContent)) {
        Write-Host "  Skipped - no content available" -ForegroundColor Gray
        return
    }

    # Suspicious TLDs commonly used for malware
    $suspiciousTLDs = @(
        '\.tk["\x27/\s>]',
        '\.ml["\x27/\s>]',
        '\.ga["\x27/\s>]',
        '\.cf["\x27/\s>]',
        '\.gq["\x27/\s>]',
        '\.xyz["\x27/\s>]',
        '\.top["\x27/\s>]',
        '\.work["\x27/\s>]',
        '\.click["\x27/\s>]',
        '\.download["\x27/\s>]'
    )

    $tldCount = 0
    foreach ($tld in $suspiciousTLDs) {
        if ($script:PageContent -match "(src|href)=[`"\x27][^`"\x27]*$tld") {
            $tldCount++
        }
    }

    $script:SuspiciousTLDs = $tldCount

    if ($tldCount -ge 3) {
        Add-Finding -Finding "Multiple suspicious TLDs detected ($tldCount found)" -Points 3
    } elseif ($tldCount -ge 1) {
        Add-Finding -Finding "Suspicious TLD detected ($tldCount found)" -Points 1
    }

    # Check for mixed content (HTTP resources on HTTPS page)
    if ($TargetUrl -match "^https://") {
        if ($script:PageContent -match '(src|href)="http://[^"]+\.(js|css)"') {
            Add-Finding -Finding "Mixed content detected (HTTP on HTTPS)" -Points 1
        }
    }

    # Check for data URIs (often used for malicious payloads)
    if ($script:PageContent -match 'src="data:text/javascript') {
        Add-Finding -Finding "Suspicious data URI JavaScript detected" -Points 2
    }

    # Check for external scripts from unusual sources
    if ($script:PageContent -match 'src="[^"]*pastebin\.com|src="[^"]*ghostbin\.') {
        Add-Finding -Finding "Script loaded from paste site" -Points 4
    }

    if ($tldCount -eq 0) {
        Write-Host "  No suspicious external resources detected" -ForegroundColor Green
    }
}

# Category 6: Blacklisted Patterns (replaces VirusTotal)
function Test-BlacklistedPatterns {
    Write-Host ""
    Write-Host "CATEGORY 6: BLACKLISTED PATTERNS" -ForegroundColor Yellow

    if ([string]::IsNullOrEmpty($script:PageContent)) {
        Write-Host "  Skipped - no content available" -ForegroundColor Gray
        return
    }

    # Known malicious domains
    $maliciousDomains = @(
        "malware-traffic-analysis",
        "coinhive.com",
        "coin-hive.com",
        "authedmine.com",
        "crypto-loot.com",
        "cryptoloot.pro",
        "webmine.pro",
        "minero.cc",
        "ppoi.org",
        "kisshentai.net/Content",
        "greenflashplayer"
    )

    $domainCount = 0
    foreach ($domain in $maliciousDomains) {
        if ($script:PageContent -match [regex]::Escape($domain)) {
            $domainCount++
            Write-Host "  Blacklisted domain found: $domain" -ForegroundColor Red
        }
    }

    $script:BlacklistedPatterns = $domainCount

    if ($domainCount -gt 0) {
        Add-Finding -Finding "Blacklisted domains detected ($domainCount found)" -Points 5
    }

    # Check for exploit kit patterns
    $exploitPatterns = @(
        "NSIS_Inetc",
        "PowerLoader",
        "Angler",
        "Magnitude",
        "RIG exploit",
        "Neutrino",
        "Nuclear",
        "Sweet Orange"
    )

    foreach ($pattern in $exploitPatterns) {
        if ($script:PageContent -match [regex]::Escape($pattern)) {
            Add-Finding -Finding "Exploit kit signature: $pattern" -Points 5
            break
        }
    }

    # Check for suspicious base64 encoded strings (potential encoded payloads)
    $base64Matches = [regex]::Matches($script:PageContent, '[A-Za-z0-9+/]{100,}={0,2}')
    $base64Count = $base64Matches.Count

    if ($base64Count -gt 5) {
        Add-Finding -Finding "Multiple large Base64 strings detected ($base64Count) - potential encoded payloads" -Points 2
    }

    if ($domainCount -eq 0) {
        Write-Host "  No blacklisted patterns detected" -ForegroundColor Green
    }
}

# Calculate threat level
function Get-ThreatLevel {
    if ($script:ThreatScore -ge 10) {
        $script:ThreatLevel = "HIGH"
    } elseif ($script:ThreatScore -ge 6) {
        $script:ThreatLevel = "MEDIUM"
    } elseif ($script:ThreatScore -ge 3) {
        $script:ThreatLevel = "LOW"
    } else {
        $script:ThreatLevel = "CLEAN"
    }
}

# Generate summary
function Show-Summary {
    Get-ThreatLevel

    Write-Host ""
    Write-Host "======================================================" -ForegroundColor Magenta
    Write-Host "WEB MALWARE DETECTION SUMMARY" -ForegroundColor Magenta
    Write-Host "======================================================" -ForegroundColor Magenta

    Write-Host ""
    Write-Host "Target: $TargetUrl" -ForegroundColor Blue
    Write-Host "Scan completed: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -ForegroundColor Blue
    Write-Host ""

    # Display threat level with color
    switch ($script:ThreatLevel) {
        "HIGH" {
            Write-Host "THREAT LEVEL: HIGH" -ForegroundColor Red
            Write-Host "Score: $($script:ThreatScore)/$($script:MaxScore)" -ForegroundColor Red
        }
        "MEDIUM" {
            Write-Host "THREAT LEVEL: MEDIUM" -ForegroundColor Yellow
            Write-Host "Score: $($script:ThreatScore)/$($script:MaxScore)" -ForegroundColor Yellow
        }
        "LOW" {
            Write-Host "THREAT LEVEL: LOW" -ForegroundColor Yellow
            Write-Host "Score: $($script:ThreatScore)/$($script:MaxScore)" -ForegroundColor Yellow
        }
        default {
            Write-Host "THREAT LEVEL: CLEAN" -ForegroundColor Green
            Write-Host "Score: $($script:ThreatScore)/$($script:MaxScore)" -ForegroundColor Green
        }
    }

    if ($script:Findings.Count -gt 0) {
        Write-Host ""
        Write-Host "Findings:" -ForegroundColor Cyan
        foreach ($finding in $script:Findings) {
            Write-Host "  - $finding"
        }
    }

    Write-Host ""
    Write-Host "Detection Details:" -ForegroundColor Cyan
    Write-Host "  Obfuscation Patterns: $($script:ObfuscationPatterns)"
    Write-Host "  Cryptominer Detected: $($script:CryptominerDetected)"
    Write-Host "  Backdoors Found: $($script:BackdoorsFound)"
    Write-Host "  Spam Keywords: $($script:SpamKeywords)"
    Write-Host "  Hidden Links: $($script:HiddenLinks)"
    Write-Host "  Defacement Detected: $($script:DefacementDetected)"
    Write-Host "  Suspicious TLDs: $($script:SuspiciousTLDs)"
    Write-Host "  Blacklisted Patterns: $($script:BlacklistedPatterns)"

    # Recommendation
    Write-Host ""
    Write-Host "RECOMMENDATION:" -ForegroundColor Cyan
    switch ($script:ThreatLevel) {
        "HIGH" {
            Write-Host "  CRITICAL: Multiple malware indicators detected. Site appears compromised." -ForegroundColor Red
            Write-Host "  Investigate immediately and consider taking the site offline." -ForegroundColor Red
        }
        "MEDIUM" {
            Write-Host "  WARNING: Suspicious patterns found. Manual review recommended." -ForegroundColor Yellow
            Write-Host "  Check source code and server logs for further investigation." -ForegroundColor Yellow
        }
        "LOW" {
            Write-Host "  NOTICE: Minor concerns detected. Review may be beneficial." -ForegroundColor Yellow
            Write-Host "  Some findings may be false positives from legitimate code." -ForegroundColor Yellow
        }
        default {
            Write-Host "  Site appears clean. No significant malware indicators detected." -ForegroundColor Green
            Write-Host "  Continue regular security monitoring." -ForegroundColor Green
        }
    }

    Write-Host ""
    Write-Host "======================================================" -ForegroundColor Magenta

    # JSON output for application parsing
    Write-Host ""
    Write-Host "--- JSON OUTPUT ---"

    $recommendation = switch ($script:ThreatLevel) {
        "HIGH" { "CRITICAL: Site appears compromised - investigate immediately" }
        "MEDIUM" { "Suspicious patterns found - manual review recommended" }
        "LOW" { "Minor concerns detected - review may be beneficial" }
        default { "Site appears clean - continue regular monitoring" }
    }

    $jsonOutput = @{
        targetUrl = $TargetUrl
        threatLevel = $script:ThreatLevel
        threatScore = $script:ThreatScore
        maxScore = $script:MaxScore
        findings = $script:Findings
        details = @{
            obfuscationPatterns = $script:ObfuscationPatterns
            cryptominerDetected = $script:CryptominerDetected
            backdoorsFound = $script:BackdoorsFound
            spamKeywords = $script:SpamKeywords
            hiddenLinks = $script:HiddenLinks
            defacementDetected = $script:DefacementDetected
            suspiciousTLDs = $script:SuspiciousTLDs
            blacklistedPatterns = $script:BlacklistedPatterns
        }
        recommendation = $recommendation
    } | ConvertTo-Json -Compress

    Write-Host $jsonOutput
}

# Show error summary with JSON output
function Show-ErrorSummary {
    Write-Host "Unable to scan target - could not retrieve content" -ForegroundColor Red
    Write-Host ""
    Write-Host "--- JSON OUTPUT ---"

    $jsonOutput = @{
        targetUrl = $TargetUrl
        threatLevel = "UNKNOWN"
        threatScore = 0
        maxScore = $script:MaxScore
        findings = @("Unable to fetch target content")
        details = @{
            obfuscationPatterns = 0
            cryptominerDetected = $false
            backdoorsFound = $false
            spamKeywords = 0
            hiddenLinks = 0
            defacementDetected = $false
            suspiciousTLDs = 0
            blacklistedPatterns = 0
        }
        recommendation = "Unable to scan - check target URL and network connectivity"
    } | ConvertTo-Json -Compress

    Write-Host $jsonOutput
}

# Main execution
try {
    if (-not (Get-PageContent)) {
        Show-ErrorSummary
        exit 1
    }

    Test-MaliciousJavaScript

    if ($Mode -eq "full") {
        Test-Backdoors
        Test-SeoSpam
        Test-Defacement
        Test-SuspiciousResources
        Test-BlacklistedPatterns
    } elseif ($Mode -eq "quick") {
        Write-Host ""
        Write-Host "Quick mode: Skipping backdoor probing, SEO spam, defacement, and detailed checks" -ForegroundColor Gray
        # Still run lightweight content-based checks
        Test-SuspiciousResources
    }

    Show-Summary

} catch {
    Write-Host "Script error: $($_.Exception.Message)" -ForegroundColor Red
    exit 1
}
