<#
.SYNOPSIS
    Generic Web Application Security Testing Script (PowerShell)

.DESCRIPTION
    Works with any website - no specific API endpoints or authentication required
    Advanced OWASP-based testing with 11 phases including ZAP integration

.PARAMETER TargetUrl
    Target web application URL

.PARAMETER Mode
    Scan mode: Quick, Full, or ZapOnly

.EXAMPLE
    .\generic-security-test.ps1 -TargetUrl "http://localhost:3000" -Mode Quick
    .\generic-security-test.ps1 -TargetUrl "http://localhost:3000" -Mode Full
#>

[CmdletBinding()]
param(
    [Parameter(Mandatory=$true)]
    [string]$TargetUrl,

    [Parameter(Mandatory=$false)]
    [ValidateSet('Quick', 'Full', 'ZapOnly')]
    [string]$Mode = 'Full'
)

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# ZAP configuration
$ZapPort = if ($env:ZAP_PORT) { $env:ZAP_PORT } else { "8090" }
$ZapHost = if ($env:ZAP_HOST) { $env:ZAP_HOST } else { "127.0.0.1" }
$ZapBaseUrl = "http://${ZapHost}:${ZapPort}"

# Helper function for authenticated web requests
function Invoke-AuthWebRequest {
    param([string]$Uri, [string]$Method = "GET", [object]$Body, [string]$ContentType, [hashtable]$Headers, [int]$TimeoutSec = 30)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "Stop" }
    if ($Body) { $params.Body = $Body }
    if ($ContentType) { $params.ContentType = $ContentType }
    $authHeaders = @{}
    if ($Headers) { $Headers.Keys | ForEach-Object { $authHeaders[$_] = $Headers[$_] } }
    if ($AuthToken) { $authHeaders["Authorization"] = "Bearer $AuthToken" }
    if ($authHeaders.Count -gt 0) { $params.Headers = $authHeaders }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

$script:AuthErrors = 0  # Track 401/403 responses

# Global variables
$script:TestsPassed = 0
$script:TestsFailed = 0
$script:HighRiskFindings = @()
$script:MediumRiskFindings = @()
$script:LowRiskFindings = @()
$script:InfoFindings = @()

# Color output functions
function Write-ColorOutput {
    param([string]$Message, [string]$Color = 'White')
    Write-Host $Message -ForegroundColor $Color
}

function Write-Header {
    param([string]$Text)
    Write-Host ""; Write-Host $Text -ForegroundColor Cyan
}

function Write-Success {
    param([string]$Text)
    Write-Host "[+] $Text" -ForegroundColor Green
}

function Write-Failure {
    param([string]$Text)
    Write-Host "[-] $Text" -ForegroundColor Red
}

# ── ZAP Helper Functions ──

function Test-ZapAvailability {
    try {
        $response = Invoke-RestMethod -Uri "$ZapBaseUrl/JSON/core/view/version/" -TimeoutSec 5 -ErrorAction Stop
        if ($response.version) { return $true }
    } catch {}
    return $false
}

function Start-ZapIfNeeded {
    if (Test-ZapAvailability) { return $true }

    $zapPath = $env:ZAP_PATH
    if (-not $zapPath -or -not (Test-Path $zapPath)) {
        Write-Host "[i] ZAP not available (ZAP_PATH not set or invalid). Skipping ZAP phases." -ForegroundColor Yellow
        return $false
    }

    Write-Host "[*] Starting OWASP ZAP daemon on port $ZapPort..." -ForegroundColor Yellow
    Start-Process -FilePath $zapPath -ArgumentList '-daemon','-port',$ZapPort,'-config','api.disablekey=true' -WindowStyle Hidden

    $maxAttempts = 150  # 150 x 2s = 5 minutes
    $attempt = 0
    Write-Host "[*] Waiting for ZAP to initialize..." -ForegroundColor Cyan
    while ($attempt -lt $maxAttempts) {
        if (Test-ZapAvailability) {
            Write-Host "[+] ZAP daemon is ready! (took $($attempt * 2) seconds)" -ForegroundColor Green
            return $true
        }
        if ($attempt % 10 -eq 0 -and $attempt -gt 0) {
            Write-Host "  ... $($attempt * 2)s elapsed" -ForegroundColor Gray
        }
        Start-Sleep -Seconds 2
        $attempt++
    }

    Write-Failure "ZAP failed to start within 5 minutes"
    return $false
}

function Invoke-ZapApi {
    param([string]$Path, [int]$TimeoutSec = 30)
    try {
        Invoke-RestMethod -Uri "$ZapBaseUrl$Path" -TimeoutSec $TimeoutSec -ErrorAction Stop
    } catch {
        $null
    }
}

# ── Phase 1: Security Headers ──

function Test-SecurityHeaders {
    Write-Header "PHASE 1: SECURITY HEADERS ANALYSIS"

    try {
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Method Get -TimeoutSec 30
        $headers = $response.Headers

        # HSTS
        if ($TargetUrl -like "https://*") {
            if ($headers['Strict-Transport-Security']) {
                Write-Success "HSTS Header Present"
                $script:TestsPassed++
            } else {
                Write-Failure "HSTS Header Missing"
                $script:TestsFailed++
                $script:LowRiskFindings += "Missing HSTS Header"
            }
        }

        # X-Frame-Options
        if ($headers['X-Frame-Options']) {
            Write-Success "X-Frame-Options Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-Frame-Options Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-Frame-Options"
        }

        # X-Content-Type-Options
        if ($headers['X-Content-Type-Options']) {
            Write-Success "X-Content-Type-Options Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-Content-Type-Options Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-Content-Type-Options"
        }

        # X-XSS-Protection
        if ($headers['X-XSS-Protection']) {
            Write-Success "X-XSS-Protection Present"
            $script:TestsPassed++
        } else {
            Write-Failure "X-XSS-Protection Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing X-XSS-Protection"
        }

        # CSP
        if ($headers['Content-Security-Policy']) {
            Write-Success "Content-Security-Policy Present"
            $script:TestsPassed++
        } else {
            Write-Failure "Content-Security-Policy Missing"
            $script:TestsFailed++
            $script:MediumRiskFindings += "Missing CSP"
        }

        # Referrer-Policy
        if ($headers['Referrer-Policy']) {
            Write-Success "Referrer-Policy Present"
            $script:TestsPassed++
        } else {
            Write-Failure "Referrer-Policy Missing"
            $script:TestsFailed++
            $script:LowRiskFindings += "Missing Referrer-Policy"
        }

    } catch {
        Write-Failure "Failed to fetch headers: $($_.Exception.Message)"
    }
}

# ── Phase 2: CORS Tests ──

function Test-CORS {
    Write-Header "PHASE 2: CORS SECURITY TESTS"

    $evilOrigin = "https://evil-site.com"

    try {
        $hdrs = @{ 'Origin' = $evilOrigin }
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -Headers $hdrs -Method Get -TimeoutSec 30

        $corsHeader = $response.Headers['Access-Control-Allow-Origin']
        if ($corsHeader -eq '*') {
            Write-Failure "CORS Wildcard Detected"
            $script:TestsFailed++
            $script:HighRiskFindings += "CORS Wildcard Origin (*)"
        } elseif ($corsHeader -eq $evilOrigin) {
            Write-Failure "Evil Origin Accepted"
            $script:TestsFailed++
            $script:HighRiskFindings += "Evil Origin Accepted"
        } else {
            Write-Success "CORS Properly Configured"
            $script:TestsPassed++
        }
    } catch {
        Write-Success "CORS Restricted (good)"
        $script:TestsPassed++
    }
}

# ── Phase 3: XSS Tests ──

function Test-XSS {
    Write-Header "PHASE 3: XSS VULNERABILITY TESTS"

    $xssPayloads = @(
        '<script>alert(''XSS'')</script>',
        'javascript:alert(''XSS'')',
        "'`"><script>alert('XSS')</script>",
        '<img src=x onerror=alert(''XSS'')>'
    )
    $params = @('q', 'search', 'query', 'name', 'message', 'text')
    $xssFound = $false

    foreach ($param in $params) {
        foreach ($payload in $xssPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -ErrorAction SilentlyContinue

                if ($response.Content -like "*$payload*") {
                    Write-Failure "XSS Vulnerability in parameter: $param"
                    Write-ColorOutput "  Payload: $payload" -Color Yellow
                    $script:HighRiskFindings += "XSS in parameter '$param'"
                    $xssFound = $true
                    break
                }
            } catch { }
        }
        if ($xssFound) { break }
    }

    if (-not $xssFound) {
        Write-Success "XSS Protection OK"
        $script:TestsPassed++
    } else {
        $script:TestsFailed++
    }
}

# ── Phase 4: SQL Injection Tests ──

function Test-SQLInjection {
    Write-Header "PHASE 4: SQL INJECTION TESTS"

    $sqlPayloads = @(
        "' OR '1'='1",
        "'; DROP TABLE users; --",
        "' UNION SELECT null, null --",
        "admin'--"
    )
    $params = @('id', 'user', 'username', 'email', 'search')
    $sqliFound = $false

    foreach ($param in $params) {
        foreach ($payload in $sqlPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -ErrorAction SilentlyContinue

                $pattern = 'sql syntax|mysql.*error|postgresql.*error|ora-[0-9]+|sqlite.*error|syntax error.*sql'
                if ($response.Content -match $pattern) {
                    Write-Failure "SQL Injection Vulnerability in parameter: $param"
                    Write-ColorOutput "  Payload: $payload" -Color Yellow
                    $script:HighRiskFindings += "SQL Injection in parameter '$param'"
                    $sqliFound = $true
                    break
                }
            } catch { }
        }
        if ($sqliFound) { break }
    }

    if (-not $sqliFound) {
        Write-Success "SQL Injection Protection OK"
        $script:TestsPassed++
    } else {
        $script:TestsFailed++
    }
}

# ── Phase 5: Information Disclosure ──

function Test-InformationDisclosure {
    Write-Header "PHASE 5: INFORMATION DISCLOSURE TESTS"

    $sensitivePaths = @('/.git', '/.env', '/robots.txt', '/admin', '/phpmyadmin', '/config.php', '/web.config')

    foreach ($path in $sensitivePaths) {
        try {
            $testUrl = $TargetUrl + $path
            $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -ErrorAction Stop

            if ($response.StatusCode -eq 200) {
                Write-Failure "Accessible: $path"
                if ($path -in @('/.git', '/.env', '/config.php', '/web.config')) {
                    $script:HighRiskFindings += "$path accessible"
                } elseif ($path -in @('/admin', '/phpmyadmin')) {
                    $script:MediumRiskFindings += "$path accessible"
                } else {
                    $script:InfoFindings += "$path accessible"
                }
            }
        } catch {
            # Path not accessible - good
        }
    }

    Write-Success "Information Disclosure Check Complete"
    $script:TestsPassed++
}

# ── Phase 6: CSRF Protection ──

function Test-CSRFProtection {
    Write-Header "PHASE 6: CSRF PROTECTION TESTING"

    try {
        $response = Invoke-AuthWebRequest -Uri $TargetUrl -TimeoutSec 15
        $content = $response.Content

        # Check if forms exist
        if ($content -match '<form') {
            if ($content -match 'csrf|_token|authenticity_token|__RequestVerificationToken') {
                Write-Success "CSRF tokens detected in forms"
                $script:TestsPassed++
            } else {
                Write-Failure "No CSRF tokens found in forms"
                Write-Host "   RISK: Medium - Forms vulnerable to Cross-Site Request Forgery" -ForegroundColor Yellow
                Write-Host "   FIX: Add CSRF tokens to all forms and validate on server-side" -ForegroundColor Yellow
                $script:TestsFailed++
                $script:MediumRiskFindings += "Missing CSRF Protection in Forms"
            }
        } else {
            Write-Host "[i] No forms detected on main page" -ForegroundColor Blue
            $script:TestsPassed++
        }
    } catch {
        Write-Failure "Failed to fetch page for CSRF check: $($_.Exception.Message)"
    }

    # Check SameSite cookie attributes
    try {
        $headResponse = Invoke-AuthWebRequest -Uri $TargetUrl -Method Head -TimeoutSec 10
        $setCookieHeader = $headResponse.Headers['Set-Cookie']
        if ($setCookieHeader) {
            if ($setCookieHeader -match 'SameSite=(Strict|Lax)') {
                Write-Success "SameSite cookie attributes detected"
                $script:TestsPassed++
            } else {
                Write-Host "[!] SameSite cookie attributes missing" -ForegroundColor Yellow
                Write-Host "   RISK: Low - Cookies vulnerable to CSRF attacks" -ForegroundColor Yellow
                Write-Host "   FIX: Add SameSite=Strict or SameSite=Lax to all cookies" -ForegroundColor Yellow
                $script:TestsFailed++
                $script:LowRiskFindings += "Missing SameSite Cookie Attributes"
            }
        } else {
            Write-Host "[i] No cookies set by server" -ForegroundColor Blue
            $script:TestsPassed++
        }
    } catch {
        # HEAD may not be supported, not a failure
    }
}

# ── Phase 7: Path Traversal ──

function Test-PathTraversal {
    Write-Header "PHASE 7: PATH TRAVERSAL DETECTION"

    $traversalPayloads = @(
        "../../../etc/passwd",
        "..%5C..%5C..%5Cwindows%5Csystem32%5Cdrivers%5Cetc%5Chosts",
        "....//....//....//etc/passwd",
        "%2e%2e%2f%2e%2e%2f%2e%2e%2fetc%2fpasswd"
    )
    $traversalParams = @('file', 'path', 'page', 'include', 'template', 'doc', 'view')
    $vulnFound = $false

    foreach ($param in $traversalParams) {
        foreach ($payload in $traversalPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -ErrorAction SilentlyContinue

                if ($response.Content -match 'root:.*:/bin/(bash|sh)|daemon:.*:/sbin') {
                    Write-Failure "Path traversal vulnerability in parameter: $param"
                    Write-Host "   RISK: High - Unix passwd file content found" -ForegroundColor Red
                    Write-Host "   FIX: Validate and sanitize file path parameters" -ForegroundColor Yellow
                    $script:HighRiskFindings += "Path Traversal in parameter '$param'"
                    $script:TestsFailed++
                    $vulnFound = $true
                    break
                }
                if ($response.Content -match '127\.0\.0\.1.*localhost|# Copyright.*Microsoft') {
                    Write-Failure "Path traversal vulnerability in parameter: $param"
                    Write-Host "   RISK: High - Windows hosts file content found" -ForegroundColor Red
                    Write-Host "   FIX: Validate and sanitize file path parameters" -ForegroundColor Yellow
                    $script:HighRiskFindings += "Path Traversal in parameter '$param'"
                    $script:TestsFailed++
                    $vulnFound = $true
                    break
                }
            } catch { }
        }
        if ($vulnFound) { break }
    }

    if (-not $vulnFound) {
        Write-Success "Path Traversal Protection OK"
        $script:TestsPassed++
    }
}

# ── Phase 8: Open Redirect ──

function Test-OpenRedirect {
    Write-Header "PHASE 8: OPEN REDIRECT TESTING"

    $redirectPayloads = @(
        "https://evil.com",
        "http://malicious-site.com",
        "//evil.com",
        "https://google.com"
    )
    $redirectParams = @('url', 'redirect', 'return', 'next', 'goto', 'continue', 'target', 'destination')
    $vulnFound = $false

    foreach ($param in $redirectParams) {
        foreach ($payload in $redirectPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=${encodedPayload}"
                # Use -MaximumRedirection 0 to capture the redirect response
                $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 10 -ErrorAction SilentlyContinue -MaximumRedirection 0
            } catch {
                # PowerShell throws on 3xx when MaximumRedirection=0 — inspect the exception
                $response = $_.Exception.Response
            }

            if ($response) {
                $statusCode = [int]$response.StatusCode
                if ($statusCode -ge 300 -and $statusCode -lt 400) {
                    $location = $null
                    if ($response.Headers) {
                        try { $location = $response.Headers['Location'] } catch {}
                        if (-not $location) {
                            try { $location = $response.Headers.Location } catch {}
                        }
                    }
                    if ($location -and $location -like "*$payload*") {
                        Write-Failure "Open redirect vulnerability in parameter: $param"
                        Write-Host "   RISK: Medium - Unvalidated redirect to external domain" -ForegroundColor Yellow
                        Write-Host "   FIX: Validate redirect URLs against whitelist" -ForegroundColor Yellow
                        $script:MediumRiskFindings += "Open Redirect in parameter '$param'"
                        $script:TestsFailed++
                        $vulnFound = $true
                        break
                    }
                }
            }
        }
        if ($vulnFound) { break }
    }

    if (-not $vulnFound) {
        Write-Success "Open Redirect Protection OK"
        $script:TestsPassed++
    }
}

# ── Phase 9: SSL/TLS Configuration ──

function Test-SSLConfiguration {
    Write-Header "PHASE 9: SSL/TLS CONFIGURATION TESTING"

    if ($TargetUrl -notlike "https://*") {
        Write-Host "[i] Target uses HTTP, SSL/TLS tests not applicable" -ForegroundColor Blue
        Write-Host "[!] Consider upgrading to HTTPS for secure communication" -ForegroundColor Yellow
        return
    }

    $uri = [Uri]$TargetUrl
    $hostname = $uri.Host
    $port = if ($uri.Port -gt 0 -and $uri.Port -ne 443) { $uri.Port } else { 443 }

    try {
        $tcpClient = New-Object System.Net.Sockets.TcpClient
        $tcpClient.Connect($hostname, $port)
        $sslStream = New-Object System.Net.Security.SslStream($tcpClient.GetStream(), $false, { $true })

        try {
            $sslStream.AuthenticateAsClient($hostname)

            # Certificate validity
            $cert = $sslStream.RemoteCertificate
            if ($cert) {
                $cert2 = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2($cert)
                if ($cert2.NotAfter -gt (Get-Date) -and $cert2.NotBefore -lt (Get-Date)) {
                    Write-Success "SSL certificate is valid (expires $($cert2.NotAfter.ToString('yyyy-MM-dd')))"
                    $script:TestsPassed++
                } else {
                    Write-Failure "SSL certificate expired or not yet valid"
                    $script:TestsFailed++
                    $script:MediumRiskFindings += "SSL Certificate Expired/Invalid"
                }
            }

            # TLS version
            $protocol = $sslStream.SslProtocol
            if ($protocol -match 'Tls12|Tls13') {
                Write-Success "Modern TLS version supported ($protocol)"
                $script:TestsPassed++
            } else {
                Write-Failure "Weak TLS version detected: $protocol"
                Write-Host "   RISK: Medium - Weak TLS version vulnerable to attacks" -ForegroundColor Yellow
                Write-Host "   FIX: Disable TLS 1.0/1.1, enable only TLS 1.2/1.3" -ForegroundColor Yellow
                $script:TestsFailed++
                $script:MediumRiskFindings += "Weak TLS Version ($protocol)"
            }

            # Cipher suite
            $cipher = $sslStream.CipherAlgorithm.ToString()
            if ($cipher -match 'Rc4|Des|Null') {
                Write-Failure "Weak cipher suite detected: $cipher"
                $script:TestsFailed++
                $script:HighRiskFindings += "Weak Cipher Suite ($cipher)"
            } else {
                Write-Success "Strong cipher suite in use ($cipher)"
                $script:TestsPassed++
            }
        } finally {
            $sslStream.Dispose()
            $tcpClient.Dispose()
        }
    } catch {
        Write-Failure "SSL/TLS connection failed: $($_.Exception.Message)"
        $script:TestsFailed++
        $script:MediumRiskFindings += "SSL/TLS Connection Failed"
    }
}

# ── Phase 10: Command Injection ──

function Test-CommandInjection {
    Write-Header "PHASE 10: COMMAND INJECTION TESTING"

    $commandPayloads = @(
        "; whoami",
        "&& id",
        "| cat /etc/passwd",
        "; ping -c 1 127.0.0.1",
        "&& sleep 5"
    )
    $commandParams = @('cmd', 'exec', 'system', 'ping', 'host', 'ip', 'domain', 'command')
    $vulnFound = $false

    foreach ($param in $commandParams) {
        foreach ($payload in $commandPayloads) {
            try {
                $encodedPayload = [System.Web.HttpUtility]::UrlEncode($payload)
                $testUrl = "${TargetUrl}?${param}=test${encodedPayload}"

                $sw = [System.Diagnostics.Stopwatch]::StartNew()
                $response = Invoke-AuthWebRequest -Uri $testUrl -TimeoutSec 15 -ErrorAction SilentlyContinue
                $sw.Stop()

                # Check for command output in response
                if ($response.Content -match 'uid=|gid=|root:|daemon:|PING.*127\.0\.0\.1') {
                    Write-Failure "Command injection vulnerability in parameter: $param"
                    Write-Host "   RISK: High - Server command execution detected" -ForegroundColor Red
                    Write-Host "   FIX: Never pass user input directly to system commands" -ForegroundColor Yellow
                    $script:HighRiskFindings += "Command Injection in parameter '$param'"
                    $script:TestsFailed++
                    $vulnFound = $true
                    break
                }

                # Time-based detection for sleep payloads
                if ($payload -like '*sleep 5*' -and $sw.Elapsed.TotalSeconds -ge 5) {
                    Write-Failure "Time-based command injection in parameter: $param (delay: $([math]::Round($sw.Elapsed.TotalSeconds, 1))s)"
                    Write-Host "   RISK: High - Blind command execution via timing attack" -ForegroundColor Red
                    Write-Host "   FIX: Never pass user input directly to system commands" -ForegroundColor Yellow
                    $script:HighRiskFindings += "Time-based Command Injection in parameter '$param'"
                    $script:TestsFailed++
                    $vulnFound = $true
                    break
                }
            } catch { }
        }
        if ($vulnFound) { break }
    }

    if (-not $vulnFound) {
        Write-Success "Command Injection Protection OK"
        $script:TestsPassed++
    }
}

# ── Phase 11: ZAP Deep Scan ──

function Invoke-ZapDeepScan {
    Write-Header "PHASE 11: OWASP ZAP VULNERABILITY SCAN"

    if (-not (Start-ZapIfNeeded)) { return }

    Write-Host "[*] Running comprehensive ZAP vulnerability scan via REST API" -ForegroundColor Cyan
    Write-Host "[*] Features: Spider crawling, passive analysis, active scanning" -ForegroundColor Blue

    # Set up ZAP context
    Invoke-ZapApi "/JSON/context/action/newContext/?contextName=GenericTest" | Out-Null
    $encodedUrl = [System.Web.HttpUtility]::UrlEncode($TargetUrl)
    Invoke-ZapApi "/JSON/context/action/includeInContext/?contextName=GenericTest&regex=${encodedUrl}.*" | Out-Null

    # Spider configuration — deep penetration testing
    Write-Host "[*] Configuring ZAP for deep penetration testing..." -ForegroundColor Yellow
    Invoke-ZapApi "/JSON/spider/action/setOptionMaxDepth/?Integer=10" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionMaxChildren/?Integer=100" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionMaxDuration/?Integer=900" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionMaxParseSizeBytes/?Integer=52428800" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionRequestWaitTime/?Integer=200" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionProcessForm/?Boolean=true" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionPostForm/?Boolean=false" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionParseComments/?Boolean=true" | Out-Null
    Invoke-ZapApi "/JSON/spider/action/setOptionParseRobotsTxt/?Boolean=true" | Out-Null

    # Active scan configuration
    Invoke-ZapApi "/JSON/ascan/action/setOptionDelayInMs/?Integer=300" | Out-Null
    Invoke-ZapApi "/JSON/ascan/action/setOptionThreadPerHost/?Integer=5" | Out-Null
    Invoke-ZapApi "/JSON/ascan/action/setOptionMaxRuleDurationInMins/?Integer=10" | Out-Null
    Invoke-ZapApi "/JSON/ascan/action/setOptionMaxScanDurationInMins/?Integer=60" | Out-Null
    Invoke-ZapApi "/JSON/ascan/action/setOptionMaxResultsToList/?Integer=500" | Out-Null
    Invoke-ZapApi "/JSON/ascan/action/setOptionAttackStrength/?String=HIGH" | Out-Null
    Invoke-ZapApi "/JSON/ascan/action/setOptionAlertThreshold/?String=LOW" | Out-Null

    # Start spider
    Write-Host "[*] Deep spidering - discovering all pages, forms, and endpoints..." -ForegroundColor Blue
    $spiderResult = Invoke-ZapApi "/JSON/spider/action/scan/?url=$TargetUrl"
    if (-not $spiderResult -or -not $spiderResult.scan) {
        Write-Failure "Failed to start spider scan"
        return
    }
    $spiderId = $spiderResult.scan
    Write-Host "[*] Started spider scan (ID: $spiderId)" -ForegroundColor Blue

    # Wait for spider to complete
    $spiderElapsed = 0
    while ($true) {
        $status = Invoke-ZapApi "/JSON/spider/view/status/?scanId=$spiderId"
        if ($status.status -eq '100') { break }
        if ($spiderElapsed % 30 -eq 0 -and $spiderElapsed -gt 0) {
            $urlResult = Invoke-ZapApi "/JSON/spider/view/results/?scanId=$spiderId"
            $urlCount = if ($urlResult.results) { $urlResult.results.Count } else { 0 }
            Write-Host "`n[*] Spider progress: $($status.status)% - $urlCount URLs found (${spiderElapsed}s)" -ForegroundColor Blue
        }
        Write-Host "." -NoNewline
        Start-Sleep -Seconds 2
        $spiderElapsed += 2
    }

    $urlResult = Invoke-ZapApi "/JSON/spider/view/results/?scanId=$spiderId"
    $urlCount = if ($urlResult.results) { $urlResult.results.Count } else { 0 }
    Write-Host ""
    Write-Success "Deep spidering completed - $urlCount URLs discovered (${spiderElapsed}s)"

    # Export spider URLs for other scripts
    if ($env:ZAP_URLS_FILE) {
        Write-Host "[*] Exporting spider URLs to $($env:ZAP_URLS_FILE)" -ForegroundColor Cyan
        if ($urlResult.results) {
            $urlResult.results | Out-File -FilePath $env:ZAP_URLS_FILE -Encoding utf8
            Write-Success "Exported $urlCount URLs for SSRF/file-upload testing"
        }
    }

    # Passive scan (automatic)
    Write-Host "[*] Running passive security scan..." -ForegroundColor Cyan
    Start-Sleep -Seconds 5

    # Active scan (skip in quick mode)
    if ($Mode -eq 'Quick') {
        Write-Host "[*] Skipping active scan in quick mode" -ForegroundColor Blue
    } else {
        Write-Host "[*] Running active vulnerability scan..." -ForegroundColor Yellow
        Write-Host "[*] HIGH attack strength, LOW threshold - this may take 15+ minutes" -ForegroundColor Red

        $activeScanResult = Invoke-ZapApi "/JSON/ascan/action/scan/?url=$TargetUrl"
        if (-not $activeScanResult -or -not $activeScanResult.scan) {
            Write-Failure "Failed to start active scan"
        } else {
            $activeScanId = $activeScanResult.scan
            Write-Host "[*] Started active scan (ID: $activeScanId)" -ForegroundColor Blue

            $activeElapsed = 0
            $lastProgress = 0
            while ($true) {
                $scanStatus = Invoke-ZapApi "/JSON/ascan/view/status/?scanId=$activeScanId"
                if ($scanStatus.status -eq '100') { break }
                if (($activeElapsed - $lastProgress) -ge 60) {
                    $currentAlerts = Invoke-ZapApi "/JSON/core/view/alerts/"
                    $alertCount = if ($currentAlerts.alerts) { $currentAlerts.alerts.Count } else { 0 }
                    Write-Host "`n[*] Active scan: $($scanStatus.status)% - $alertCount issues (${activeElapsed}s)" -ForegroundColor Red
                    $lastProgress = $activeElapsed
                }
                Write-Host "." -NoNewline
                Start-Sleep -Seconds 5
                $activeElapsed += 5
            }

            $finalAlerts = Invoke-ZapApi "/JSON/core/view/alerts/"
            $finalCount = if ($finalAlerts.alerts) { $finalAlerts.alerts.Count } else { 0 }
            Write-Host ""
            Write-Success "Active scan completed - $finalCount security issues found (${activeElapsed}s)"
        }
    }

    # Parse ZAP findings
    Write-Host ""
    Write-Header "ZAP VULNERABILITY RESULTS"
    Write-Host ("=" * 40)

    $alerts = Invoke-ZapApi "/JSON/core/view/alerts/"
    if (-not $alerts -or -not $alerts.alerts) {
        Write-Host "[i] No ZAP alerts returned" -ForegroundColor Blue
        return
    }

    # Categorize findings by risk level (deduplicated)
    $highNames = $alerts.alerts | Where-Object { $_.risk -eq 'High' } | Select-Object -ExpandProperty name -Unique
    $mediumNames = $alerts.alerts | Where-Object { $_.risk -eq 'Medium' } | Select-Object -ExpandProperty name -Unique
    $lowNames = $alerts.alerts | Where-Object { $_.risk -eq 'Low' } | Select-Object -ExpandProperty name -Unique
    $infoNames = $alerts.alerts | Where-Object { $_.risk -eq 'Informational' } | Select-Object -ExpandProperty name -Unique

    # Add high risk findings
    foreach ($name in $highNames) {
        $script:HighRiskFindings += $name
    }

    # Consolidate medium risk findings (CSP + cookie grouping)
    $cspAdded = $false; $cookieAdded = $false
    foreach ($name in $mediumNames) {
        if ($name -match 'Content Security Policy|CSP:') {
            if (-not $cspAdded) { $script:MediumRiskFindings += "CSP Configuration Issues"; $cspAdded = $true }
        } elseif ($name -match 'cookie') {
            if (-not $cookieAdded) { $script:MediumRiskFindings += "Cookie Security Issues"; $cookieAdded = $true }
        } else {
            $script:MediumRiskFindings += $name
        }
    }

    # Consolidate low risk findings (header grouping)
    $headerAdded = $false
    foreach ($name in $lowNames) {
        if ($name -match 'header|x-content-type|x-frame|x-xss') {
            if (-not $headerAdded) { $script:LowRiskFindings += "Security Header Issues"; $headerAdded = $true }
        } else {
            $script:LowRiskFindings += $name
        }
    }

    # Filter noisy informational findings
    $noisePatterns = @('User Agent Fuzzer', 'Modern Web Application', 'Retrieved from Cache', 'Storable and Cacheable Content', 'Non-Storable Content', 'Information Disclosure - Suspicious Comments', 'Timestamp Disclosure', 'Re-examine Cache-control Directives')
    foreach ($name in $infoNames) {
        $isNoise = $false
        foreach ($p in $noisePatterns) {
            if ($name -like "$p*") { $isNoise = $true; break }
        }
        if (-not $isNoise) { $script:InfoFindings += $name }
    }

    # Display ZAP risk summary
    $hc = ($highNames | Measure-Object).Count
    $mc = ($mediumNames | Measure-Object).Count
    $lc = ($lowNames | Measure-Object).Count
    $ic = ($infoNames | Measure-Object).Count

    if ($hc -gt 0) { Write-Host "HIGH RISK: $hc" -ForegroundColor Red }
    else { Write-Host "HIGH RISK: 0 (Excellent)" -ForegroundColor Green }

    if ($mc -gt 0) { Write-Host "MEDIUM RISK: $mc" -ForegroundColor Yellow }
    else { Write-Host "MEDIUM RISK: 0 (Good)" -ForegroundColor Green }

    if ($lc -gt 0) { Write-Host "LOW RISK: $lc" -ForegroundColor Blue }
    else { Write-Host "LOW RISK: 0 (Good)" -ForegroundColor Green }

    Write-Host "INFORMATIONAL: $ic" -ForegroundColor Gray

    # Show top vulnerability types
    if ($hc -gt 0 -or $mc -gt 0) {
        Write-Host ""
        Write-Host "TOP VULNERABILITY CLASSES:" -ForegroundColor Red
        $alertGroups = $alerts.alerts | Group-Object -Property name | Sort-Object -Property Count -Descending | Select-Object -First 10
        foreach ($group in $alertGroups) {
            Write-Host "  * $($group.Name) ($($group.Count) instances)" -ForegroundColor Yellow
        }
    }

    $totalIssues = $hc + $mc + $lc + $ic
    Write-Host ""
    Write-Success "Deep penetration testing completed - $totalIssues total issues"
}

# ── Summary ──

function Show-Summary {
    Write-Host ""
    Write-Header "SECURITY ASSESSMENT SUMMARY"
    Write-Host ("=" * 50)

    $totalFindings = $script:HighRiskFindings.Count + $script:MediumRiskFindings.Count + $script:LowRiskFindings.Count

    Write-ColorOutput "Target: $TargetUrl" -Color Blue
    Write-ColorOutput "Scan Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -Color Blue
    Write-ColorOutput "Tests Passed: $script:TestsPassed" -Color Green
    Write-ColorOutput "Tests Failed: $script:TestsFailed" -Color Red

    Write-Host ""

    # Risk Level
    $riskLevel = "LOW"
    $riskColor = "Green"
    if ($script:HighRiskFindings.Count -gt 0) {
        $riskLevel = "HIGH"
        $riskColor = "Red"
    } elseif ($script:MediumRiskFindings.Count -gt 3) {
        $riskLevel = "MEDIUM"
        $riskColor = "Yellow"
    }

    Write-Host "Overall Risk Level: " -NoNewline
    Write-Host $riskLevel -ForegroundColor $riskColor

    if ($totalFindings -gt 0) {
        Write-Host ""; Write-ColorOutput "VULNERABILITY FINDINGS:" -Color Yellow

        if ($script:HighRiskFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "HIGH RISK ($($script:HighRiskFindings.Count)):" -Color Red
            foreach ($finding in $script:HighRiskFindings) {
                Write-ColorOutput "  * $finding" -Color Red
            }
        }

        if ($script:MediumRiskFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "MEDIUM RISK ($($script:MediumRiskFindings.Count)):" -Color Yellow
            foreach ($finding in $script:MediumRiskFindings) {
                Write-ColorOutput "  * $finding" -Color Yellow
            }
        }

        if ($script:LowRiskFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "LOW RISK ($($script:LowRiskFindings.Count)):" -Color Gray
            foreach ($finding in $script:LowRiskFindings) {
                Write-ColorOutput "  * $finding" -Color Gray
            }
        }

        if ($script:InfoFindings.Count -gt 0) {
            Write-Host ""; Write-ColorOutput "INFORMATIONAL ($($script:InfoFindings.Count)):" -Color Cyan
            foreach ($finding in $script:InfoFindings) {
                Write-ColorOutput "  * $finding" -Color Cyan
            }
        }
    } else {
        Write-Host ""; Write-Success "No security findings detected!"
    }

    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "[!]  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    Write-Host ""; Write-Host ("=" * 50)
}

# ── Main execution ──

try {
    Add-Type -AssemblyName System.Web -ErrorAction SilentlyContinue

    Write-Host ""
    Write-ColorOutput "GENERIC WEB SECURITY SCANNER (PowerShell)" -Color Magenta
    Write-Host ("=" * 50)
    Write-ColorOutput "Target: $TargetUrl" -Color Cyan
    Write-ColorOutput "Mode: $Mode" -Color Cyan
    if ($AuthToken) { Write-Host "[*] Bearer token authentication enabled" -ForegroundColor Green }
    if ($AuthCookie -and -not $AuthToken) { Write-Host "[*] Cookie authentication enabled" -ForegroundColor Green }
    if (-not $AuthToken -and -not $AuthCookie) { Write-Host "[i] Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

    Write-Host ("=" * 50)

    # Run tests based on mode
    switch ($Mode) {
        'ZapOnly' {
            Invoke-ZapDeepScan
        }
        'Quick' {
            Test-SecurityHeaders
            Test-CORS
            Test-CSRFProtection
            Test-PathTraversal
            Test-SQLInjection
            Test-InformationDisclosure
            Test-SSLConfiguration
        }
        default {
            # Full mode: all 11 phases
            Test-SecurityHeaders
            Test-CORS
            Test-XSS
            Test-SQLInjection
            Test-InformationDisclosure
            Test-CSRFProtection
            Test-PathTraversal
            Test-OpenRedirect
            Test-SSLConfiguration
            Test-CommandInjection
            Invoke-ZapDeepScan
        }
    }

    # Show summary
    Show-Summary

    Write-Host ""
    Write-Success "Security scan completed!"
    Write-Host ""

} catch {
    Write-Failure "Script error: $($_.Exception.Message)"
    exit 1
}
