# Generic Web Application Security Testing Suite

A comprehensive bash-based security testing toolkit that works with any website - no specific API endpoints or authentication required. Features complete security testing including OWASP ZAP integration.

## 🚀 Features

### Core Security Tests (Always Available)

- **Security Headers Analysis**: HSTS, CSP, X-Frame-Options, etc.
- **CORS Security Testing**: Wildcard origins, evil origin rejection
- **XSS Vulnerability Detection**: Reflected XSS testing with multiple payloads
- **SQL Injection Testing**: Multiple injection techniques and error detection
- **Information Disclosure**: Sensitive file/directory detection
- **Requires**: Only `curl` (usually pre-installed)

### ZAP Vulnerability Scanning (Full Professional Suite)

- **Everything from Core Tests** + **Advanced vulnerability scanning**
- **Spider Crawling**: Complete content discovery and mapping
- **Passive Analysis**: Security issue detection during crawling
- **Active Scanning**: Real vulnerability testing (XSS, SQLi, etc.)
- **Detailed Terminal Results**: Rich terminal output with detailed vulnerability explanations
- **Risk Assessment**: Automated scoring and categorization
- **Requires**: Only OWASP ZAP

## 📦 Installation

### Prerequisites

**Core Security Tests (Always Available):**

- ✅ Works immediately - no installation needed
- ✅ Requires: Only `curl` (usually pre-installed)

**ZAP Vulnerability Scanning (Full Suite):**

- ✅ Requires: Only OWASP ZAP (bash + curl only)

### Quick Installation

**macOS:**

```bash
brew install --cask owasp-zap
```

**Linux (Ubuntu/Debian):**

```bash
sudo apt update && sudo apt install zaproxy
```

**Linux (RHEL/CentOS/Fedora):**

```bash
sudo dnf install owasp-zap
# or: sudo yum install owasp-zap
```

**Windows:**

- Download from: https://www.zaproxy.org/download/
- Run the installer

**That's it!** No Python, no complex dependencies.

### Make Script Executable

```bash
chmod +x generic-security-test.sh
```

## 🔧 Usage

### Quick Start

```bash
# Install ZAP (one-time setup)
brew install --cask owasp-zap  # macOS

# Make script executable
chmod +x generic-security-test.sh

# Run security scans
./generic-security-test.sh https://example.com          # Full scan (5-45 min)
./generic-security-test.sh https://example.com quick    # Quick scan (1-2 min)
./generic-security-test.sh https://example.com zap-only # ZAP-only scan

# ZAP comprehensive scan only - 15-30 min
./generic-security-test.sh https://example.com zap-only
```

## 🎯 Mode Comparison

| **Mode**           | **Tests Included**                           | **Time**  | **Use Case**                |
| ------------------ | -------------------------------------------- | --------- | --------------------------- |
| **Quick**          | 🛡️ Headers + 🌐 CORS + 📋 Info disclosure    | 1-2 min   | CI/CD, first check          |
| **Full (Default)** | Everything above + ⚡ XSS + 💉 SQLi + 🕷️ ZAP | 5-45 min  | Complete security audit     |
| **ZAP-Only**       | 🕷️ ZAP comprehensive scan only               | 15-30 min | Deep vulnerability analysis |

## 🔍 Detailed Test Coverage

### 🛡️ Core Security Tests (Always Available)

**Phase 1: Security Headers Analysis**

- ✅ **HSTS** - Prevents protocol downgrade attacks, ensures HTTPS-only access
- ✅ **X-Frame-Options** - Prevents clickjacking by blocking iframe embedding
- ✅ **X-Content-Type-Options** - Stops MIME-type sniffing vulnerabilities
- ✅ **X-XSS-Protection** - Legacy XSS filtering (browser-based protection)
- ✅ **Content-Security-Policy** - Controls resource loading, prevents XSS/code injection
- ✅ **Referrer-Policy** - Controls referrer information leakage across origins

**Phase 2: CORS Security Testing**

- ✅ **Wildcard Origin Protection** - Detects dangerous `Access-Control-Allow-Origin: *`
- ✅ **Evil Origin Rejection** - Tests if malicious domains are accepted
- ✅ **Endpoint-Specific Testing** - Checks `/api`, `/admin`, `/login`, `/auth` paths
- ✅ **Credentials + Wildcard Detection** - Finds dangerous CORS + credentials combinations

**Phase 3: XSS Vulnerability Detection**

- ✅ **Reflected XSS** - Tests 8 common parameters (`q`, `search`, `query`, `name`, etc.)
- ✅ **Multiple Payloads** - `<script>`, `javascript:`, `<img>`, `<svg>` attack vectors
- ✅ **Context-Aware Testing** - Detects unescaped user input in HTML responses
- ✅ **GET Parameter Injection** - URL parameter-based XSS detection

**Phase 4: SQL Injection Testing**

- ✅ **Error-Based Detection** - Tests for database error messages (MySQL, PostgreSQL, Oracle, MSSQL)
- ✅ **Multiple Injection Types** - Union-based, boolean-based, comment-based attacks
- ✅ **Common Parameters** - Tests `id`, `user`, `email`, `search`, `category` parameters
- ✅ **Database Fingerprinting** - Identifies database type from error responses

**Phase 5: Information Disclosure**

- ✅ **Sensitive Files** - Tests for `.git`, `.env`, `/admin`, `/phpmyadmin` exposure
- ✅ **Content Validation** - Verifies actual sensitive content (not just 200 status)
- ✅ **Admin Interface Detection** - Identifies accessible admin panels
- ✅ **Configuration File Leakage** - Detects environment variables and secrets

### 🕷️ ZAP Professional Vulnerability Scanning (50+ Rules)

**Active Scan Rules (40+ Vulnerability Tests):**

- 🚨 **Injection Attacks**: SQL injection (multiple databases), XSS (reflected/stored), Command injection, Code injection, XPath injection
- 🚨 **Authentication Bypasses**: Path traversal, Directory browsing, Parameter tampering, External redirects
- 🚨 **Critical CVEs**: Log4Shell (CVE-2021-44228), Spring4Shell (CVE-2022-22965), Heartbleed OpenSSL
- 🚨 **Framework Vulnerabilities**: PHP-CGI RCE, Server-side template injection, Buffer overflows
- 🚨 **Information Disclosure**: Source code disclosure, Hidden file discovery, Environment file leaks (.env, .htaccess)
- 🚨 **Advanced Attacks**: XXE (XML External Entity), CRLF injection, XSLT injection, Format string errors
- 🚨 **Cloud Security**: Cloud metadata attacks (AWS, GCP, Azure), ELMAH information leaks
- 🚨 **Database-Specific**: Time-based SQL injection for MySQL, PostgreSQL, Oracle, MSSQL, SQLite, Hypersonic

**Passive Scan Rules (20+ Security Analysis):**

- 🔍 **Security Headers** - Missing security headers analysis during crawling
- 🔍 **Cookie Security** - Insecure cookie attributes (HttpOnly, Secure, SameSite)
- 🔍 **SSL/TLS Issues** - Certificate problems, weak ciphers, protocol issues
- 🔍 **Information Leakage** - Debug information, error messages, version disclosure
- 🔍 **Content Analysis** - Dangerous content types, insecure links, mixed content

**Spider + Discovery:**

- 🕸️ **Complete Site Mapping** - Discovers all pages, forms, parameters, endpoints
- 🕸️ **Content Discovery** - Finds hidden directories, backup files, admin panels
- 🕸️ **Parameter Discovery** - Identifies all input points for vulnerability testing
- 🕸️ **Technology Detection** - Fingerprints frameworks, servers, databases, languages

## 📄 Output & Reports

All results displayed in **terminal only** - no files generated, clean immediate feedback with detailed explanations.

### 🛡️ Core Security Tests Output

**Real-time colored terminal results** showing exactly what needs to be fixed:

```bash
🛡️ PHASE 1: SECURITY HEADERS ANALYSIS
Testing: HSTS Header Present... ✅ PASS
Testing: X-Frame-Options Header... ❌ FAIL
🔒 MISSING: X-Frame-Options Header
   RISK: Medium - Vulnerable to clickjacking attacks
   WHAT: Prevents page from being embedded in frames/iframes
   FIX: Add header: X-Frame-Options: DENY or X-Frame-Options: SAMEORIGIN
   EXAMPLE: In nginx: add_header X-Frame-Options "SAMEORIGIN" always;

🌐 PHASE 2: CORS SECURITY TESTS
✅ PASS: CORS Wildcard Protection
✅ PASS: Evil Origin Rejection

⚡ PHASE 3: XSS VULNERABILITY TESTS
Testing parameter 'search' for XSS... ✅ PROTECTED
Testing parameter 'query' for XSS... ❌ VULNERABLE
🚨 FOUND: Reflected XSS Vulnerability in parameter 'query'
   RISK: High - Remote code execution in user browsers
   PAYLOAD: <script>alert('XSS')</script>
   FIX: Sanitize ALL user input, use output encoding, implement CSP
```

### 🕷️ ZAP Professional Scan Output

**Comprehensive vulnerability analysis** with industry-standard categorization:

```bash
🕷️ PHASE 6: OWASP ZAP VULNERABILITY SCAN
🕸️ Spidering target for content discovery... ✅ Completed (45 pages found)
🔍 Running passive security scan... ✅ Completed
⚡ Running active vulnerability scan... ✅ Completed

📊 ZAP FINDINGS SUMMARY:
🔴 HIGH RISK (2): SQL Injection, Stored XSS
🟡 MEDIUM RISK (5): Missing Security Headers, CSRF Vulnerability
🟢 LOW RISK (12): Information Disclosure, Weak SSL Ciphers
```

### 📊 Final Assessment Report

**Executive summary** with actionable risk scoring:

```bash
📊 SECURITY ASSESSMENT SUMMARY
================================
Target: https://example.com
Test Success Rate: 73% (8/11)
Overall Risk Level: MEDIUM
Risk Score: 45/100 (lower is better)

🔍 DETAILED VULNERABILITY FINDINGS:
🔴 HIGH RISK FINDINGS (2):
   • SQL Injection in parameter 'id'
   • Stored XSS in contact form

🟡 MEDIUM RISK FINDINGS (5):
   • Missing HSTS Header
   • Missing Content-Security-Policy Header
   • CSRF Token Missing
```

**Each finding includes:**

- 🎯 **RISK LEVEL**: High/Medium/Low with clear explanations
- 🔍 **TECHNICAL DETAILS**: What the vulnerability is and why it matters
- 🛠️ **EXACT FIXES**: Copy-paste solutions for your web server/framework
- 📋 **EXAMPLES**: Real configuration snippets you can use immediately

**Exit codes indicate overall security status:**

- `0`: Low risk - Good security posture
- `1`: Medium risk - Some issues to address
- `2`: High risk - Critical vulnerabilities found

## 🎯 Risk Assessment

The tool provides automatic risk scoring:

- **🔴 HIGH RISK**: Immediate attention required (XSS, SQLi, etc.)
- **🟡 MEDIUM RISK**: Should be addressed (Missing headers, etc.)
- **🟢 LOW RISK**: Best practice improvements

Exit codes indicate overall risk level:

- `0`: Low risk
- `1`: Medium risk
- `2`: High risk

The assessment concludes with a detailed breakdown of all findings, prioritized by risk level, so you know exactly what to fix first.

## 🛠️ Configuration

### Environment Variables

```bash
export ZAP_PORT=8090              # ZAP daemon port
export ZAP_HOST=127.0.0.1         # ZAP daemon host
export TIMEOUT=300                # Scan timeout in seconds
```

**Note:** ZAP automatically starts in daemon mode when needed - no manual setup required!

## 🚨 Security Considerations

### ⚠️ **CRITICAL WARNING: Deep Penetration Testing Dangers**

**🏠 RECOMMENDED: Local Testing Only**

```bash
# SAFE: Test your development environment
npm run dev
./generic-security-test.sh http://localhost:3000 full
```

**⚡ DANGERS of Running on Live Websites:**

**🚨 Server Impact:**

- **DoS/Crash Risk** - 5 concurrent threads + unlimited requests can overwhelm servers
- **Database Locks** - SQL injection tests can freeze tables and crash databases
- **Resource Exhaustion** - Intensive testing can consume all CPU/memory

**💥 Data Damage:**

- **Test Data Pollution** - ZAP injects thousands of test payloads into your database
- **Unwanted File Uploads** - May upload test files to your server
- **Form Spam** - Creates fake accounts, orders, and contact form submissions

**⚖️ Legal Consequences:**

- **Computer Fraud & Abuse Act** violations (US federal crime)
- **Unauthorized Access** charges in most countries
- **Civil Lawsuits** from affected companies
- **Criminal Prosecution** - "just testing" is not a legal defense

**🎯 What Deep Penetration Mode Actually Does:**

```bash
# Example of aggressive testing ZAP performs:
POST /api/users {"name": "<script>alert('XSS')</script>"}
POST /login {"user": "admin'; DROP TABLE users; --"}
GET /admin/../../../../etc/passwd
GET /api/delete/1 OR 1=1--
# × Thousands of similar requests
```

### 🛡️ Safe Usage Guidelines

1. **🏠 Local development only** - Perfect for localhost testing
2. **📝 Written authorization required** - Never test sites you don't own
3. **🔒 Own websites only** - Even then, use staging/development environments
4. **⚠️ Live site testing** - Possible but carries all above risks
5. **🌐 Network isolation** - Consider running from isolated environments

## 🔧 Troubleshooting

### Common Issues

**🛡️ Bot Protection Causing False Failures**

**Problem:** Your security headers are perfect, but the scanner shows failures.

```bash
# Example: Scanner shows all headers missing
❌ FAIL: HSTS Header Present
❌ FAIL: X-Frame-Options Header
# But headers are actually configured correctly!
```

**Root Cause:** Hosting platforms (Vercel, Cloudflare, etc.) block automated requests.

**How to Detect:**

```bash
# Manual test shows bot protection
curl -sI https://yoursite.com
# Returns: HTTP/2 403 or challenge page instead of real headers
```

**Solutions:**

1. **Test the correct URL** (check for redirects):

   ```bash
   # ❌ Wrong: Tests redirect response
   ./generic-security-test.sh https://example.com

   # ✅ Correct: Tests actual site
   ./generic-security-test.sh https://www.example.com
   ```

2. **Temporarily disable bot protection** (in your hosting dashboard)

3. **Test locally** (bypasses hosting protection):

   ```bash
   npm run dev  # or your local server
   ./generic-security-test.sh http://localhost:3000 quick
   ```

4. **Use browser-based tools** (not blocked):
   - Visit https://securityheaders.com
   - Test your site (gets real browser treatment)

**Platform Examples:**

- ✅ **GitHub**: Permits security scanners (built for developers)
- ⚠️ **Vercel/Netlify**: May block automated requests (prioritizes performance)
- ⚠️ **Cloudflare**: Often blocks bots (strong DDoS protection)

**🔒 HSTS Missing on HTTP (Expected Behavior)**

**Problem:** Scanner reports "Missing HSTS Header" on localhost.

```bash
# Testing HTTP localhost
./generic-security-test.sh http://localhost:3000 quick
❌ FAIL: HSTS Header Present  # This is actually CORRECT!
```

**Explanation:** HSTS (HTTP Strict Transport Security) is **only for HTTPS sites**.

- ✅ **HTTPS sites**: Should have HSTS header
- ✅ **HTTP sites**: Should NOT have HSTS header (meaningless on HTTP)
- ✅ **Your localhost HTTP**: Missing HSTS is the **correct behavior**

**Solution:** This is not a real failure - your configuration is working properly!

**📊 Interpreting Results: Real vs False Failures**

| Test Environment     | Expected Score           | What Missing Headers Mean          |
| -------------------- | ------------------------ | ---------------------------------- |
| **Production HTTPS** | 100% (6/6 headers)       | Real security issues               |
| **Localhost HTTP**   | 88% (5/6, no HSTS)       | Perfect (HSTS not needed on HTTP)  |
| **Bot-blocked site** | 0-33% (all/most missing) | False failure (site is protected)  |
| **Redirect URL**     | Variable                 | Test the final destination instead |

**🎯 Quick Check:**

```bash
# If you see this pattern, you're likely hitting bot protection:
HTTP/2 403
x-vercel-mitigated: challenge
# OR
HTTP/2 503
cf-mitigated: challenge  # Cloudflare
```

---

**Permission denied:**

```bash
chmod +x generic-security-test.sh
```

## ❓ Frequently Asked Questions

### Q: How do I install the requirements?

**A: Just install OWASP ZAP:**

```bash
# macOS
brew install --cask owasp-zap

# Linux
sudo apt install zaproxy

# Then run scans
./generic-security-test.sh https://example.com
```

### Q: Do I need to install anything?

**A: Only if you want ZAP vulnerability scanning:**

| **What You Want**          | **Installation Required?** | **What to Install**    |
| -------------------------- | -------------------------- | ---------------------- |
| **Core Security Tests**    | ❌ **Nothing needed**      | Works with just `curl` |
| **ZAP Vulnerability Scan** | ✅ **Yes**                 | Only OWASP ZAP         |

### Q: What does `-daemon` mean?

**A: It starts ZAP in headless mode** (no GUI) so it can run in the background and be controlled via API calls. Perfect for automated scanning.

### Q: What's the difference between Core Tests and ZAP Scanning?

**A: Two complementary approaches:**

| **Core Security Tests**                      | **ZAP Vulnerability Scanning**              |
| -------------------------------------------- | ------------------------------------------- |
| ✅ Works immediately (just needs `curl`)     | ⚠️ Requires only OWASP ZAP (no Python!)     |
| ✅ Headers, CORS, XSS, SQLi, info disclosure | ✅ Everything from Core + deep ZAP analysis |
| ✅ Professional risk scoring                 | ✅ Spider crawling + content discovery      |
| ✅ Detailed terminal output with fixes       | ✅ Rich terminal output with ZAP findings   |
| ⚡ **Fast** (1-2 min)                        | 🔍 **Comprehensive** (15-30 min)            |

### Q: How do I control which tests to run?

**A: Use the mode parameter:**

```bash
# QUICK: Core security tests only (fast)
./generic-security-test.sh https://example.com quick
# ✅ 1-2 minutes, headers + CORS + info disclosure

# FULL: Core tests + ZAP vulnerability scanning (comprehensive)
./generic-security-test.sh https://example.com
# ✅ 15-30 minutes, everything + ZAP analysis

# ZAP-ONLY: Just ZAP vulnerability scanning
./generic-security-test.sh https://example.com zap-only
# ✅ ZAP spider + passive + active scanning
```

### Q: Why are my security headers failing when I know they're configured?

**A: This is usually bot protection blocking the scanner.**

**Common signs:**

- All or most headers show as "missing"
- Site works fine in browsers
- `curl -sI yoursite.com` returns 403 or challenge page

**Solutions:**

1. Test locally: `./generic-security-test.sh http://localhost:3000 quick`
2. Use browser tools: https://securityheaders.com
3. Check the [Bot Protection troubleshooting section](#common-issues)

Your headers are likely working perfectly - the scanner just can't see them!

## 📈 Performance Notes

- **Basic scan**: ~2-5 minutes
- **Advanced ZAP scan**: ~10-30 minutes depending on site complexity
- **Resource usage**: ZAP can use significant CPU/memory for large sites

## 📜 License

MIT License - Use responsibly and only on systems you own or have explicit permission to test.

## 📝 Quick Summary

**🎯 Goal:** Test any website's security without needing authentication or specific knowledge.

**🚀 Getting Started (Zero Dependencies):**

1. Download scripts
2. `./generic-security-test.sh https://example.com quick`
3. ✅ **Done!** Professional security assessment in 1-2 minutes

**🚀 Simple Setup (Professional Suite):**

1. **Install ZAP**: `brew install --cask owasp-zap` (macOS) or `apt install zaproxy` (Linux)
2. **Run scan**: `./generic-security-test.sh https://example.com`
3. ✅ **Done!** Comprehensive assessment in 15-30 minutes

**📊 Three Test Modes:**

- ✅ **Quick Mode**: Headers, CORS, info disclosure (needs only `curl`)
- ✅ **Full Mode**: Everything above + XSS + SQL injection + ZAP vulnerability scanning
- ✅ **ZAP-Only**: Just the professional ZAP scan with 50+ rules + terminal results (needs only ZAP!)

**⚡ Performance:**

- **Basic**: 1-2 minutes, no setup required
- **Advanced**: 15-30 minutes, full professional suite

**Platform Support:**

- **macOS/Linux**: Simple package manager installation (`brew`, `apt`, `dnf`)
- **Windows**: Manual download and install from official website

---

**Legal Disclaimer:** This tool is for authorized security testing only. Users are responsible for ensuring they have proper authorization before testing any systems.

---

# 汎用 Web アプリケーションセキュリティテストスイート

特定の API エンドポイントや認証を必要としない、あらゆるウェブサイトで動作する包括的な bash ベースのセキュリティテストツールキット。Python の依存関係なしで OWASP ZAP 統合を含む完全なセキュリティテストを提供

## 🚀 機能

### コアセキュリティテスト（常に利用可能）

- **セキュリティヘッダー分析**: HSTS、CSP、X-Frame-Options など
- **CORS セキュリティテスト**: ワイルドカードオリジン、悪意のあるオリジン拒否
- **XSS 脆弱性検出**: 複数のペイロードによる反射型 XSS テスト
- **SQL インジェクションテスト**: 複数のインジェクション技術とエラー検出
- **情報漏洩**: 機密ファイル/ディレクトリの検出
- **必要なもの**: `curl`のみ（通常はプリインストール済み）

### ZAP 脆弱性スキャン（完全プロフェッショナルスイート）

- **コアテストのすべて** + **高度な脆弱性スキャン**
- **スパイダークローリング**: 完全なコンテンツ発見とマッピング
- **パッシブ分析**: クローリング中のセキュリティ問題検出
- **アクティブスキャン**: 実際の脆弱性テスト（XSS、SQLi など）
- **詳細なターミナル結果**: 詳細な脆弱性説明付きの豊富なターミナル出力
- **リスク評価**: 自動スコアリングと分類
- **必要なもの**: OWASP ZAP のみ

## 📦 インストール

### 前提条件

**コアセキュリティテスト（常に利用可能）:**

- ✅ すぐに動作 - インストール不要
- ✅ 必要なもの: `curl`のみ（通常はプリインストール済み）

**ZAP 脆弱性スキャン（完全スイート）:**

- ✅ 必要なもの: OWASP ZAP のみ（bash + curl のみ、Python なし！）

### クイックインストール

**macOS:**

```bash
brew install --cask owasp-zap
```

**Linux (Ubuntu/Debian):**

```bash
sudo apt update && sudo apt install zaproxy
```

**Linux (RHEL/CentOS/Fedora):**

```bash
sudo dnf install owasp-zap
# または: sudo yum install owasp-zap
```

**Windows:**

- ダウンロード: https://www.zaproxy.org/download/
- インストーラーを実行

**以上です！** Python 不要、複雑な依存関係なし。

### スクリプトを実行可能にする

```bash
chmod +x generic-security-test.sh
```

## 🔧 使用方法

### クイックスタート

```bash
# ZAPをインストール（初回のみ）
brew install --cask owasp-zap  # macOS

# スクリプトを実行可能にする
chmod +x generic-security-test.sh

# セキュリティスキャンを実行
./generic-security-test.sh https://example.com          # フルスキャン（5-45分）
./generic-security-test.sh https://example.com quick    # クイックスキャン（1-2分）
./generic-security-test.sh https://example.com zap-only # ZAPのみスキャン

# ZAP包括スキャンのみ - 15-30分
./generic-security-test.sh https://example.com zap-only
```

## 🎯 モード比較

| **モード**             | **含まれるテスト**                     | **時間** | **用途**             |
| ---------------------- | -------------------------------------- | -------- | -------------------- |
| **Quick**              | 🛡️ ヘッダー + 🌐 CORS + 📋 情報漏洩    | 1-2 分   | CI/CD、初回チェック  |
| **Full（デフォルト）** | 上記すべて + ⚡ XSS + 💉 SQLi + 🕷️ ZAP | 5-45 分  | 完全セキュリティ監査 |
| **ZAP-Only**           | 🕷️ ZAP 包括スキャンのみ                | 15-30 分 | 深い脆弱性分析       |

## 🔍 詳細テストカバレッジ

### 🛡️ コアセキュリティテスト（常に利用可能）

**フェーズ 1: セキュリティヘッダー分析**

- ✅ **HSTS** - プロトコルダウングレード攻撃を防止、HTTPS のみアクセスを確保
- ✅ **X-Frame-Options** - iframe 埋め込みをブロックしてクリックジャッキングを防止
- ✅ **X-Content-Type-Options** - MIME タイプスニッフィング脆弱性を停止
- ✅ **X-XSS-Protection** - レガシー XSS フィルタリング（ブラウザベース保護）
- ✅ **Content-Security-Policy** - リソース読み込みを制御、XSS/コードインジェクションを防止
- ✅ **Referrer-Policy** - オリジン間のリファラー情報漏洩を制御

**フェーズ 2: CORS セキュリティテスト**

- ✅ **ワイルドカードオリジン保護** - 危険な `Access-Control-Allow-Origin: *` を検出
- ✅ **悪意のあるオリジン拒否** - 悪意のあるドメインが受け入れられるかをテスト
- ✅ **エンドポイント固有テスト** - `/api`、`/admin`、`/login`、`/auth` パスを確認
- ✅ **認証情報 + ワイルドカード検出** - 危険な CORS + 認証情報の組み合わせを発見

**フェーズ 3: XSS 脆弱性検出**

- ✅ **反射型 XSS** - 8 つの一般的なパラメータ（`q`、`search`、`query`、`name` など）をテスト
- ✅ **複数ペイロード** - `<script>`、`javascript:`、`<img>`、`<svg>` 攻撃ベクターを使用
- ✅ **コンテキスト対応テスト** - HTML レスポンス内のエスケープされていないユーザー入力を検出
- ✅ **GET パラメータインジェクション** - URL パラメータベースの XSS 検出

**フェーズ 4: SQL インジェクションテスト**

- ✅ **エラーベース検出** - データベースエラーメッセージをテスト（MySQL、PostgreSQL、Oracle、MSSQL）
- ✅ **複数インジェクションタイプ** - Union ベース、Boolean ベース、コメントベース攻撃
- ✅ **一般的なパラメータ** - `id`、`user`、`email`、`search`、`category` パラメータをテスト
- ✅ **データベースフィンガープリンティング** - エラーレスポンスからデータベースタイプを特定

**フェーズ 5: 情報漏洩検出**

- ✅ **機密ファイル** - `.git`、`.env`、`/admin`、`/phpmyadmin` の露出をテスト
- ✅ **コンテンツ検証** - 実際の機密コンテンツを確認（200 ステータスだけでなく）
- ✅ **管理インターフェース検出** - アクセス可能な管理パネルを特定
- ✅ **設定ファイル漏洩** - 環境変数やシークレットの検出

### 🕷️ ZAP プロフェッショナル脆弱性スキャン（50+ ルール）

**アクティブスキャンルール（40+ 脆弱性テスト）:**

- 🚨 **インジェクション攻撃**: SQL インジェクション（複数データベース）、XSS（反射型/格納型）、コマンドインジェクション、コードインジェクション、XPath インジェクション
- 🚨 **認証バイパス**: パストラバーサル、ディレクトリブラウジング、パラメータ改竄、外部リダイレクト
- 🚨 **重要な CVE**: Log4Shell（CVE-2021-44228）、Spring4Shell（CVE-2022-22965）、Heartbleed OpenSSL
- 🚨 **フレームワーク脆弱性**: PHP-CGI RCE、サーバーサイドテンプレートインジェクション、バッファオーバーフロー
- 🚨 **情報漏洩**: ソースコード開示、隠しファイル発見、環境ファイル漏洩（.env、.htaccess）
- 🚨 **高度な攻撃**: XXE（XML External Entity）、CRLF インジェクション、XSLT インジェクション、フォーマット文字列エラー
- 🚨 **クラウドセキュリティ**: クラウドメタデータ攻撃（AWS、GCP、Azure）、ELMAH 情報漏洩
- 🚨 **データベース固有**: MySQL、PostgreSQL、Oracle、MSSQL、SQLite、Hypersonic の時間ベース SQL インジェクション

**パッシブスキャンルール（20+ セキュリティ分析）:**

- 🔍 **セキュリティヘッダー** - クローリング中の不足セキュリティヘッダー分析
- 🔍 **Cookie セキュリティ** - 安全でない Cookie 属性（HttpOnly、Secure、SameSite）
- 🔍 **SSL/TLS 問題** - 証明書問題、弱い暗号、プロトコル問題
- 🔍 **情報漏洩** - デバッグ情報、エラーメッセージ、バージョン開示
- 🔍 **コンテンツ分析** - 危険なコンテンツタイプ、安全でないリンク、混在コンテンツ

**スパイダー + 発見:**

- 🕸️ **完全サイトマッピング** - すべてのページ、フォーム、パラメータ、エンドポイントを発見
- 🕸️ **コンテンツ発見** - 隠しディレクトリ、バックアップファイル、管理パネルを発見
- 🕸️ **パラメータ発見** - 脆弱性テスト用のすべての入力ポイントを特定
- 🕸️ **技術検出** - フレームワーク、サーバー、データベース、言語をフィンガープリント

## 📄 出力とレポート

すべての結果を**ターミナルのみ**で表示 - ファイル生成なし、詳細な説明付きのクリーンな即座のフィードバック。

### 🛡️ コアセキュリティテスト出力

**修正すべき内容を正確に示すリアルタイムカラーターミナル結果:**

```bash
🛡️ フェーズ 1: セキュリティヘッダー分析
Testing: HSTS Header Present... ✅ PASS
Testing: X-Frame-Options Header... ❌ FAIL
🔒 不足: X-Frame-Options ヘッダー
   リスク: 中 - クリックジャッキング攻撃に脆弱
   内容: ページがフレーム/iframe に埋め込まれることを防ぐ
   修正: ヘッダーを追加: X-Frame-Options: DENY または X-Frame-Options: SAMEORIGIN
   例: nginx の場合: add_header X-Frame-Options "SAMEORIGIN" always;

🌐 フェーズ 2: CORS セキュリティテスト
✅ PASS: CORS ワイルドカード保護
✅ PASS: 悪意のあるオリジン拒否

⚡ フェーズ 3: XSS 脆弱性テスト
パラメータ 'search' の XSS テスト... ✅ 保護済み
パラメータ 'query' の XSS テスト... ❌ 脆弱
🚨 発見: パラメータ 'query' で反射型 XSS 脆弱性
   リスク: 高 - ユーザーブラウザでのリモートコード実行
   ペイロード: <script>alert('XSS')</script>
   修正: すべてのユーザー入力をサニタイズ、出力エンコーディング使用、CSP 実装
```

### 🕷️ ZAP プロフェッショナルスキャン出力

**業界標準分類による包括的脆弱性分析:**

```bash
🕷️ フェーズ 6: OWASP ZAP 脆弱性スキャン
🕸️ コンテンツ発見のためのターゲットスパイダリング... ✅ 完了（45 ページ発見）
🔍 パッシブセキュリティスキャン実行中... ✅ 完了
⚡ アクティブ脆弱性スキャン実行中... ✅ 完了

📊 ZAP 検出結果サマリー:
🔴 高リスク (2): SQL インジェクション、格納型 XSS
🟡 中リスク (5): セキュリティヘッダー不足、CSRF 脆弱性
🟢 低リスク (12): 情報漏洩、弱い SSL 暗号
```

### 📊 最終評価レポート

**実行可能なリスクスコアリング付きエグゼクティブサマリー:**

```bash
📊 セキュリティ評価サマリー
================================
ターゲット: https://example.com
テスト成功率: 73% (8/11)
全体的リスクレベル: 中
リスクスコア: 45/100 (低いほど良い)

🔍 詳細脆弱性検出結果:
🔴 高リスク検出結果 (2):
   • パラメータ 'id' での SQL インジェクション
   • お問い合わせフォームでの格納型 XSS

🟡 中リスク検出結果 (5):
   • HSTS ヘッダー不足
   • Content-Security-Policy ヘッダー不足
   • CSRF トークン不足
```

**各検出結果には以下が含まれます:**

- 🎯 **リスクレベル**: 高/中/低の明確な説明
- 🔍 **技術詳細**: 脆弱性の内容とその重要性
- 🛠️ **正確な修正**: あなたのウェブサーバー/フレームワーク用のコピー&ペースト解決策
- 📋 **例**: すぐに使用できる実際の設定スニペット

**終了コードは全体的なセキュリティ状況を示します:**

- `0`: 低リスク - 良好なセキュリティ体制
- `1`: 中リスク - 対処すべき問題がある
- `2`: 高リスク - 重要な脆弱性が発見された

## 🎯 リスク評価

ツールは自動リスクスコアリングを提供します:

- **🔴 高リスク**: 即座の対応が必要（XSS、SQLi など）
- **🟡 中リスク**: 対処すべき（ヘッダー不足など）
- **🟢 低リスク**: ベストプラクティスの改善

終了コードは全体的なリスクレベルを示します:

- `0`: 低リスク
- `1`: 中リスク
- `2`: 高リスク

評価は、リスクレベル別に優先順位付けされたすべての検出結果の詳細な内訳で終了するため、最初に修正すべき内容を正確に把握できます。

## 🛠️ 設定

### 環境変数

```bash
export ZAP_PORT=8090              # ZAPデーモンポート
export ZAP_HOST=127.0.0.1         # ZAPデーモンホスト
export TIMEOUT=300                # スキャンタイムアウト（秒）
```

**注意:** ZAP は必要に応じて自動的にデーモンモードで起動します - 手動設定は不要です！

## 🚨 セキュリティ上の考慮事項

### ⚠️ **重要警告: ディープ侵入テストの危険性**

**🏠 推奨: ローカルテストのみ**

```bash
# 安全: 開発環境でのテスト
npm run dev
./generic-security-test.sh http://localhost:3000 full
```

**⚡ 本番ウェブサイトでの実行の危険性:**

**🚨 サーバーへの影響:**

- **DoS/クラッシュリスク** - 5 つの並行スレッド + 無制限リクエストでサーバー過負荷
- **データベースロック** - SQL インジェクションテストでテーブル凍結、データベースクラッシュ
- **リソース枯渇** - 集約的テストで CPU/メモリを全て消費

**💥 データ損傷:**

- **テストデータ汚染** - ZAP が数千のテストペイロードをデータベースに注入
- **不要なファイルアップロード** - テストファイルをサーバーにアップロードする可能性
- **フォームスパム** - 偽アカウント、注文、お問い合わせフォーム送信を作成

**⚖️ 法的結果:**

- **コンピュータ詐欺防止法** 違反（米国連邦犯罪）
- **不正アクセス** 多くの国で刑事告発
- **民事訴訟** 影響を受けた企業からの訴訟
- **刑事起訴** - 「テストしただけ」は法的弁護にならない

**🎯 ディープ侵入テストモードが実際に行うこと:**

```bash
# ZAPが実行する攻撃的テストの例:
POST /api/users {"name": "<script>alert('XSS')</script>"}
POST /login {"user": "admin'; DROP TABLE users; --"}
GET /admin/../../../../etc/passwd
GET /api/delete/1 OR 1=1--
# × 類似のリクエストを数千回
```

### 🛡️ 安全な使用ガイドライン

1. **🏠 ローカル開発のみ** - localhost テストに最適
2. **📝 書面での許可が必要** - 所有していないサイトはテストしない
3. **🔒 自分のウェブサイトのみ** - それでもステージング/開発環境を使用
4. **⚠️ 本番サイトテスト** - 可能だが上記のリスクを全て伴う
5. **🌐 ネットワーク分離** - 分離された環境からの実行を検討

## 🔧 トラブルシューティング

### よくある問題

**🛡️ ボット保護による偽の失敗**

**問題:** あなたのセキュリティヘッダーは完璧ですが、スキャナーは失敗を表示します。

```bash
# 例: スキャナーがすべてのヘッダーが不足していると表示
❌ FAIL: HSTS Header Present
❌ FAIL: X-Frame-Options Header
# しかし、ヘッダーは実際には正しく設定されています！
```

**根本原因:** ホスティングプラットフォーム（Vercel、Cloudflare など）が自動化されたリクエストをブロックしています。

**検出方法:**

```bash
# 手動テストでボット保護を確認
curl -sI https://yoursite.com
# 返り値: HTTP/2 403 または実際のヘッダーの代わりにチャレンジページ
```

**解決策:**

1. **正しい URL をテスト**（リダイレクトを確認）:

   ```bash
   # ❌ 間違い: リダイレクト応答をテスト
   ./generic-security-test.sh https://example.com

   # ✅ 正しい: 実際のサイトをテスト
   ./generic-security-test.sh https://www.example.com
   ```

2. **ボット保護を一時的に無効化**（ホスティングダッシュボードで）

3. **ローカルでテスト**（ホスティング保護をバイパス）:

   ```bash
   npm run dev  # またはあなたのローカルサーバー
   ./generic-security-test.sh http://localhost:3000 quick
   ```

4. **ブラウザベースのツールを使用**（ブロックされない）:
   - https://securityheaders.com にアクセス
   - あなたのサイトをテスト（実際のブラウザ扱いを受ける）

**プラットフォーム例:**

- ✅ **GitHub**: セキュリティスキャナーを許可（開発者向けに構築）
- ⚠️ **Vercel/Netlify**: 自動化されたリクエストをブロックする場合がある（パフォーマンス優先）
- ⚠️ **Cloudflare**: 多くの場合ボットをブロック（強力な DDoS 保護）

**🔒 HTTP での HSTS 不足（期待される動作）**

**問題:** スキャナーが localhost で「HSTS ヘッダーが不足」と報告。

```bash
# HTTP localhostをテスト
./generic-security-test.sh http://localhost:3000 quick
❌ FAIL: HSTS Header Present  # これは実際には正しいです！
```

**説明:** HSTS（HTTP Strict Transport Security）は**HTTPS サイトのみ**のものです。

- ✅ **HTTPS サイト**: HSTS ヘッダーがあるべき
- ✅ **HTTP サイト**: HSTS ヘッダーがあるべきでない（HTTP では意味がない）
- ✅ **あなたの localhost HTTP**: HSTS の不足は**正しい動作**

**解決策:** これは実際の失敗ではありません - あなたの設定は正しく動作しています！

**📊 結果の解釈: 実際の失敗 vs 偽の失敗**

| テスト環境               | 期待されるスコア           | ヘッダー不足の意味                 |
| ------------------------ | -------------------------- | ---------------------------------- |
| **本番 HTTPS**           | 100%（6/6 ヘッダー）       | 実際のセキュリティ問題             |
| **ローカル HTTP**        | 88%（5/6、HSTS なし）      | 完璧（HTTP では HSTS 不要）        |
| **ボットブロックサイト** | 0-33%（すべて/大部分不足） | 偽の失敗（サイトは保護されている） |
| **リダイレクト URL**     | 可変                       | 代わりに最終的な宛先をテスト       |

**🎯 クイックチェック:**

```bash
# このパターンが見える場合、ボット保護に当たっている可能性があります:
HTTP/2 403
x-vercel-mitigated: challenge
# または
HTTP/2 503
cf-mitigated: challenge  # Cloudflare
```

---

**権限が拒否される:**

```bash
chmod +x generic-security-test.sh
```

## ❓ よくある質問

### Q: 要件をインストールするにはどうすればよいですか？

**A: OWASP ZAP をインストールするだけです:**

```bash
# macOS
brew install --cask owasp-zap

# Linux
sudo apt install zaproxy

# その後スキャンを実行
./generic-security-test.sh https://example.com
```

### Q: 何かインストールする必要がありますか？

**A: ZAP 脆弱性スキャンが必要な場合のみ:**

| **必要なもの**             | **インストール必要？** | **何をインストール** |
| -------------------------- | ---------------------- | -------------------- |
| **コアセキュリティテスト** | ❌ **何も不要**        | `curl`だけで動作     |
| **ZAP 脆弱性スキャン**     | ✅ **はい**            | OWASP ZAP のみ       |

### Q: `-daemon`とは何ですか？

**A: ZAP をヘッドレスモードで起動します**（GUI なし）でバックグラウンドで実行でき、API 呼び出しで制御できます。自動化されたスキャンに最適です。

### Q: コアテストと ZAP スキャンの違いは何ですか？

**A: 2 つの補完的なアプローチ:**

| **コアセキュリティテスト**              | **ZAP 脆弱性スキャン**                     |
| --------------------------------------- | ------------------------------------------ |
| ✅ すぐに動作（`curl`のみ必要）         | ⚠️ OWASP ZAP のみ必要（Python なし！）     |
| ✅ ヘッダー、CORS、XSS、SQLi、情報漏洩  | ✅ コアのすべて + 深い ZAP 分析            |
| ✅ プロフェッショナルリスクスコアリング | ✅ スパイダークローリング + コンテンツ発見 |
| ✅ 修正付き詳細ターミナル出力           | ✅ ZAP 検出結果付き豊富なターミナル出力    |
| ⚡ **高速**（1-2 分）                   | 🔍 **包括的**（15-30 分）                  |

### Q: どのテストを実行するかを制御するにはどうすればよいですか？

**A: モードパラメータを使用します:**

```bash
# クイック: コアセキュリティテストのみ（高速）
./generic-security-test.sh https://example.com quick
# ✅ 1-2分、ヘッダー + CORS + 情報漏洩

# フル: コアテスト + ZAP脆弱性スキャン（包括的）
./generic-security-test.sh https://example.com
# ✅ 15-30分、すべて + ZAP分析

# ZAPのみ: ZAP脆弱性スキャンのみ
./generic-security-test.sh https://example.com zap-only
# ✅ ZAPスパイダー + パッシブ + アクティブスキャン
```

### Q: セキュリティヘッダーが設定されているのに失敗するのはなぜですか？

**A: これは通常、ボット保護がスキャナーをブロックしているためです。**

**よくあるサイン:**

- すべてまたは大部分のヘッダーが「不足」として表示される
- サイトはブラウザでは正常に動作する
- `curl -sI yoursite.com`が 403 またはチャレンジページを返す

**解決策:**

1. ローカルでテスト: `./generic-security-test.sh http://localhost:3000 quick`
2. ブラウザツールを使用: https://securityheaders.com
3. [ボット保護トラブルシューティングセクション](#よくある問題)を確認

あなたのヘッダーはおそらく完璧に動作しています - スキャナーがそれらを見ることができないだけです！

## 📈 パフォーマンス注記

- **基本スキャン**: 約 2-5 分
- **高度な ZAP スキャン**: サイトの複雑さに応じて約 10-30 分
- **リソース使用量**: ZAP は大きなサイトでは大量の CPU/メモリを使用する可能性があります

## 📜 ライセンス

MIT ライセンス - 責任を持って使用し、所有するシステムまたは明確な許可を得たシステムでのみ使用してください。

## 📝 クイックサマリー

**🎯 目標:** 認証や特定の知識を必要とせずに、あらゆるウェブサイトのセキュリティをテストする。

**🚀 はじめに（依存関係ゼロ）:**

1. スクリプトをダウンロード
2. `./generic-security-test.sh https://example.com quick`
3. 1-2 分でプロフェッショナルセキュリティ評価

**🚀 簡単セットアップ（プロフェッショナルスイート）:**

1. **ZAP をインストール**: `brew install --cask owasp-zap`（macOS）または`apt install zaproxy`（Linux）
2. **スキャンを実行**: `./generic-security-test.sh https://example.com`
3. 15-30 分で包括的評価

**📊 3 つのテストモード:**

- ✅ **クイックモード** (quick): ヘッダー、CORS、情報漏洩（`curl`のみ必要）
- ✅ **フルモード** (full): 上記すべて + XSS + SQL インジェクション + ZAP 脆弱性スキャン
- ✅ **ZAP のみ** (zap-only): 50 以上のルール + ターミナル結果を含むプロフェッショナル ZAP スキャンのみ（ZAP のみ必要！）

**⚡ パフォーマンス:**

- **基本**: 1-2 分、セットアップ不要
- **高度**: 15-30 分、完全プロフェッショナルスイート

**プラットフォームサポート:**

- **macOS/Linux**: シンプルなパッケージマネージャーインストール（`brew`、`apt`、`dnf`）
- **Windows**: 公式ウェブサイトから手動ダウンロードおよびインストール

---

**⚠️ 法的免責事項:** このツールは許可されたセキュリティテストのみを目的としています。ユーザーはシステムをテストする前に適切な許可を得ていることを確認する責任があります。
