#!/bin/bash

# General Web Application Security Testing Script
# Basic security testing with foundational ZAP integration
# Focuses on general security headers, CORS, XSS, SQL injection, and info disclosure
# Does NOT include authentication/login testing - use specialized scripts for that
# Usage: ./general-security-test.sh [TARGET_URL] [--quick]

set -e

# Check for help flag first
if [[ "$1" == "--help" || "$1" == "-h" ]]; then
    echo -e "${PURPLE}🔐 GENERAL WEB APPLICATION SECURITY SCANNER ${NC}"
    echo -e "${PURPLE}======================================================${NC}"
    echo -e "${BLUE}A comprehensive security testing tool for web applications${NC}"
    echo -e "${BLUE}Focuses on general security headers, CORS, XSS, SQL injection, and info disclosure${NC}"
    echo -e "${GRAY}Does NOT include authentication/login testing - use specialized scripts for that${NC}\n"
    
    echo -e "${YELLOW}USAGE:${NC}"
    echo -e "  $0 <TARGET_URL> [MODE] [CLEAN_OUTPUT]\n"
    
    echo -e "${YELLOW}ARGUMENTS:${NC}"
    echo -e "  TARGET_URL    Target web application URL (required)"
    echo -e "  MODE          Testing mode (optional, default: full)"
    echo -e "  CLEAN_OUTPUT  Output format (optional, default: false)\n"
    
    echo -e "${YELLOW}MODES:${NC}"
    echo -e "  quick    Fast core tests (headers, CORS, CSRF, path traversal, SQLi, SSL)"
    echo -e "  full     Complete analysis (all quick tests + XSS, open redirect, command injection, ZAP)\n"
    
    echo -e "${YELLOW}EXAMPLES:${NC}"
    echo -e "  $0 http://localhost:3000 full"
    echo -e "  $0 https://example.com quick"
    echo -e "  $0 http://app.local full true\n"
    
    echo -e "${YELLOW}FEATURES:${NC}"
    echo -e "  • Security headers analysis (HSTS, CSP, X-Frame-Options)"
    echo -e "  • CORS misconfiguration detection"
    echo -e "  • XSS vulnerability testing"
    echo -e "  • SQL injection detection"
    echo -e "  • CSRF protection testing"
    echo -e "  • Path traversal detection"
    echo -e "  • Open redirect testing"
    echo -e "  • SSL/TLS configuration analysis"
    echo -e "  • Command injection detection"
    echo -e "  • Information disclosure checks"
    echo -e "  • ZAP integration for comprehensive scanning\n"
    
    echo -e "${YELLOW}NOTES:${NC}"
    echo -e "  • Requires ZAP proxy to be running for full mode"
    echo -e "  • Authentication testing is handled by separate scripts"
    echo -e "  • Results include risk levels and fix recommendations\n"
    
    exit 0
fi

# Configuration
TARGET_URL="${1:-}"
MODE="${2:-full}"  # full, quick
CLEAN_OUTPUT="${3:-false}"  # clean output for desktop apps
AUTH_COOKIE="${AUTH_COOKIE:-}"  # Authentication cookie for authenticated scanning
AUTH_TOKEN="${AUTH_TOKEN:-}"  # Bearer token for authenticated scanning
ZAP_PORT="${ZAP_PORT:-8090}"
ZAP_HOST="${ZAP_HOST:-127.0.0.1}"
TIMEOUT="${TIMEOUT:-300}"

# Colors - disable for clean output
if [ "$CLEAN_OUTPUT" = "true" ]; then
    RED=''
    GREEN=''
    YELLOW=''
    BLUE=''
    PURPLE=''
    CYAN=''
    GRAY=''
    NC=''
else
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    BLUE='\033[0;34m'
    PURPLE='\033[0;35m'
    CYAN='\033[0;36m'
    GRAY='\033[0;90m'
    NC='\033[0m'
fi

# Test tracking
TESTS_PASSED=0
TESTS_FAILED=0
TESTS_TOTAL=0
HIGH_RISK_FINDINGS=0
MEDIUM_RISK_FINDINGS=0
LOW_RISK_FINDINGS=0

# Findings tracking arrays
HIGH_RISK_LIST=()
MEDIUM_RISK_LIST=()
LOW_RISK_LIST=()
INFO_LIST=()

# Validate input
if [ -z "$TARGET_URL" ]; then
    echo -e "${RED}❌ Error: Target URL is required${NC}"
    echo -e "${BLUE}Usage: $0 <TARGET_URL> [mode]${NC}"
    echo -e "${BLUE}Modes: ${NC}"
    echo -e "${BLUE}  quick   - Fast core tests only${NC}"
    echo -e "${BLUE}  full    - Full security analysis with ZAP${NC}"
    echo -e "${BLUE}Example: $0 http://localhost:3000 full${NC}"
    echo -e "${GRAY}Note: This script focuses on general security tests.${NC}"
    echo -e "${GRAY}      For authentication testing, use specialized scripts.${NC}"
    exit 1
fi

# Ensure URL has protocol
if [[ ! "$TARGET_URL" =~ ^https?:// ]]; then
    TARGET_URL="https://$TARGET_URL"
fi

echo -e "${PURPLE}🔐 GENERAL WEB APPLICATION SECURITY SCANNER ${NC}"
echo -e "${PURPLE}======================================================${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${NC}"
echo -e "${BLUE}Mode: ${MODE}${NC}"
echo -e "${BLUE}Timestamp: $(date)${NC}"
if [ -n "$AUTH_TOKEN" ]; then
    echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
fi
if [ -n "$AUTH_COOKIE" ]; then
    echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
fi
echo -e ""

# Helper function for authenticated curl requests (supports cookie and bearer token)
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

# Function to log results
log_result() {
    echo "$1"
}

# Function to get detailed finding explanations
get_finding_details() {
    local finding_type="$1"
    
    case "$finding_type" in
        "HSTS")
            echo "🔒 MISSING: HTTP Strict Transport Security (HSTS)"
            echo "   RISK: Medium - Allows protocol downgrade attacks"
            echo "   WHAT: Forces browsers to use HTTPS, prevents man-in-the-middle attacks"
            echo "   FIX: Add header: Strict-Transport-Security: max-age=31536000; includeSubDomains"
            ;;
        "X-Frame-Options")
            echo "🔒 MISSING: X-Frame-Options Header"
            echo "   RISK: Medium - Vulnerable to clickjacking attacks"
            echo "   WHAT: Prevents page from being embedded in frames/iframes"
            echo "   FIX: Add header: X-Frame-Options: DENY or X-Frame-Options: SAMEORIGIN"
            ;;
        "X-Content-Type-Options")
            echo "🔒 MISSING: X-Content-Type-Options Header"
            echo "   RISK: Low - MIME type sniffing vulnerabilities"
            echo "   WHAT: Prevents browsers from MIME-sniffing responses"
            echo "   FIX: Add header: X-Content-Type-Options: nosniff"
            ;;
        "Content-Security-Policy")
            echo "🔒 MISSING: Content Security Policy (CSP)"
            echo "   RISK: Medium - No protection against XSS/injection attacks"
            echo "   WHAT: Controls which resources browser can load"
            echo "   FIX: Add header: Content-Security-Policy: default-src 'self'"
            ;;
        "CORS-Wildcard")
            echo "🚨 FOUND: CORS Wildcard Origin (*)"
            echo "   RISK: High - Allows any website to access your API"
            echo "   WHAT: Access-Control-Allow-Origin: * allows all domains"
            echo "   FIX: Specify exact origins: Access-Control-Allow-Origin: https://yourdomain.com"
            ;;
        "XSS-Reflected")
            echo "🚨 FOUND: Reflected XSS Vulnerability"
            echo "   RISK: High - Remote code execution in user browsers"
            echo "   WHAT: User input reflected in HTML without proper escaping"
            echo "   FIX: Sanitize ALL user input, use output encoding, implement CSP"
            ;;
        "SQL-Injection")
            echo "🚨 FOUND: SQL Injection Vulnerability"
            echo "   RISK: High - Database compromise, data theft"
            echo "   WHAT: User input directly concatenated into SQL queries"
            echo "   FIX: Use parameterized queries/prepared statements ALWAYS"
            ;;
        "Info-Git")
            echo "   RISK: High - Source code disclosure"
            echo "   WHAT: Git repository accessible via web browser"
            echo "   FIX: Block access to .git in web server config"
            ;;
        "Info-Env")
            echo "   RISK: High - Environment variables/secrets exposed"
            echo "   WHAT: Environment configuration file accessible"
            echo "   FIX: Block access to .env files in web server"
            ;;
    esac
}

# Function to run a test with detailed reporting
run_test() {
    local test_name="$1"
    local test_input="$2"
    local success_pattern="$3"
    local failure_is_good="${4:-false}"
    local finding_type="${5:-}"

    TESTS_TOTAL=$((TESTS_TOTAL + 1))

    echo -n "Testing: $test_name... "

    # Use direct string matching instead of eval to prevent command injection
    result="$test_input"
    
    # Check result
    local test_passed=false
    if [ "$failure_is_good" = "true" ]; then
        if ! echo "$result" | grep -qi "$success_pattern"; then
            test_passed=true
        fi
    else
        if echo "$result" | grep -qi "$success_pattern"; then
            test_passed=true
        fi
    fi
    
    if [ "$test_passed" = "true" ]; then
        echo -e "${GREEN}PASS${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
        log_result "✅ PASS: $test_name"
    else
        echo -e "${RED}FAIL${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        log_result "❌ FAIL: $test_name"
        
        if [ -n "$finding_type" ]; then
            echo -e "${YELLOW}$(get_finding_details "$finding_type" | head -1)${NC}"
            get_finding_details "$finding_type" | tail -n +2
            echo ""
        fi
    fi
}

# Function to check if ZAP is running
check_zap_status() {
    local response=$(curl -s --connect-timeout 5 --max-time 10 "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/version/" 2>/dev/null)
    
    if echo "$response" | grep -q '"version"'; then
        return 0
    else
        return 1
    fi
}

# Function to start ZAP if needed
start_zap_if_needed() {
    if ! check_zap_status; then
        echo -e "${YELLOW}🕷️ Starting OWASP ZAP...${NC}"
        
        if [ -x "/Applications/ZAP.app/Contents/Java/zap.sh" ]; then
            echo -e "${BLUE}🚀 Launching ZAP daemon on port $ZAP_PORT...${NC}"
            /Applications/ZAP.app/Contents/Java/zap.sh -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
            ZAP_PID=$!
        elif command -v zap.sh >/dev/null 2>&1; then
            echo -e "${BLUE}🚀 Launching ZAP daemon on port $ZAP_PORT...${NC}"
            zap.sh -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
            ZAP_PID=$!
        elif command -v owasp-zap >/dev/null 2>&1; then
            echo -e "${BLUE}🚀 Launching ZAP daemon on port $ZAP_PORT...${NC}"
            owasp-zap -daemon -port "$ZAP_PORT" -config api.disablekey=true > /tmp/zap_startup.log 2>&1 &
            ZAP_PID=$!
        else
            echo -e "${RED}❌ OWASP ZAP not found. Please install ZAP or run manually on port $ZAP_PORT${NC}"
            return 1
        fi
        
        echo -e "${YELLOW}🔄 ZAP is initializing...${NC}"
        echo -n "Progress: "
        
        local max_attempts=60  # 2 minutes
        local attempt=0
        
        while [ $attempt -lt $max_attempts ]; do
            if check_zap_status; then
                echo -e "\n${GREEN}✅ ZAP daemon is ready! (took $((attempt * 2)) seconds)${NC}"
                return 0
            fi
            
            if [ $((attempt % 10)) -eq 0 ]; then
                local elapsed=$((attempt * 2))
                echo -n "${elapsed}s "
            else
                echo -n "."
            fi
            
            if ! kill -0 $ZAP_PID 2>/dev/null; then
                echo -e "\n${RED}❌ ZAP process died during startup${NC}"
                return 1
            fi
            
            attempt=$((attempt + 1))
            sleep 2
        done
        
        echo -e "\n${RED}❌ ZAP failed to start within timeout${NC}"
        return 1
    else
        echo -e "${GREEN}✅ ZAP is already running${NC}"
    fi
}

# Function to run security header tests (Days 1-2)
run_security_headers_test() {
    echo -e "\n${YELLOW}🛡️ PHASE 1: SECURITY HEADERS ANALYSIS${NC}"
    
    headers=$(auth_curl -sI "$TARGET_URL" 2>/dev/null || echo "")
    
    # Check for HSTS (only for HTTPS sites)
    if [[ "$TARGET_URL" == https://* ]]; then
        run_test "HSTS Header Present" "$headers" "Strict-Transport-Security" "false" "HSTS"
        if ! echo "$headers" | grep -qi "Strict-Transport-Security"; then
            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
            MEDIUM_RISK_LIST+=("Missing HSTS Header")
        fi
    else
        echo -n "Testing: HSTS Header Present... "
        echo -e "${BLUE}SKIP${NC}"
        echo -e "${GRAY}ℹ️  SKIP: HSTS not applicable to HTTP sites${NC}"
    fi
    
    # Check for X-Frame-Options (MEDIUM - enables clickjacking attacks)
    run_test "X-Frame-Options Header" "$headers" "X-Frame-Options" "false" "X-Frame-Options"
    if ! echo "$headers" | grep -qi "X-Frame-Options"; then
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Missing X-Frame-Options Header")
    fi
    
    # Check for X-Content-Type-Options
    run_test "X-Content-Type-Options Header" "$headers" "X-Content-Type-Options" "false" "X-Content-Type-Options"
    if ! echo "$headers" | grep -qi "X-Content-Type-Options"; then
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
        LOW_RISK_LIST+=("Missing X-Content-Type-Options Header")
    fi
    
    # Check for Content-Security-Policy
    csp_header=$(echo "$headers" | grep -i "Content-Security-Policy" || echo "")
    if [ -z "$csp_header" ]; then
        echo -n "Testing: Content-Security-Policy Header... "
        echo -e "${RED}FAIL${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Missing Content-Security-Policy Header")
        log_result "❌ FAIL: Content-Security-Policy Header"
    else
        # CSP exists, now analyze its security
        echo -n "Testing: Content-Security-Policy Header... "
        if echo "$csp_header" | grep -qi "unsafe-eval\|unsafe-inline\|\*"; then
            echo -e "${YELLOW}WEAK${NC}"
            TESTS_FAILED=$((TESTS_FAILED + 1))
            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
            MEDIUM_RISK_LIST+=("Weak Content-Security-Policy (unsafe directives)")
            log_result "⚠️ WEAK: Content-Security-Policy Header - contains unsafe directives"
        else
            echo -e "${GREEN}PASS${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
            log_result "✅ PASS: Content-Security-Policy Header"
        fi
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
}

# Function to run CORS tests (Day 3)
run_cors_tests() {
    echo -e "\n${YELLOW}🌐 PHASE 2: CORS SECURITY TESTS${NC}"
    
    evil_origin="https://evil-site.com"
    cors_response=$(auth_curl -s -H "Origin: $evil_origin" \
        -H "Access-Control-Request-Method: POST" \
        -X OPTIONS "$TARGET_URL" 2>/dev/null || echo "")
    
    # Check if CORS allows any origin (bad)
    if echo "$cors_response" | grep -q "Access-Control-Allow-Origin: \*"; then
        echo -e "${RED}FAIL: CORS Wildcard Protection${NC}"
        get_finding_details "CORS-Wildcard"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("CORS Wildcard Origin (*)")
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN}PASS: CORS Wildcard Protection${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Test if evil origin is accepted (bad)
    if echo "$cors_response" | grep -q "$evil_origin"; then
        echo -e "${RED}FAIL: Evil Origin Rejection${NC}"
        echo -e "${YELLOW}🚨 FOUND: Evil Origin Accepted${NC}"
        echo "   RISK: High - Malicious websites can access your resources"
        echo "   FIX: Implement proper origin whitelist checking"
        echo ""
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Evil Origin Accepted")
        TESTS_FAILED=$((TESTS_FAILED + 1))
    else
        echo -e "${GREEN}PASS: Evil Origin Rejection${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
}

# Function to run XSS tests (Day 3)
run_xss_tests() {
    echo -e "\n${YELLOW}⚡ PHASE 3: XSS VULNERABILITY TESTS${NC}"
    
    # Basic XSS payloads
    xss_payloads=(
        "<script>alert('XSS')</script>"
        "javascript:alert('XSS')"
        "<img src=x onerror=alert('XSS')>"
    )
    
    # Common parameters (general, non-authentication)
    common_params=("q" "search" "query" "data" "input")
    
    for param in "${common_params[@]}"; do
        for payload in "${xss_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed 's/ /%20/g' | sed 's/</%3C/g' | sed 's/>/%3E/g')
            response=$(auth_curl -s "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")
            
            if echo "$response" | grep -q "$payload"; then
                echo -e "${RED}🚨 Potential XSS in parameter: $param${NC}"
                echo "VULN_DETAILS: XSS_REFLECTED | Parameter: $param | Location: ${TARGET_URL}?${param}=${encoded_payload} | Payload: $payload"
                get_finding_details "XSS-Reflected"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("XSS Vulnerability in parameter '$param'")
                break
            fi
        done
    done
    
    run_test "XSS Protection Test" "Completed XSS testing" "Completed"
}

# Function to run SQL injection tests (Day 3)
run_sql_injection_tests() {
    echo -e "\n${YELLOW}💉 PHASE 4: SQL INJECTION TESTS${NC}"
    
    # Basic SQL injection payloads (general)
    sql_payloads=(
        "' OR '1'='1"
        "'; DROP TABLE test; --"
        "1' AND '1'='1"
    )
    
    # Common vulnerable parameters (general, non-authentication)
    sql_params=("id" "search" "query")
    
    for param in "${sql_params[@]}"; do
        for payload in "${sql_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed "s/ /%20/g" | sed "s/'/%27/g")
            response=$(auth_curl -s "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")
            
            if echo "$response" | grep -qiE "(sql syntax|mysql.*error|postgresql.*error|syntax error.*sql)"; then
                echo -e "${RED}🚨 Potential SQL injection in parameter: $param${NC}"
                echo "VULN_DETAILS: SQL_INJECTION | Parameter: $param | Location: ${TARGET_URL}?${param}=${encoded_payload} | Payload: $payload"
                get_finding_details "SQL-Injection"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("SQL Injection in parameter '$param'")
                break
            fi
        done
    done
    
    run_test "SQL Injection Protection Test" "Completed SQL injection testing" "Completed"
}

# Function to run information disclosure tests (Day 3)
run_info_disclosure_tests() {
    echo -e "\n${YELLOW}📋 PHASE 5: INFORMATION DISCLOSURE TESTS${NC}"
    
    # Basic sensitive paths (non-authentication related)
    sensitive_paths=(
        "/.git"
        "/.env"
    )
    
    for path in "${sensitive_paths[@]}"; do
        response=$(auth_curl -s "${TARGET_URL}${path}" 2>/dev/null || echo "ERROR")
        status_code=$(auth_curl -s -o /dev/null -w "%{http_code}" "${TARGET_URL}${path}" 2>/dev/null || echo "000")
        
        if [ "$status_code" = "200" ]; then
            case "$path" in
                "/.git")
                    if echo "$response" | grep -qi "objects\\|refs\\|HEAD"; then
                        # Single output format to avoid duplicate detection
                        echo -e "${RED}❌ FAIL: .git directory accessible${NC}"
                        echo "VULN_DETAILS: EXPOSED_GIT | Location: ${TARGET_URL}${path} | Status: ${status_code}"
                        get_finding_details "Info-Git"
                        echo ""
                        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                        HIGH_RISK_LIST+=(".git directory accessible")
                    fi
                    ;;
                "/.env")
                    if echo "$response" | grep -qE "^[A-Z_]+=|APP_|DB_"; then
                        # Single output format to avoid duplicate detection
                        echo -e "${RED}❌ FAIL: .env file accessible${NC}"
                        echo "VULN_DETAILS: EXPOSED_ENV | Location: ${TARGET_URL}${path} | Status: ${status_code}"
                        get_finding_details "Info-Env"
                        echo ""
                        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                        HIGH_RISK_LIST+=(".env file accessible")
                    fi
                    ;;
            esac
        fi
    done
    
    run_test "Information Disclosure Test" "Completed path enumeration" "Completed"
}

# Function to run basic ZAP scan (Day 4)
run_basic_zap_scan() {
    echo -e "\n${YELLOW}🕷️ PHASE 6: BASIC ZAP VULNERABILITY SCAN${NC}"
    
    if ! start_zap_if_needed; then
        echo -e "${RED}❌ Cannot run ZAP scan - ZAP not available${NC}"
        return 1
    fi
    
    echo "🔍 Running basic vulnerability scan..."
    
    # Set up ZAP context
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/context/action/newContext/?contextName=GeneralTest" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/context/action/includeInContext/?contextName=GeneralTest&regex=${TARGET_URL}.*" >/dev/null 2>&1
    
    # Basic spider configuration
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxDepth/?Integer=3" >/dev/null 2>&1
    curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/setOptionMaxDuration/?Integer=120" >/dev/null 2>&1
    
    # Spider the target
    echo "🕸️ Basic spidering..."
    
    # URL encode the target URL for ZAP API
    encoded_url=$(echo "$TARGET_URL" | sed 's/:/%3A/g' | sed 's/\//%2F/g')
    spider_response=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/action/scan/?url=${encoded_url}")
    
    # Debug output to see what ZAP returned
    echo "Debug - ZAP spider response: $spider_response"
    
    # Try to extract scan ID - handle both with and without jq
    if command -v jq >/dev/null 2>&1; then
        spider_id=$(echo "$spider_response" | jq -r '.scan // empty' 2>/dev/null)
    else
        # Fallback parsing without jq
        spider_id=$(echo "$spider_response" | grep -o '"scan":"[^"]*"' | cut -d'"' -f4)
    fi
    
    echo "Debug - Extracted spider ID: '$spider_id'"
    
    if [ -z "$spider_id" ] || [ "$spider_id" = "empty" ] || [ "$spider_id" = "null" ]; then
        echo -e "${RED}❌ Failed to start spider scan${NC}"
        echo -e "${YELLOW}Response from ZAP: $spider_response${NC}"
        return 1
    fi
    
    # Wait for spider to complete
    while true; do
        if command -v jq >/dev/null 2>&1; then
            spider_status=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/status/?scanId=${spider_id}" | jq -r '.status // "0"' 2>/dev/null)
        else
            # Fallback parsing without jq
            spider_status=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/status/?scanId=${spider_id}" | grep -o '"status":"[^"]*"' | cut -d'"' -f4)
        fi
        
        if [ "$spider_status" = "100" ]; then
            break
        fi
        echo -n "."
        sleep 2
    done
    
    # Get URL count
    if command -v jq >/dev/null 2>&1; then
        urls_found=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | jq -r '.results | length // 0' 2>/dev/null)
    else
        # Fallback - just count URLs in results
        urls_found=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/spider/view/results/?scanId=${spider_id}" | grep -o '"results":\[' | wc -l)
        [ "$urls_found" -eq 0 ] && urls_found="unknown"
    fi
    echo -e " ${GREEN}✅ Basic spidering completed - ${urls_found} URLs discovered${NC}"
    
    # Passive scan (automatic)
    echo "🔍 Running passive security scan..."
    sleep 3
    
    # Get basic vulnerability findings
    alerts=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/")
    
    # Count findings
    if command -v jq >/dev/null 2>&1; then
        zap_high=$(jq -r '[.alerts[] | select(.risk == "High")] | length // 0' <<< "$alerts" 2>/dev/null)
        zap_medium=$(jq -r '[.alerts[] | select(.risk == "Medium")] | length // 0' <<< "$alerts" 2>/dev/null)
        zap_low=$(jq -r '[.alerts[] | select(.risk == "Low")] | length // 0' <<< "$alerts" 2>/dev/null)
        
        # Add ZAP findings to lists (deduplicated) and count unique findings
        high_findings=$(jq -r '.alerts[] | select(.risk == "High") | .name' <<< "$alerts" 2>/dev/null | sort -u)
        if [ -n "$high_findings" ]; then
            zap_high_unique=0
            while IFS= read -r finding; do
                if [ -n "$finding" ]; then
                    HIGH_RISK_LIST+=("$finding")
                    zap_high_unique=$((zap_high_unique + 1))
                fi
            done <<< "$high_findings"
            HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + zap_high_unique))
        fi
        
        medium_findings=$(jq -r '.alerts[] | select(.risk == "Medium") | .name' <<< "$alerts" 2>/dev/null | sort -u)
        if [ -n "$medium_findings" ]; then
            zap_medium_unique=0
            while IFS= read -r finding; do
                if [ -n "$finding" ]; then
                    MEDIUM_RISK_LIST+=("$finding")
                    zap_medium_unique=$((zap_medium_unique + 1))
                fi
            done <<< "$medium_findings"
            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + zap_medium_unique))
        fi
        
        low_findings=$(jq -r '.alerts[] | select(.risk == "Low") | .name' <<< "$alerts" 2>/dev/null | sort -u)
        if [ -n "$low_findings" ]; then
            zap_low_unique=0
            while IFS= read -r finding; do
                if [ -n "$finding" ]; then
                    LOW_RISK_LIST+=("$finding")
                    zap_low_unique=$((zap_low_unique + 1))
                fi
            done <<< "$low_findings"
            LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + zap_low_unique))
        fi
        
        echo -e "${GREEN}✅ ZAP findings parsed: High=$zap_high, Medium=$zap_medium, Low=$zap_low${NC}"
        
        # Output detailed ZAP findings for UI parsing
        if [ "$zap_high" -gt 0 ] || [ "$zap_medium" -gt 0 ] || [ "$zap_low" -gt 0 ]; then
            echo -e "\n${BLUE}🔍 DETAILED ZAP FINDINGS:${NC}"
            
            # Parse and output detailed findings
            jq -r '.alerts[] | 
                "ZAP FINDING: " + .name + 
                "\nRisk: " + .risk + 
                "\nDescription: " + .description[0:200] + 
                "\nSolution: " + .solution[0:200] + 
                "\nInstance: " + (.instances[0].uri // "N/A") + 
                "\nMethod: " + (.instances[0].method // "N/A") + 
                "\nParameter: " + (.instances[0].param // "N/A") + 
                "\nEvidence: " + (.instances[0].evidence // "N/A")[0:100] + 
                "\n---"' <<< "$alerts" 2>/dev/null | while IFS= read -r line; do
                if [[ "$line" == ZAP* ]]; then
                    echo -e "${RED}🚨 $line${NC}"
                elif [[ "$line" == Risk:* ]]; then
                    risk_level=$(echo "$line" | cut -d':' -f2 | tr -d ' ')
                    case "$risk_level" in
                        "High") echo -e "${RED}   $line${NC}" ;;
                        "Medium") echo -e "${YELLOW}   $line${NC}" ;;
                        "Low") echo -e "${BLUE}   $line${NC}" ;;
                        *) echo -e "${GRAY}   $line${NC}" ;;
                    esac
                elif [[ "$line" == ---* ]]; then
                    echo ""
                else
                    echo -e "${GRAY}   $line${NC}"
                fi
            done
        fi
    else
        # Fallback parsing without jq - basic counting
        zap_high=$(echo "$alerts" | grep -c '"risk":"High"' 2>/dev/null || echo "0")
        zap_medium=$(echo "$alerts" | grep -c '"risk":"Medium"' 2>/dev/null || echo "0")
        zap_low=$(echo "$alerts" | grep -c '"risk":"Low"' 2>/dev/null || echo "0")
        
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + zap_high))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + zap_medium))
        LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + zap_low))
        
        echo -e "${YELLOW}⚠️ jq not found - basic ZAP parsing: High=$zap_high, Medium=$zap_medium, Low=$zap_low${NC}"
        echo -e "${YELLOW}⚠️ Install jq for detailed vulnerability information${NC}"
    fi
    
    echo -e "${GREEN}✅ Basic ZAP scan completed${NC}"
}

# Function to generate summary report
generate_summary() {
    echo -e "\n${PURPLE}📊 SECURITY ASSESSMENT SUMMARY${NC}"
    echo -e "${PURPLE}================================${NC}"
    
    # Calculate scores
    local success_rate=0
    if [ $TESTS_TOTAL -gt 0 ]; then
        success_rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    fi
    
    # Calculate overall risk
    local total_findings=$((HIGH_RISK_FINDINGS + MEDIUM_RISK_FINDINGS + LOW_RISK_FINDINGS))
    local risk_score=$((HIGH_RISK_FINDINGS * 10 + MEDIUM_RISK_FINDINGS * 3 + LOW_RISK_FINDINGS * 1))
    
    # Cap risk score at 100
    if [ $risk_score -gt 100 ]; then
        risk_score=100
    fi
    
    # Determine risk level
    local risk_level="LOW"
    local risk_color=$GREEN
    if [ $HIGH_RISK_FINDINGS -gt 0 ]; then
        risk_level="HIGH"
        risk_color=$RED
    elif [ $MEDIUM_RISK_FINDINGS -gt 2 ]; then
        risk_level="MEDIUM"
        risk_color=$YELLOW
    fi
    
    # Display summary
    echo -e "${BLUE}Target: $TARGET_URL${NC}"
    echo -e "${BLUE}Scan Date: $(date)${NC}"
    echo -e "${BLUE}Test Success Rate: $success_rate% ($TESTS_PASSED/$TESTS_TOTAL)${NC}"
    echo -e "${risk_color}Overall Risk Level: $risk_level${NC}"
    echo -e "${BLUE}Risk Score: $risk_score/100${NC}"
    
    # Show findings
    if [ $total_findings -gt 0 ]; then
        echo -e "\n${YELLOW}🔍 VULNERABILITY FINDINGS:${NC}"
        
        # High Risk Findings
        if [ $HIGH_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${RED}🔴 HIGH RISK FINDINGS ($HIGH_RISK_FINDINGS):${NC}"
            for finding in "${HIGH_RISK_LIST[@]}"; do
                echo -e "${RED}   • $finding${NC}"
                # Add detailed info for Vulnerable JS Library
                if [[ "$finding" == "Vulnerable JS Library" ]]; then
                    if command -v jq >/dev/null 2>&1; then
                        vuln_details=$(curl -s "http://${ZAP_HOST}:${ZAP_PORT}/JSON/core/view/alerts/" 2>/dev/null | jq -r '.alerts[] | select(.risk == "High" and .name == "Vulnerable JS Library") | .other' 2>/dev/null | head -1)
                        if [[ "$vuln_details" =~ nextjs.*version.*([0-9]+\.[0-9]+\.[0-9]+) ]]; then
                            echo -e "${RED}     📦 Library: Next.js ${BASH_REMATCH[1]}${NC}"
                            echo -e "${RED}     🚨 Multiple CVEs detected (CVE-2024-47831, CVE-2024-56332, etc.)${NC}"
                            echo -e "${RED}     🔧 Fix: Upgrade to Next.js 14.2.15+ or latest${NC}"
                        fi
                    fi
                fi
            done
        fi
        
        # Medium Risk Findings
        if [ $MEDIUM_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${YELLOW}🟡 MEDIUM RISK FINDINGS ($MEDIUM_RISK_FINDINGS):${NC}"
            for finding in "${MEDIUM_RISK_LIST[@]}"; do
                echo -e "${YELLOW}   • $finding${NC}"
            done
        fi
        
        # Low Risk Findings
        if [ $LOW_RISK_FINDINGS -gt 0 ]; then
            echo -e "\n${GREEN}🟢 LOW RISK FINDINGS ($LOW_RISK_FINDINGS):${NC}"
            for finding in "${LOW_RISK_LIST[@]}"; do
                echo -e "${GREEN}   • $finding${NC}"
            done
        fi
    else
        echo -e "\n${GREEN}✅ No security findings detected!${NC}"
    fi
    
    echo -e "\n${GREEN}✅ Security assessment completed${NC}"
}

# Function to run CSRF protection tests (Phase 7)
run_csrf_tests() {
    echo -e "\n${YELLOW}🛡️ PHASE 7: CSRF PROTECTION TESTING${NC}"
    
    # Test main page for forms with CSRF tokens
    main_page=$(auth_curl -s "$TARGET_URL" 2>/dev/null || echo "")
    
    # Check if forms exist
    if echo "$main_page" | grep -qi "<form"; then
        # Look for CSRF tokens in forms
        if echo "$main_page" | grep -qiE "csrf|_token|authenticity_token|__RequestVerificationToken"; then
            echo -e "${GREEN}✅ CSRF tokens detected in forms${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
        else
            echo -e "${RED}❌ No CSRF tokens found in forms${NC}"
            echo -e "${YELLOW}🚨 POTENTIAL: Missing CSRF Protection${NC}"
            echo "   RISK: Medium - Forms vulnerable to Cross-Site Request Forgery"
            echo "   WHAT: Forms without CSRF tokens can be submitted by malicious websites"
            echo "   FIX: Add CSRF tokens to all forms and validate on server-side"
            echo ""
            MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
            MEDIUM_RISK_LIST+=("Missing CSRF Protection in Forms")
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
    else
        echo -e "${BLUE}ℹ️  No forms detected on main page${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Check SameSite cookie attributes
    cookie_response=$(auth_curl -s -I "$TARGET_URL" 2>/dev/null || echo "")
    if echo "$cookie_response" | grep -qi "set-cookie:"; then
        if echo "$cookie_response" | grep -qi "samesite=strict\|samesite=lax"; then
            echo -e "${GREEN}✅ SameSite cookie attributes detected${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
        else
            echo -e "${YELLOW}⚠️ SameSite cookie attributes missing${NC}"
            echo "   RISK: Low - Cookies vulnerable to CSRF attacks"
            echo "   FIX: Add SameSite=Strict or SameSite=Lax to all cookies"
            LOW_RISK_FINDINGS=$((LOW_RISK_FINDINGS + 1))
            LOW_RISK_LIST+=("Missing SameSite Cookie Attributes")
            TESTS_FAILED=$((TESTS_FAILED + 1))
        fi
    else
        echo -e "${BLUE}ℹ️  No cookies set by server${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
}

# Function to run path traversal tests (Phase 8)
run_path_traversal_tests() {
    echo -e "\n${YELLOW}📁 PHASE 8: PATH TRAVERSAL DETECTION${NC}"
    
    # Path traversal payloads
    traversal_payloads=(
        "../../../etc/passwd"
        "..%5C..%5C..%5Cwindows%5Csystem32%5Cdrivers%5Cetc%5Chosts"
        "....//....//....//etc/passwd"
        "%2e%2e%2f%2e%2e%2f%2e%2e%2fetc%2fpasswd"
    )
    
    # Common vulnerable parameters
    traversal_params=("file" "path" "page" "include" "template" "doc" "view")
    
    local vulnerabilities_found=0
    
    for param in "${traversal_params[@]}"; do
        for payload in "${traversal_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed 's/\//%2F/g')
            response=$(auth_curl -s "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")
            
            # Check for Unix/Linux system files
            if echo "$response" | grep -qE "root:.*:/bin/(bash|sh)|daemon:.*:/sbin"; then
                echo -e "${RED}🚨 Path traversal vulnerability in parameter: $param${NC}"
                echo "VULN_DETAILS: PATH_TRAVERSAL | Parameter: $param | Payload: $payload"
                echo "   RISK: High - Unauthorized file system access detected"
                echo "   EVIDENCE: Unix passwd file content found"
                echo "   FIX: Validate and sanitize file path parameters, use whitelist approach"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Path Traversal in parameter '$param'")
                vulnerabilities_found=1
                break
            fi
            
            # Check for Windows system files
            if echo "$response" | grep -qE "127\.0\.0\.1.*localhost|# Copyright.*Microsoft"; then
                echo -e "${RED}🚨 Path traversal vulnerability in parameter: $param${NC}"
                echo "VULN_DETAILS: PATH_TRAVERSAL | Parameter: $param | Payload: $payload"
                echo "   RISK: High - Unauthorized file system access detected"
                echo "   EVIDENCE: Windows hosts file content found"
                echo "   FIX: Validate and sanitize file path parameters, use whitelist approach"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Path Traversal in parameter '$param'")
                vulnerabilities_found=1
                break
            fi
        done
        if [ $vulnerabilities_found -eq 1 ]; then
            break
        fi
    done
    
    run_test "Path Traversal Protection Test" "Completed path traversal testing" "Completed"
}

# Function to run open redirect tests (Phase 9)
run_open_redirect_tests() {
    echo -e "\n${YELLOW}🔀 PHASE 9: OPEN REDIRECT TESTING${NC}"
    
    # Redirect payloads
    redirect_payloads=(
        "https://evil.com"
        "http://malicious-site.com"
        "//evil.com"
        "https://google.com"
    )
    
    # Common redirect parameters
    redirect_params=("url" "redirect" "return" "next" "goto" "continue" "target" "destination")
    
    local vulnerabilities_found=0
    
    for param in "${redirect_params[@]}"; do
        for payload in "${redirect_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed 's/:/%3A/g' | sed 's/\//%2F/g')
            
            # Get response headers to check for redirects
            response=$(auth_curl -s -I "${TARGET_URL}?${param}=${encoded_payload}" 2>/dev/null || echo "")
            
            # Check for redirect status codes and location header
            if echo "$response" | grep -qE "HTTP/[0-9]\.[0-9] 30[12]"; then
                location_header=$(echo "$response" | grep -i "location:" | head -1)
                if echo "$location_header" | grep -qF "$payload"; then
                    echo -e "${RED}🚨 Open redirect vulnerability in parameter: $param${NC}"
                    echo "VULN_DETAILS: OPEN_REDIRECT | Parameter: $param | Redirect: $payload"
                    echo "   RISK: Medium - Unvalidated redirect to external domain"
                    echo "   EVIDENCE: Location header contains external URL"
                    echo "   FIX: Validate redirect URLs against whitelist, use relative URLs"
                    echo ""
                    MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                    MEDIUM_RISK_LIST+=("Open Redirect in parameter '$param'")
                    vulnerabilities_found=1
                    break
                fi
            fi
        done
        if [ $vulnerabilities_found -eq 1 ]; then
            break
        fi
    done
    
    run_test "Open Redirect Protection Test" "Completed redirect testing" "Completed"
}

# Function to run SSL/TLS tests (Phase 10)
run_ssl_tests() {
    echo -e "\n${YELLOW}🔒 PHASE 10: SSL/TLS CONFIGURATION TESTING${NC}"
    
    # Only test HTTPS URLs
    if [[ "$TARGET_URL" =~ ^https:// ]]; then
        # Extract domain from URL
        domain=$(echo "$TARGET_URL" | sed 's|https\?://||' | cut -d'/' -f1 | cut -d':' -f1)
        
        # Test SSL certificate with OpenSSL
        if command -v openssl >/dev/null 2>&1; then
            # Test certificate validity with timeout
            cert_info=$(echo "" | gtimeout 10 openssl s_client -servername "$domain" -connect "$domain:443" 2>/dev/null || echo "" | timeout 10 openssl s_client -servername "$domain" -connect "$domain:443" 2>/dev/null || echo | openssl s_client -servername "$domain" -connect "$domain:443" 2>/dev/null | head -20)
            
            if echo "$cert_info" | grep -q "Verify return code: 0 (ok)"; then
                echo -e "${GREEN}✅ SSL certificate is valid${NC}"
                TESTS_PASSED=$((TESTS_PASSED + 1))
            else
                echo -e "${YELLOW}⚠️ SSL certificate issues detected${NC}"
                verify_error=$(echo "$cert_info" | grep "Verify return code:" | head -1)
                echo "   ISSUE: $verify_error"
                echo "   RISK: Medium - SSL certificate validation problems"
                echo "   FIX: Ensure valid, trusted SSL certificate is properly configured"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("SSL Certificate Validation Issues")
                TESTS_FAILED=$((TESTS_FAILED + 1))
            fi
            TESTS_TOTAL=$((TESTS_TOTAL + 1))
            
            # Check TLS version support
            if echo "$cert_info" | grep -q "Protocol.*TLSv1\.[23]"; then
                echo -e "${GREEN}✅ Modern TLS version supported${NC}"
                TESTS_PASSED=$((TESTS_PASSED + 1))
            else
                echo -e "${YELLOW}⚠️ Weak TLS version detected${NC}"
                protocol=$(echo "$cert_info" | grep "Protocol:" | head -1)
                echo "   DETECTED: $protocol"
                echo "   RISK: Medium - Weak TLS version vulnerable to attacks"
                echo "   FIX: Disable TLS 1.0/1.1, enable only TLS 1.2/1.3"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("Weak TLS Version Support")
                TESTS_FAILED=$((TESTS_FAILED + 1))
            fi
            TESTS_TOTAL=$((TESTS_TOTAL + 1))
            
            # Check for weak ciphers
            if echo "$cert_info" | grep -qiE "cipher.*rc4|cipher.*des|cipher.*md5"; then
                echo -e "${RED}❌ Weak cipher suites detected${NC}"
                weak_cipher=$(echo "$cert_info" | grep -iE "cipher.*rc4|cipher.*des|cipher.*md5" | head -1)
                echo "   DETECTED: $weak_cipher"
                echo "   RISK: High - Weak encryption vulnerable to attacks"
                echo "   FIX: Disable weak ciphers (RC4, DES, MD5), use strong AES ciphers"
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Weak Cipher Suites")
                TESTS_FAILED=$((TESTS_FAILED + 1))
            else
                echo -e "${GREEN}✅ Strong cipher suites in use${NC}"
                TESTS_PASSED=$((TESTS_PASSED + 1))
            fi
            TESTS_TOTAL=$((TESTS_TOTAL + 1))
        else
            echo -e "${BLUE}ℹ️  OpenSSL not available, skipping detailed SSL tests${NC}"
        fi
    else
        echo -e "${BLUE}ℹ️  Target uses HTTP, SSL/TLS tests not applicable${NC}"
    fi
}

# Function to run command injection tests (Phase 11)
run_command_injection_tests() {
    echo -e "\n${YELLOW}💻 PHASE 11: COMMAND INJECTION TESTING${NC}"
    
    # Command injection payloads
    command_payloads=(
        "; whoami"
        "&& id"
        "| cat /etc/passwd"
        "; ping -c 1 127.0.0.1"
        "&& sleep 5"
    )
    
    # Common vulnerable parameters
    command_params=("cmd" "exec" "system" "ping" "host" "ip" "domain" "command")
    
    local vulnerabilities_found=0
    
    for param in "${command_params[@]}"; do
        for payload in "${command_payloads[@]}"; do
            encoded_payload=$(echo "$payload" | sed 's/ /%20/g' | sed 's/;/%3B/g' | sed 's/|/%7C/g')
            
            # Time the request for sleep-based detection
            start_time=$(date +%s)
            response=$(auth_curl -s --max-time 15 "${TARGET_URL}?${param}=test${encoded_payload}" 2>/dev/null || echo "")
            end_time=$(date +%s)
            response_time=$((end_time - start_time))
            
            # Check for command output in response
            if echo "$response" | grep -qiE "uid=|gid=|root:|daemon:|PING.*127\.0\.0\.1"; then
                echo -e "${RED}🚨 Command injection vulnerability in parameter: $param${NC}"
                echo "VULN_DETAILS: COMMAND_INJECTION | Parameter: $param | Payload: $payload"
                echo "   RISK: High - Server command execution detected"
                echo "   EVIDENCE: Command output found in response"
                echo "   FIX: Never pass user input directly to system commands, use parameterization"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Command Injection in parameter '$param'")
                vulnerabilities_found=1
                break
            fi
            
            # Check for time-based injection (sleep command)
            if [[ "$payload" == *"sleep 5"* ]] && [ $response_time -ge 5 ]; then
                echo -e "${RED}🚨 Time-based command injection in parameter: $param${NC}"
                echo "VULN_DETAILS: COMMAND_INJECTION_BLIND | Parameter: $param | Delay: ${response_time}s"
                echo "   RISK: High - Blind command execution via timing attack"
                echo "   EVIDENCE: Response delayed by $response_time seconds"
                echo "   FIX: Never pass user input directly to system commands, use parameterization"
                echo ""
                HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
                HIGH_RISK_LIST+=("Time-based Command Injection in parameter '$param'")
                vulnerabilities_found=1
                break
            fi
        done
        if [ $vulnerabilities_found -eq 1 ]; then
            break
        fi
    done
    
    run_test "Command Injection Protection Test" "Completed command injection testing" "Completed"
}

# Cleanup function
cleanup() {
    rm -f /tmp/zap_startup.log 2>/dev/null
}

# Set up cleanup on script exit
trap cleanup EXIT

# Main execution
echo "GENERAL WEB APPLICATION SECURITY SCAN"
echo "Target: $TARGET_URL"
echo "Started: $(date)"
echo "========================================"

# Run tests based on mode
case $MODE in
    "quick")
        run_security_headers_test
        run_cors_tests  
        run_csrf_tests
        run_path_traversal_tests
        run_sql_injection_tests
        run_info_disclosure_tests
        run_ssl_tests
        ;;
    *)
        run_security_headers_test
        run_cors_tests
        run_xss_tests
        run_sql_injection_tests
        run_info_disclosure_tests
        run_csrf_tests
        run_path_traversal_tests
        run_open_redirect_tests
        run_ssl_tests
        run_command_injection_tests
        if [ "$MODE" != "quick" ]; then
            run_basic_zap_scan
        fi
        ;;
esac

# Generate final summary
generate_summary
