#!/bin/bash
# File Upload Security Testing Script - CWE-434, CWE-22
# Usage: ./file-upload-security-test.sh [TARGET_URL] [MODE]

set -e
[[ "$1" == "-h" ]] && { echo "📁 File Upload Security Scanner - Tests unrestricted upload, extension/MIME bypass, path traversal"; exit 0; }

TARGET_URL="${1:-}"; MODE="${2:-full}"; CLEAN_OUTPUT="${3:-false}"
UPLOAD_ENDPOINT="${UPLOAD_ENDPOINT:-/upload}"
AUTH_COOKIE="${AUTH_COOKIE:-}"
AUTH_TOKEN="${AUTH_TOKEN:-}"

[ -z "$TARGET_URL" ] && { echo "❌ Error: Target URL required"; exit 1; }
[[ ! "$TARGET_URL" =~ ^https?:// ]] && TARGET_URL="https://$TARGET_URL"
TARGET_URL="${TARGET_URL%/}"

[[ "$CLEAN_OUTPUT" = "true" ]] && { RED=''; GREEN=''; YELLOW=''; BLUE=''; PURPLE=''; CYAN=''; NC=''; } || \
{ RED='\033[0;31m'; GREEN='\033[0;32m'; YELLOW='\033[1;33m'; BLUE='\033[0;34m'; PURPLE='\033[0;35m'; CYAN='\033[0;36m'; NC='\033[0m'; }

TESTS_PASSED=0; TESTS_FAILED=0; TESTS_TOTAL=0; HIGH_RISK_FINDINGS=0; HIGH_RISK_LIST=()
AUTH_ERRORS=0  # Track 401/403 responses

# Helper function for authenticated curl requests (supports cookie and bearer token)
auth_curl() {
    if [ -n "$AUTH_TOKEN" ]; then
        curl -H "Authorization: Bearer $AUTH_TOKEN" "$@"
    elif [ -n "$AUTH_COOKIE" ]; then
        curl --cookie "$AUTH_COOKIE" "$@"
    else
        curl "$@"
    fi
}

echo -e "${PURPLE}📁 FILE UPLOAD SECURITY SCANNER${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${UPLOAD_ENDPOINT} | Mode: ${MODE}${NC}"
[ -n "$AUTH_TOKEN" ] && echo -e "${GREEN}🔐 Bearer token authentication enabled${NC}"
[ -n "$AUTH_COOKIE" ] && echo -e "${GREEN}🔐 Cookie authentication enabled${NC}"
[ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ] && echo -e "${YELLOW}ℹ️ Running without authentication${NC}"

echo -e ""

# Create test files
create_test_files() {
    TEMP_DIR=$(mktemp -d)
    echo '<?php echo "VULN_MARKER"; ?>' > "$TEMP_DIR/test.php"
    echo '<?php echo "VULN_MARKER"; ?>' > "$TEMP_DIR/test.php.jpg"
    echo '<?php echo "VULN_MARKER"; ?>' > "$TEMP_DIR/test.phtml"
    echo '<script>alert("XSS")</script>' > "$TEMP_DIR/test.html"
    echo '<script>alert("XSS")</script>' > "$TEMP_DIR/test.svg"
    echo '#!/bin/bash' > "$TEMP_DIR/test.sh"
    printf 'GIF89a<?php echo "VULN_MARKER"; ?>' > "$TEMP_DIR/polyglot.gif"
}

cleanup() { rm -rf "$TEMP_DIR" 2>/dev/null; }
trap cleanup EXIT

test_unrestricted_upload() {
    echo -e "\n${YELLOW}📤 PHASE 1: UNRESTRICTED FILE UPLOAD${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local vuln=false
    
    local dangerous_files=("test.php" "test.phtml" "test.html" "test.sh")
    for file in "${dangerous_files[@]}"; do
        local response=$(auth_curl -s -X POST "${TARGET_URL}${UPLOAD_ENDPOINT}" \
            -F "file=@${TEMP_DIR}/${file}" --max-time 15 2>/dev/null || echo "")
        
        if echo "$response" | grep -qiE "success|uploaded|saved|created|url|path"; then
            vuln=true
            echo -e "${RED}  ⚠️ Dangerous file accepted: ${file}${NC}"
        fi
    done
    
    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: Unrestricted file upload${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Unrestricted File Upload (CWE-434)")
    else
        echo -e "${GREEN}✅ PASS: Dangerous files blocked${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_extension_bypass() {
    echo -e "\n${YELLOW}🔄 PHASE 2: EXTENSION BYPASS${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local vuln=false
    
    local bypass_files=("test.php.jpg" "test.php%00.jpg" "test.php;.jpg" "test.pHp" "test.php5" "test.phar")
    for file in "${bypass_files[@]}"; do
        cp "$TEMP_DIR/test.php" "$TEMP_DIR/$file" 2>/dev/null
        local response=$(auth_curl -s -X POST "${TARGET_URL}${UPLOAD_ENDPOINT}" \
            -F "file=@${TEMP_DIR}/${file}" --max-time 15 2>/dev/null || echo "")
        
        if echo "$response" | grep -qiE "success|uploaded|saved"; then
            vuln=true
            echo -e "${RED}  ⚠️ Extension bypass: ${file}${NC}"
        fi
    done
    
    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: Extension bypass possible${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Extension Bypass (CWE-434)")
    else
        echo -e "${GREEN}✅ PASS: Extension bypass blocked${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_mime_type_bypass() {
    echo -e "\n${YELLOW}📋 PHASE 3: MIME TYPE BYPASS${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local vuln=false
    
    # Upload PHP with image MIME type
    local response=$(auth_curl -s -X POST "${TARGET_URL}${UPLOAD_ENDPOINT}" \
        -F "file=@${TEMP_DIR}/test.php;type=image/jpeg" --max-time 15 2>/dev/null || echo "")
    
    if echo "$response" | grep -qiE "success|uploaded|saved"; then
        vuln=true
        echo -e "${RED}  ⚠️ MIME type bypass accepted${NC}"
    fi
    
    # Test polyglot
    response=$(auth_curl -s -X POST "${TARGET_URL}${UPLOAD_ENDPOINT}" \
        -F "file=@${TEMP_DIR}/polyglot.gif" --max-time 15 2>/dev/null || echo "")
    
    if echo "$response" | grep -qiE "success|uploaded|saved"; then
        echo -e "${YELLOW}  ⚠️ Polyglot file accepted (may need manual verification)${NC}"
    fi
    
    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: MIME type bypass possible${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("MIME Type Bypass (CWE-434)")
    else
        echo -e "${GREEN}✅ PASS: MIME type validation working${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

test_path_traversal() {
    echo -e "\n${YELLOW}📂 PHASE 4: PATH TRAVERSAL IN FILENAME${NC}"
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local vuln=false
    
    local traversal_names=("../test.txt" "..\\test.txt" "....//test.txt" "%2e%2e/test.txt" "..%252ftest.txt")
    for name in "${traversal_names[@]}"; do
        local response=$(auth_curl -s -X POST "${TARGET_URL}${UPLOAD_ENDPOINT}" \
            -F "file=@${TEMP_DIR}/test.php;filename=${name}" --max-time 15 2>/dev/null || echo "")
        
        if echo "$response" | grep -qiE "success|uploaded|saved"; then
            # Check if path was sanitized
            if echo "$response" | grep -qE "\.\./|\.\.\\\\"; then
                vuln=true
                echo -e "${RED}  ⚠️ Path traversal accepted: ${name}${NC}"
            fi
        fi
    done
    
    if [ "$vuln" = true ]; then
        echo -e "${RED}❌ FAIL: Path traversal in filename${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Path Traversal Upload (CWE-22)")
    else
        echo -e "${GREEN}✅ PASS: Path traversal blocked${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

generate_summary() {
    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}📊 FILE UPLOAD SECURITY SUMMARY${NC}"
    local rate=0; [ $TESTS_TOTAL -gt 0 ] && rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    echo -e "Tests: ${TESTS_TOTAL} | ${GREEN}Passed: ${TESTS_PASSED}${NC} | ${RED}Failed: ${TESTS_FAILED}${NC} | Rate: ${rate}%"
    echo -e "${RED}High Risk: ${HIGH_RISK_FINDINGS}${NC}"
    [ ${#HIGH_RISK_LIST[@]} -gt 0 ] && { echo -e "${RED}🚨 FINDINGS:${NC}"; for f in "${HIGH_RISK_LIST[@]}"; do echo -e "  ${RED}• ${f}${NC}"; done; }

    # Warn about authentication errors
    if [ $AUTH_ERRORS -gt 0 ]; then
        echo -e "\n${YELLOW}⚠️  WARNING: ${AUTH_ERRORS} requests returned 401/403 Unauthorized${NC}"
        if [ -z "$AUTH_TOKEN" ] && [ -z "$AUTH_COOKIE" ]; then
            echo -e "${YELLOW}   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints.${NC}"
        else
            echo -e "${YELLOW}   Provided credentials may be invalid or expired.${NC}"
        fi
    fi

    local score=$((100 - HIGH_RISK_FINDINGS * 25)); [ $score -lt 0 ] && score=0
    echo -e "Score: $score/100"
    echo -e "${PURPLE}======================================================${NC}"
}

# Test upload endpoints discovered by ZAP spider
test_spider_upload_endpoints() {
    if [ -z "${ZAP_URLS_FILE:-}" ] || [ ! -f "$ZAP_URLS_FILE" ]; then
        return
    fi

    # Filter for URLs containing upload/file/attach/import keywords
    local upload_urls=$(grep -iE "upload|file|attach|import|media|image|document" "$ZAP_URLS_FILE" 2>/dev/null | head -10)
    local url_count=$(echo "$upload_urls" | grep -c '.' 2>/dev/null || echo "0")

    if [ "$url_count" -eq 0 ]; then
        return
    fi

    echo -e "\n${YELLOW}📤 PHASE 5: UPLOAD ENDPOINTS FROM SPIDER${NC}"
    echo -e "${CYAN}Testing ${url_count} upload-related URLs discovered by ZAP spider...${NC}"

    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    local spider_upload_vuln=false

    while IFS= read -r upload_url; do
        [ -z "$upload_url" ] && continue

        # Try uploading a dangerous file to this endpoint
        local response=$(auth_curl -s -X POST "$upload_url" \
            -F "file=@${TEMP_DIR}/test.php" --max-time 15 2>/dev/null || echo "")

        if echo "$response" | grep -qiE "success|uploaded|saved|created|url|path"; then
            spider_upload_vuln=true
            echo -e "${RED}  ⚠️ Dangerous file accepted at spider URL: ${upload_url}${NC}"
        fi
    done <<< "$upload_urls"

    if [ "$spider_upload_vuln" = true ]; then
        echo -e "${RED}❌ FAIL: Unrestricted upload at spider-discovered endpoint${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1)); HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Unrestricted Upload at Spider URL (CWE-434)")
    else
        echo -e "${GREEN}✅ PASS: Spider upload endpoints protected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

create_test_files
test_unrestricted_upload
test_extension_bypass
[ "$MODE" = "full" ] && { test_mime_type_bypass; test_path_traversal; }
test_spider_upload_endpoints
generate_summary
