# File Upload Security Testing Script (PowerShell) - CWE-434, CWE-22
param([string]$TargetUrl, [ValidateSet("quick","full")][string]$Mode="full", [string]$UploadEndpoint="/upload", [switch]$Help)

if ($Help) { Write-Host "[*] File Upload Security Scanner"; exit 0 }
if ([string]::IsNullOrEmpty($TargetUrl)) { Write-Host "Error: TargetUrl required" -ForegroundColor Red; exit 1 }
if ($TargetUrl -notmatch "^https?://") { $TargetUrl = "https://$TargetUrl" }
$TargetUrl = $TargetUrl.TrimEnd('/')

# Authentication support (Cookie or Bearer token)
$AuthCookie = $env:AUTH_COOKIE
$AuthToken = $env:AUTH_TOKEN

# Helper function for authenticated REST requests (file uploads)
function Invoke-AuthRestMethod {
    param([string]$Uri, [string]$Method = "POST", [hashtable]$Form, [int]$TimeoutSec = 15)
    $params = @{ Uri = $Uri; Method = $Method; TimeoutSec = $TimeoutSec; ErrorAction = "SilentlyContinue" }
    if ($Form) { $params.Form = $Form }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) {
                try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {}
            }
        }
        $params.WebSession = $session
    }
    Invoke-RestMethod @params
}

$script:TestsPassed=0; $script:TestsFailed=0; $script:TestsTotal=0; $script:HighRiskFindings=0; $script:HighRiskList=@()

Write-Host "`n[*] FILE UPLOAD SECURITY SCANNER" -ForegroundColor Magenta
Write-Host "Target: $TargetUrl$UploadEndpoint | Mode: $Mode" -ForegroundColor Blue
if ($AuthToken) { Write-Host "[*] Bearer token authentication enabled" -ForegroundColor Green }
if ($AuthCookie -and -not $AuthToken) { Write-Host "[*] Cookie authentication enabled" -ForegroundColor Green }

# Auth verification helper
function Invoke-AuthWebRequest {
    param([string]$Uri, [int]$TimeoutSec = 5)
    $params = @{ Uri = $Uri; TimeoutSec = $TimeoutSec; UseBasicParsing = $true; ErrorAction = "SilentlyContinue" }
    $headers = @{}
    if ($AuthToken) { $headers["Authorization"] = "Bearer $AuthToken" }
    if ($headers.Count -gt 0) { $params.Headers = $headers }
    if ($AuthCookie -and -not $AuthToken) {
        $session = New-Object Microsoft.PowerShell.Commands.WebRequestSession
        $AuthCookie -split ';' | ForEach-Object {
            $parts = $_.Trim() -split '=', 2
            if ($parts.Count -eq 2) { try { $session.Cookies.Add((New-Object System.Net.Cookie($parts[0], $parts[1], "/", ([Uri]$Uri).Host))) } catch {} }
        }
        $params.WebSession = $session
    }
    Invoke-WebRequest @params
}

if (-not $AuthToken -and -not $AuthCookie) { Write-Host "[i] Running without authentication (unauthenticated scan)" -ForegroundColor Yellow }

$script:AuthErrors=0  # Track 401/403 responses

Write-Host ""

$TempDir = New-Item -ItemType Directory -Path ([System.IO.Path]::GetTempPath() + [System.Guid]::NewGuid().ToString()) -Force

function New-TestFiles {
    '<?php echo "VULN"; ?>' | Out-File -FilePath "$TempDir\test.php" -Encoding ascii
    '<?php echo "VULN"; ?>' | Out-File -FilePath "$TempDir\test.php.jpg" -Encoding ascii
    '<script>alert(1)</script>' | Out-File -FilePath "$TempDir\test.html" -Encoding ascii
    'test content' | Out-File -FilePath "$TempDir\test.txt" -Encoding ascii
    # Polyglot GIF: GIF89a header followed by PHP code
    $polyglotBytes = [byte[]]@(0x47,0x49,0x46,0x38,0x39,0x61) + [System.Text.Encoding]::ASCII.GetBytes('<?php echo "VULN"; ?>')
    [System.IO.File]::WriteAllBytes("$TempDir\polyglot.gif", $polyglotBytes)
}

function Test-UnrestrictedUpload {
    Write-Host "`n[*] PHASE 1: UNRESTRICTED UPLOAD" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $files = @("test.php", "test.html")
    foreach ($file in $files) {
        try {
            $filePath = "$TempDir\$file"
            $form = @{ file = Get-Item -Path $filePath }
            $r = Invoke-AuthRestMethod -Uri "$TargetUrl$UploadEndpoint" -Method Post -Form $form -TimeoutSec 15
            if ($r -match "success|uploaded|saved") { $vuln = $true; Write-Host "  [!] Accepted: $file" -ForegroundColor Red }
        } catch {}
    }
    if ($vuln) {
        Write-Host "[-] FAIL: Unrestricted upload" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Unrestricted Upload"
    } else { Write-Host "[+] PASS: Blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

function Test-ExtensionBypass {
    Write-Host "`n[*] PHASE 2: EXTENSION BYPASS" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false
    $bypasses = @("test.php.jpg", "test.pHp", "test.php5")
    foreach ($bypass in $bypasses) {
        Copy-Item "$TempDir\test.php" "$TempDir\$bypass" -Force
        try {
            $form = @{ file = Get-Item -Path "$TempDir\$bypass" }
            $r = Invoke-AuthRestMethod -Uri "$TargetUrl$UploadEndpoint" -Method Post -Form $form -TimeoutSec 15
            if ($r -match "success|uploaded") { $vuln = $true; Write-Host "  [!] Bypass: $bypass" -ForegroundColor Red }
        } catch {}
    }
    if ($vuln) {
        Write-Host "[-] FAIL: Extension bypass" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Extension Bypass"
    } else { Write-Host "[+] PASS: Blocked" -ForegroundColor Green; $script:TestsPassed++ }
}

# Phase 3: MIME Type Bypass
function Test-MimeTypeBypass {
    Write-Host "`n[*] PHASE 3: MIME TYPE BYPASS" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false

    try {
        # Upload PHP file with image/jpeg Content-Type using HttpClient for precise control
        $httpClient = New-Object System.Net.Http.HttpClient
        if ($AuthToken) { $httpClient.DefaultRequestHeaders.Authorization = New-Object System.Net.Http.Headers.AuthenticationHeaderValue("Bearer", $AuthToken) }

        $uploadUri = "$TargetUrl$UploadEndpoint"

        # Test 1: PHP with image/jpeg MIME type
        $phpContent = [System.IO.File]::ReadAllBytes("$TempDir\test.php")
        $multipartContent = New-Object System.Net.Http.MultipartFormDataContent
        $byteContent = New-Object System.Net.Http.ByteArrayContent($phpContent)
        $byteContent.Headers.ContentType = [System.Net.Http.Headers.MediaTypeHeaderValue]::Parse("image/jpeg")
        $multipartContent.Add($byteContent, "file", "test.php")

        $response = $httpClient.PostAsync($uploadUri, $multipartContent).Result
        $responseBody = $response.Content.ReadAsStringAsync().Result
        if ($responseBody -match "success|uploaded|saved") {
            $vuln = $true
            Write-Host "  [!] MIME type bypass accepted (PHP as image/jpeg)" -ForegroundColor Red
        }

        # Test 2: Polyglot file (GIF header + PHP)
        $polyglotContent = [System.IO.File]::ReadAllBytes("$TempDir\polyglot.gif")
        $multipartContent2 = New-Object System.Net.Http.MultipartFormDataContent
        $byteContent2 = New-Object System.Net.Http.ByteArrayContent($polyglotContent)
        $byteContent2.Headers.ContentType = [System.Net.Http.Headers.MediaTypeHeaderValue]::Parse("image/gif")
        $multipartContent2.Add($byteContent2, "file", "polyglot.gif")

        $response2 = $httpClient.PostAsync($uploadUri, $multipartContent2).Result
        $responseBody2 = $response2.Content.ReadAsStringAsync().Result
        if ($responseBody2 -match "success|uploaded|saved") {
            Write-Host "  [!] Polyglot file accepted (may need manual verification)" -ForegroundColor Yellow
        }

        $httpClient.Dispose()
    } catch {
        Write-Host "  [i] MIME type bypass test error: $($_.Exception.Message)" -ForegroundColor Gray
    }

    if ($vuln) {
        Write-Host "[-] FAIL: MIME type bypass possible" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "MIME Type Bypass (CWE-434)"
    } else {
        Write-Host "[+] PASS: MIME type validation working" -ForegroundColor Green; $script:TestsPassed++
    }
}

# Phase 4: File Path Traversal in filename
function Test-FilePathTraversal {
    Write-Host "`n[*] PHASE 4: PATH TRAVERSAL IN FILENAME" -ForegroundColor Yellow
    $script:TestsTotal++; $vuln = $false

    $traversalNames = @("../test.txt", "..\test.txt", "....//test.txt", "%2e%2e/test.txt", "..%252ftest.txt")

    try {
        $httpClient = New-Object System.Net.Http.HttpClient
        if ($AuthToken) { $httpClient.DefaultRequestHeaders.Authorization = New-Object System.Net.Http.Headers.AuthenticationHeaderValue("Bearer", $AuthToken) }

        $uploadUri = "$TargetUrl$UploadEndpoint"
        $fileContent = [System.IO.File]::ReadAllBytes("$TempDir\test.txt")

        foreach ($name in $traversalNames) {
            try {
                $multipartContent = New-Object System.Net.Http.MultipartFormDataContent
                $byteContent = New-Object System.Net.Http.ByteArrayContent($fileContent)
                $byteContent.Headers.ContentType = [System.Net.Http.Headers.MediaTypeHeaderValue]::Parse("text/plain")
                $multipartContent.Add($byteContent, "file", $name)

                $response = $httpClient.PostAsync($uploadUri, $multipartContent).Result
                $responseBody = $response.Content.ReadAsStringAsync().Result
                if ($responseBody -match "success|uploaded|saved") {
                    # Check if path was NOT sanitized
                    if ($responseBody -match '\.\./|\.\.\\') {
                        $vuln = $true
                        Write-Host "  [!] Path traversal accepted: $name" -ForegroundColor Red
                    }
                }
            } catch { }
        }

        $httpClient.Dispose()
    } catch {
        Write-Host "  [i] Path traversal test error: $($_.Exception.Message)" -ForegroundColor Gray
    }

    if ($vuln) {
        Write-Host "[-] FAIL: Path traversal in filename" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Path Traversal Upload (CWE-22)"
    } else {
        Write-Host "[+] PASS: Path traversal blocked" -ForegroundColor Green; $script:TestsPassed++
    }
}

# Phase 5: Test upload endpoints discovered by ZAP spider
function Test-SpiderUploadEndpoints {
    $zapUrlsFile = $env:ZAP_URLS_FILE
    if (-not $zapUrlsFile -or -not (Test-Path $zapUrlsFile)) { return }

    # Filter for URLs containing upload/file/attach/import keywords
    $uploadUrls = Get-Content $zapUrlsFile -ErrorAction SilentlyContinue | Where-Object {
        $_ -match 'upload|file|attach|import|media|image|document'
    } | Select-Object -First 10

    if (-not $uploadUrls -or $uploadUrls.Count -eq 0) { return }

    Write-Host "`n[*] PHASE 5: UPLOAD ENDPOINTS FROM SPIDER" -ForegroundColor Yellow
    Write-Host "[*] Testing $($uploadUrls.Count) upload-related URLs discovered by ZAP spider..." -ForegroundColor Cyan

    $script:TestsTotal++
    $spiderUploadVuln = $false

    foreach ($uploadUrl in $uploadUrls) {
        if ([string]::IsNullOrWhiteSpace($uploadUrl)) { continue }

        try {
            $form = @{ file = Get-Item -Path "$TempDir\test.php" }
            $r = Invoke-AuthRestMethod -Uri $uploadUrl -Method Post -Form $form -TimeoutSec 15
            if ($r -match "success|uploaded|saved|created|url|path") {
                $spiderUploadVuln = $true
                Write-Host "  [!] Dangerous file accepted at spider URL: $uploadUrl" -ForegroundColor Red
            }
        } catch { }
    }

    if ($spiderUploadVuln) {
        Write-Host "[-] FAIL: Unrestricted upload at spider-discovered endpoint" -ForegroundColor Red
        $script:TestsFailed++; $script:HighRiskFindings++; $script:HighRiskList += "Unrestricted Upload at Spider URL (CWE-434)"
    } else {
        Write-Host "[+] PASS: Spider upload endpoints protected" -ForegroundColor Green; $script:TestsPassed++
    }
}

function Show-Summary {
    Write-Host "`n======================================================" -ForegroundColor Magenta
    Write-Host " FILE UPLOAD SUMMARY" -ForegroundColor Magenta
    $rate = if($script:TestsTotal -gt 0){[Math]::Round(($script:TestsPassed/$script:TestsTotal)*100)}else{0}
    Write-Host "Tests: $($script:TestsTotal) | Passed: $($script:TestsPassed) | Failed: $($script:TestsFailed) | Rate: $rate%"
    if ($script:HighRiskList.Count -gt 0) { $script:HighRiskList | ForEach-Object { Write-Host "  [!] $_" -ForegroundColor Red } }

    # Warn about authentication errors
    if ($script:AuthErrors -gt 0) {
        Write-Host ""
        Write-Host "[!]  WARNING: $($script:AuthErrors) requests returned 401/403 Unauthorized" -ForegroundColor Yellow
        if (-not $AuthToken -and -not $AuthCookie) {
            Write-Host "   Results may have FALSE NEGATIVES. Provide credentials to test protected endpoints." -ForegroundColor Yellow
        } else {
            Write-Host "   Provided credentials may be invalid or expired." -ForegroundColor Yellow
        }
    }

    $score = 100 - ($script:HighRiskFindings * 25); if($score -lt 0){$score=0}
    Write-Host "Score: $score/100" -ForegroundColor $(if($score -ge 80){"Green"}else{"Red"})
    Remove-Item -Path $TempDir -Recurse -Force -ErrorAction SilentlyContinue
}

New-TestFiles
Test-UnrestrictedUpload
Test-ExtensionBypass
if ($Mode -eq "full") { Test-MimeTypeBypass; Test-FilePathTraversal }
Test-SpiderUploadEndpoints
Show-Summary
