﻿<#
.SYNOPSIS
    Dependency Security Testing Script for Windows/PowerShell
    Tests for outdated dependencies and known CVEs.

.DESCRIPTION
    This script performs comprehensive dependency security testing.
    It covers CWE-1104 (Use of Unmaintained Third Party Components)
    and CWE-1035 (Using Components with Known Vulnerabilities).

.PARAMETER ProjectPath
    The path to the project to scan.

.PARAMETER Mode
    Test mode: 'quick' for essential tests only, 'full' for comprehensive testing.

.PARAMETER ScanType
    Scan type: 'auto', 'npm', 'pip', 'composer', 'maven', 'gradle', 'ruby', 'dotnet'

.PARAMETER CleanOutput
    Suppress colored output for logging purposes.

.PARAMETER Help
    Display help information.

.EXAMPLE
    .\dependency-security-test.ps1 -ProjectPath "C:\Projects\MyApp"
    .\dependency-security-test.ps1 -ProjectPath "." -Mode full
#>

param(
    [Parameter(Position=0)]
    [string]$ProjectPath,

    [Parameter()]
    [string]$TargetUrl,

    [Parameter()]
    [ValidateSet('quick', 'full')]
    [string]$Mode = 'quick',

    [Parameter()]
    [ValidateSet('auto', 'npm', 'pip', 'composer', 'maven', 'gradle', 'ruby', 'dotnet', 'cargo', 'go')]
    [string]$ScanType = 'auto',

    [Parameter()]
    [switch]$CleanOutput,

    [Parameter()]
    [switch]$Help
)

# If TargetUrl is provided but ProjectPath is not, use current directory
if ($TargetUrl -and -not $ProjectPath) {
    $ProjectPath = Get-Location
}

# Color output functions
function Write-Success { param([string]$Message) if (-not $CleanOutput) { Write-Host "[+] $Message" -ForegroundColor Green } else { Write-Host "[+] $Message" } }
function Write-Failure { param([string]$Message) if (-not $CleanOutput) { Write-Host "[-] $Message" -ForegroundColor Red } else { Write-Host "[-] $Message" } }
function Write-Info { param([string]$Message) if (-not $CleanOutput) { Write-Host "[*] $Message" -ForegroundColor Blue } else { Write-Host "[*] $Message" } }
function Write-Warning { param([string]$Message) if (-not $CleanOutput) { Write-Host "[!] $Message" -ForegroundColor Yellow } else { Write-Host "[!] $Message" } }
function Write-Phase { param([string]$Message) if (-not $CleanOutput) { Write-Host "`n=== $Message ===" -ForegroundColor Cyan } else { Write-Host "`n=== $Message ===" } }

# Test tracking variables
$script:TESTS_PASSED = 0
$script:TESTS_FAILED = 0
$script:TESTS_TOTAL = 0
$script:HIGH_RISK_FINDINGS = @()
$script:MEDIUM_RISK_FINDINGS = @()
$script:LOW_RISK_FINDINGS = @()
$script:DetectedTypes = @()

function Show-Help {
    Write-Host @"

Dependency Security Testing Script
===================================

Usage: .\dependency-security-test.ps1 -ProjectPath <path> [options]

Options:
    -ProjectPath    Path to the project to scan (required)
    -Mode           Test mode: 'quick' or 'full' (default: quick)
    -ScanType       Scan type: 'auto', 'npm', 'pip', 'composer', etc. (default: auto)
    -CleanOutput    Suppress colored output
    -Help           Show this help message

Examples:
    .\dependency-security-test.ps1 -ProjectPath "C:\Projects\MyApp"
    .\dependency-security-test.ps1 -ProjectPath "." -Mode full
    .\dependency-security-test.ps1 -ProjectPath "." -ScanType npm

Test Phases:
    Phase 1: Dependency Discovery
    Phase 2: Known Vulnerability Scanning (CWE-1035)
    Phase 3: Outdated Dependency Detection (CWE-1104)

Supported Package Managers:
    - npm/yarn (Node.js)
    - pip (Python)
    - composer (PHP)
    - maven/gradle (Java)
    - bundler (Ruby)
    - NuGet (.NET)
    - cargo (Rust)
    - go modules (Go)

"@
}

function Get-FindingDetails {
    param([string]$FindingType)
    
    switch ($FindingType) {
        "known_vuln" {
            return @{
                CWE = "CWE-1035"
                Risk = "HIGH"
                Description = "Using Components with Known Vulnerabilities"
                Impact = "Remote code execution, data breach, privilege escalation"
                Remediation = "Update to patched version immediately"
            }
        }
        "outdated" {
            return @{
                CWE = "CWE-1104"
                Risk = "MEDIUM"
                Description = "Use of Unmaintained Third Party Components"
                Impact = "Missing security patches, compatibility issues"
                Remediation = "Update to latest stable version"
            }
        }
        default {
            return @{
                CWE = "Unknown"
                Risk = "INFO"
                Description = "Unknown vulnerability"
                Impact = "Unknown"
                Remediation = "Further investigation required"
            }
        }
    }
}

function Get-ProjectTypes {
    param([string]$Path)
    
    $types = @()
    
    # Node.js
    if (Test-Path "$Path\package.json") {
        $types += "npm"
    }
    
    # Python
    if ((Test-Path "$Path\requirements.txt") -or (Test-Path "$Path\Pipfile") -or (Test-Path "$Path\pyproject.toml")) {
        $types += "pip"
    }
    
    # PHP
    if (Test-Path "$Path\composer.json") {
        $types += "composer"
    }
    
    # Java Maven
    if (Test-Path "$Path\pom.xml") {
        $types += "maven"
    }
    
    # Java Gradle
    if ((Test-Path "$Path\build.gradle") -or (Test-Path "$Path\build.gradle.kts")) {
        $types += "gradle"
    }
    
    # Ruby
    if (Test-Path "$Path\Gemfile") {
        $types += "ruby"
    }
    
    # .NET
    if ((Get-ChildItem -Path $Path -Filter "*.csproj" -ErrorAction SilentlyContinue) -or 
        (Get-ChildItem -Path $Path -Filter "*.fsproj" -ErrorAction SilentlyContinue) -or
        (Test-Path "$Path\packages.config")) {
        $types += "dotnet"
    }
    
    # Rust
    if (Test-Path "$Path\Cargo.toml") {
        $types += "cargo"
    }
    
    # Go
    if (Test-Path "$Path\go.mod") {
        $types += "go"
    }
    
    return $types
}

function Invoke-DependencyDiscovery {
    Write-Phase "Phase 1: Dependency Discovery"
    
    if ($ScanType -eq 'auto') {
        $script:DetectedTypes = Get-ProjectTypes -Path $ProjectPath
        if ($script:DetectedTypes.Count -eq 0) {
            Write-Warning "No recognized package manager files found"
            return $false
        }
        Write-Info "Detected project types: $($script:DetectedTypes -join ', ')"
    } else {
        $script:DetectedTypes = @($ScanType)
        Write-Info "Using specified type: $ScanType"
    }
    
    foreach ($projectType in $script:DetectedTypes) {
        $script:TESTS_TOTAL++
        Write-Info "Scanning $projectType dependencies..."
        
        switch ($projectType) {
            "npm" {
                if ((Test-Path "$ProjectPath\package-lock.json") -or (Test-Path "$ProjectPath\yarn.lock")) {
                    try {
                        $npmList = & npm ls --all --json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
                        $depCount = ($npmList | ConvertTo-Json -Depth 100 | Select-String -Pattern '"version"' -AllMatches).Matches.Count
                        Write-Success "Found approximately $depCount npm dependencies (including transitive)"
                        $script:TESTS_PASSED++
                    } catch {
                        Write-Success "npm project detected"
                        $script:TESTS_PASSED++
                    }
                } else {
                    Write-Warning "No package-lock.json or yarn.lock found - run 'npm install' first"
                    $script:TESTS_FAILED++
                }
            }
            "pip" {
                if (Test-Path "$ProjectPath\requirements.txt") {
                    $depCount = (Get-Content "$ProjectPath\requirements.txt" | Where-Object { $_ -match '\S' -and $_ -notmatch '^#' }).Count
                    Write-Success "Found $depCount direct Python dependencies"
                    $script:TESTS_PASSED++
                } elseif (Test-Path "$ProjectPath\Pipfile.lock") {
                    Write-Success "Pipfile.lock found"
                    $script:TESTS_PASSED++
                } else {
                    Write-Warning "No requirements.txt or Pipfile.lock found"
                    $script:TESTS_FAILED++
                }
            }
            "composer" {
                if (Test-Path "$ProjectPath\composer.lock") {
                    $composerLock = Get-Content "$ProjectPath\composer.lock" | ConvertFrom-Json -ErrorAction SilentlyContinue
                    $depCount = $composerLock.packages.Count
                    Write-Success "Found $depCount PHP dependencies"
                    $script:TESTS_PASSED++
                } else {
                    Write-Warning "No composer.lock found - run 'composer install' first"
                    $script:TESTS_FAILED++
                }
            }
            "dotnet" {
                $csprojFiles = Get-ChildItem -Path $ProjectPath -Filter "*.csproj" -Recurse -ErrorAction SilentlyContinue
                if ($csprojFiles) {
                    Write-Success "Found $($csprojFiles.Count) .NET project file(s)"
                    $script:TESTS_PASSED++
                } else {
                    Write-Warning "No .NET project files found"
                    $script:TESTS_FAILED++
                }
            }
            "ruby" {
                if (Test-Path "$ProjectPath\Gemfile.lock") {
                    $gemContent = Get-Content "$ProjectPath\Gemfile.lock"
                    $depCount = ($gemContent | Where-Object { $_ -match '^\s{4}[a-z]' }).Count
                    Write-Success "Found approximately $depCount Ruby gems"
                    $script:TESTS_PASSED++
                } else {
                    Write-Warning "No Gemfile.lock found - run 'bundle install' first"
                    $script:TESTS_FAILED++
                }
            }
            "cargo" {
                if (Test-Path "$ProjectPath\Cargo.lock") {
                    $cargoContent = Get-Content "$ProjectPath\Cargo.lock"
                    $depCount = ($cargoContent | Where-Object { $_ -match '^\[\[package\]\]' }).Count
                    Write-Success "Found approximately $depCount Rust crates"
                    $script:TESTS_PASSED++
                } else {
                    Write-Warning "No Cargo.lock found - run 'cargo build' first"
                    $script:TESTS_FAILED++
                }
            }
            "go" {
                if (Test-Path "$ProjectPath\go.sum") {
                    $goSum = Get-Content "$ProjectPath\go.sum"
                    $depCount = [math]::Floor($goSum.Count / 2)
                    Write-Success "Found approximately $depCount Go modules"
                    $script:TESTS_PASSED++
                } else {
                    Write-Warning "No go.sum found - run 'go mod tidy' first"
                    $script:TESTS_FAILED++
                }
            }
            default {
                Write-Warning "Unknown project type: $projectType"
                $script:TESTS_FAILED++
            }
        }
    }
    
    return $true
}

function Invoke-VulnerabilityScan {
    Write-Phase "Phase 2: Known Vulnerability Scanning (CWE-1035)"
    
    foreach ($projectType in $script:DetectedTypes) {
        switch ($projectType) {
            "npm" { Invoke-NpmAudit }
            "pip" { Invoke-PipAudit }
            "composer" { Invoke-ComposerAudit }
            "dotnet" { Invoke-DotnetAudit }
            "ruby" { Invoke-RubyAudit }
            "cargo" { Invoke-CargoAudit }
            "go" { Invoke-GoVulncheck }
        }
    }
    
    # Try Snyk if available (full mode)
    if ($Mode -eq 'full') {
        $snykPath = Get-Command snyk -ErrorAction SilentlyContinue
        if ($snykPath) {
            Invoke-SnykScan
        }
    }
}

function Invoke-NpmAudit {
    $script:TESTS_TOTAL++
    Write-Info "Running npm audit..."
    
    $npmPath = Get-Command npm -ErrorAction SilentlyContinue
    if (-not $npmPath) {
        Write-Warning "npm not installed - skipping npm vulnerability scan"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $auditOutput = & npm audit --json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
        Pop-Location
        
        if ($auditOutput -and $auditOutput.metadata) {
            $vulnerabilities = $auditOutput.metadata.vulnerabilities
            $critical = if ($vulnerabilities.critical) { $vulnerabilities.critical } else { 0 }
            $high = if ($vulnerabilities.high) { $vulnerabilities.high } else { 0 }
            $moderate = if ($vulnerabilities.moderate) { $vulnerabilities.moderate } else { 0 }
            $low = if ($vulnerabilities.low) { $vulnerabilities.low } else { 0 }
            
            $total = $critical + $high + $moderate + $low
            
            if ($total -gt 0) {
                Write-Warning "Found $total npm vulnerabilities"
                
                if ($critical -gt 0) {
                    Write-Failure "  CRITICAL: $critical"
                    $script:HIGH_RISK_FINDINGS += "$critical critical npm vulnerabilities - CWE-1035"
                }
                if ($high -gt 0) {
                    Write-Failure "  HIGH: $high"
                    $script:HIGH_RISK_FINDINGS += "$high high severity npm vulnerabilities - CWE-1035"
                }
                if ($moderate -gt 0) {
                    Write-Warning "  MODERATE: $moderate"
                    $script:MEDIUM_RISK_FINDINGS += "$moderate moderate npm vulnerabilities - CWE-1035"
                }
                if ($low -gt 0) {
                    Write-Info "  LOW: $low"
                    $script:LOW_RISK_FINDINGS += "$low low severity npm vulnerabilities - CWE-1035"
                }
                
                $script:TESTS_FAILED++
            } else {
                Write-Success "No npm vulnerabilities found"
                $script:TESTS_PASSED++
            }
        } else {
            Write-Success "No npm vulnerabilities found"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Warning "Error running npm audit: $($_.Exception.Message)"
        $script:TESTS_PASSED++
    }
}

function Invoke-PipAudit {
    $script:TESTS_TOTAL++
    Write-Info "Scanning Python dependencies for vulnerabilities..."
    
    # Try pip-audit
    $pipAuditPath = Get-Command pip-audit -ErrorAction SilentlyContinue
    if ($pipAuditPath) {
        try {
            Push-Location $ProjectPath
            $auditOutput = & pip-audit --format json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
            Pop-Location
            
            if ($auditOutput -and $auditOutput.Count -gt 0) {
                $vulnCount = $auditOutput.Count
                Write-Warning "pip-audit found $vulnCount vulnerabilities"
                $script:MEDIUM_RISK_FINDINGS += "$vulnCount Python package vulnerabilities - CWE-1035"
                $script:TESTS_FAILED++
            } else {
                Write-Success "No Python vulnerabilities found"
                $script:TESTS_PASSED++
            }
        } catch {
            Write-Warning "Error running pip-audit: $($_.Exception.Message)"
            $script:TESTS_PASSED++
        }
        return
    }
    
    # Try safety
    $safetyPath = Get-Command safety -ErrorAction SilentlyContinue
    if ($safetyPath) {
        try {
            Push-Location $ProjectPath
            $safetyOutput = & safety check --json 2>$null
            Pop-Location
            
            if ($safetyOutput -match "vulnerability") {
                Write-Warning "Safety found vulnerabilities"
                $script:MEDIUM_RISK_FINDINGS += "Python package vulnerabilities found by safety - CWE-1035"
                $script:TESTS_FAILED++
            } else {
                Write-Success "No Python vulnerabilities found"
                $script:TESTS_PASSED++
            }
        } catch {
            Write-Warning "Error running safety: $($_.Exception.Message)"
            $script:TESTS_PASSED++
        }
        return
    }
    
    Write-Warning "No Python vulnerability scanner available (install pip-audit or safety)"
    $script:TESTS_PASSED++
}

function Invoke-ComposerAudit {
    $script:TESTS_TOTAL++
    Write-Info "Scanning PHP dependencies for vulnerabilities..."
    
    $composerPath = Get-Command composer -ErrorAction SilentlyContinue
    if (-not $composerPath) {
        Write-Warning "Composer not installed - skipping PHP vulnerability scan"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $auditOutput = & composer audit --format=json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
        Pop-Location
        
        if ($auditOutput -and $auditOutput.advisories -and $auditOutput.advisories.Count -gt 0) {
            $vulnCount = $auditOutput.advisories.Count
            Write-Warning "Composer audit found $vulnCount vulnerabilities"
            $script:MEDIUM_RISK_FINDINGS += "$vulnCount PHP package vulnerabilities - CWE-1035"
            $script:TESTS_FAILED++
        } else {
            Write-Success "No PHP vulnerabilities found"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Warning "Error running composer audit"
        $script:TESTS_PASSED++
    }
}

function Invoke-DotnetAudit {
    $script:TESTS_TOTAL++
    Write-Info "Scanning .NET dependencies for vulnerabilities..."
    
    $dotnetPath = Get-Command dotnet -ErrorAction SilentlyContinue
    if (-not $dotnetPath) {
        Write-Warning "dotnet CLI not installed - skipping .NET vulnerability scan"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $auditOutput = & dotnet list package --vulnerable --format json 2>$null
        Pop-Location
        
        if ($auditOutput -match "vulnerability|vulnerable") {
            Write-Warning ".NET vulnerabilities detected"
            $script:MEDIUM_RISK_FINDINGS += ".NET package vulnerabilities found - CWE-1035"
            $script:TESTS_FAILED++
        } else {
            Write-Success "No .NET vulnerabilities found"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Warning "Error running dotnet list package --vulnerable"
        $script:TESTS_PASSED++
    }
}

function Invoke-RubyAudit {
    $script:TESTS_TOTAL++
    Write-Info "Scanning Ruby dependencies for vulnerabilities..."
    
    $bundleAuditPath = Get-Command bundle-audit -ErrorAction SilentlyContinue
    if (-not $bundleAuditPath) {
        Write-Warning "bundle-audit not installed - skipping Ruby vulnerability scan"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $auditOutput = & bundle-audit check 2>$null
        Pop-Location
        
        if ($auditOutput -match "vulnerabilities found|Insecure") {
            Write-Warning "bundle-audit found vulnerabilities"
            $script:MEDIUM_RISK_FINDINGS += "Ruby gem vulnerabilities found - CWE-1035"
            $script:TESTS_FAILED++
        } else {
            Write-Success "No Ruby vulnerabilities found"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Warning "Error running bundle-audit"
        $script:TESTS_PASSED++
    }
}

function Invoke-CargoAudit {
    $script:TESTS_TOTAL++
    Write-Info "Scanning Rust dependencies for vulnerabilities..."
    
    $cargoAuditPath = Get-Command cargo-audit -ErrorAction SilentlyContinue
    if (-not $cargoAuditPath) {
        Write-Warning "cargo-audit not installed - skipping Rust vulnerability scan"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $auditOutput = & cargo audit --json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
        Pop-Location
        
        if ($auditOutput -and $auditOutput.vulnerabilities -and $auditOutput.vulnerabilities.Count -gt 0) {
            $vulnCount = $auditOutput.vulnerabilities.Count
            Write-Warning "cargo-audit found $vulnCount vulnerabilities"
            $script:MEDIUM_RISK_FINDINGS += "$vulnCount Rust crate vulnerabilities - CWE-1035"
            $script:TESTS_FAILED++
        } else {
            Write-Success "No Rust vulnerabilities found"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Warning "Error running cargo-audit"
        $script:TESTS_PASSED++
    }
}

function Invoke-GoVulncheck {
    $script:TESTS_TOTAL++
    Write-Info "Scanning Go dependencies for vulnerabilities..."
    
    $govulncheckPath = Get-Command govulncheck -ErrorAction SilentlyContinue
    if (-not $govulncheckPath) {
        Write-Warning "govulncheck not installed - skipping Go vulnerability scan"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $vulnOutput = & govulncheck ./... 2>$null
        Pop-Location
        
        if ($vulnOutput -match "Vulnerability|GO-") {
            Write-Warning "govulncheck found vulnerabilities"
            $script:MEDIUM_RISK_FINDINGS += "Go module vulnerabilities found - CWE-1035"
            $script:TESTS_FAILED++
        } else {
            Write-Success "No Go vulnerabilities found"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Warning "Error running govulncheck"
        $script:TESTS_PASSED++
    }
}

function Invoke-SnykScan {
    $script:TESTS_TOTAL++
    Write-Info "Running Snyk vulnerability scan..."
    
    try {
        Push-Location $ProjectPath
        $snykOutput = & snyk test --json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
        Pop-Location
        
        if ($snykOutput -and $snykOutput.vulnerabilities) {
            $vulnCount = $snykOutput.vulnerabilities.Count
            $critical = ($snykOutput.vulnerabilities | Where-Object { $_.severity -eq "critical" }).Count
            $high = ($snykOutput.vulnerabilities | Where-Object { $_.severity -eq "high" }).Count
            
            Write-Warning "Snyk found $vulnCount vulnerabilities"
            
            if ($critical -gt 0 -or $high -gt 0) {
                $script:HIGH_RISK_FINDINGS += "$vulnCount vulnerabilities found by Snyk - CWE-1035"
            } else {
                $script:MEDIUM_RISK_FINDINGS += "$vulnCount vulnerabilities found by Snyk - CWE-1035"
            }
            $script:TESTS_FAILED++
        } else {
            Write-Success "Snyk found no vulnerabilities"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Info "Snyk scan completed (check auth status if no results)"
        $script:TESTS_PASSED++
    }
}

function Invoke-OutdatedCheck {
    Write-Phase "Phase 3: Outdated Dependency Detection (CWE-1104)"
    
    foreach ($projectType in $script:DetectedTypes) {
        switch ($projectType) {
            "npm" { Test-NpmOutdated }
            "pip" { Test-PipOutdated }
            "composer" { Test-ComposerOutdated }
            "dotnet" { Test-DotnetOutdated }
        }
    }
}

function Test-NpmOutdated {
    $script:TESTS_TOTAL++
    Write-Info "Checking for outdated npm packages..."
    
    $npmPath = Get-Command npm -ErrorAction SilentlyContinue
    if (-not $npmPath) {
        Write-Warning "npm not installed - skipping outdated check"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $outdatedOutput = & npm outdated --json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
        Pop-Location
        
        if ($outdatedOutput -and ($outdatedOutput | Get-Member -MemberType NoteProperty).Count -gt 0) {
            $outdatedCount = ($outdatedOutput | Get-Member -MemberType NoteProperty).Count
            Write-Warning "Found $outdatedCount outdated npm packages"
            
            if ($outdatedCount -gt 10) {
                $script:MEDIUM_RISK_FINDINGS += "$outdatedCount outdated npm packages - CWE-1104"
            } else {
                $script:LOW_RISK_FINDINGS += "$outdatedCount outdated npm packages - CWE-1104"
            }
            $script:TESTS_FAILED++
        } else {
            Write-Success "All npm packages are up to date"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Success "All npm packages are up to date"
        $script:TESTS_PASSED++
    }
}

function Test-PipOutdated {
    $script:TESTS_TOTAL++
    Write-Info "Checking for outdated Python packages..."
    
    $pipPath = Get-Command pip -ErrorAction SilentlyContinue
    if (-not $pipPath) {
        Write-Warning "pip not installed - skipping outdated check"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        $outdatedOutput = & pip list --outdated --format=json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
        
        if ($outdatedOutput -and $outdatedOutput.Count -gt 0) {
            Write-Warning "Found $($outdatedOutput.Count) outdated Python packages"
            $script:LOW_RISK_FINDINGS += "$($outdatedOutput.Count) outdated Python packages - CWE-1104"
            $script:TESTS_FAILED++
        } else {
            Write-Success "All Python packages are up to date"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Success "All Python packages are up to date"
        $script:TESTS_PASSED++
    }
}

function Test-ComposerOutdated {
    $script:TESTS_TOTAL++
    Write-Info "Checking for outdated PHP packages..."
    
    $composerPath = Get-Command composer -ErrorAction SilentlyContinue
    if (-not $composerPath) {
        Write-Warning "Composer not installed - skipping outdated check"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $outdatedOutput = & composer outdated --format=json 2>$null | ConvertFrom-Json -ErrorAction SilentlyContinue
        Pop-Location
        
        if ($outdatedOutput -and $outdatedOutput.installed) {
            $outdatedCount = $outdatedOutput.installed.Count
            Write-Warning "Found $outdatedCount outdated PHP packages"
            $script:LOW_RISK_FINDINGS += "$outdatedCount outdated PHP packages - CWE-1104"
            $script:TESTS_FAILED++
        } else {
            Write-Success "All PHP packages are up to date"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Success "All PHP packages are up to date"
        $script:TESTS_PASSED++
    }
}

function Test-DotnetOutdated {
    $script:TESTS_TOTAL++
    Write-Info "Checking for outdated .NET packages..."
    
    $dotnetPath = Get-Command dotnet -ErrorAction SilentlyContinue
    if (-not $dotnetPath) {
        Write-Warning "dotnet CLI not installed - skipping outdated check"
        $script:TESTS_PASSED++
        return
    }
    
    try {
        Push-Location $ProjectPath
        $outdatedOutput = & dotnet list package --outdated 2>$null
        Pop-Location
        
        if ($outdatedOutput -match "has the following updates") {
            Write-Warning "Outdated .NET packages detected"
            $script:LOW_RISK_FINDINGS += "Outdated .NET packages found - CWE-1104"
            $script:TESTS_FAILED++
        } else {
            Write-Success "All .NET packages are up to date"
            $script:TESTS_PASSED++
        }
    } catch {
        Write-Success "All .NET packages are up to date"
        $script:TESTS_PASSED++
    }
}

function Show-Summary {
    Write-Phase "Dependency Security Test Summary"
    
    $totalFindings = $script:HIGH_RISK_FINDINGS.Count + $script:MEDIUM_RISK_FINDINGS.Count + $script:LOW_RISK_FINDINGS.Count
    
    Write-Host "`nProject: $ProjectPath"
    Write-Host "Mode: $Mode"
    Write-Host "Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')"
    Write-Host ""
    Write-Host "Tests Passed: $($script:TESTS_PASSED)/$($script:TESTS_TOTAL)"
    Write-Host "Tests Failed: $($script:TESTS_FAILED)/$($script:TESTS_TOTAL)"
    
    if ($totalFindings -gt 0) {
        Write-Host "`n--- Findings ---" -ForegroundColor Yellow
        
        if ($script:HIGH_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nHIGH RISK:" -ForegroundColor Red
            foreach ($finding in $script:HIGH_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Red
            }
        }
        
        if ($script:MEDIUM_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nMEDIUM RISK:" -ForegroundColor Yellow
            foreach ($finding in $script:MEDIUM_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Yellow
            }
        }
        
        if ($script:LOW_RISK_FINDINGS.Count -gt 0) {
            Write-Host "`nLOW RISK:" -ForegroundColor Cyan
            foreach ($finding in $script:LOW_RISK_FINDINGS) {
                Write-Host "  - $finding" -ForegroundColor Cyan
            }
        }
    }
    
    # Calculate security score
    $maxScore = 100
    $deductions = ($script:HIGH_RISK_FINDINGS.Count * 20) + ($script:MEDIUM_RISK_FINDINGS.Count * 10) + ($script:LOW_RISK_FINDINGS.Count * 5)
    $score = [Math]::Max(0, $maxScore - $deductions)
    
    Write-Host "`n--- Security Score ---"
    $scoreColor = if ($score -ge 80) { "Green" } elseif ($score -ge 60) { "Yellow" } else { "Red" }
    Write-Host "Score: $score/100" -ForegroundColor $scoreColor
    
    Write-Host "`n--- CWEs Tested ---"
    Write-Host "CWE-1035: Using Components with Known Vulnerabilities"
    Write-Host "CWE-1104: Use of Unmaintained Third Party Components"
    
    Write-Host "`n--- Recommendations ---"
    Write-Host "1. Run 'npm audit fix' / 'pip-audit --fix' to auto-fix vulnerabilities"
    Write-Host "2. Review and update dependencies regularly"
    Write-Host "3. Use dependabot or renovate for automated updates"
    Write-Host "4. Consider using lockfiles for reproducible builds"
}

# Main execution
if ($Help) {
    Show-Help
    exit 0
}

if (-not $ProjectPath) {
    Write-Host "Error: Project path is required." -ForegroundColor Red
    Write-Host "Usage: .\dependency-security-test.ps1 -ProjectPath <path>"
    Write-Host "Use -Help for more information."
    exit 1
}

# Resolve path
if ($ProjectPath -eq ".") {
    $ProjectPath = Get-Location
} elseif (-not [System.IO.Path]::IsPathRooted($ProjectPath)) {
    $ProjectPath = Join-Path (Get-Location) $ProjectPath
}

# Validate path
if (-not (Test-Path $ProjectPath -PathType Container)) {
    Write-Host "Error: Project path does not exist: $ProjectPath" -ForegroundColor Red
    exit 1
}

$ProjectPath = (Resolve-Path $ProjectPath).Path

Write-Host "`n========================================" -ForegroundColor Cyan
Write-Host "Dependency Security Testing Script" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Project: $ProjectPath"
Write-Host "Mode: $Mode"
Write-Host "Started: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')"

# Run test phases
$discovered = Invoke-DependencyDiscovery

if ($discovered -and $script:DetectedTypes.Count -gt 0) {
    Invoke-VulnerabilityScan
    Invoke-OutdatedCheck
}

# Show summary
Show-Summary

Write-Host "`nDependency security testing completed." -ForegroundColor Cyan
