#!/bin/bash

# Authentication & Session Security Testing Script
# Comprehensive testing for authentication and session management vulnerabilities
# Covers: CWE-384, CWE-521, CWE-307, CWE-200, CWE-330, CWE-613, CWE-287, CWE-306
# Usage: ./auth-security-test.sh [TARGET_URL] [--quick]

set -e

# Check for help flag first
if [[ "$1" == "--help" || "$1" == "-h" ]]; then
    echo -e "${PURPLE}🔐 AUTHENTICATION & SESSION SECURITY SCANNER ${NC}"
    echo -e "${PURPLE}======================================================${NC}"
    echo -e "${BLUE}Comprehensive security testing for authentication mechanisms${NC}"
    echo -e "${GRAY}Tests session management, password policies, and auth bypass vectors${NC}\n"
    
    echo -e "${YELLOW}USAGE:${NC}"
    echo -e "  $0 <TARGET_URL> [MODE] [CLEAN_OUTPUT]\n"
    
    echo -e "${YELLOW}ARGUMENTS:${NC}"
    echo -e "  TARGET_URL     Target web application URL (required)"
    echo -e "  MODE           Testing mode (optional, default: full)"
    echo -e "  CLEAN_OUTPUT   Output format (optional, default: false)\n"
    
    echo -e "${YELLOW}MODES:${NC}"
    echo -e "  quick    Fast core tests (session, cookies, basic auth checks)"
    echo -e "  full     Complete analysis (all quick tests + brute force, enumeration, timing)\n"
    
    echo -e "${YELLOW}ENVIRONMENT VARIABLES:${NC}"
    echo -e "  LOGIN_ENDPOINT     Login endpoint path (default: /login)"
    echo -e "  LOGOUT_ENDPOINT    Logout endpoint path (default: /logout)"
    echo -e "  USERNAME_FIELD     Username field name (default: username)"
    echo -e "  PASSWORD_FIELD     Password field name (default: password)"
    echo -e "  VALID_USERNAME     Valid username for testing (default: admin)"
    echo -e "  VALID_PASSWORD     Valid password for testing (default: admin123)\n"
    
    echo -e "${YELLOW}EXAMPLES:${NC}"
    echo -e "  $0 http://localhost:3000 full"
    echo -e "  $0 https://example.com quick"
    echo -e "  LOGIN_ENDPOINT=/api/auth $0 http://app.local full\n"
    
    echo -e "${YELLOW}VULNERABILITY COVERAGE:${NC}"
    echo -e "  • Session Fixation (CWE-384)"
    echo -e "  • Weak Password Policy (CWE-521)"
    echo -e "  • Brute Force / No Rate Limiting (CWE-307)"
    echo -e "  • User Enumeration (CWE-200)"
    echo -e "  • Weak Session Tokens (CWE-330)"
    echo -e "  • Missing Session Timeout (CWE-613)"
    echo -e "  • Authentication Bypass (CWE-287)"
    echo -e "  • Missing Cookie Security Flags (CWE-1004)\n"
    
    exit 0
fi

# Configuration
TARGET_URL="${1:-}"
MODE="${2:-full}"
CLEAN_OUTPUT="${3:-false}"

# Configurable endpoints
LOGIN_ENDPOINT="${LOGIN_ENDPOINT:-/login}"
LOGOUT_ENDPOINT="${LOGOUT_ENDPOINT:-/logout}"
REGISTER_ENDPOINT="${REGISTER_ENDPOINT:-/register}"
RESET_ENDPOINT="${RESET_ENDPOINT:-/reset-password}"
USERNAME_FIELD="${USERNAME_FIELD:-username}"
PASSWORD_FIELD="${PASSWORD_FIELD:-password}"
VALID_USERNAME="${VALID_USERNAME:-admin}"
VALID_PASSWORD="${VALID_PASSWORD:-admin123}"

# Colors - disable for clean output
if [ "$CLEAN_OUTPUT" = "true" ]; then
    RED=''
    GREEN=''
    YELLOW=''
    BLUE=''
    PURPLE=''
    CYAN=''
    GRAY=''
    NC=''
else
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    BLUE='\033[0;34m'
    PURPLE='\033[0;35m'
    CYAN='\033[0;36m'
    GRAY='\033[0;90m'
    NC='\033[0m'
fi

# Global curl options — prevent hangs on network issues
CURL_OPTS="--connect-timeout 10 --max-time 30"

# Test tracking
TESTS_PASSED=0
TESTS_FAILED=0
TESTS_TOTAL=0
HIGH_RISK_FINDINGS=0
MEDIUM_RISK_FINDINGS=0
LOW_RISK_FINDINGS=0

# Findings tracking arrays
HIGH_RISK_LIST=()
MEDIUM_RISK_LIST=()
LOW_RISK_LIST=()
INFO_LIST=()

# Validate input
if [ -z "$TARGET_URL" ]; then
    echo -e "${RED}❌ Error: Target URL is required${NC}"
    echo -e "${BLUE}Usage: $0 <TARGET_URL> [mode]${NC}"
    echo -e "${BLUE}Modes: quick, full${NC}"
    echo -e "${BLUE}Example: $0 http://localhost:3000 full${NC}"
    exit 1
fi

# Ensure URL has protocol
if [[ ! "$TARGET_URL" =~ ^https?:// ]]; then
    TARGET_URL="https://$TARGET_URL"
fi

# Remove trailing slash
TARGET_URL="${TARGET_URL%/}"

# Authentication credentials (passed via environment from scanner app)
AUTH_COOKIE="${AUTH_COOKIE:-}"
AUTH_TOKEN="${AUTH_TOKEN:-}"

echo -e "${PURPLE}🔐 AUTHENTICATION & SESSION SECURITY SCANNER ${NC}"
echo -e "${PURPLE}======================================================${NC}"
echo -e "${BLUE}Target: ${TARGET_URL}${NC}"
echo -e "${BLUE}Mode: ${MODE}${NC}"
echo -e "${BLUE}Login Endpoint: ${LOGIN_ENDPOINT}${NC}"
echo -e "${BLUE}Timestamp: $(date)${NC}\n"

if [ -z "$AUTH_COOKIE" ] && [ -z "$AUTH_TOKEN" ]; then
    echo -e "${YELLOW}⚠️ No session cookie provided. Some tests (session fixation, authenticated endpoint checks) will be limited.${NC}"
    echo -e "${YELLOW}   Provide a session cookie via the Authentication panel for more accurate results.${NC}\n"
fi

# Function to get detailed finding explanations
get_finding_details() {
    local finding_type="$1"
    
    case "$finding_type" in
        "SESSION-FIXATION")
            echo "🚨 FOUND: Session Fixation Vulnerability (CWE-384)"
            echo "   RISK: High - Attackers can hijack user sessions"
            echo "   WHAT: Session ID not regenerated after authentication"
            echo "   FIX: Regenerate session ID upon successful login"
            ;;
        "WEAK-PASSWORD")
            echo "🚨 FOUND: Weak Password Policy (CWE-521)"
            echo "   RISK: Medium - Easy to guess/brute force passwords"
            echo "   WHAT: No minimum length, complexity, or common password checks"
            echo "   FIX: Enforce min 8 chars, mixed case, numbers, special chars"
            ;;
        "NO-RATE-LIMIT")
            echo "🚨 FOUND: No Rate Limiting (CWE-307)"
            echo "   RISK: High - Vulnerable to brute force attacks"
            echo "   WHAT: No limit on login attempts"
            echo "   FIX: Implement rate limiting, account lockout, CAPTCHA"
            ;;
        "USER-ENUM")
            echo "🚨 FOUND: User Enumeration (CWE-200)"
            echo "   RISK: Medium - Attackers can discover valid usernames"
            echo "   WHAT: Different responses for valid/invalid usernames"
            echo "   FIX: Use generic error messages for all login failures"
            ;;
        "WEAK-TOKEN")
            echo "🚨 FOUND: Weak Session Token (CWE-330)"
            echo "   RISK: High - Session tokens can be predicted"
            echo "   WHAT: Predictable or low-entropy session identifiers"
            echo "   FIX: Use cryptographically secure random token generation"
            ;;
        "NO-TIMEOUT")
            echo "🚨 FOUND: No Session Timeout (CWE-613)"
            echo "   RISK: Medium - Sessions remain active indefinitely"
            echo "   WHAT: No idle or absolute session timeout"
            echo "   FIX: Implement 15-30 min idle timeout, 8-24hr absolute timeout"
            ;;
        "AUTH-BYPASS")
            echo "🚨 FOUND: Authentication Bypass (CWE-287)"
            echo "   RISK: Critical - Access without valid credentials"
            echo "   WHAT: Authentication can be bypassed"
            echo "   FIX: Enforce authentication on all protected resources"
            ;;
        "COOKIE-SECURE")
            echo "🔒 MISSING: Secure Cookie Flag (CWE-614)"
            echo "   RISK: Medium - Cookie transmitted over HTTP"
            echo "   WHAT: Session cookie missing Secure flag"
            echo "   FIX: Set Secure flag on all session cookies"
            ;;
        "COOKIE-HTTPONLY")
            echo "🔒 MISSING: HttpOnly Cookie Flag (CWE-1004)"
            echo "   RISK: Medium - Cookie accessible via JavaScript"
            echo "   WHAT: Session cookie missing HttpOnly flag"
            echo "   FIX: Set HttpOnly flag to prevent XSS cookie theft"
            ;;
        "COOKIE-SAMESITE")
            echo "🔒 MISSING: SameSite Cookie Attribute"
            echo "   RISK: Medium - Vulnerable to CSRF attacks"
            echo "   WHAT: Session cookie missing SameSite attribute"
            echo "   FIX: Set SameSite=Strict or SameSite=Lax"
            ;;
        "CONCURRENT-SESSIONS")
            echo "⚠️ FOUND: Concurrent Sessions Allowed"
            echo "   RISK: Low - Multiple simultaneous sessions permitted"
            echo "   WHAT: Same user can have multiple active sessions"
            echo "   FIX: Consider limiting concurrent sessions per user"
            ;;
    esac
}

# Function to extract cookies from response
extract_cookies() {
    local response="$1"
    echo "$response" | grep -i "Set-Cookie" | sed 's/Set-Cookie: //i' | tr -d '\r'
}

# Function to extract session ID from cookies
extract_session_id() {
    local cookies="$1"
    # Try common session cookie names
    for name in "JSESSIONID" "PHPSESSID" "ASP.NET_SessionId" "sessionid" "session" "sid" "connect.sid" "token"; do
        local value=$(echo "$cookies" | grep -i "$name" | sed -E "s/.*$name=([^;]+).*/\1/i" | head -1)
        if [ -n "$value" ]; then
            echo "$value"
            return 0
        fi
    done
    # Return any cookie value as fallback
    echo "$cookies" | head -1 | sed -E 's/([^=]+)=([^;]+).*/\2/'
}

# Baseline fingerprinting - detect catch-all responses
get_body_length() {
    echo -n "$1" | wc -c | tr -d ' '
}

capture_auth_baselines() {
    # Fetch root page without auth to establish baseline
    BASELINE_ROOT_BODY=$(curl -s "${TARGET_URL}/" --max-time 10 2>/dev/null || echo "")
    BASELINE_ROOT_LEN=$(get_body_length "$BASELINE_ROOT_BODY")

    # Fetch a random non-existent path
    BASELINE_404_BODY=$(curl -s "${TARGET_URL}/nonexistent_xyzzy_$(date +%s)" --max-time 10 2>/dev/null || echo "")
    BASELINE_404_LEN=$(get_body_length "$BASELINE_404_BODY")
}

is_catch_all_response() {
    local body_len="$1"
    if [ "$BASELINE_ROOT_LEN" -gt 0 ]; then
        local diff=$((body_len - BASELINE_ROOT_LEN))
        diff=${diff#-}  # absolute value
        local threshold=$((BASELINE_ROOT_LEN / 10))
        [ "$threshold" -lt 50 ] && threshold=50
        if [ "$diff" -le "$threshold" ]; then
            return 0
        fi
    fi
    if [ "$BASELINE_404_LEN" -gt 0 ]; then
        local diff=$((body_len - BASELINE_404_LEN))
        diff=${diff#-}  # absolute value
        local threshold=$((BASELINE_404_LEN / 10))
        [ "$threshold" -lt 50 ] && threshold=50
        if [ "$diff" -le "$threshold" ]; then
            return 0
        fi
    fi
    return 1
}

capture_auth_baselines

# PHASE 1: Cookie Security Flags Testing
run_cookie_security_tests() {
    echo -e "\n${YELLOW}🍪 PHASE 1: COOKIE SECURITY FLAGS ANALYSIS${NC}"
    
    # Get cookies from login page or main page
    local response=$(curl -sI -c - $CURL_OPTS "${TARGET_URL}${LOGIN_ENDPOINT}" 2>/dev/null || curl -sI -c - $CURL_OPTS "${TARGET_URL}" 2>/dev/null)
    local cookies=$(echo "$response" | grep -i "Set-Cookie" || echo "")
    
    if [ -z "$cookies" ]; then
        echo -e "${GRAY}ℹ️  No cookies set on initial request, attempting login...${NC}"
        # Try to get cookies via login
        response=$(curl -si -c - $CURL_OPTS -X POST "${TARGET_URL}${LOGIN_ENDPOINT}" \
            -d "${USERNAME_FIELD}=${VALID_USERNAME}&${PASSWORD_FIELD}=${VALID_PASSWORD}" \
            -H "Content-Type: application/x-www-form-urlencoded" 2>/dev/null || echo "")
        cookies=$(echo "$response" | grep -i "Set-Cookie" || echo "")
    fi
    
    if [ -z "$cookies" ]; then
        echo -e "${GRAY}ℹ️  No cookies detected - may use different auth mechanism (JWT, etc.)${NC}"
        INFO_LIST+=("No traditional session cookies detected")
        return
    fi
    
    echo -e "${CYAN}Found cookies:${NC}"
    echo "$cookies" | head -5
    echo ""
    
    # Test Secure flag
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    if echo "$cookies" | grep -qi "Secure"; then
        echo -e "${GREEN}✅ PASS: Secure flag present${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    else
        echo -e "${RED}❌ FAIL: Secure flag missing${NC}"
        get_finding_details "COOKIE-SECURE"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Missing Secure cookie flag")
    fi
    
    # Test HttpOnly flag
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    if echo "$cookies" | grep -qi "HttpOnly"; then
        echo -e "${GREEN}✅ PASS: HttpOnly flag present${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    else
        echo -e "${RED}❌ FAIL: HttpOnly flag missing${NC}"
        get_finding_details "COOKIE-HTTPONLY"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Missing HttpOnly cookie flag")
    fi
    
    # Test SameSite attribute
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    if echo "$cookies" | grep -qi "SameSite"; then
        echo -e "${GREEN}✅ PASS: SameSite attribute present${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    else
        echo -e "${RED}❌ FAIL: SameSite attribute missing${NC}"
        get_finding_details "COOKIE-SAMESITE"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Missing SameSite cookie attribute")
    fi
}

# PHASE 2: Session Fixation Testing
run_session_fixation_tests() {
    echo -e "\n${YELLOW}🔄 PHASE 2: SESSION FIXATION TESTING${NC}"

    # If no AUTH_COOKIE provided, we cannot reliably test session fixation
    if [ -z "$AUTH_COOKIE" ]; then
        echo -e "${GRAY}ℹ️  No session cookie provided - skipping session fixation test${NC}"
        echo -e "${GRAY}   Provide a session cookie via the Authentication panel for accurate session fixation testing${NC}"
        INFO_LIST+=("Session fixation test skipped - provide session cookie for accurate testing")
        return
    fi

    # Step 1: Get a pre-login session ID (unauthenticated request)
    local pre_login_response=$(curl -si -c - $CURL_OPTS "${TARGET_URL}${LOGIN_ENDPOINT}" 2>/dev/null || curl -si -c - $CURL_OPTS "${TARGET_URL}" 2>/dev/null)
    local pre_login_cookies=$(extract_cookies "$pre_login_response")
    local pre_login_session=$(extract_session_id "$pre_login_cookies")

    if [ -z "$pre_login_session" ]; then
        echo -e "${GRAY}ℹ️  No pre-login session ID detected - server may not issue sessions before login${NC}"
        INFO_LIST+=("No pre-login session ID - may be secure by design")
        return
    fi

    echo -e "${CYAN}Pre-login session ID: ${pre_login_session:0:20}...${NC}"

    # Step 2: Extract session ID from the provided AUTH_COOKIE (post-login session)
    local post_login_session=$(echo "$AUTH_COOKIE" | grep -oiE '(JSESSIONID|session_id|sessionid|PHPSESSID|ASP\.NET_SessionId|connect\.sid|sid)=[^;]+' | head -1 | cut -d'=' -f2-)

    if [ -z "$post_login_session" ]; then
        # Try to extract any cookie value that looks like a session token
        post_login_session=$(echo "$AUTH_COOKIE" | tr ';' '\n' | head -1 | cut -d'=' -f2- | xargs)
    fi

    if [ -z "$post_login_session" ]; then
        echo -e "${GRAY}ℹ️  Could not extract session ID from provided cookie${NC}"
        INFO_LIST+=("Session fixation test inconclusive - could not parse session from cookie")
        return
    fi

    echo -e "${CYAN}Post-login session ID (from cookie): ${post_login_session:0:20}...${NC}"

    # Step 3: Compare - if they differ, session was regenerated on login (PASS)
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    if [ "$pre_login_session" = "$post_login_session" ]; then
        echo -e "${RED}❌ FAIL: Session ID not regenerated after login (session fixation)${NC}"
        get_finding_details "SESSION-FIXATION"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Session Fixation Vulnerability")
    else
        echo -e "${GREEN}✅ PASS: Session ID regenerated after login${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# PHASE 3: Session Token Entropy Testing
run_session_token_entropy_tests() {
    echo -e "\n${YELLOW}🎲 PHASE 3: SESSION TOKEN ENTROPY ANALYSIS${NC}"
    
    local tokens=()
    local token_lengths=()
    
    echo -e "${CYAN}Collecting session tokens...${NC}"
    
    # Collect multiple session tokens
    for i in {1..5}; do
        local response=$(curl -si -c - $CURL_OPTS "${TARGET_URL}${LOGIN_ENDPOINT}" 2>/dev/null || curl -si -c - $CURL_OPTS "${TARGET_URL}" 2>/dev/null)
        local cookies=$(extract_cookies "$response")
        local token=$(extract_session_id "$cookies")
        
        if [ -n "$token" ]; then
            tokens+=("$token")
            token_lengths+=(${#token})
            echo -e "${GRAY}  Token $i: ${token:0:30}... (length: ${#token})${NC}"
        fi
        sleep 0.5
    done
    
    if [ ${#tokens[@]} -lt 2 ]; then
        echo -e "${GRAY}ℹ️  Not enough tokens collected for entropy analysis${NC}"
        return
    fi
    
    # Check for patterns/predictability
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Check minimum length (should be at least 128 bits = 16 bytes = 32 hex chars)
    local min_length=${token_lengths[0]}
    for len in "${token_lengths[@]}"; do
        if [ "$len" -lt "$min_length" ]; then
            min_length=$len
        fi
    done
    
    if [ "$min_length" -lt 16 ]; then
        echo -e "${RED}❌ FAIL: Session tokens too short (${min_length} chars)${NC}"
        get_finding_details "WEAK-TOKEN"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Weak Session Token (too short)")
    else
        # Check for sequential/predictable patterns
        local all_unique=true
        for ((i=0; i<${#tokens[@]}; i++)); do
            for ((j=i+1; j<${#tokens[@]}; j++)); do
                if [ "${tokens[$i]}" = "${tokens[$j]}" ]; then
                    all_unique=false
                    break 2
                fi
            done
        done
        
        if [ "$all_unique" = true ]; then
            echo -e "${GREEN}✅ PASS: Session tokens appear random and unique${NC}"
            TESTS_PASSED=$((TESTS_PASSED + 1))
        else
            echo -e "${RED}❌ FAIL: Duplicate session tokens detected${NC}"
            get_finding_details "WEAK-TOKEN"
            echo ""
            TESTS_FAILED=$((TESTS_FAILED + 1))
            HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
            HIGH_RISK_LIST+=("Duplicate Session Tokens")
        fi
    fi
}

# PHASE 4: User Enumeration Testing
run_user_enumeration_tests() {
    echo -e "\n${YELLOW}👤 PHASE 4: USER ENUMERATION TESTING${NC}"
    
    # Test with valid username
    local valid_user_response=$(curl -si $CURL_OPTS -X POST "${TARGET_URL}${LOGIN_ENDPOINT}" \
        -d "${USERNAME_FIELD}=${VALID_USERNAME}&${PASSWORD_FIELD}=wrongpassword123" \
        -H "Content-Type: application/x-www-form-urlencoded" 2>/dev/null || echo "ERROR")
    
    # Test with invalid username
    local invalid_user_response=$(curl -si $CURL_OPTS -X POST "${TARGET_URL}${LOGIN_ENDPOINT}" \
        -d "${USERNAME_FIELD}=nonexistentuser12345&${PASSWORD_FIELD}=wrongpassword123" \
        -H "Content-Type: application/x-www-form-urlencoded" 2>/dev/null || echo "ERROR")
    
    # Extract response bodies (remove headers)
    local valid_body=$(echo "$valid_user_response" | sed -n '/^\r*$/,$p' | tail -n +2)
    local invalid_body=$(echo "$invalid_user_response" | sed -n '/^\r*$/,$p' | tail -n +2)
    
    # Extract status codes
    local valid_status=$(echo "$valid_user_response" | head -1 | grep -oE '[0-9]{3}' | head -1)
    local invalid_status=$(echo "$invalid_user_response" | head -1 | grep -oE '[0-9]{3}' | head -1)
    
    echo -e "${CYAN}Valid username response: HTTP ${valid_status}${NC}"
    echo -e "${CYAN}Invalid username response: HTTP ${invalid_status}${NC}"
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    # Check for user enumeration via different responses
    local enumeration_detected=false
    
    # Different status codes
    if [ "$valid_status" != "$invalid_status" ]; then
        enumeration_detected=true
        echo -e "${GRAY}  Different HTTP status codes detected${NC}"
    fi
    
    # Different response length (significant difference)
    local valid_len=${#valid_body}
    local invalid_len=${#invalid_body}
    local len_diff=$((valid_len - invalid_len))
    if [ ${len_diff#-} -gt 50 ]; then
        enumeration_detected=true
        echo -e "${GRAY}  Significant response length difference: ${valid_len} vs ${invalid_len}${NC}"
    fi
    
    # Check for specific enumeration messages
    if echo "$valid_body" | grep -qiE "invalid password|wrong password|password incorrect|incorrect password"; then
        if ! echo "$invalid_body" | grep -qiE "invalid password|wrong password|password incorrect|incorrect password"; then
            enumeration_detected=true
            echo -e "${GRAY}  Different error messages for valid/invalid usernames${NC}"
        fi
    fi
    
    if echo "$invalid_body" | grep -qiE "user not found|username not found|no such user|invalid username|user does not exist"; then
        enumeration_detected=true
        echo -e "${GRAY}  Specific 'user not found' message detected${NC}"
    fi
    
    if [ "$enumeration_detected" = true ]; then
        echo -e "${RED}❌ FAIL: User enumeration possible${NC}"
        get_finding_details "USER-ENUM"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("User Enumeration Vulnerability")
    else
        echo -e "${GREEN}✅ PASS: No obvious user enumeration detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# PHASE 5: Brute Force / Rate Limiting Testing
run_rate_limiting_tests() {
    echo -e "\n${YELLOW}🚫 PHASE 5: BRUTE FORCE / RATE LIMITING TESTING${NC}"
    
    local attempts=10
    local success_count=0
    local blocked=false
    local response_codes=()
    
    echo -e "${CYAN}Attempting ${attempts} rapid login requests...${NC}"
    
    for i in $(seq 1 $attempts); do
        local response=$(curl -si $CURL_OPTS -X POST "${TARGET_URL}${LOGIN_ENDPOINT}" \
            -d "${USERNAME_FIELD}=${VALID_USERNAME}&${PASSWORD_FIELD}=wrongpassword${i}" \
            -H "Content-Type: application/x-www-form-urlencoded" \
            --max-time 10 2>/dev/null || echo "TIMEOUT")
        
        local status=$(echo "$response" | head -1 | grep -oE '[0-9]{3}' | head -1)
        response_codes+=("$status")
        
        # Check for rate limiting indicators
        if echo "$response" | grep -qiE "rate limit|too many|blocked|try again later|429|captcha"; then
            blocked=true
            echo -e "${GREEN}  Attempt $i: Rate limiting detected (HTTP ${status})${NC}"
            break
        elif [ "$status" = "429" ]; then
            blocked=true
            echo -e "${GREEN}  Attempt $i: HTTP 429 Too Many Requests${NC}"
            break
        else
            echo -e "${GRAY}  Attempt $i: HTTP ${status}${NC}"
            success_count=$((success_count + 1))
        fi
    done
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    if [ "$blocked" = true ]; then
        echo -e "${GREEN}✅ PASS: Rate limiting detected after ${success_count} attempts${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    else
        echo -e "${RED}❌ FAIL: No rate limiting detected after ${attempts} attempts${NC}"
        get_finding_details "NO-RATE-LIMIT"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("No Rate Limiting on Login")
    fi
}

# PHASE 6: Weak Password Policy Testing
run_password_policy_tests() {
    echo -e "\n${YELLOW}🔑 PHASE 6: WEAK PASSWORD POLICY TESTING${NC}"
    
    # Common weak passwords to test
    local weak_passwords=("123456" "password" "admin" "test" "1234" "a" "abc")
    local weak_accepted=false
    local accepted_passwords=()
    
    echo -e "${CYAN}Testing if weak passwords are accepted...${NC}"
    
    # Test on registration endpoint if available
    for pwd in "${weak_passwords[@]}"; do
        local test_username="testuser$(date +%s)"
        local response=$(curl -si $CURL_OPTS -X POST "${TARGET_URL}${REGISTER_ENDPOINT}" \
            -d "${USERNAME_FIELD}=${test_username}&${PASSWORD_FIELD}=${pwd}&email=test$(date +%s)@test.com" \
            -H "Content-Type: application/x-www-form-urlencoded" \
            --max-time 10 2>/dev/null || echo "")

        local status=$(echo "$response" | head -1 | grep -oE '[0-9]{3}' | head -1)

        # Check if registration succeeded or password was accepted
        if [ "$status" = "200" ] || [ "$status" = "201" ] || [ "$status" = "302" ]; then
            if ! echo "$response" | grep -qiE "password.*weak|password.*short|password.*simple|password.*common|password.*required|password.*minimum"; then
                # Verify by attempting to login with the same credentials
                local login_response=$(curl -si $CURL_OPTS -X POST "${TARGET_URL}${LOGIN_ENDPOINT}" \
                    -d "${USERNAME_FIELD}=${test_username}&${PASSWORD_FIELD}=${pwd}" \
                    -H "Content-Type: application/x-www-form-urlencoded" \
                    --max-time 10 2>/dev/null || echo "")
                local login_status=$(echo "$login_response" | head -1 | grep -oE '[0-9]{3}' | head -1)

                # Confirm: must produce a session cookie or redirect to authenticated area
                if echo "$login_response" | grep -qiE "Set-Cookie:.*session|Set-Cookie:.*token|Set-Cookie:.*sid" || \
                   ([ "$login_status" = "302" ] && echo "$login_response" | grep -qiE "location:.*(dashboard|home|profile|welcome)"); then
                    weak_accepted=true
                    accepted_passwords+=("$pwd")
                    echo -e "${RED}  Weak password '${pwd}' CONFIRMED accepted (login verified)${NC}"
                else
                    echo -e "${GRAY}  Weak password '${pwd}' registration returned ${status} but login not confirmed (likely false positive)${NC}"
                fi
            fi
        fi
    done
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    if [ "$weak_accepted" = true ]; then
        echo -e "${RED}❌ FAIL: Weak password policy detected${NC}"
        echo -e "${GRAY}  Accepted weak passwords: ${accepted_passwords[*]}${NC}"
        get_finding_details "WEAK-PASSWORD"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("Weak Password Policy")
    else
        echo -e "${GREEN}✅ PASS: Password policy appears enforced (or registration unavailable)${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# PHASE 7: Authentication Bypass Testing
run_auth_bypass_tests() {
    echo -e "\n${YELLOW}🚪 PHASE 7: AUTHENTICATION BYPASS TESTING${NC}"
    
    # Common protected endpoints to test
    local protected_endpoints=("/admin" "/dashboard" "/api/users" "/api/admin" "/profile" "/settings" "/account" "/user" "/panel" "/management")
    local bypass_found=false
    local bypassed_endpoints=()
    
    echo -e "${CYAN}Testing access to protected endpoints without authentication...${NC}"
    
    for endpoint in "${protected_endpoints[@]}"; do
        local response=$(curl -si $CURL_OPTS "${TARGET_URL}${endpoint}" 2>/dev/null || echo "")
        local status=$(echo "$response" | head -1 | grep -oE '[0-9]{3}' | head -1)

        # Check if endpoint is accessible (not 401, 403, or redirect to login)
        if [ "$status" = "200" ]; then
            # Verify it's not a login page or error page
            if ! echo "$response" | grep -qiE "login|sign in|authenticate|unauthorized|forbidden|access denied"; then
                # Baseline check: compare body length to root/404
                local body=$(echo "$response" | sed -n '/^\r*$/,$p' | tail -n +2)
                local body_len=$(get_body_length "$body")
                if is_catch_all_response "$body_len"; then
                    echo -e "${GRAY}  ${endpoint}: HTTP ${status} - Same as root (likely catch-all)${NC}"
                    continue
                fi
                bypass_found=true
                bypassed_endpoints+=("$endpoint")
                echo -e "${RED}  ${endpoint}: HTTP ${status} - Accessible without auth!${NC}"
            else
                echo -e "${GRAY}  ${endpoint}: HTTP ${status} - Login page/redirect${NC}"
            fi
        else
            echo -e "${GRAY}  ${endpoint}: HTTP ${status}${NC}"
        fi
    done
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    if [ "$bypass_found" = true ]; then
        echo -e "${RED}❌ FAIL: Authentication bypass detected${NC}"
        echo -e "${GRAY}  Bypassed endpoints: ${bypassed_endpoints[*]}${NC}"
        get_finding_details "AUTH-BYPASS"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("Authentication Bypass")
    else
        echo -e "${GREEN}✅ PASS: No obvious authentication bypass detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
    
    # Test SQL injection in login
    echo -e "\n${CYAN}Testing SQL injection in authentication...${NC}"
    
    local sqli_payloads=("admin'--" "' OR '1'='1" "admin' OR '1'='1'--" "' OR 1=1--" "admin'/*")
    local sqli_bypass=false
    
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    for payload in "${sqli_payloads[@]}"; do
        local response=$(curl -si $CURL_OPTS -X POST "${TARGET_URL}${LOGIN_ENDPOINT}" \
            -d "${USERNAME_FIELD}=$(printf '%s' "$payload" | jq -sRr @uri)&${PASSWORD_FIELD}=anything" \
            -H "Content-Type: application/x-www-form-urlencoded" \
            --max-time 10 2>/dev/null || echo "")
        
        local status=$(echo "$response" | head -1 | grep -oE '[0-9]{3}' | head -1)
        
        # Check for successful login (redirect or success indicators)
        if [ "$status" = "302" ] || [ "$status" = "303" ]; then
            if echo "$response" | grep -qiE "location:.*(dashboard|admin|home|profile|welcome)"; then
                sqli_bypass=true
                echo -e "${RED}  SQL injection payload succeeded: ${payload}${NC}"
                break
            fi
        fi
        
        if echo "$response" | grep -qiE "welcome|logged in|success|dashboard"; then
            sqli_bypass=true
            echo -e "${RED}  SQL injection payload succeeded: ${payload}${NC}"
            break
        fi
    done
    
    if [ "$sqli_bypass" = true ]; then
        echo -e "${RED}❌ FAIL: SQL injection authentication bypass possible${NC}"
        TESTS_FAILED=$((TESTS_FAILED + 1))
        HIGH_RISK_FINDINGS=$((HIGH_RISK_FINDINGS + 1))
        HIGH_RISK_LIST+=("SQL Injection Auth Bypass")
    else
        echo -e "${GREEN}✅ PASS: No SQL injection auth bypass detected${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    fi
}

# PHASE 8: Session Timeout Testing
run_session_timeout_tests() {
    echo -e "\n${YELLOW}⏰ PHASE 8: SESSION TIMEOUT ANALYSIS${NC}"
    
    echo -e "${CYAN}Analyzing session configuration...${NC}"
    
    # Get initial session
    local login_response=$(curl -si -c - $CURL_OPTS -X POST "${TARGET_URL}${LOGIN_ENDPOINT}" \
        -d "${USERNAME_FIELD}=${VALID_USERNAME}&${PASSWORD_FIELD}=${VALID_PASSWORD}" \
        -H "Content-Type: application/x-www-form-urlencoded" 2>/dev/null || echo "")
    
    local cookies=$(extract_cookies "$login_response")
    
    if [ -z "$cookies" ]; then
        echo -e "${GRAY}ℹ️  Could not establish session for timeout testing${NC}"
        return
    fi
    
    # Check for session cookie max-age or expires
    TESTS_TOTAL=$((TESTS_TOTAL + 1))
    
    local has_expiry=false
    
    if echo "$cookies" | grep -qiE "max-age|expires"; then
        has_expiry=true
        local max_age=$(echo "$cookies" | grep -oiE "max-age=[0-9]+" | head -1 | cut -d= -f2)
        local expires=$(echo "$cookies" | grep -oiE "expires=[^;]+" | head -1 | cut -d= -f2-)
        
        if [ -n "$max_age" ]; then
            local hours=$((max_age / 3600))
            echo -e "${CYAN}  Session max-age: ${max_age}s (~${hours} hours)${NC}"
            
            if [ "$max_age" -gt 86400 ]; then # More than 24 hours
                echo -e "${YELLOW}⚠️ WARNING: Long session timeout (${hours} hours)${NC}"
                MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
                MEDIUM_RISK_LIST+=("Long session timeout (${hours}h)")
            fi
        fi
        
        if [ -n "$expires" ]; then
            echo -e "${CYAN}  Session expires: ${expires}${NC}"
        fi
    fi
    
    if [ "$has_expiry" = true ]; then
        echo -e "${GREEN}✅ PASS: Session has expiration configured${NC}"
        TESTS_PASSED=$((TESTS_PASSED + 1))
    else
        echo -e "${RED}❌ FAIL: No session expiration detected${NC}"
        get_finding_details "NO-TIMEOUT"
        echo ""
        TESTS_FAILED=$((TESTS_FAILED + 1))
        MEDIUM_RISK_FINDINGS=$((MEDIUM_RISK_FINDINGS + 1))
        MEDIUM_RISK_LIST+=("No Session Timeout")
    fi
}

# Generate summary report
generate_summary() {
    echo -e "\n${PURPLE}======================================================${NC}"
    echo -e "${PURPLE}📊 AUTHENTICATION SECURITY SCAN SUMMARY${NC}"
    echo -e "${PURPLE}======================================================${NC}"
    
    echo -e "\n${BLUE}Target: ${TARGET_URL}${NC}"
    echo -e "${BLUE}Scan completed: $(date)${NC}\n"
    
    # Test statistics
    local pass_rate=0
    if [ $TESTS_TOTAL -gt 0 ]; then
        pass_rate=$((TESTS_PASSED * 100 / TESTS_TOTAL))
    fi
    
    echo -e "${CYAN}Test Results:${NC}"
    echo -e "  Total Tests: ${TESTS_TOTAL}"
    echo -e "  ${GREEN}Passed: ${TESTS_PASSED}${NC}"
    echo -e "  ${RED}Failed: ${TESTS_FAILED}${NC}"
    echo -e "  Pass Rate: ${pass_rate}%\n"
    
    # Risk summary
    echo -e "${CYAN}Risk Summary:${NC}"
    echo -e "  ${RED}High Risk: ${HIGH_RISK_FINDINGS}${NC}"
    echo -e "  ${YELLOW}Medium Risk: ${MEDIUM_RISK_FINDINGS}${NC}"
    echo -e "  ${BLUE}Low Risk: ${LOW_RISK_FINDINGS}${NC}\n"
    
    # List findings
    if [ ${#HIGH_RISK_LIST[@]} -gt 0 ]; then
        echo -e "${RED}🚨 HIGH RISK FINDINGS:${NC}"
        for finding in "${HIGH_RISK_LIST[@]}"; do
            echo -e "  ${RED}• ${finding}${NC}"
        done
        echo ""
    fi
    
    if [ ${#MEDIUM_RISK_LIST[@]} -gt 0 ]; then
        echo -e "${YELLOW}⚠️ MEDIUM RISK FINDINGS:${NC}"
        for finding in "${MEDIUM_RISK_LIST[@]}"; do
            echo -e "  ${YELLOW}• ${finding}${NC}"
        done
        echo ""
    fi
    
    if [ ${#LOW_RISK_LIST[@]} -gt 0 ]; then
        echo -e "${BLUE}ℹ️ LOW RISK FINDINGS:${NC}"
        for finding in "${LOW_RISK_LIST[@]}"; do
            echo -e "  ${BLUE}• ${finding}${NC}"
        done
        echo ""
    fi
    
    # Overall security score
    local score=100
    score=$((score - HIGH_RISK_FINDINGS * 20))
    score=$((score - MEDIUM_RISK_FINDINGS * 10))
    score=$((score - LOW_RISK_FINDINGS * 5))
    if [ $score -lt 0 ]; then score=0; fi
    
    echo -e "${CYAN}Overall Security Score: ${NC}"
    if [ $score -ge 80 ]; then
        echo -e "  ${GREEN}${score}/100 - GOOD${NC}"
    elif [ $score -ge 60 ]; then
        echo -e "  ${YELLOW}${score}/100 - NEEDS IMPROVEMENT${NC}"
    else
        echo -e "  ${RED}${score}/100 - CRITICAL${NC}"
    fi
    
    echo -e "\n${PURPLE}======================================================${NC}"
}

# Main execution
main() {
    # Run all phases based on mode
    run_cookie_security_tests
    run_session_fixation_tests
    run_session_token_entropy_tests
    run_user_enumeration_tests
    
    if [ "$MODE" = "full" ]; then
        run_rate_limiting_tests
        run_password_policy_tests
        run_auth_bypass_tests
        run_session_timeout_tests
    fi
    
    generate_summary
}

# Execute main
main
